/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.service.cache;

import java.util.*;
import java.lang.reflect.Array;

import jp.ossc.nimbus.core.*;

/**
 * ۃLbV}bvT[rXB<p>
 *
 * @author M.Takata
 */
public abstract class AbstractCacheMapService<K, V> extends ServiceBase
 implements CacheRemoveListener<V>, AbstractCacheMapServiceMBean<K, V>{
    
    private static final long serialVersionUID = 6779186037980520151L;
    
    /**
     * LbṼL[ƃLbVQƂ̃}bvB<p>
     */
    protected Map<K, KeyCachedReference<K, V>> references;
    
    /**
     * OverflowControllerT[rX̃T[rXzB<p>
     */
    protected ServiceName[] overflowControllerServiceNames;
    
    /**
     * OverflowControllerT[rX̃XgB<p>
     */
    protected List<OverflowController<V>> overflowControllers;
    
    /**
     * T[rX~ɁALbVNA邩ǂ̃tOB<p>
     * ftHǵAfalseB<br>
     */
    protected boolean isClearOnStop;
    
    /**
     * T[rXjɁALbVNA邩ǂ̃tOB<p>
     * ftHǵAtrueB<br>
     */
    protected boolean isClearOnDestroy = true;
    
    // AbstractCacheMapServiceMBeanJavaDoc
    @Override
    public void setOverflowControllerServiceNames(ServiceName[] names){
        overflowControllerServiceNames = names;
    }
    // AbstractCacheMapServiceMBeanJavaDoc
    @Override
    public ServiceName[] getOverflowControllerServiceNames(){
        return overflowControllerServiceNames;
    }
    
    // AbstractCacheMapServiceMBeanJavaDoc
    @Override
    public void setClearOnStop(boolean isClear){
        isClearOnStop = isClear;
    }
    // AbstractCacheMapServiceMBeanJavaDoc
    @Override
    public boolean isClearOnStop(){
        return isClearOnStop;
    }
    
    // AbstractCacheMapServiceMBeanJavaDoc
    @Override
    public void setClearOnDestroy(boolean isClear){
        isClearOnDestroy = isClear;
    }
    // AbstractCacheMapServiceMBeanJavaDoc
    @Override
    public boolean isClearOnDestroy(){
        return isClearOnDestroy;
    }
    
    /**
     * OverflowControllerݒ肷B
     */
    public void setOverflowControllers(List<OverflowController<V>> overflowControllers) {
		this.overflowControllers = overflowControllers;
	}
    
	/**
     * T[rX̐OsB<p>
     * CX^Xϐ̏sB<br>
     *
     * @exception OɎsꍇ
     */
    @Override
    public void preCreateService() throws Exception{
        super.preCreateService();
        references = Collections.synchronizedMap(new HashMap<K, KeyCachedReference<K, V>>());
        overflowControllers = new ArrayList<OverflowController<V>>();
    }
    
    /**
     * T[rX̊JnOsB<p>
     * QverflowControllerT[rX̎擾sB<br>
     *
     * @exception JnOɎsꍇ
     */
    @Override
    public void preStartService() throws Exception{
        super.preStartService();
        if(overflowControllerServiceNames != null){
            for(int i = 0; i < overflowControllerServiceNames.length; i++){
                overflowControllers.add(
                    ServiceManagerFactory
                        .<OverflowController<V>>getServiceObject(overflowControllerServiceNames[i])
                );
            }
        }
    }
    
    /**
     * T[rX̒~㏈sB<p>
     * QverflowControllerT[rX̎QƂjB<br>
     *
     * @exception ~㏈Ɏsꍇ
     */
    @Override
    public void postStopService() throws Exception{
        if(isClearOnStop()){
            clear();
        }
        if(overflowControllers != null){
            overflowControllers.clear();
        }
        super.postStopService();
    }
    
    /**
     * T[rX̔j㏈sB<p>
     * CX^Xϐ̎QƂjB<br>
     *
     * @exception j㏈Ɏsꍇ
     */
    @Override
    public void postDestroyService() throws Exception{
        if(isClearOnDestroy()){
            clear();
        }
        references = null;
        overflowControllers = null;
        super.postDestroyService();
    }
    
    // CacheMapJavaDoc
    @Override
    public KeyCachedReference<K, V> getCachedReference(Object key){
        if(references == null){
            return null;
        }
        return references.get(key);
    }
    
    // CacheMapJavaDoc
    @Override
    public int size(){
        if(references == null){
            return 0;
        }
        return references.size();
    }
    
    // CacheMapJavaDoc
    @Override
    public boolean isEmpty(){
        if(references == null){
            return true;
        }
        return references.isEmpty();
    }
    
    // CacheMapJavaDoc
    @Override
    public boolean containsKey(Object key){
        if(references == null){
            return false;
        }
        return references.containsKey(key);
    }
    
    // CacheMapJavaDoc
    @Override
    public boolean containsValue(Object value){
        if(references == null){
            return false;
        }
        synchronized(references){
            for(K key : references.keySet()){
                final Object obj = get(key, false);
                if(value == null){
                    if(obj == null){
                        return true;
                    }
                }else if(value.equals(obj)){
                    return true;
                }
            }
        }
        return false;
    }
    
    // CacheMapJavaDoc
    @Override
    public V get(Object key){
        return get(key, true);
    }
    
    /**
     * w肳ꂽL[̃LbV擾B<p>
     *
     * @param key LbṼL[
     * @param notify LbVɃANZXXiɒʒm邩ǂ̃tOBʒmꍇAtrue
     */
    protected V get(Object key, boolean notify){
        if(references == null){
            return null;
        }
        synchronized(references){
            final CachedReference<V> ref = references.get(key);
            if(ref == null){
                return null;
            }
            return ref.get(this, notify);
        }
    }
    
    // CacheMapJavaDoc
    @Override
    public V put(K key, V value){
        if(references == null){
            return null;
        }
        synchronized(references){
            V oldVal = null;
            if(references.containsKey(key)){
                oldVal = remove(key);
            }
            final KeyCachedReference<K, V> ref = createKeyCachedReference(key, value);
            put(key, ref);
            return oldVal;
        }
    }
    
    /**
     * w肵L[̃LbVQƂǉB<p>
     *
     * @param key LbṼL[
     * @param ref LbVQ
     */
    protected void put(K key, KeyCachedReference<K, V> ref){
        if(references == null || getState().compareTo(State.STOPPED) > 0 || ref == null){
            return;
        }
        synchronized(references){
            ref.addCacheRemoveListener(this);
            references.put(key, ref);
            if(overflowControllers.size() != 0){
                for(OverflowController<V> controller : overflowControllers){
                    controller.control(ref);
                }
            }
        }
    }
    
    /**
     * L[tLbVQƂ𐶐B<p>
     *
     * @param key LbṼL[
     * @param obj LbVIuWFNg
     * @return L[tLbVQ
     */
    protected abstract KeyCachedReference<K, V> createKeyCachedReference(
        K key,
        V obj
    );
    
    // CacheMapJavaDoc
    @Override
    public V remove(Object key){
        if(references == null){
            return null;
        }
        V val = null;
        synchronized(references){
            final CachedReference<V> ref = references.remove(key);
            if(ref != null){
                val = ref.get(this, false);
                ref.remove(this);
            }
        }
        return val;
    }
    
    // CacheMapJavaDoc
    @Override
    public void putAll(Map<? extends K,? extends V> map){
        if(references == null || map == null || map.size() == 0){
            return;
        }
        for(Map.Entry<? extends K,? extends V> entry : map.entrySet()){
            put(entry.getKey(), entry.getValue());
        }
    }
    
    // CacheMapJavaDoc
    @Override
    public void clear(){
        if(references == null || references.size() == 0){
            return;
        }
        synchronized(references){
            final Object[] keys = references.keySet().toArray();
            for(int i = 0; i < keys.length; i++){
                remove(keys[i]);
            }
        }
    }
    
    // CacheMapJavaDoc
    @Override
    public Set<K> keySet(){
        return new KeySet<K>();
    }
    
    // CacheMapJavaDoc
    @Override
    public Collection<V> values(){
        return new ValuesCollection<V>();
    }
    
    // CacheMapJavaDoc
    @Override
    public Set<Map.Entry<K, V>> entrySet(){
        return new EntrySet<K, V>();
    }
    
    // CacheRemoveListenerJavaDoc
    @SuppressWarnings("unchecked")
    @Override
    public void removed(CachedReference<V> ref){
        if(references != null && ref instanceof KeyCachedReference){
            final KeyCachedReference<K, V> keyRef = (KeyCachedReference<K, V>)ref;
            references.remove(keyRef.getKey());
        }
    }
    
    /**
     * LbV}bṽL[WB<p>
     *
     * @author M.Takata
     * @see AbstractCacheMapService#keySet()
     */
    protected class KeySet<K2>
     implements Set<K2>, java.io.Serializable{
        
        private static final long serialVersionUID = 8251697022788153149L;
        
        private Collection<K> collection;
        
        /**
         * CX^X𐶐B<p>
         */
        public KeySet(){
            if(references != null){
                collection = references.keySet();
            }
        }
        
        // SetJavaDoc
        @Override
        public int size(){
            return AbstractCacheMapService.this.size();
        }
        
        // SetJavaDoc
        @Override
        public boolean isEmpty(){
            return AbstractCacheMapService.this.isEmpty();
        }
        
        // SetJavaDoc
        @Override
        public boolean contains(Object o){
            return AbstractCacheMapService.this.containsKey(o);
        }
        
        // SetJavaDoc
        @Override
        public boolean containsAll(Collection<?> c){
            boolean result = true;
            for(Object val : c){
                result &= AbstractCacheMapService.this.containsKey(val);
                if(!result){
                    break;
                }
            }
            return result;
        }
        
        // SetJavaDoc
        @Override
        public Iterator<K2> iterator(){
            return new KeyIterator<K2>();
        }
        
        // SetJavaDoc
        @Override
        public Object[] toArray(){
            if(references == null || references.size() == 0){
                return new Object[0];
            }
            final Object[] result = new Object[references.size()];
            synchronized(references){
                int count = 0;
                final Iterator<K> keys = references.keySet().iterator();
                while(keys.hasNext()){
                    result[count++] = keys.next();
                }
            }
            return result;
        }
        
        // SetJavaDoc
        @SuppressWarnings("unchecked")
        @Override
        public <T> T[] toArray(T[] a){
            if(references == null || references.size() == 0){
                if(a.length == 0){
                    return a;
                }else{
                    for(int i = 0; i < a.length; i++){
                        a[i] = null;
                    }
                    return a;
                }
            }
            T[] result = null;
            synchronized(references){
                final int length = references.size();
                if(a.length >= length){
                    result = a;
                }else{
                    result = (T[])Array.newInstance(
                        a.getClass().getComponentType(),
                        length
                    );
                }
                int count = 0;
                final Iterator<K> keys = references.keySet().iterator();
                while(keys.hasNext()){
                    result[count++] = (T)keys.next();
                }
            }
            return result;
        }
        
        // SetJavaDoc
        @Override
        public boolean add(K2 o){
            throw new UnsupportedOperationException();
        }
        
        // SetJavaDoc
        @Override
        public boolean addAll(Collection<? extends K2> c){
            throw new UnsupportedOperationException();
        }
        
        // SetJavaDoc
        @Override
        public boolean remove(Object o){
            if(references == null || references.size() == 0){
                return false;
            }
            final Object removed = AbstractCacheMapService.this.remove(o);
            return removed != null;
        }
        
        // SetJavaDoc
        @Override
        public boolean removeAll(Collection<?> c){
            boolean result = false;
            for(Object val : c){
                result |= remove(val);
            }
            return result;
        }
        
        // SetJavaDoc
        @Override
        public boolean retainAll(Collection<?> c){
            if(references == null || references.size() == 0){
                return false;
            }
            boolean result = false;
            synchronized(references){
                final Object[] keys = references.keySet().toArray();
                for(int i = 0; i < keys.length; i++){
                    if(!c.contains(keys[i])){
                        result = remove(keys[i]);
                    }
                }
            }
            return result;
        }
        
        // SetJavaDoc
        @Override
        public void clear(){
            AbstractCacheMapService.this.clear();
        }
        
        // SetJavaDoc
        @SuppressWarnings("unchecked")
        @Override
        public boolean equals(Object o){
            if(o == null){
                return false;
            }
            if(o == this){
                return true;
            }
            if(o instanceof KeySet){
                final KeySet<K2> cmp = (KeySet<K2>)o;
                if(collection == null){
                    return cmp.collection == null;
                }else if(collection.equals(cmp.collection)){
                    return true;
                }
            }
            return false;
        }
        
        // SetJavaDoc
        @Override
        public int hashCode(){
            return collection == null ? 0 : collection.hashCode();
        }
        
        @Override
        public String toString(){
            return collection == null
                 ? super.toString() : collection.toString();
        }
    }
    
    /**
     * LbV}bṽL[JԂB<p>
     *
     * @author M.Takata
     * @see AbstractCacheMapService.KeySet#iterator()
     */
    protected class KeyIterator<K2>
     implements Iterator<K2>, java.io.Serializable{
        
        private static final long serialVersionUID = 8251697022788153149L;
        
        private Iterator<K> iterator;
        private K current;
        
        /**
         * CX^X𐶐B<p>
         */
        public KeyIterator(){
            if(references != null){
                iterator = new HashSet<K>(references.keySet()).iterator();
            }
        }
        
        // IteratorJavaDoc
        @Override
        public boolean hasNext(){
            return iterator == null ? false : iterator.hasNext();
        }
        
        // IteratorJavaDoc
        @SuppressWarnings("unchecked")
        @Override
        public K2 next(){
            if(iterator == null){
                throw new NoSuchElementException();
            }
            current = null;
            current = iterator.next();
            return (K2)current;
        }
        
        // IteratorJavaDoc
        @Override
        public void remove(){
            if(current == null){
                throw new IllegalStateException();
            }
            AbstractCacheMapService.this.remove(current);
            iterator.remove();
            if(!iterator.hasNext()){
                current = null;
            }
        }
        
        @Override
        public String toString(){
            return iterator == null
                 ? super.toString() : iterator.toString();
        }
    }
    
    /**
     * LbV}bv̒lWB<p>
     *
     * @author M.Takata
     * @see AbstractCacheMapService#values()
     */
    protected class ValuesCollection<V2>
     implements Collection<V2>, java.io.Serializable{
        
        private static final long serialVersionUID = -3991603731459237593L;
        
        private Collection<KeyCachedReference<K, V>> collection;
        
        /**
         * CX^X𐶐B<p>
         */
        public ValuesCollection(){
            if(references != null){
                collection = references.values();
            }
        }
        
        // CollectionJavaDoc
        @Override
        public int size(){
            return AbstractCacheMapService.this.size();
        }
        
        // CollectionJavaDoc
        @Override
        public boolean isEmpty(){
            return AbstractCacheMapService.this.isEmpty();
        }
        
        // CollectionJavaDoc
        @Override
        public boolean contains(Object o){
            return AbstractCacheMapService.this.containsValue(o);
        }
        
        // CollectionJavaDoc
        @Override
        public boolean containsAll(Collection<?> c){
            boolean result = true;
            for(Object obj : c){
                result &= AbstractCacheMapService.this.containsValue(
                    obj
                );
                if(!result){
                    break;
                }
            }
            return result;
        }
        
        // CollectionJavaDoc
        @Override
        public Iterator<V2> iterator(){
            return new ValuesIterator<V2>();
        }
        
        // CollectionJavaDoc
        @Override
        public Object[] toArray(){
            if(references == null || references.size() == 0){
                return new Object[0];
            }
            final Object[] result = new Object[references.size()];
            synchronized(references){
                int count = 0;
                for(CachedReference<V> ref : references.values()){
                    if(ref != null){
                        result[count++] = ref.get(this);
                    }else{
                        result[count++] = null;
                    }
                }
            }
            return result;
        }
        
        // CollectionJavaDoc
        @SuppressWarnings("unchecked")
        @Override
        public <T> T[] toArray(T[] a){
            if(references == null || references.size() == 0){
                if(a.length == 0){
                    return a;
                }else{
                    for(int i = 0; i < a.length; i++){
                        a[i] = null;
                    }
                    return a;
                }
            }
            T[] result = null;
            synchronized(references){
                final int length = references.size();
                if(a.length >= length){
                    result = a;
                }else{
                    result = (T[])Array.newInstance(
                        a.getClass().getComponentType(),
                        length
                    );
                }
                int count = 0;
                for(CachedReference<V> ref : references.values()){
                    if(ref != null){
                        result[count++] = (T)ref.get(this);
                    }else{
                        result[count++] = null;
                    }
                }
            }
            return result;
        }
        
        // CollectionJavaDoc
        @Override
        public boolean add(V2 o){
            throw new UnsupportedOperationException();
        }
        
        // CollectionJavaDoc
        @Override
        public boolean addAll(Collection<? extends V2> c){
            throw new UnsupportedOperationException();
        }
        
        // CollectionJavaDoc
        @Override
        public boolean remove(Object o){
            if(references == null || references.size() == 0){
                return false;
            }
            boolean result = false;
            synchronized(references){
                for(KeyCachedReference<K, V>ref : references.values()){
                    V val = null;
                    if(ref != null){
                        val = ref.get(this, false);
                    }
                    Object removed = null;
                    if(val == null){
                        if(o == null){
                            if(ref != null){
                                removed = AbstractCacheMapService.this
                                    .remove(ref.getKey());
                            }
                        }
                    }else if(val.equals(o)){
                        if(ref != null){
                            removed = AbstractCacheMapService.this
                                .remove(ref.getKey());
                        }
                    }
                    if(removed != null){
                        result = true;
                    }
                }
            }
            return result;
        }
        
        // CollectionJavaDoc
        @Override
        public boolean removeAll(Collection<?> c){
            boolean result = false;
            for(Object val : c){
                result |= remove(val);
            }
            return result;
        }
        
        // CollectionJavaDoc
        @Override
        public boolean retainAll(Collection<?> c){
            if(references == null || references.size() == 0){
                return false;
            }
            boolean result = false;
            synchronized(references){
                for(KeyCachedReference<K, V>ref : references.values()){
                    V val = null;
                    if(ref != null){
                        val = ref.get(this, false);
                    }
                    if(!c.contains(val)){
                        if(ref != null){
                            if(AbstractCacheMapService.this
                                .remove(ref.getKey()) != null){
                                result = true;
                            }
                        }
                    }
                }
            }
            return result;
        }
        
        // CollectionJavaDoc
        @Override
        public void clear(){
            AbstractCacheMapService.this.clear();
        }
        
        // CollectionJavaDoc
        @SuppressWarnings("unchecked")
        @Override
        public boolean equals(Object o){
            if(o == null){
                return false;
            }
            if(o == this){
                return true;
            }
            if(o instanceof ValuesCollection){
                final ValuesCollection<V2> cmp = (ValuesCollection<V2>)o;
                if(collection == null){
                    return cmp.collection == null;
                }else if(collection.equals(cmp.collection)){
                    return true;
                }
            }
            return false;
        }
        
        // CollectionJavaDoc
        @Override
        public int hashCode(){
            return collection == null ? 0 : collection.hashCode();
        }
        
        @Override
        public String toString(){
            return collection == null
                 ? super.toString() : collection.toString();
        }
    }
    
    /**
     * LbV}bv̒lJԂB<p>
     *
     * @author M.Takata
     * @see AbstractCacheMapService.ValuesCollection#iterator()
     */
    protected class ValuesIterator<V2>
     implements Iterator<V2>, java.io.Serializable{
        
        private static final long serialVersionUID = 1885683078757668887L;
        
        private Iterator<KeyCachedReference<K, V>> iterator;
        private KeyCachedReference<K, V> current;
        
        /**
         * CX^X𐶐B<p>
         */
        public ValuesIterator(){
            if(references != null){
                iterator = new HashSet<KeyCachedReference<K, V>>(references.values()).iterator();
            }
        }
        
        // IteratorJavaDoc
        @Override
        public boolean hasNext(){
            return iterator == null ? false : iterator.hasNext();
        }
        
        // IteratorJavaDoc
        @SuppressWarnings("unchecked")
        @Override
        public V2 next(){
            if(iterator == null){
                throw new NoSuchElementException();
            }
            current = null;
            current = iterator.next();
            return (V2)current.get(this);
        }
        
        // IteratorJavaDoc
        @Override
        public void remove(){
            if(current == null){
                throw new IllegalStateException();
            }
            AbstractCacheMapService.this.remove(current.getKey());
            iterator.remove();
            if(!iterator.hasNext()){
                current = null;
            }
        }
        
        @Override
        public String toString(){
            return iterator == null
                 ? super.toString() : iterator.toString();
        }
    }
    
    /**
     * LbV}bṽ}bvGgWB<p>
     *
     * @author M.Takata
     * @see AbstractCacheMapService#entrySet()
     */
    protected class EntrySet<K2, V2>
     implements Set<Map.Entry<K2, V2>>, java.io.Serializable{
        
        private static final long serialVersionUID = -6274837740283895811L;
        
        private Collection<K2> collection;
        
        /**
         * CX^X𐶐B<p>
         */
        @SuppressWarnings("unchecked")
        public EntrySet(){
            if(references != null){
                collection = (Collection<K2>)references.keySet();
            }
        }
        
        // SetJavaDoc
        @Override
        public int size(){
            return AbstractCacheMapService.this.size();
        }
        
        // SetJavaDoc
        @Override
        public boolean isEmpty(){
            return AbstractCacheMapService.this.isEmpty();
        }
        
        // SetJavaDoc
        @SuppressWarnings("unchecked")
        @Override
        public boolean contains(Object o){
            if(o instanceof Map.Entry){
                final Map.Entry<K2, V2> entry = (Map.Entry<K2, V2>)o;
                return AbstractCacheMapService.this.containsKey(entry.getKey());
            }
            return false;
        }
        
        // SetJavaDoc
        @Override
        public boolean containsAll(Collection<?> c){
            boolean result = true;
            for(Object val : c){
                result &= contains(val);
                if(!result){
                    break;
                }
            }
            return result;
        }
        
        // SetJavaDoc
        @Override
        public Iterator<Map.Entry<K2, V2>> iterator(){
            return new EntryIterator<K2, V2>();
        }
        
        // SetJavaDoc
        @SuppressWarnings("unchecked")
        @Override
        public Object[] toArray(){
            if(references == null || references.size() == 0){
                return new Object[0];
            }
            final Entry<K2, V2>[] result = new Entry[references.size()];
            synchronized(references){
                int count = 0;
                final Iterator<K> keys = references.keySet().iterator();
                while(keys.hasNext()){
                    result[count++] = new Entry<K2, V2>((K2)keys.next());
                }
            }
            return result;
        }
        
        // SetJavaDoc
        @SuppressWarnings("unchecked")
        @Override
        public <T> T[] toArray(T[] a){
            if(references == null || references.size() == 0){
                if(a.length == 0){
                    return a;
                }else{
                    for(int i = 0; i < a.length; i++){
                        a[i] = null;
                    }
                    return a;
                }
            }
            T[] result = null;
            synchronized(references){
                final int length = references.size();
                if(a.length >= length){
                    result = a;
                }else{
                    result = (T[])Array.newInstance(
                        a.getClass().getComponentType(),
                        length
                    );
                }
                int count = 0;
                final Iterator<K> keys = references.keySet().iterator();
                while(keys.hasNext()){
                    result[count++] = (T)new Entry<K,V>(keys.next());
                }
            }
            return result;
        }
        
        // SetJavaDoc
        @Override
        public boolean add(Map.Entry<K2, V2> o){
            throw new UnsupportedOperationException();
        }
        
        // SetJavaDoc
        @Override
        public boolean addAll(Collection<? extends Map.Entry<K2, V2>> c){
            throw new UnsupportedOperationException();
        }
        
        // SetJavaDoc
        @SuppressWarnings("unchecked")
        @Override
        public boolean remove(Object o){
            if(references == null || references.size() == 0){
                return false;
            }
            if(o instanceof Map.Entry){
                final Map.Entry<K2, V2> entry = (Map.Entry<K2, V2>)o;
                final V removed
                     = AbstractCacheMapService.this.remove(entry.getKey());
                return removed != null;
            }
            return false;
        }
        
        // SetJavaDoc
        @Override
        public boolean removeAll(Collection<?> c){
            boolean result = false;
            for(Object val : c){
                result |= remove(val);
            }
            return result;
        }
        
        // SetJavaDoc
        @SuppressWarnings("unchecked")
        @Override
        public boolean retainAll(Collection<?> c){
            if(references == null || references.size() == 0){
                return false;
            }
            boolean result = false;
            synchronized(references){
                final Object[] keys = references.keySet().toArray();
                final Object[] entries = c.toArray();
                final Set<K2> retainsKeys = new HashSet<K2>();
                for(int i = 0; i < entries.length; i++){
                    if(entries[i] instanceof Map.Entry){
                        final Map.Entry<K2, V2> entry = (Map.Entry<K2, V2>)entries[i];
                        retainsKeys.add(entry.getKey());
                    }
                }
                for(int i = 0; i < keys.length; i++){
                    if(!retainsKeys.contains(keys[i])){
                        final Object removed = AbstractCacheMapService.
                            this.remove(keys[i]);
                        result |= removed != null;
                    }
                }
            }
            return result;
        }
        
        // SetJavaDoc
        @Override
        public void clear(){
            AbstractCacheMapService.this.clear();
        }
        
        // SetJavaDoc
        @SuppressWarnings("unchecked")
        @Override
        public boolean equals(Object o){
            if(o == null){
                return false;
            }
            if(o == this){
                return true;
            }
            if(o instanceof EntrySet){
                final EntrySet<K2, V2> cmp = (EntrySet<K2, V2>)o;
                if(collection == null){
                    return cmp.collection == null;
                }else if(collection.equals(cmp.collection)){
                    return true;
                }
            }
            return false;
        }
        
        // SetJavaDoc
        @Override
        public int hashCode(){
            return collection == null ? 0 : collection.hashCode();
        }
    }
    
    /**
     * LbV}bṽ}bvGgJԂB<p>
     *
     * @author M.Takata
     * @see AbstractCacheMapService.EntrySet#iterator()
     */
    protected class EntryIterator<K2, V2>
     implements Iterator<Map.Entry<K2, V2>>, java.io.Serializable{
        
        private static final long serialVersionUID = -5858884191202944380L;
        
        private Iterator<K2> iterator;
        private K2 current;
        
        /**
         * CX^X𐶐B<p>
         */
        @SuppressWarnings("unchecked")
        public EntryIterator(){
            if(references != null){
                iterator = new HashSet<K2>((Set<K2>)references.keySet()).iterator();
            }
        }
        
        // IteratorJavaDoc
        @Override
        public boolean hasNext(){
            return iterator == null ? false : iterator.hasNext();
        }
        
        // IteratorJavaDoc
        @Override
        public Map.Entry<K2, V2> next(){
            if(iterator == null){
                throw new NoSuchElementException();
            }
            current = null;
            current = iterator.next();
            return new Entry<K2, V2>(current);
        }
        
        // IteratorJavaDoc
        @Override
        public void remove(){
            if(current == null){
                throw new IllegalStateException();
            }
            AbstractCacheMapService.this.remove(current);
            iterator.remove();
            if(!iterator.hasNext()){
                current = null;
            }
        }
    }
    
    /**
     * LbV}bṽ}bvGg<p>
     *
     * @author M.Takata
     * @see AbstractCacheMapService.EntryIterator#next()
     */
    protected class Entry<K2, V2> implements Map.Entry<K2, V2>, java.io.Serializable{
        
        private static final long serialVersionUID = -2047042147063848911L;
        
        private K2 key;
        
        /**
         * CX^X𐶐B<p>
         */
        public Entry(K2 key){
            this.key = key;
        }
        
        // Map.EntryJavaDoc
        @Override
        public K2 getKey(){
            return key;
        }
        
        // Map.EntryJavaDoc
        @SuppressWarnings("unchecked")
        @Override
        public V2 getValue(){
            return (V2)AbstractCacheMapService.this.get(key);
        }
        
        // Map.EntryJavaDoc
        @SuppressWarnings("unchecked")
        @Override
        public V2 setValue(V2 value){
            final V result = AbstractCacheMapService.this.remove(key);
            AbstractCacheMapService.this.put((K)key, (V)value);
            return (V2)result;
        }
        
        // Map.EntryJavaDoc
        @SuppressWarnings("unchecked")
        @Override
        public boolean equals(Object o){
            if(o == null){
                return false;
            }
            if(o == this){
                return true;
            }
            if(o instanceof Entry){
                final Entry<K2, V2> cmp = (Entry<K2, V2>)o;
                if(key.equals(cmp.key)){
                    return true;
                }
            }
            return false;
        }
        
        // Map.EntryJavaDoc
        @Override
        public int hashCode(){
            return key.hashCode();
        }
    }
}
