/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.service.cache;

import java.util.*;

import jp.ossc.nimbus.core.ServiceBase;

/**
 * TCYӂꌟ؃T[rXB<p>
 * ȉɁAq[v̎gpTCYőq[v̔𒴂Ƃӂ邠ӂꌟ؃T[rX̃T[rX`B<br>
 * <pre>
 * &lt;?xml version="1.0" encoding="Shift_JIS"?&gt;
 * 
 * &lt;nimbus&gt;
 *     
 *     &lt;manager name="Sample"&gt;
 *         
 *         &lt;service name="MemorySizeOverflowValidator"
 *                  code="jp.ossc.nimbus.service.cache.MemorySizeOverflowValidatorService"/&gt;
 *         
 *     &lt;/manager&gt;
 *     
 * &lt;/nimbus&gt;
 * </pre>
 *
 * @author M.Takata
 */
public class MemorySizeOverflowValidatorService<E> extends ServiceBase
 implements OverflowValidator<E>, CacheRemoveListener<E>, java.io.Serializable,
            MemorySizeOverflowValidatorServiceMBean{
    
    private static final long serialVersionUID = -8937874822673039671L;
    
    private static final char KILO_UNIT = 'K';
    private static final char MEGA_UNIT = 'M';
    private static final char GIGA_UNIT = 'G';
    
    private static final long KILO_BYTE = 1024;
    private static final long MEGA_BYTE = KILO_BYTE * KILO_BYTE;
    private static final long GIGA_BYTE = MEGA_BYTE * KILO_BYTE;
    
    private String maxHeapMemorySizeStr = "64M";
    private long maxHeapMemorySize = 64 * MEGA_BYTE;
    
    private String highHeapMemorySizeStr = "32M";
    private long highHeapMemorySize = 32 * MEGA_BYTE;
    
    private Set<CachedReference<E>> references;
    
    {
        final Runtime runtime = Runtime.getRuntime();
        try{
            maxHeapMemorySize = runtime.maxMemory();
            maxHeapMemorySizeStr = Long.toString(maxHeapMemorySize);
            highHeapMemorySize = maxHeapMemorySize / 2;
            highHeapMemorySizeStr = Long.toString(highHeapMemorySize);
        }catch(NoSuchMethodError err){
        }
    }
    
    /**
     * T[rX̐sB<p>
     * CX^Xϐ̏sB
     *
     * @exception Exception T[rX̐Ɏsꍇ
     */
    @Override
    public void createService() throws Exception{
        references = Collections.synchronizedSet(new HashSet<CachedReference<E>>());
    }
    
    /**
     * T[rX̊JnsB<p>
     * ̑Ó`FbNsB<br>
     *
     * @exception Exception T[rX̊JnɎsꍇ
     */
    @Override
    public void startService() throws Exception{
        if(maxHeapMemorySize <= highHeapMemorySize){
            throw new IllegalArgumentException(
                "maxHeapMemorySize must be larger than highHeapMemorySize."
            );
        }
    }
    
    /**
     * T[rX̔jsB<p>
     * CX^Xϐ̊JsB
     *
     * @exception Exception T[rX̔jɎsꍇ
     */
    @Override
    public void destroyService() throws Exception{
        reset();
        references = null;
    }
    
    // MemorySizeOverflowValidatorServiceMBeanJavaDoc
    @Override
    public void setMaxHeapMemorySize(String size)
     throws IllegalArgumentException{
        maxHeapMemorySize = convertMemorySize(size);
        maxHeapMemorySizeStr = size;
    }
    
    // MemorySizeOverflowValidatorServiceMBeanJavaDoc
    @Override
    public String getMaxHeapMemorySize(){
        return maxHeapMemorySizeStr;
    }
    
    // MemorySizeOverflowValidatorServiceMBeanJavaDoc
    @Override
    public void setHighHeapMemorySize(String size)
     throws IllegalArgumentException{
        highHeapMemorySize = convertMemorySize(size);
        highHeapMemorySizeStr = size;
    }
    
    // MemorySizeOverflowValidatorServiceMBeanJavaDoc
    @Override
    public String getHighHeapMemorySize(){
        return highHeapMemorySizeStr;
    }
    
    // MemorySizeOverflowValidatorServiceMBeanJavaDoc
    @Override
    public int size(){
        return references == null ? 0 : references.size();
    }
    
    private long convertMemorySize(String size)
     throws IllegalArgumentException{
        long value = 0L;
        boolean isValid = true;
        
        if(size == null){
            isValid = false;
        }else{
            final int length = size.length();
            if(length == 0){
                isValid = false;
            }else{
                final char unit = Character.toUpperCase(
                    size.charAt(length - 1)
                );
                String tmpSize = null;
                long unitValue = 0;
                switch(unit){
                case KILO_UNIT:
                    tmpSize = size.substring(0, length - 1);
                    unitValue = KILO_BYTE;
                    break;
                case MEGA_UNIT:
                    tmpSize = size.substring(0, length - 1);
                    unitValue = MEGA_BYTE;
                    break;
                case GIGA_UNIT:
                    tmpSize = size.substring(0, length - 1);
                    unitValue = GIGA_BYTE;
                    break;
                default:
                    tmpSize = size;
                    unitValue = 1;
                }
                try{
                    value = (long)(Double.parseDouble(tmpSize) * (long)unitValue);
                }catch(NumberFormatException e){
                    isValid = false;
                }
            }
        }
        if(value < 0){
            isValid = false;
        }
        if(!isValid){
            throw new IllegalArgumentException("Invalid size : " + size);
        }
        return value;
    }
    
    /**
     * ׃q[vTCY擾B<p>
     *
     * @return ׃q[vTCY
     */
    protected long getHighHeapMemorySizeValue(){
        return highHeapMemorySize;
    }
    
    /**
     * őq[vTCY擾B<p>
     *
     * @return őq[vTCY
     */
    protected long getMaxHeapMemorySizeValue(){
        return maxHeapMemorySize;
    }
    
    /**
     * LbVQƂǉB<p>
     * œnꂽLbVQƂێBɁA{@link CachedReference#addCacheRemoveListener(CacheRemoveListener)}ŁA{@link CacheRemoveListener}ƂĎgo^B<br>
     *
     * @param ref LbVQ
     */
    @Override
    public void add(CachedReference<E> ref){
        if(references == null || ref == null){
            return;
        }
        synchronized(references){
            if(!references.contains(ref)){
                references.add(ref);
                ref.addCacheRemoveListener(this);
            }
        }
    }
    
    /**
     * LbVQƂ폜B<p>
     * œnꂽLbVQƂŕێĂꍇ́AjBɁA{@link CachedReference#removeCacheRemoveListener(CacheRemoveListener)}ŁA{@link CacheRemoveListener}ƂĎgo^B<br>
     *
     * @param ref LbVQ
     */
    @Override
    public void remove(CachedReference<E> ref){
        if(references == null || ref == null){
            return;
        }
        synchronized(references){
            if(references.contains(ref)){
                references.remove(ref);
                ref.removeCacheRemoveListener(this);
            }
        }
    }
    
    /**
     * q[v̎gpłӂꌟ؂sB<p>
     * ȉ̌vZŁAӂꐔvZBAAvZʂ̏ꍇ́A0ƂB<br>
     * LbVTCY~igpq[v]׃q[vjiőq[v]׃q[vj
     *
     * @return ӂꌟ؂sʂӂꂪꍇAӂꐔԂBӂȂꍇ́A0Ԃ
     */
    @Override
    public int validate(){
        if(references == null || references.size() == 0){
            return 0;
        }
        synchronized(references){
            if(getState() != State.STARTED){
                return 0;
            }
            final Runtime runtime = Runtime.getRuntime();
            final long currentHeap = runtime.totalMemory();
            final long currentFree = runtime.freeMemory();
            final long currentUse = currentHeap - currentFree;
            final long highHeap = getHighHeapMemorySizeValue();
            if(currentUse < highHeap){
                return 0;
            }
            final long maxHeap = getMaxHeapMemorySizeValue();
            float rate
                 = ((float)(currentUse - highHeap)) / (maxHeap - highHeap);
            rate = Math.min(rate, 1.0F);
            final int overflowSize = (int)(references.size() * rate);
            return overflowSize > 0 ? overflowSize : 0;
        }
    }
    
    /**
     * ӂꌟ؂s邽߂ɕێĂB<p>
     * {@link #add(CachedReference)}œnꂽLbVQƂSĔjB<br>
     */
    @Override
    public void reset(){
        if(references != null){
            references.clear();
        }
    }
    
    /**
     * LbV폜ꂽLbVQƂ̒ʒm󂯂B<p>
     * {@link #remove(CachedReference)}ĂяoB<br>
     *
     * @param ref LbV폜ꂽLbVQ
     */
    @Override
    public void removed(CachedReference<E> ref){
        remove(ref);
    }
}
