/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.service.journal.editor;

import jp.ossc.nimbus.core.ServiceBase;
import jp.ossc.nimbus.service.journal.JournalEditor;
import jp.ossc.nimbus.service.journal.EditorFinder;

/**
 * bytez𕶎ɕҏWJournalEditorT[rXB<p>
 * 
 * @author M.Takata
 */
public class ByteArrayJournalEditorService extends ServiceBase
 implements JournalEditor, ByteArrayJournalEditorServiceMBean{
    
    private static final long serialVersionUID = -6026767269577892520L;
    private static final int CONVERT_HEX_VALUE = 0;
    private static final int CONVERT_DECIMAL_VALUE = 1;
    private static final int CONVERT_OCTAL_VALUE = 2;
    private static final int CONVERT_ENCODE_VALUE = 3;
    private static final int CONVERT_LENGTH_VALUE = 4;
    
    private String convertMode = CONVERT_HEX;
    private int convertModeValue;
    private String encode;
    private int maxArraySize = -1;
    
    @Override
    public void setMaxArraySize(int max){
        maxArraySize = max;
    }
    @Override
    public int getMaxArraySize(){
        return maxArraySize;
    }
    
    @Override
    public void setConvertMode(String mode) throws java.io.UnsupportedEncodingException{
        if(mode != null){
            if(mode.equals(CONVERT_HEX)){
                convertModeValue = CONVERT_HEX_VALUE;
            }else if(mode.equals(CONVERT_DECIMAL)){
                convertModeValue = CONVERT_DECIMAL_VALUE;
            }else if(mode.equals(CONVERT_OCTAL)){
                convertModeValue = CONVERT_OCTAL_VALUE;
            }else if(mode.equals(CONVERT_LENGTH)){
                convertModeValue = CONVERT_LENGTH_VALUE;
            }else{
                new String(new byte[0], mode);
                encode = mode;
                convertModeValue = CONVERT_ENCODE_VALUE;
            }
            convertMode = mode;
        }
    }
    @Override
    public String getConvertMode(){
        return convertMode;
    }
    
    @Override
    public Object toObject(EditorFinder finder, String key, Object value){
        final byte[] bytes = (byte[])value;
        switch(convertModeValue){
        case CONVERT_HEX_VALUE:
            return toHexStringFormat(finder, key, bytes);
        case CONVERT_DECIMAL_VALUE:
            return toDecimalStringFormat(finder, key, bytes);
        case CONVERT_OCTAL_VALUE:
            return toOctalStringFormat(finder, key, bytes);
        case CONVERT_LENGTH_VALUE:
            return toLengthStringFormat(finder, key, bytes);
        case CONVERT_ENCODE_VALUE:
        default:
            return toEncodeStringFormat(finder, key, bytes);
        }
    }
    
    protected String toHexStringFormat(
        EditorFinder finder,
        String key,
        byte[] bytes
    ){
        StringBuilder buf = new StringBuilder();
        for(int i = 0; i < bytes.length; i++){
            if(maxArraySize >= 0 && i >= maxArraySize){
                buf.append("... length=" + bytes.length);
                break;
            }
            String hex = Integer.toHexString(((int)bytes[i]) & 0x000000FF).toUpperCase();
            if(hex.length() == 1){
                buf.append('0');
            }
            buf.append(hex);
            if(i != bytes.length - 1){
                buf.append(' ');
            }
        }
        return buf.toString();
    }
    
    protected String toDecimalStringFormat(
        EditorFinder finder,
        String key,
        byte[] bytes
    ){
        StringBuilder buf = new StringBuilder();
        for(int i = 0; i < bytes.length; i++){
            if(maxArraySize >= 0 && i >= maxArraySize){
                buf.append("... length=" + bytes.length);
                break;
            }
            String decimal = Byte.toString(bytes[i]);
            buf.append(decimal);
            if(i != bytes.length - 1){
                buf.append(' ');
            }
        }
        return buf.toString();
    }
    
    protected String toOctalStringFormat(
        EditorFinder finder,
        String key,
        byte[] bytes
    ){
        StringBuilder buf = new StringBuilder();
        for(int i = 0; i < bytes.length; i++){
            if(maxArraySize >= 0 && i >= maxArraySize){
                buf.append("... length=" + bytes.length);
                break;
            }
            String octal = Integer.toOctalString(((int)bytes[i]) & 0x000000FF).toUpperCase();
            if(octal.length() == 1){
                buf.append('0').append('0');
            }else if(octal.length() == 2){
                buf.append('0');
            }
            buf.append(octal);
            if(i != bytes.length - 1){
                buf.append(' ');
            }
        }
        return buf.toString();
    }
    
    protected String toEncodeStringFormat(
        EditorFinder finder,
        String key,
        byte[] bytes
    ){
        try{
            return new String(bytes, encode);
        }catch(java.io.UnsupportedEncodingException e){
            return new String(bytes);
        }
    }
    
    protected String toLengthStringFormat(
        EditorFinder finder,
        String key,
        byte[] bytes
    ){
        return new StringBuilder().append(Byte.TYPE.getName()).append('[').append(bytes.length).append(']').toString();
    }
}