/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 *
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.service.publish.websocket.tomcat;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.util.Set;

import jp.ossc.nimbus.beans.dataset.DataSet;
import jp.ossc.nimbus.beans.dataset.Header;
import jp.ossc.nimbus.beans.dataset.Record;
import jp.ossc.nimbus.beans.dataset.RecordList;
import jp.ossc.nimbus.core.ServiceBase;
import jp.ossc.nimbus.core.ServiceBaseSupport;
import jp.ossc.nimbus.core.ServiceManagerFactory;
import jp.ossc.nimbus.core.ServiceName;
import jp.ossc.nimbus.service.context.Context;
import jp.ossc.nimbus.service.journal.EditorFinder;
import jp.ossc.nimbus.service.journal.Journal;
import jp.ossc.nimbus.service.log.Logger;
import jp.ossc.nimbus.service.publish.Message;
import jp.ossc.nimbus.service.publish.MessageListener;
import jp.ossc.nimbus.service.publish.MessageReceiver;
import jp.ossc.nimbus.service.sequence.Sequence;
import jp.ossc.nimbus.service.websocket.AuthenticateException;
import jp.ossc.nimbus.service.websocket.Authenticator;
import jp.ossc.nimbus.service.websocket.jetty.PingSender;
import jp.ossc.nimbus.util.converter.BindingStreamConverter;
import jp.ossc.nimbus.util.converter.ConvertException;
import jp.ossc.nimbus.util.converter.StreamConverter;
import jp.ossc.nimbus.util.converter.StringStreamConverter;

import org.apache.catalina.websocket.MessageInbound;
import org.apache.catalina.websocket.WsOutbound;


/**
 * org.apache.catalina.websocket.MessageInboundpWebSocketMpT[rXB<p>
 * WebSocket̒ʐMDataSetōsBWebSocketւ̃NGXgŁA{@link MessageReceiver}ɔzM̓o^yщsAMessageReceiverM{@link Message}NCAgɔzMB<br>
 *
 * @author M.Ishida
 */
public class DataSetWebSocketService extends MessageInbound
 implements ServiceBaseSupport ,MessageListener, DataSetWebSocketServiceMBean{

    protected static final String PUBLISH_HEADER_PROPERTY_NAME_SUBJECT = "subject";
    protected static final String PUBLISH_HEADER_PROPERTY_NAME_KEY = "key";

    protected static final String PUBLISH_HEADER_SCHEMA
        = ':' + PUBLISH_HEADER_PROPERTY_NAME_SUBJECT + ",java.lang.String\n"
        + ':' + PUBLISH_HEADER_PROPERTY_NAME_KEY + ",java.lang.String";

    protected ServiceName messageReceiverServiceName;
    protected ServiceName reuqestDataSetConverterServiceName;
    protected ServiceName sendDataSetConverterServiceName;
    protected ServiceName requestJournalServiceName;
    protected ServiceName requestEditorFinderServiceName;
    protected String requestIdKey = DEFAULT_REQUEST_ID_KEY;
    protected String requestJournalKey = DEFAULT_REQUEST_JOURNAL_KEY;
    protected String requestMessageJournalKey = DEFAULT_REQUEST_MESSAGE_JOURNAL_KEY;
    protected String requestDataSetJournalKey = DEFAULT_REQUEST_DATASET_JOURNAL_KEY;
    protected String idJournalKey = DEFAULT_ID_JOURNAL_KEY;
    protected String exceptionJournalKey = DEFAULT_EXCEPTION_JOURNAL_KEY;
    protected ServiceName sendJournalServiceName;
    protected ServiceName sendEditorFinderServiceName;
    protected String sendJournalKey = DEFAULT_SEND_JOURNAL_KEY;
    protected String sendMessageJournalKey = DEFAULT_SEND_MESSAGE_JOURNAL_KEY;
    protected String sendDataSetJournalKey = DEFAULT_SEND_DATASET_JOURNAL_KEY;
    protected String sendBytesJournalKey = DEFAULT_SEND_BYTES_JOURNAL_KEY;
    protected ServiceName sequenceServiceName;
    protected ServiceName contextServiceName;
    protected ServiceName authenticatorServiceName;
    protected String characterEncoding = "UTF-8";
    protected int sendMode = SEND_MODE_STRING;
    protected boolean isAppendPublishHeader = true;
    protected String publishHeaderName = PUBLISH_HEADER_DEFAULT_NAME;
    protected boolean isSendBySubject = true;
    protected ServiceName loggerServiceName;
    protected transient Logger logger;

    protected int readTimeout = -1;
    protected int inboundByteBufferSize = -1;
    protected int inboundCharBufferSize = -1;
    protected int outboundByteBufferSize = -1;
    protected int outboundCharBufferSize = -1;
    protected int maxBinaryMessageSize = -1;
    protected int maxTextMessageSize = -1;

    protected String requestConvertErrorMessageId = MSG_ID_REQUEST_CONVERT_ERROR;
    protected String unknownRequestObjectErrorMessageId = MSG_ID_UNKNOWN_REQUEST_OBJECT_ERROR;
    protected String noAuthenticatedErrorMessageId = MSG_ID_NO_AUTHENTICATED_ERROR;
    protected String requestProcessErrorMessageId = MSG_ID_REQUEST_PROCESS_ERROR;
    protected String sendMessageConvertErrorMessageId = MSG_ID_SEND_MESSAGE_CONVERT_ERROR;
    protected String sendMessageErrorMessageId = MSG_ID_SEND_MESSAGE_ERROR;
    protected String sendPongErrorMessageId = MSG_ID_SEND_PONG_ERROR;
    protected String logoutErrorMessageId = MSG_ID_LOGOUT_ERROR;
    protected String closeErrorMessageId = MSG_ID_CLOSE_ERROR;
    protected String messageLargeErrorMessageId = MSG_ID_MESSAGE_TOO_LARGE_ERROR;

    protected int closeCodeForNormal = 1000;
    protected int closeCodeForRequestConvertError = 1003;
    protected int closeCodeForUnknownRequestObjectError = 1011;
    protected int closeCodeForNoAuthenticatedError = 1008;
    protected int closeCodeForRequestProcessError = 1011;
    protected int closeCodeForSendMessageError = 1001;
    protected int closeCodeForRequestMesssageTooLarge=1009;

    protected ServiceBase serviceBase;

    protected MessageReceiver messageReceiver;
    protected BindingStreamConverter requestDataSetConverter;
    protected StreamConverter sendDataSetConverter;
    protected Journal requestJournal;
    protected EditorFinder requestEditorFinder;
    protected Journal sendJournal;
    protected EditorFinder sendEditorFinder;
    protected Sequence sequence;
    protected Context<String, Object> context;
    protected StringStreamConverter stringStreamConverter;
    protected PingSender pingSender;
    protected Authenticator authenticator;
    protected long receiveRequestCount;
    protected long receiveMessageCount;
    protected long sendMessageCount;
    protected String id;

    protected WsOutbound outbound;

    @Override
    public void setMessageReceiverServiceName(ServiceName name){
        messageReceiverServiceName = name;
    }
    @Override
    public ServiceName getMessageReceiverServiceName(){
        return messageReceiverServiceName;
    }

    @Override
    public void setRequestDataSetConverterServiceName(ServiceName name){
        reuqestDataSetConverterServiceName = name;
    }
    @Override
    public ServiceName getRequestDataSetConverterServiceName(){
        return reuqestDataSetConverterServiceName;
    }

    @Override
    public void setSendDataSetConverterServiceName(ServiceName name){
        sendDataSetConverterServiceName = name;
    }
    @Override
    public ServiceName getSendDataSetConverterServiceName(){
        return sendDataSetConverterServiceName;
    }

    @Override
    public void setRequestJournalServiceName(ServiceName name){
        requestJournalServiceName = name;
    }
    @Override
    public ServiceName getRequestJournalServiceName(){
        return requestJournalServiceName;
    }

    @Override
    public void setRequestEditorFinderServiceName(ServiceName name){
        requestEditorFinderServiceName = name;
    }
    @Override
    public ServiceName getRequestEditorFinderServiceName(){
        return requestEditorFinderServiceName;
    }

    @Override
    public void setRequestIDKey(String key){
        requestIdKey = key;
    }
    @Override
    public String getRequestIDKey(){
        return requestIdKey;
    }

    @Override
    public void setRequestJournalKey(String key){
        requestJournalKey = key;
    }
    @Override
    public String getRequestJournalKey(){
        return requestJournalKey;
    }

    @Override
    public void setRequestMessageJournalKey(String key){
        requestMessageJournalKey = key;
    }
    @Override
    public String getRequestMessageJournalKey(){
        return requestMessageJournalKey;
    }

    @Override
    public void setRequestDataSetJournalKey(String key){
        requestDataSetJournalKey = key;
    }
    @Override
    public String getRequestDataSetJournalKey(){
        return requestDataSetJournalKey;
    }

    @Override
    public void setIdJournalKey(String key){
        idJournalKey = key;
    }
    @Override
    public String getIdJournalKey(){
        return idJournalKey;
    }

    @Override
    public void setExceptionJournalKey(String key){
        exceptionJournalKey = key;
    }
    @Override
    public String getExceptionJournalKey(){
        return exceptionJournalKey;
    }

    @Override
    public void setSendJournalServiceName(ServiceName name){
        sendJournalServiceName = name;
    }
    @Override
    public ServiceName getSendJournalServiceName(){
        return sendJournalServiceName;
    }

    @Override
    public void setSendEditorFinderServiceName(ServiceName name){
        sendEditorFinderServiceName = name;
    }
    @Override
    public ServiceName getSendEditorFinderServiceName(){
        return sendEditorFinderServiceName;
    }

    @Override
    public void setSendJournalKey(String key){
        sendJournalKey = key;
    }
    @Override
    public String getSendJournalKey(){
        return sendJournalKey;
    }

    @Override
    public void setSendMessageJournalKey(String key){
        sendMessageJournalKey = key;
    }
    @Override
    public String getSendMessageJournalKey(){
        return sendMessageJournalKey;
    }

    @Override
    public void setSendDataSetJournalKey(String key){
        sendDataSetJournalKey = key;
    }
    @Override
    public String getSendDataSetJournalKey(){
        return sendDataSetJournalKey;
    }

    @Override
    public void setSendBytesJournalKey(String key){
        sendBytesJournalKey = key;
    }
    @Override
    public String getSendBytesJournalKey(){
        return sendBytesJournalKey;
    }

    @Override
    public void setSequenceServiceName(ServiceName name){
        sequenceServiceName = name;
    }
    @Override
    public ServiceName getSequenceServiceName(){
        return sequenceServiceName;
    }

    @Override
    public void setContextServiceName(ServiceName name){
        contextServiceName = name;
    }
    @Override
    public ServiceName getContextServiceName(){
        return contextServiceName;
    }

    @Override
    public void setAuthenticatorServiceName(ServiceName name){
        authenticatorServiceName = name;
    }
    @Override
    public ServiceName getAuthenticatorServiceName(){
        return authenticatorServiceName;
    }

    @Override
    public void setCharacterEncoding(String encoding){
        characterEncoding = encoding;
    }
    @Override
    public String getCharacterEncoding(){
        return characterEncoding;
    }

    @Override
    public void setSendMode(int mode) throws IllegalArgumentException{
        switch(mode){
        case SEND_MODE_STRING:
        case SEND_MODE_BINARY:
            break;
        default:
            throw new IllegalArgumentException("Illegal send mode. mode=" + mode);
        }
        sendMode = mode;
    }
    @Override
    public int getSendMode(){
        return sendMode;
    }

    @Override
    public void setAppendPublishHeader(boolean isAppend){
        isAppendPublishHeader = isAppend;
    }
    @Override
    public boolean isAppendPublishHeader(){
        return isAppendPublishHeader;
    }

    @Override
    public void setPublishHeaderName(String name){
        publishHeaderName = name;
    }
    @Override
    public String getPublishHeaderName(){
        return publishHeaderName;
    }

    @Override
    public void setSendBySubject(boolean isSend){
        isSendBySubject = isSend;
    }
    @Override
    public boolean isSendBySubject(){
        return isSendBySubject;
    }

    @Override
    public Set<String> getSubjects(){
	return messageReceiver == null ? null : messageReceiver.getSubjects(this);
    }

    @Override
    public void setRequestConvertErrorMessageId(String id){
        requestConvertErrorMessageId = id;
    }
    @Override
    public String getRequestConvertErrorMessageId(){
        return requestConvertErrorMessageId;
    }

    @Override
    public void setUnknownRequestObjectErrorMessageId(String id){
        unknownRequestObjectErrorMessageId = id;
    }
    @Override
    public String getUnknownRequestObjectErrorMessageId(){
        return unknownRequestObjectErrorMessageId;
    }

    @Override
    public void setNoAuthenticatedErrorMessageId(String id){
        noAuthenticatedErrorMessageId = id;
    }
    @Override
    public String getNoAuthenticatedErrorMessageId(){
        return noAuthenticatedErrorMessageId;
    }

    @Override
    public void setRequestProcessErrorMessageId(String id){
        requestProcessErrorMessageId = id;
    }
    @Override
    public String getRequestProcessErrorMessageId(){
        return requestProcessErrorMessageId;
    }

    @Override
    public void setSendMessageConvertErrorMessageId(String id){
        sendMessageConvertErrorMessageId = id;
    }
    @Override
    public String getSendMessageConvertErrorMessageId(){
        return sendMessageConvertErrorMessageId;
    }

    @Override
    public void setSendMessageErrorMessageId(String id){
        sendMessageErrorMessageId = id;
    }
    @Override
    public String getSendMessageErrorMessageId(){
        return sendMessageErrorMessageId;
    }

    @Override
    public void setLogoutErrorMessageId(String id){
        logoutErrorMessageId = id;
    }
    @Override
    public String getLogoutErrorMessageId(){
        return logoutErrorMessageId;
    }

    @Override
    public void setCloseCodeForRequestConvertError(int code){
        closeCodeForRequestConvertError = code;
    }
    @Override
    public int getCloseCodeForRequestConvertError(){
        return closeCodeForRequestConvertError;
    }

    @Override
    public void setCloseCodeForUnknownRequestObjectError(int code){
        closeCodeForUnknownRequestObjectError = code;
    }
    @Override
    public int getCloseCodeForUnknownRequestObjectError(){
        return closeCodeForUnknownRequestObjectError;
    }

    @Override
    public void setCloseCodeForNoAuthenticatedError(int code){
        closeCodeForNoAuthenticatedError = code;
    }
    @Override
    public int getCloseCodeForNoAuthenticatedError(){
        return closeCodeForNoAuthenticatedError;
    }

    @Override
    public void setCloseCodeForRequestProcessError(int code){
        closeCodeForRequestProcessError = code;
    }
    @Override
    public int getCloseCodeForRequestProcessError(){
        return closeCodeForRequestProcessError;
    }

    @Override
    public void setCloseCodeForSendMessageError(int code){
        closeCodeForSendMessageError = code;
    }
    @Override
    public int getCloseCodeForSendMessageError(){
        return closeCodeForSendMessageError;
    }

    @Override
    public void setCloseCodeForRequestMesssageTooLarge(
    	int code) {
        closeCodeForRequestMesssageTooLarge = code;
    }
    @Override
    public int getCloseCodeForRequestMesssageTooLarge() {
        return closeCodeForRequestMesssageTooLarge;
    }

    @Override
    public long getReceiveRequestCount(){
        return receiveRequestCount;
    }

    @Override
    public long getReceiveMessageCount(){
        return receiveMessageCount;
    }

    @Override
    public long getSendMessageCount(){
        return sendMessageCount;
    }

    @Override
    public void resetCount(){
        receiveRequestCount = 0;
        receiveMessageCount = 0;
        sendMessageCount = 0;
    }

    @Override
    public String getId(){
        return id;
    }

    public void setMessageReceiver(MessageReceiver receiver){
        messageReceiver = receiver;
    }

    public void setRequestDataSetConverter(BindingStreamConverter converter){
        requestDataSetConverter = converter;
    }

    public void setSendDataSetConverter(StreamConverter converter){
        sendDataSetConverter = converter;
    }

    public void setPingSender(PingSender sender){
        pingSender = sender;
    }

    public void setRequestJournal(Journal journal){
        requestJournal = journal;
    }

    public void setRequestEditorFinder(EditorFinder finder){
        requestEditorFinder = finder;
    }

    public void setSendJournal(Journal journal){
        sendJournal = journal;
    }

    public void setSendEditorFinder(EditorFinder finder){
        sendEditorFinder = finder;
    }

    public void setSequence(Sequence seq){
        sequence = seq;
    }

    public void setContext(Context<String, Object> ctx){
        context = ctx;
    }

    public void setAuthenticator(Authenticator auth){
        authenticator = auth;
    }

    public ServiceName getLoggerServiceName() {
        return loggerServiceName;
    }

    public void setLoggerServiceName(ServiceName loggerServiceName) {
        this.loggerServiceName = loggerServiceName;
    }

    protected Logger getLogger(){
        return logger;
    }

    @Override
    public void setReadTimeout(int readTimeout){
        this.readTimeout = readTimeout;
    }

    @Override
    public int getReadTimeout(){
        return readTimeout;
    }

    public int getInboundByteBufferSize() {
        return getByteBufferMaxSize();
    }

    public void setInboundByteBufferSize(int inboundByteBufferSize) {
	setByteBufferMaxSize(inboundByteBufferSize);
    }

    public int getInboundCharBufferSize() {
        return getCharBufferMaxSize();
    }

    public void setInboundCharBufferSize(int inboundCharBufferSize) {
	setCharBufferMaxSize(inboundCharBufferSize);
    }

    @Override
    public int getOutboundByteBufferSize() {
        return outboundByteBufferSize;
    }

    @Override
    public void setOutboundByteBufferSize(int outboundByteBufferSize) {
        this.outboundByteBufferSize = outboundByteBufferSize;
    }

    @Override
    public int getOutboundCharBufferSize() {
        return outboundCharBufferSize;
    }

    @Override
    public void setOutboundCharBufferSize(int outboundCharBufferSize) {
        this.outboundCharBufferSize = outboundCharBufferSize;
    }

    public int getMaxBinaryMessageSize() {
        return maxBinaryMessageSize;
    }
    public void setMaxBinaryMessageSize(int maxBinaryMessageSize) {
        this.maxBinaryMessageSize = maxBinaryMessageSize;
    }
    public int getMaxTextMessageSize() {
        return maxTextMessageSize;
    }
    public void setMaxTextMessageSize(int maxTextMessageSize) {
        this.maxTextMessageSize = maxTextMessageSize;
    }

    @Override
    public void setServiceBase(ServiceBase service){
	serviceBase = service;
    }

    @Override
    public void createService() throws Exception {
    }

    @Override
    public void startService() throws Exception {
        if(messageReceiverServiceName != null){
            messageReceiver = ServiceManagerFactory.getServiceObject(messageReceiverServiceName);
        }
        if(messageReceiver == null){
            throw new IllegalArgumentException("MessageReceiver is null.");
        }
        if(reuqestDataSetConverterServiceName != null){
            requestDataSetConverter = ServiceManagerFactory.getServiceObject(reuqestDataSetConverterServiceName);
        }
        if(requestDataSetConverter == null){
            throw new IllegalArgumentException("RequestDataSetConverter is null.");
        }
        if(sendDataSetConverterServiceName != null){
            sendDataSetConverter = ServiceManagerFactory.getServiceObject(sendDataSetConverterServiceName);
        }
        if(sendDataSetConverter == null){
            throw new IllegalArgumentException("SendDataSetConverter is null.");
        }
        if(requestJournalServiceName != null){
            requestJournal = ServiceManagerFactory.getServiceObject(requestJournalServiceName);
        }
        if(requestEditorFinderServiceName != null){
            requestEditorFinder = ServiceManagerFactory.getServiceObject(requestEditorFinderServiceName);
        }
        if(sendJournalServiceName != null){
            sendJournal = ServiceManagerFactory.getServiceObject(sendJournalServiceName);
        }
        if(sendEditorFinderServiceName != null){
            sendEditorFinder = ServiceManagerFactory.getServiceObject(sendEditorFinderServiceName);
        }
        if(sequenceServiceName != null){
            sequence = ServiceManagerFactory.getServiceObject(sequenceServiceName);
        }
        if(contextServiceName != null){
            context = ServiceManagerFactory.getServiceObject(contextServiceName);
        }
        if(authenticatorServiceName != null){
            authenticator = ServiceManagerFactory.getServiceObject(authenticatorServiceName);
        }
        stringStreamConverter = new StringStreamConverter();
        if(characterEncoding != null){
            stringStreamConverter.setCharacterEncodingToStream(characterEncoding);
        }
        if(loggerServiceName != null){
            logger = ServiceManagerFactory.getServiceObject(loggerServiceName);
        }
        if(logger == null){
            logger = serviceBase.getLogger();
        }
        if(inboundByteBufferSize != -1){
            setByteBufferMaxSize(inboundByteBufferSize);
        }
        if(inboundCharBufferSize != -1){
            setCharBufferMaxSize(inboundCharBufferSize);
        }
        if(outboundByteBufferSize != -1){
            super.setOutboundByteBufferSize(outboundByteBufferSize);
        }
        if(outboundCharBufferSize != -1){
            super.setOutboundCharBufferSize(outboundCharBufferSize);
        }
    }

    @Override
    public void stopService() throws Exception {
    }

    @Override
    public void destroyService() throws Exception {
    }

    @Override
    protected void onOpen(WsOutbound outbound){
        synchronized(this){
            this.outbound = outbound;
        }
    }

    @Override
    protected void onClose(int closeCode){
        try{
            messageReceiver.removeMessageListener(this);
        }catch(Exception e){
        }
        if(authenticator != null && id != null){
            try{
                authenticator.logout(id, closeCode == 1000 ? false : true);
            }catch(AuthenticateException e){
                getLogger().write(logoutErrorMessageId, e);
            }
        }
        synchronized(this){
            outbound = null;
        }
        serviceBase.stop();
        serviceBase.destroy();
    }

    @Override
    protected void onBinaryMessage(ByteBuffer message) {
	ByteArrayOutputStream baos = new ByteArrayOutputStream();
	while(message.hasRemaining()){
	    baos.write(message.get());
	    if(maxBinaryMessageSize != -1 && baos.size() > maxBinaryMessageSize){
                synchronized(this){
                    if(outbound != null){
                        try{
                            outbound.close(closeCodeForRequestMesssageTooLarge, null);
                        } catch(IOException ioe){
                            getLogger().write(closeErrorMessageId, ioe);
                        }
                        outbound = null;
                    }
                }
                getLogger().write(messageLargeErrorMessageId, maxBinaryMessageSize);
                return;
	    }
	}
        onMessage(new ByteArrayInputStream(baos.toByteArray()), baos.toByteArray());
    }

    @Override
    protected void onTextMessage(CharBuffer message)  {
	ByteArrayOutputStream baos = new ByteArrayOutputStream();
	while(message.hasRemaining()){
	    baos.write(message.get());
	    if(maxTextMessageSize != -1 && baos.size() > maxTextMessageSize){
                synchronized(this){
                    if(outbound != null){
                        try{
                            outbound.close(closeCodeForRequestMesssageTooLarge, null);
                        } catch(IOException ioe){
                            getLogger().write(closeErrorMessageId, ioe);
                        }
                        outbound = null;
                    }
                }
                getLogger().write(messageLargeErrorMessageId, maxTextMessageSize);
                return;
	    }
	}
        onMessage(new ByteArrayInputStream(baos.toByteArray()), baos.toByteArray());
    }

    protected void onMessage(InputStream is, Object data){
        receiveRequestCount++;
        try{
            if(requestJournal != null){
                requestJournal.startJournal(requestJournalKey, requestEditorFinder);
                if(sequence != null){
                    String sequenceId = sequence.increment();
                    if(context != null){
                        context.put(requestIdKey, sequenceId);
                    }
                    requestJournal.setRequestId(sequenceId);
                }else if(context != null){
                    requestJournal.setRequestId(
                        (String)context.get(requestIdKey)
                    );
                }
                if(id != null){
                    requestJournal.addInfo(idJournalKey, id);
                }
                requestJournal.addInfo(requestMessageJournalKey, data);
            }

            Object message = null;
            try{
                message = requestDataSetConverter.convertToObject(is);
            }catch(ConvertException e){
                getLogger().write(requestConvertErrorMessageId, e, data);
                if(requestJournal != null){
                    requestJournal.addInfo(exceptionJournalKey, e);
                }
                synchronized(this){
                    if(outbound != null){
                        try{
                            outbound.close(closeCodeForRequestConvertError, null);
                        } catch(IOException ioe){
                            if(requestJournal != null){
                                requestJournal.addInfo(exceptionJournalKey, ioe);
                            }
                        }
                        outbound = null;
                    }
                }
                return;
            }
            if(!(message instanceof DataSet)){
                getLogger().write(unknownRequestObjectErrorMessageId, message.getClass());
                synchronized(this){
                    if(outbound != null){
                        try{
                            outbound.close(closeCodeForUnknownRequestObjectError, null);
                        } catch(IOException ioe){
                            if(requestJournal != null){
                                requestJournal.addInfo(exceptionJournalKey, ioe);
                            }
                        }
                        outbound = null;
                    }
                }
                return;
            }
            onMessage((DataSet)message);
        }finally{
            if(requestJournal != null){
                requestJournal.endJournal();
            }
        }
    }

    protected void onMessage(DataSet ds){
        String name = ds.getName();
        if(requestJournal != null){
            requestJournal.addInfo(requestDataSetJournalKey, ds);
        }
        if(AuthenticateDataSet.NAME.equals(name)){
            try{
                id = onReceiveAuthenticate(ds);
                if(id != null){
                    requestJournal.addInfo(idJournalKey, id);
                }
            }catch(Exception e){
                getLogger().write(
                    requestProcessErrorMessageId,
                    e,
                    "authenticate",
                    ds.getHeader(AuthenticateDataSet.HEADER_AUTH)
                );
                if(requestJournal != null){
                    requestJournal.addInfo(exceptionJournalKey, e);
                }
                synchronized(this){
                    if(outbound != null){
                    try{
                        outbound.close(closeCodeForRequestProcessError, null);
                    } catch(IOException ioe){
                            if(requestJournal != null){
                                requestJournal.addInfo(exceptionJournalKey, ioe);
                            }
                    }
                    outbound = null;
                    }
                }
                return;
            }
        }else if(AddSubjectDataSet.NAME.equals(name)){
            if(!isAuthenticated()){
                getLogger().write(noAuthenticatedErrorMessageId, ds);
                synchronized(this){
                    if(outbound != null){
                    try{
                        outbound.close(closeCodeForNoAuthenticatedError, null);
                    } catch(IOException ioe){
                            if(requestJournal != null){
                                requestJournal.addInfo(exceptionJournalKey, ioe);
                            }
                    }
                    outbound = null;
                    }
                }
                return;
            }
            try{
                onReceiveAddSubject(ds);
            }catch(Exception e){
                getLogger().write(
                    requestProcessErrorMessageId,
                    e,
                    "addSubject",
                    ds.getHeader(AddSubjectDataSet.HEADER_ADD_SUBJECT)
                );
                if(requestJournal != null){
                    requestJournal.addInfo(exceptionJournalKey, e);
                }
                synchronized(this){
                    if(outbound != null){
                    try{
                        outbound.close(closeCodeForRequestProcessError, null);
                    } catch(IOException ioe){
                            if(requestJournal != null){
                                requestJournal.addInfo(exceptionJournalKey, ioe);
                            }
                    }
                    outbound = null;
                    }
                }
                return;
            }
        }else if(RemoveSubjectDataSet.NAME.equals(name)){
            if(!isAuthenticated()){
                getLogger().write(noAuthenticatedErrorMessageId, ds);
                synchronized(this){
                    if(outbound != null){
                    try{
                        outbound.close(closeCodeForNoAuthenticatedError, null);
                    } catch(IOException ioe){
                            if(requestJournal != null){
                                requestJournal.addInfo(exceptionJournalKey, ioe);
                            }
                    }
                    outbound = null;
                    }
                }
                return;
            }
            try{
                onReceiveRemoveSubject(ds);
            }catch(Exception e){
                getLogger().write(
                    requestProcessErrorMessageId,
                    e,
                    "removeSubject",
                    ds.getHeader(RemoveSubjectDataSet.HEADER_REMOVE_SUBJECT)
                );
                if(requestJournal != null){
                    requestJournal.addInfo(exceptionJournalKey, e);
                }
                synchronized(this){
                    if(outbound != null){
                    try{
                        outbound.close(closeCodeForRequestProcessError, null);
                    } catch(IOException ioe){
                            if(requestJournal != null){
                                requestJournal.addInfo(exceptionJournalKey, ioe);
                            }
                    }
                    outbound = null;
                    }
                }
                return;
            }
        }else if(ByeDataSet.NAME.equals(name)){
            onReceiveBye(ds);
        }else{
            try{
                onReceiveExtentionDataSet(ds);
            }catch(Exception e){
                getLogger().write(
                    requestProcessErrorMessageId,
                    e,
                    "extentionRequest",
                    ds
                );
                if(requestJournal != null){
                    requestJournal.addInfo(exceptionJournalKey, e);
                }
                synchronized(this){
                    if(outbound != null){
                    try{
                        outbound.close(closeCodeForRequestProcessError, null);
                    } catch(IOException ioe){
                            if(requestJournal != null){
                                requestJournal.addInfo(exceptionJournalKey, ioe);
                            }
                    }
                    outbound = null;
                    }
                }
                return;
            }
        }
    }

    protected String onReceiveAuthenticate(DataSet ds) throws Exception{
        Header header = ds.getHeader(AuthenticateDataSet.HEADER_AUTH);
        if(authenticator != null){
            boolean isSuccess = authenticator.login(
                header.getStringProperty(AuthenticateDataSet.HEADER_AUTH_PROPERTY_ID),
                header.getStringProperty(AuthenticateDataSet.HEADER_AUTH_PROPERTY_TICKET)
            );
            return isSuccess ? header.getStringProperty(AuthenticateDataSet.HEADER_AUTH_PROPERTY_ID) : null;
        }
        return header.getStringProperty(AuthenticateDataSet.HEADER_AUTH_PROPERTY_ID);
    }

    protected boolean isAuthenticated(){
        return authenticator == null ? true : id != null;
    }

    protected void onReceiveAddSubject(DataSet ds) throws Exception{
        Header header = ds.getHeader(AddSubjectDataSet.HEADER_ADD_SUBJECT);
        messageReceiver.addSubject(
            this,
            header.getStringProperty(AddSubjectDataSet.HEADER_ADD_SUBJECT_PROPERTY_SUBJECT),
            (String[])header.getProperty(AddSubjectDataSet.HEADER_ADD_SUBJECT_PROPERTY_KEYS)
        );
    }

    protected void onReceiveRemoveSubject(DataSet ds) throws Exception{
        Header header = ds.getHeader(RemoveSubjectDataSet.HEADER_REMOVE_SUBJECT);
        messageReceiver.removeSubject(
            this,
            header.getStringProperty(RemoveSubjectDataSet.HEADER_REMOVE_SUBJECT_PROPERTY_SUBJECT),
            (String[])header.getProperty(RemoveSubjectDataSet.HEADER_REMOVE_SUBJECT_PROPERTY_KEYS)
        );
    }

    protected void onReceiveBye(DataSet ds){
        synchronized(this){
            if(outbound != null){
                try{
                    outbound.close(closeCodeForNormal, null);
                } catch(IOException ioe){
                    getLogger().write(closeErrorMessageId, ioe);
                }
                outbound = null;
            }
        }
    }

    protected void onReceiveExtentionDataSet(DataSet ds) throws Exception{
        throw new UnsupportedOperationException("Unsupported DataSet. dataSet=" + ds);
    }

    @Override
    public void onMessage(Message message) {
        receiveMessageCount++;

        Set<String> subjects = message.getSubjects();
        if(subjects.size() <= 1){
            try{
                if(sendJournal != null){
                    sendJournal.startJournal(sendJournalKey, sendEditorFinder);
                    sendJournal.addInfo(sendMessageJournalKey, message);
                    if(id != null){
                        sendJournal.addInfo(idJournalKey, id);
                    }
                }
                DataSet ds = null;
                try{
                    ds = messageToDataSet(message.getSubject(), message);
                }catch(Exception e){
                    getLogger().write(sendMessageConvertErrorMessageId, e, message);
                    if(sendJournal != null){
                        sendJournal.addInfo(exceptionJournalKey, e);
                    }
                    return;
                }
                if(isAppendPublishHeader){
                    ds = appendPublishHeader(message.getSubject(), message, ds);
                }
                send(ds);
            }finally{
                if(sendJournal != null){
                    sendJournal.endJournal();
                }
            }
        }else{
            if(isSendBySubject){
                for(String subject : subjects){
                    Set<String> keys = messageReceiver.getKeys(this, subject);
                    if(keys == null || !keys.contains(message.getKey(subject))){
                        continue;
                    }
                    try{
                        if(sendJournal != null){
                            sendJournal.startJournal(sendJournalKey, sendEditorFinder);
                            sendJournal.addInfo(sendMessageJournalKey, message);
                            if(id != null){
                                sendJournal.addInfo(idJournalKey, id);
                            }
                        }
                        DataSet ds = null;
                        try{
                            ds = messageToDataSet(subject, message);
                        }catch(Exception e){
                            getLogger().write(sendMessageConvertErrorMessageId, e, message);
                            if(sendJournal != null){
                                sendJournal.addInfo(exceptionJournalKey, e);
                            }
                            continue;
                        }
                        if(isAppendPublishHeader){
                            ds = appendPublishHeader(subject, message, ds);
                        }
                        send(ds);
                    }finally{
                        if(sendJournal != null){
                            sendJournal.endJournal();
                        }
                    }
                }
            }else{
                try{
                    if(sendJournal != null){
                        sendJournal.startJournal(sendJournalKey, sendEditorFinder);
                        sendJournal.addInfo(sendMessageJournalKey, message);
                        if(id != null){
                            sendJournal.addInfo(idJournalKey, id);
                        }
                    }
                    DataSet ds = null;
                    try{
                        ds = messageToDataSet(message.getSubject(), message);
                    }catch(Exception e){
                        getLogger().write(sendMessageConvertErrorMessageId, e, message);
                        if(sendJournal != null){
                            sendJournal.addInfo(exceptionJournalKey, e);
                        }
                        return;
                    }
                    if(isAppendPublishHeader){
                        for(String subject : subjects){
                            if(isAppendPublishHeader){
                                ds = appendPublishHeader(subject, message, ds);
                            }
                        }
                    }
                    send(ds);
                }finally{
                    if(sendJournal != null){
                        sendJournal.endJournal();
                    }
                }
            }
        }
    }

    protected DataSet appendPublishHeader(String subject, Message message, DataSet ds){
        if(isSendBySubject){
            Header header = ds.getHeader(publishHeaderName);
            if(header == null){
                ds.setHeaderSchema(publishHeaderName, PUBLISH_HEADER_SCHEMA);
                header = ds.getHeader(publishHeaderName);
            }
            header.setProperty(PUBLISH_HEADER_PROPERTY_NAME_SUBJECT, subject);
            header.setProperty(PUBLISH_HEADER_PROPERTY_NAME_KEY, message.getKey(subject));
        }else{
            RecordList list = ds.getRecordList(publishHeaderName);
            if(list == null){
                ds.setRecordListSchema(publishHeaderName, PUBLISH_HEADER_SCHEMA);
                list = ds.getRecordList(publishHeaderName);
            }
            Record record = list.createRecord();
            record.setProperty(PUBLISH_HEADER_PROPERTY_NAME_SUBJECT, subject);
            record.setProperty(PUBLISH_HEADER_PROPERTY_NAME_KEY, message.getKey(subject));
            list.addRecord(record);
        }
        return ds;
    }

    protected DataSet messageToDataSet(String subject, Message message) throws Exception{
        return (DataSet)message.getObject();
    }

    protected void send(DataSet ds){
        if(sendJournal != null){
            sendJournal.addInfo(sendDataSetJournalKey, ds);
        }
        byte[] bytes = null;
        try{
            InputStream is = sendDataSetConverter.convertToStream(ds);
            bytes = toBytes(is);
            if(bytes == null || bytes.length == 0){
                return;
            }
        }catch(ConvertException e){
            getLogger().write(sendMessageConvertErrorMessageId, e, ds);
            return;
        }catch(IOException e){
            getLogger().write(sendMessageConvertErrorMessageId, e, ds);
            return;
        }
        try{
            send(bytes);
        }catch(IOException e){
            getLogger().write(sendMessageErrorMessageId, e, ds);
            return;
        }
    }

    protected void send(byte[] bytes) throws IOException{
        if(sendJournal != null){
            sendJournal.addInfo(sendBytesJournalKey, bytes);
        }
        try{
            switch(sendMode){
            case SEND_MODE_BINARY:
                synchronized(this){
                    if(outbound != null){
                	ByteBuffer buffer = ByteBuffer.wrap(bytes);
                        outbound.writeBinaryMessage(buffer);
                        outbound.flush();
                    }
                }
                break;
            case SEND_MODE_STRING:
            default:
                synchronized(this){
                    if(outbound != null){
                        String msg = characterEncoding == null ? new String(bytes) : new String(bytes, characterEncoding);
                        CharBuffer buffer = CharBuffer.wrap(msg);
                        outbound.writeTextMessage(buffer);
                        outbound.flush();
                    }
                }
            }
            sendMessageCount++;
        }catch(IOException e){
            if(sendJournal != null){
                sendJournal.addInfo(exceptionJournalKey, e);
            }
            throw e;
        }
    }

    protected byte[] toBytes(InputStream is) throws IOException{
        byte[] bytes = new byte[1024];
        final ByteArrayOutputStream baos = new ByteArrayOutputStream();
        int length = 0;
        while((length = is.read(bytes)) != -1){
            baos.write(bytes, 0, length);
        }
        return baos.toByteArray();
    }

    public static class AuthenticateDataSet extends DataSet{

        private static final long serialVersionUID = -1186408833828282199L;

        public static final String NAME = "Authenticate";

        public static final String HEADER_AUTH = "Authenticate";

        public static final String HEADER_AUTH_PROPERTY_ID = "id";
        public static final String HEADER_AUTH_PROPERTY_TICKET = "ticket";

        public static final String HEADER_SCHEMA_AUTH
            = ':' + HEADER_AUTH_PROPERTY_ID + ",java.lang.String\n"
            + ':' + HEADER_AUTH_PROPERTY_TICKET + ",java.lang.String"
            ;

        public AuthenticateDataSet(){
            setName(NAME);
            setHeaderSchema(HEADER_AUTH, HEADER_SCHEMA_AUTH);
        }
    }

    public static class AddSubjectDataSet extends DataSet{

        private static final long serialVersionUID = -3694906611138615108L;

        public static final String NAME = "AddSubject";

        public static final String HEADER_ADD_SUBJECT = "AddSubject";

        public static final String HEADER_ADD_SUBJECT_PROPERTY_SUBJECT = "subject";
        public static final String HEADER_ADD_SUBJECT_PROPERTY_KEYS = "keys";

        public static final String HEADER_SCHEMA_ADD_SUBJECT
            = ':' + HEADER_ADD_SUBJECT_PROPERTY_SUBJECT + ",java.lang.String\n"
            + ':' + HEADER_ADD_SUBJECT_PROPERTY_KEYS + ",java.lang.String[]"
            ;

        public AddSubjectDataSet(){
            setName(NAME);
            setHeaderSchema(HEADER_ADD_SUBJECT, HEADER_SCHEMA_ADD_SUBJECT);
        }
    }

    public static class RemoveSubjectDataSet extends DataSet{

        private static final long serialVersionUID = -252165941836313576L;

        public static final String NAME = "RemoveSubject";

        public static final String HEADER_REMOVE_SUBJECT = "RemoveSubject";

        public static final String HEADER_REMOVE_SUBJECT_PROPERTY_SUBJECT = "subject";
        public static final String HEADER_REMOVE_SUBJECT_PROPERTY_KEYS = "keys";

        public static final String HEADER_SCHEMA_REMOVE_SUBJECT
            = ':' + HEADER_REMOVE_SUBJECT_PROPERTY_SUBJECT + ",java.lang.String\n"
            + ':' + HEADER_REMOVE_SUBJECT_PROPERTY_KEYS + ",java.lang.String[]"
            ;

        public RemoveSubjectDataSet(){
            setName(NAME);
            setHeaderSchema(HEADER_REMOVE_SUBJECT, HEADER_SCHEMA_REMOVE_SUBJECT);
        }
    }

    public static class ByeDataSet extends DataSet{

        private static final long serialVersionUID = -5788398695541333870L;

        public static final String NAME = "Bye";

        public ByeDataSet(){
            setName(NAME);
        }
    }

}