/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.util.converter;

import java.util.Map;
import java.util.HashMap;
import java.lang.reflect.InvocationTargetException;

import javax.servlet.ServletRequest;
import javax.servlet.http.HttpServletRequest;

import jp.ossc.nimbus.beans.IndexedProperty;
import jp.ossc.nimbus.beans.NestedProperty;
import jp.ossc.nimbus.beans.NoSuchPropertyException;
import jp.ossc.nimbus.beans.Property;
import jp.ossc.nimbus.beans.PropertyAccess;
import jp.ossc.nimbus.beans.dataset.DataSet;
import jp.ossc.nimbus.beans.dataset.PropertySetException;
import jp.ossc.nimbus.beans.dataset.Record;
import jp.ossc.nimbus.beans.dataset.RecordList;
import jp.ossc.nimbus.service.beanflow.BeanFlow;
import jp.ossc.nimbus.service.beanflow.BeanFlowException;
import jp.ossc.nimbus.service.beanflow.BeanFlowFactory;

/**
 * T[ubgNGXgp[^DataSetRo[^B<p>
 * {@link javax.servlet.ServletRequest#getParameterValues(String)}Ŏ擾łp[^{@link jp.ossc.nimbus.beans.dataset.DataSet DataSet}{@link jp.ossc.nimbus.beans.dataset.Header Header}A{@link jp.ossc.nimbus.beans.dataset.RecordList RecordList}̎{@link Record Record}̃vpeBɐݒ肵āADataSetIuWFNgɕϊB<br>
 * NGXgp[^ŁAf[^Zbgƃf[^Zbgɂǂ̂悤ɒlݒ肷邩̃vpeB\w肷鎖ŁANGXgp[^DataSetƂ̃}bsOsB<br>
 * <p>
 * DataSet́A{@link #setDataSet(String, jp.ossc.nimbus.beans.dataset.DataSet) setDataSet(String, DataSet)}ŗ\߂Convertergɓo^ĂB̏ꍇA1f[^ZbgƂȂB<br>
 * ܂́A{@link #setBeanFlowFactory(BeanFlowFactory)}Őݒ肵BeanFlowFactoryɁADataSet߂lƂBeanFlow`ĂB̏ꍇABeanFlowf[^ZbgƂȂB<br>
 * <p>
 * NGXgp[^̎w@́Aȉ̒ʂB܂AvpeB\́A{@link jp.ossc.nimbus.beans.PropertyFactory PropertyFactory}QƁB<br>
 * <table border="1">
 *   <tr>
 *     <td>NGXgp[^Header̃}bsO@</td>
 *     <td>&lt;input name="ds1:Header(h1).prop1" type="text" value="a"&gt;</td>
 *   </tr>
 *   <tr>
 *     <td rowspan="2">NGXgp[^RecordList̃}bsO@</td>
 *     <td>&lt;input name="ds1:RecordList(l1).prop1" type="text" value="a"&gt;<br>&lt;input name="ds1:RecordList(l1).prop1" type="text" value="a"&gt;</td>
 *   </tr>
 *   <tr>
 *     <td>&lt;input name="ds1:RecordList[0](l1).prop1" type="text" value="a"&gt;<br>&lt;input name="ds1:RecordList[1](l1).prop1" type="text" value="a"&gt;</td>
 *   </tr>
 * </table>
 * <p>
 * ܂Af[^Zbg̎ẃASẴNGXgp[^𓯂DataSetɊi[ꍇ͈ꊇŎw肷鎖łAȉ̂悤ɂB<br>
 * <pre>
 *   &lt;input type="hidden" name="ds" value="ds1"&gt;
 *   &lt;input name=":Header(h1).prop1" type="text" value="a"&gt;
 *   &lt;input name=":Header(h1).prop2" type="text" value="b"&gt;
 * </pre>
 * 
 * @author M.Takata
 */
public class DataSetServletRequestParameterConverter implements Converter{
    
    public static final String DEFAULT_DATASET_PARAMETER_NAME = "ds";
    
    public static final String DEFAULT_DATASET_DELIMITER = ":";
    
    public static final String DEFAULT_DATASET_PREFIX = "dataset";
    
    /**
     * f[^Zbg}bsOB<p>
     */
    protected Map<String, DataSet> dataSetMap = new HashMap<String, DataSet>();
    
    /**
     * BeanFlowFactoryB<p>
     */
    protected BeanFlowFactory beanFlowFactory;
    
    /**
     * PropertyLbVMapB<p>
     */
    protected PropertyAccess propertyCache = new PropertyAccess();
    
    /**
     * f[^Zbg肷p[^B<p>
     */
    protected String dataSetParameterName = DEFAULT_DATASET_PARAMETER_NAME;
    
    /**
     * f[^Zbg̋؂qB<p>
     */
    protected String datasetDelimiter = DEFAULT_DATASET_DELIMITER;
    
    /**
     * f[^ZbgpX猈肷ꍇɁApXɕtOuB<p>
     * ftHǵA{@link #DEFAULT_DATASET_PREFIX}B<br>
     */
    protected String dataSetPathPrefix = DEFAULT_DATASET_PREFIX;
    
    /**
     * f[^Zbgɑ݂Ȃp[^𖳎邩ǂ̃tOB<p>
     * ftHǵAfalseŁAϊG[ƂB<br>
     */
    protected boolean isIgnoreUnknownParameter;
    
    /**
     * f[^Zbgƃf[^Zbg̃}bsOݒ肷B<p>
     * T[ubgNGXgp[^f[^ZbgϊsۂɁAf[^Zbgf[^Zbg肷̂ɎgpB<br>
     * 
     * @param name f[^Zbg
     * @param dataSet f[^Zbg
     */
    public void setDataSet(String name, DataSet dataSet){
        if(dataSet.getName() == null){
            dataSet.setName(name);
        }
        dataSetMap.put(name, dataSet);
    }
    
    /**
     * DataSetBeanFlowŎ擾ꍇɎgp{@link BeanFlowFactory}ݒ肷B<p>
     *
     * @param factory BeanFlowFactory
     */
    public void setBeanFlowFactory(BeanFlowFactory factory){
        beanFlowFactory = factory;
    }
    
    /**
     * DataSetꊇŎw肷郊NGXgp[^ݒ肷B<p>
     * ftHǵA{@link #DEFAULT_DATASET_PARAMETER_NAME}B<br>
     *
     * @param name DataSetꊇŎw肷郊NGXgp[^
     */
    public void setDataSetParameterName(String name){
        dataSetParameterName = name;
    }
    
    /**
     * DataSetꊇŎw肷郊NGXgp[^擾B<p>
     *
     * @return DataSetꊇŎw肷郊NGXgp[^
     */
    public String getDataSetParameterName(){
        return dataSetParameterName;
    }
    
    /**
     * f[^Zbg̋؂qݒ肷B<p>
     * ftHǵA{@link #DEFAULT_DATASET_DELIMITER}B<br>
     *
     * @param delim f[^Zbg̋؂q
     */
    public void setDataSetDelimiter(String delim){
        datasetDelimiter = delim;
    }
    
    /**
     * f[^Zbg̋؂q擾B<p>
     *
     * @return f[^Zbg̋؂q
     */
    public String getDataSetDelimiter(){
        return datasetDelimiter;
    }
    
    /**
     * DataSetpX猈肷ꍇ̑Ouݒ肷B<p>
     * ftHǵA{@link #DEFAULT_DATASET_PREFIX}B<br>
     *
     * @param prefix DataSetpX猈肷ꍇ̑Ou
     */
    public void setDataSetPathPrefix(String prefix){
        dataSetPathPrefix = prefix;
    }
    
    /**
     * DataSetpX猈肷ꍇ̑Ou擾B<p>
     *
     * @return DataSetpX猈肷ꍇ̑Ou
     */
    public String getDataSetPathPrefix(){
        return dataSetPathPrefix;
    }
    
    /**
     * f[^Zbgɑ݂Ȃp[^𖳎邩ǂݒ肷B<p>
     * ftHǵAfalseŁAϊG[ƂȂB<br>
     * 
     * @param isIgnore truȅꍇA
     */
    public void setIgnoreUnknownParameter(boolean isIgnore){
        isIgnoreUnknownParameter = isIgnore;
    }
    
    /**
     * f[^Zbgɑ݂Ȃp[^𖳎邩ǂ𔻒肷B<p>
     * 
     * @return truȅꍇA
     */
    public boolean isIgnoreUnknownParameter(){
        return isIgnoreUnknownParameter;
    }
    
    /**
     * w肳ꂽIuWFNgϊB<p>
     *
     * @param obj ϊΏۂ̃IuWFNg
     * @return ϊ̃IuWFNg
     * @exception ConvertException ϊɎsꍇ
     */
    @SuppressWarnings("unchecked")
    @Override
    public Object convert(Object obj) throws ConvertException{
        ServletRequest request = (ServletRequest)obj;
        final Map<String, String[]> paramMap = (Map<String, String[]>)request.getParameterMap();
        if(paramMap == null || paramMap.size() == 0){
            return null;
        }
        String defaultDsName = request.getParameter(dataSetParameterName);
        if((defaultDsName == null || defaultDsName.length() == 0)
            && request instanceof HttpServletRequest){
            HttpServletRequest httpReq = (HttpServletRequest)request;
            String path = httpReq.getPathInfo();
            if(path == null){
                path = httpReq.getServletPath();
            }
            if(path != null){
                int index = path.lastIndexOf('.');
                if(index != -1){
                    path = path.substring(0, index);
                }
                defaultDsName = dataSetPathPrefix + path;
            }
        }
        final Map<String, DataSet> currentDsMap = new HashMap<String, DataSet>();
        for(Map.Entry<String, String[]> entry : paramMap.entrySet()){
            final String key = entry.getKey();
            final int index = key.indexOf(datasetDelimiter);
            if(index == -1 || index == key.length() - 1){
                continue;
            }
            String dsName = null;
            if(index == 0){
                dsName = defaultDsName;
            }else{
                dsName = key.substring(0, index);
            }
            if(dsName == null){
                continue;
            }
            DataSet ds = (DataSet)currentDsMap.get(dsName);
            if(ds == null){
                if(dataSetMap.containsKey(dsName)){
                    ds = ((DataSet)dataSetMap.get(dsName)).cloneSchema();
                }else if(beanFlowFactory != null
                            && beanFlowFactory.containsFlow(dsName)
                ){
                    BeanFlow beanFlow = null;
                    try{
                        beanFlow = beanFlowFactory.createFlow(dsName);
                    }catch(BeanFlowException e){
                        throw new ConvertException(e);
                    }
                    Object ret = null;
                    try{
                        ret = beanFlow.execute(null);
                    }catch(Exception e){
                        throw new ConvertException("Exception occured in BeanFlow '" + dsName + "'", e);
                    }
                    if(!(ret instanceof DataSet)){
                        throw new ConvertException("Result of BeanFlow '" + dsName + "' is not DataSet.");
                    }
                    ds = (DataSet)ret;
                }else{
                    if(isIgnoreUnknownParameter){
                        continue;
                    }else{
                        throw new ConvertException("Unknown DataSet : " + dsName);
                    }
                }
                currentDsMap.put(dsName, ds);
            }
            final String propStr = key.substring(index + 1);
            Property prop = null;
            try{
                prop = propertyCache.getProperty(propStr);
            }catch(IllegalArgumentException e){
                throw new ConvertException("Parameter '" + key + "' is illegal.", e);
            }
            final String[] vals = entry.getValue();
            try{
                if(prop instanceof NestedProperty){
                    Property thisProp = ((NestedProperty)prop).getThisProperty();
                    if(thisProp instanceof NestedProperty){
                        Property nestedProp = ((NestedProperty)prop).getNestedProperty();
                        Property nestedProp2 = ((NestedProperty)thisProp).getNestedProperty();
                        if(nestedProp2 instanceof IndexedProperty){
                            Property thisProp2 = ((NestedProperty)thisProp).getThisProperty();
                            Object thisObj = thisProp2.getProperty(ds);
                            if(thisObj == null){
                                if(isIgnoreUnknownParameter){
                                    continue;
                                }else{
                                    throw new ConvertException("Parameter '" + key + "' is illegal.");
                                }
                            }
                            if(thisObj instanceof RecordList){
                                setRecordListProperty(
                                    (RecordList)thisObj,
                                    nestedProp.getPropertyName(),
                                    ((IndexedProperty)nestedProp2).getIndex(),
                                    vals
                                );
                            }else{
                                // 肦Ȃ
                                prop.setProperty(
                                    ds,
                                    vals[vals.length - 1]
                                );
                            }
                        }else{
                            Object thisObj = thisProp.getProperty(ds);
                            if(thisObj == null){
                                if(isIgnoreUnknownParameter){
                                    continue;
                                }else{
                                    throw new ConvertException("Parameter '" + key + "' is illegal.");
                                }
                            }
                            if(thisObj instanceof RecordList){
                                setRecordListProperty(
                                    (RecordList)thisObj,
                                    nestedProp.getPropertyName(),
                                    vals
                                );
                            }else if(thisObj instanceof Record){
                                setRecordProperty(
                                    (Record)thisObj,
                                    nestedProp.getPropertyName(),
                                    nestedProp.getPropertyType(thisObj),
                                    vals
                                );
                            }else{
                                nestedProp.setProperty(
                                    thisObj,
                                    vals[vals.length - 1]
                                );
                            }
                        }
                    }else{
                        Object thisObj = thisProp.getProperty(ds);
                        if(thisObj == null){
                            if(isIgnoreUnknownParameter){
                                continue;
                            }else{
                                throw new ConvertException("Parameter '" + key + "' is illegal.");
                            }
                        }
                        Property nestedProp = ((NestedProperty)prop).getNestedProperty();
                        if(thisObj instanceof RecordList){
                            setRecordListProperty(
                                (RecordList)thisObj,
                                nestedProp.getPropertyName(),
                                vals
                            );
                        }else if(thisObj instanceof Record){
                            setRecordProperty(
                                (Record)thisObj,
                                nestedProp.getPropertyName(),
                                nestedProp.getPropertyType(thisObj),
                                vals
                            );
                        }else{
                            nestedProp.setProperty(
                                thisObj,
                                vals[vals.length - 1]
                            );
                        }
                    }
                }else{
                    throw new ConvertException("Parameter '" + key + "' is illegal.");
                }
            }catch(PropertySetException e){
                Throwable cause = e.getCause();
                if(cause instanceof ConvertException){
                    throw (ConvertException)cause;
                }
                if(isIgnoreUnknownParameter){
                    continue;
                }else{
                    throw new ConvertException("Parameter '" + key + "' is illegal.", e);
                }
            }catch(NoSuchPropertyException e){
                if(isIgnoreUnknownParameter){
                    continue;
                }else{
                    throw new ConvertException("Parameter '" + key + "' is illegal.", e);
                }
            }catch(InvocationTargetException e){
                throw new ConvertException("Parameter '" + key + "' is illegal.", e.getTargetException());
            }
        }
        if(currentDsMap.size() == 0){
            return null;
        }else if(currentDsMap.size() == 1){
            return currentDsMap.values().iterator().next();
        }else{
            return currentDsMap;
        }
    }
    
    protected void setRecordProperty(
        Record record,
        String name,
        Class<?> propType,
        String[] vals
    ) throws PropertySetException{
        if(propType == null
            || propType.equals(java.lang.Object.class)
            || propType.equals(String[].class)
        ){
            record.setProperty(
                name,
                vals
            );
        }else{
            if(propType.isArray() && vals.length != 1){
                record.setParseProperty(
                    name,
                    vals
                );
            }else{
                record.setParseProperty(
                    name,
                    vals[vals.length - 1]
                );
            }
        }
    }
    
    protected void setRecordListProperty(
        RecordList recList,
        String name,
        String[] vals
    ) throws PropertySetException{
        for(int i = 0; i < vals.length; i++){
            Record rec = null;
            if(recList.size() > i){
                rec = recList.getRecord(i);
            }else{
                rec = recList.createRecord();
                recList.addRecord(rec);
            }
            rec.setParseProperty(name, vals[i]);
        }
    }
    
    protected void setRecordListProperty(
        RecordList recList,
        String name,
        int index,
        String[] vals
    ) throws PropertySetException{
        Record rec = null;
        if(recList.size() > index){
            rec = recList.getRecord(index);
        }else{
            for(int i = recList.size(); i <= index; i++){
                rec = recList.createRecord();
                recList.addRecord(rec);
            }
        }
        rec.setParseProperty(name, vals[0]);
    }
}
