/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.core;

import java.io.*;
import java.util.*;
import org.w3c.dom.*;

import jp.ossc.nimbus.beans.ServiceNameEditor;

/**
 * T[rX`&lt;service&gt;vf^f[^B<p>
 * T[rX`t@C&lt;service&gt;vfɋLqꂽei[郁^f[^ReiłB<p>
 *
 * @author M.Takata
 * @see <a href="nimbus-service_2_0.xsd">T[rX`t@CXL[}`</a>
 */
public class ServiceMetaData extends ObjectMetaData implements Serializable{
    
    private static final long serialVersionUID = 1524948064493968357L;
    
    /**
     * &lt;service&gt;vf̗vfB<p>
     */
    public static final String SERVICE_TAG_NAME = "service";
    
    public static final String INSTANCE_TYPE_SINGLETON = "singleton";
    public static final String INSTANCE_TYPE_FACTORY = "factory";
    public static final String INSTANCE_TYPE_THREADLOCAL = "threadlocal";
    public static final String INSTANCE_TYPE_TEMPLATE = "template";
    
    private static final String DEPENDS_TAG_NAME = "depends";
    private static final String OPT_CONF_TAG_NAME = "optional-config";
    private static final String NAME_ATTRIBUTE_NAME = "name";
    private static final String INIT_STATE_ATTRIBUTE_NAME = "initState";
    private static final String INSTANCE_ATTRIBUTE_NAME = "instance";
    private static final String MANAGEMENT_ATTRIBUTE_NAME = "management";
    private static final String CREATE_TEMPLATE_ATTRIBUTE_NAME = "createTemplate";
    private static final String TEMPLATE_ATTRIBUTE_NAME = "template";
    
    private String name;
    
    private String initState = Service.State.STARTED.toString();
    
    private Service.State initStateValue = Service.State.STARTED;
    
    private String instance = INSTANCE_TYPE_SINGLETON;
    
    private boolean isFactory;
    
    private boolean isTemplate;
    
    private boolean isManagement;
    
    private boolean isCreateTemplate = true;
    
    private transient Element optionalConfig;
    
    private List<DependsMetaData> depends = new ArrayList<DependsMetaData>();
    
    private ManagerMetaData manager;
    
    private String template;
    
    /**
     * ̃CX^X𐶐B<p>
     */
    public ServiceMetaData(){
    }
    
    /**
     * w肵T[rXANX̃^f[^CX^X𐶐B<p>
     * @param name T[rX
     * @param code NX
     */
    public ServiceMetaData(String name, String code){
        setName(name);
        setCode(code);
    }
    
    /**
     * evf̃^f[^CX^X𐶐B<p>
     * ServiceMetaData̐evf́A&lt;manager&gt;vf\ManagerMetaDataA܂́A&lt;depends&gt;vf\DependsMetaDatałB<br>
     * 
     * @param loader [hServiceLoader
     * @param parent evf̃^f[^
     * @param manager &lt;service&gt;vfŒ`ꂽT[rXo^{@link ServiceManager}&lt;manager&gt;vf\ManagerMetaData
     * @see ManagerMetaData
     * @see ServiceMetaData.DependsMetaData
     */
    public ServiceMetaData(
        ServiceLoader loader,
        MetaData parent,
        ManagerMetaData manager
    ){
        super(loader, parent, manager == null ? null : manager.getName());
        this.manager = manager;
    }
    
    /**
     * &lt;service&gt;vfname̒l擾B<p>
     * 
     * @return name̒l
     */
    public String getName(){
        return name;
    }
    
    /**
     * &lt;service&gt;vfname̒lݒ肷B<p>
     * 
     * @param name name̒l
     */
    public void setName(String name){
        this.name = name;
    }
    
    /**
     * &lt;service&gt;vfinitState̒l擾B<p>
     * 
     * @return initState̒l
     */
    public String getInitState(){
        return initState;
    }
    
    /**
     * &lt;service&gt;vfinitState̒lݒ肷B<p>
     * 
     * @param state initState̒l
     */
    public void setInitState(String state){
        Service.State stateValue = Service.State.toState(state);
        if(!stateValue.equals(Service.State.CREATED)
                && !stateValue.equals(Service.State.STARTED)){
            throw new IllegalArgumentException("Illegal initState : " + state);
        }
        initStateValue = stateValue;
        initState = state;
    }
    
    /**
     * &lt;service&gt;vfinstance̒l擾B<p>
     * 
     * @return instance̒l
     */
    public String getInstance(){
        return instance;
    }
    
    /**
     * &lt;service&gt;vfinstance̒lݒ肷B<p>
     * 
     * @param val instance̒l
     */
    public void setInstance(String val){
        if(INSTANCE_TYPE_SINGLETON.equals(val)){
            instance = val;
        }else if(INSTANCE_TYPE_FACTORY.equals(val)
             || INSTANCE_TYPE_THREADLOCAL.equals(val)){
            isFactory = true;
            instance = val;
        }else if(INSTANCE_TYPE_TEMPLATE.equals(val)){
            isTemplate = true;
            instance = val;
        }else{
            throw new IllegalArgumentException(val);
        }
    }
    
    /**
     * &lt;service&gt;vf\T[rXt@Ngǂ𔻒肷B<p>
     *
     * @return &lt;service&gt;vf\T[rXt@Ng̏ꍇtrue
     */
    public boolean isFactory(){
        return isFactory;
    }
    
    /**
     * &lt;service&gt;vf\T[rXt@Ngǂݒ肷B<p>
     *
     * @param isFactory truȅꍇA&lt;service&gt;vf\T[rXt@Ng
     */
    public void setFactory(boolean isFactory){
        this.isFactory = isFactory;
    }
    
    /**
     * &lt;service&gt;vftemplate̒l擾B<p>
     * 
     * @return template̒l
     */
    public String getTemplateName(){
        return template;
    }
    
    /**
     * &lt;service&gt;vftemplate̒lݒ肷B<p>
     * 
     * @param name template̒l
     */
    public void setTemplateName(String name){
        this.template = name;
    }
    
    /**
     * &lt;service&gt;vf\T[rXev[gǂ𔻒肷B<p>
     *
     * @return &lt;service&gt;vf\T[rXev[g̏ꍇtrue
     */
    public boolean isTemplate(){
        return isTemplate;
    }
    
    /**
     * &lt;service&gt;vf\T[rXev[gǂݒ肷B<p>
     *
     * @param isTemplate truȅꍇA&lt;service&gt;vf\T[rXev[g
     */
    public void setTemplate(boolean isTemplate){
        this.isTemplate = isTemplate;
    }
    
    /**
     * &lt;service&gt;vfmanagement̒l擾B<p>
     * 
     * @return management̒l
     */
    public boolean isManagement(){
        return isManagement;
    }
    
    /**
     * &lt;service&gt;vfmanagement̒lݒ肷B<p>
     * 
     * @param flg management̒l
     */
    public void setManagement(boolean flg){
        isManagement = flg;
    }
    
    /**
     * &lt;service&gt;vfcreateTemplate̒l擾B<p>
     * 
     * @return createTemplate̒l
     */
    public boolean isCreateTemplate(){
        return isCreateTemplate;
    }
    
    /**
     * &lt;service&gt;vfcreateTemplate̒lݒ肷B<p>
     * 
     * @param flg createTemplate̒l
     */
    public void setCreateTemplate(boolean flg){
        isCreateTemplate = flg;
    }
    
    /**
     * &lt;service&gt;vfinitState̒l擾B<p>
     * 
     * @return initState̒l
     */
    public Service.State getInitStateValue(){
        return initStateValue;
    }
    
    /**
     * &lt;service&gt;vfinitState̒lݒ肷B<p>
     * 
     * @param state initState̒l
     */
    public void setInitStateValue(Service.State state){
        if(state == null
                || (!state.equals(Service.State.CREATED)
                        && !state.equals(Service.State.STARTED))){
            throw new IllegalArgumentException("Illegal initState : " + state);
        }
        initStateValue = state;
        initState = initStateValue.toString();
    }
    
    /**
     * &lt;service&gt;vf̎qvf&lt;optional-config&gt;vf擾B<p>
     *
     * @return qvf&lt;optional-config&gt;vf
     */
    public Element getOptionalConfig(){
        return optionalConfig;
    }
    
    /**
     * &lt;service&gt;vf̎qvf&lt;optional-config&gt;vfݒ肷B<p>
     *
     * @param option qvf&lt;optional-config&gt;vf
     */
    public void setOptionalConfig(Element option){
        optionalConfig = option;
    }
    
    /**
     * &lt;service&gt;vf̎qvf&lt;depends&gt;vf\{@link ServiceMetaData.DependsMetaData}̃Xg擾B<p>
     *
     * @return qvf&lt;depends&gt;vf\DependsMetaDatãXg
     */
    public List<DependsMetaData> getDepends(){
        return depends;
    }
    
    /**
     * &lt;service&gt;vf̎qvf&lt;depends&gt;vf\{@link ServiceMetaData.DependsMetaData}ǉB<p>
     *
     * @param depends qvf&lt;depends&gt;vf\DependsMetaData
     */
    public void addDepends(DependsMetaData depends){
        this.depends.add(depends);
    }
    
    /**
     * &lt;service&gt;vf̎qvf&lt;depends&gt;vf\{@link ServiceMetaData.DependsMetaData}폜B<p>
     *
     * @param depends qvf&lt;depends&gt;vf\DependsMetaData
     */
    public void removeDepends(DependsMetaData depends){
        this.depends.remove(depends);
    }
    
    /**
     * &lt;service&gt;vf̎qvf&lt;depends&gt;vf\{@link ServiceMetaData.DependsMetaData}Sč폜B<p>
     */
    public void clearDepends(){
        this.depends.clear();
    }
    
    /**
     * &lt;service&gt;vfŒ`ꂽT[rXo^T[rX}l[W`&lt;manager&gt;vf\ManagerMetaData擾B<p>
     *
     * @return &lt;service&gt;vfŒ`ꂽT[rXo^T[rX}l[W`&lt;manager&gt;vf\ManagerMetaData
     */
    public ManagerMetaData getManager(){
        return manager;
    }
    
    /**
     * &lt;service&gt;vfŒ`ꂽT[rXo^T[rX}l[W`&lt;manager&gt;vf\ManagerMetaDataݒ肷B<p>
     *
     * @param manager &lt;service&gt;vfŒ`ꂽT[rXo^T[rX}l[W`&lt;manager&gt;vf\ManagerMetaData
     */
    public void setManager(ManagerMetaData manager){
        this.manager = manager;
    }
    
    /**
     * &lt;depends&gt;vf𐶐B<p>
     *
     * @param managerName ˑT[rXo^Ă}l[W
     * @param serviceName ˑT[rX̃T[rX
     * @return &lt;depends&gt;vf̃^f[^
     */
    public DependsMetaData createDependsMetaData(
        String managerName,
        String serviceName
    ){
        return new DependsMetaData(
            managerName,
            serviceName
        );
    }
    
    /**
     * vfserviceł鎖`FbNB<p>
     *
     * @param element servicevf
     * @exception DeploymentException vfservicełȂꍇ
     */
    protected void checkTagName(Element element) throws DeploymentException{
        if(!element.getTagName().equals(SERVICE_TAG_NAME)){
            throw new DeploymentException(
                "Tag must be " + SERVICE_TAG_NAME + " : "
                 + element.getTagName()
            );
        }
    }
    
    /**
     * ev[gKp^f[^𐶐B<p>
     *
     * @param manager ServiceManager
     * @return ev[gKp̃^f[^
     * @exception ServiceNotFoundException ev[gƂĎw肳ĂT[rX^f[^Ȃꍇ
     */
    public ServiceMetaData applyTemplate(ServiceManager manager) throws ServiceNotFoundException{
        if(getTemplateName() == null){
            return this;
        }
        ServiceNameEditor editor = new ServiceNameEditor();
        if(getManager() != null){
            editor.setServiceManagerName(getManager().getName());
        }
        editor.setAsText(getTemplateName());
        ServiceName temlateServiceName = (ServiceName)editor.getValue();
        ServiceMetaData templateData = null;
        if(manager != null && manager.getServiceName().equals(temlateServiceName.getServiceManagerName())){
            templateData = manager.getServiceMetaData(
                temlateServiceName.getServiceName()
            );
        }
        if(templateData == null){
            templateData = ServiceManagerFactory.getServiceMetaData(temlateServiceName);
        }
        templateData = templateData.applyTemplate(manager);
        
        ServiceMetaData result = (ServiceMetaData)clone();
        if(result.constructor == null){
            result.constructor = templateData.constructor;
        }
        if(templateData.fields.size() != 0){
            Iterator<Map.Entry<String, FieldMetaData>> entries = templateData.fields.entrySet().iterator();
            while(entries.hasNext()){
                Map.Entry<String, FieldMetaData> entry = entries.next();
                if(!result.fields.containsKey(entry.getKey())){
                    result.fields.put(entry.getKey(), entry.getValue());
                }
            }
        }
        if(templateData.attributes.size() != 0){
            Iterator<Map.Entry<String, AttributeMetaData>> entries = templateData.attributes.entrySet().iterator();
            while(entries.hasNext()){
                Map.Entry<String, AttributeMetaData> entry = entries.next();
                if(!result.attributes.containsKey(entry.getKey())){
                    result.attributes.put(entry.getKey(), entry.getValue());
                }
            }
        }
        if(templateData.invokes.size() != 0){
            result.invokes.addAll(templateData.invokes);
        }
        if(result.optionalConfig == null){
            result.optionalConfig = templateData.optionalConfig;
        }
        if(templateData.depends.size() != 0){
            result.depends.addAll(templateData.depends);
        }
        return result;
    }
    
    /**
     * &lt;service&gt;vfElementp[XāAg̏Ayюqvf̃^f[^̐sB<p>
     *
     * @param element &lt;service&gt;vfElement
     * @exception DeploymentException &lt;service&gt;vf̉́Ǎʂɂ郁^f[^̐Ɏsꍇ
     */
    @Override
    public void importXML(Element element) throws DeploymentException{
        
        name = getUniqueAttribute(element, NAME_ATTRIBUTE_NAME);
        if(name != null){
            // VXevpeB̒u
            name = Utility.replaceSystemProperty(name, false);
            if(getServiceLoader() != null){
                // T[rX[_\vpeB̒u
                name = Utility.replaceServiceLoderConfig(
                    getServiceLoader().getConfig(),
                    name,
                    false
                );
            }
            if(manager != null){
                // }l[WvpeB̒u
                name = Utility.replaceManagerProperty(manager, name, false);
            }
            // T[ovpeB̒u
            name = Utility.replaceServerProperty(name, true);
        }
        
        String tmpInitState
             = getOptionalAttribute(element, INIT_STATE_ATTRIBUTE_NAME);
        if(tmpInitState != null){
            // VXevpeB̒u
            tmpInitState = Utility.replaceSystemProperty(tmpInitState, false);
            if(getServiceLoader() != null){
                // T[rX[_\vpeB̒u
                tmpInitState = Utility.replaceServiceLoderConfig(
                    getServiceLoader().getConfig(),
                    tmpInitState,
                    false
                );
            }
            if(manager != null){
                // }l[WvpeB̒u
                tmpInitState = Utility.replaceManagerProperty(manager, tmpInitState, false);
            }
            // T[ovpeB̒u
            tmpInitState = Utility.replaceServerProperty(tmpInitState, true);
            try{
                setInitState(tmpInitState);
            }catch(IllegalArgumentException e){
                throw new DeploymentException("Illegal init state. serviceName=" + name + ", initState=" + tmpInitState, e);
            }
        }
        
        String tmpInstance = getOptionalAttribute(
            element,
            INSTANCE_ATTRIBUTE_NAME,
            INSTANCE_TYPE_SINGLETON
        );
        try{
            setInstance(tmpInstance);
        }catch(IllegalArgumentException e){
            throw new DeploymentException("Illegal instance. serviceName=" + name + ", instance=" + tmpInstance, e);
        }
        final String management = getOptionalAttribute(
            element,
            MANAGEMENT_ATTRIBUTE_NAME
        );
        if(management != null){
            isManagement = Boolean.valueOf(management).booleanValue();
        }
        final String createTemplate = getOptionalAttribute(
            element,
            CREATE_TEMPLATE_ATTRIBUTE_NAME
        );
        if(createTemplate != null){
            isCreateTemplate = Boolean.valueOf(createTemplate).booleanValue();
        }
        
        template = getOptionalAttribute(
            element,
            TEMPLATE_ATTRIBUTE_NAME
        );
        
        super.importXML(element);
    }
    
    protected void importXMLInner(Element element, IfDefMetaData ifdefData) throws DeploymentException{
        
        super.importXMLInner(element, ifdefData);
        
        final boolean ifdefMatch
            = ifdefData == null ? true : ifdefData.isMatch();
        
        final Element optionalConfig = getOptionalChild(element, OPT_CONF_TAG_NAME);
        if(ifdefMatch){
            this.optionalConfig = optionalConfig;
        }
        
        final Iterator<Element> dependsElements = getChildrenByTagName(
            element,
            DEPENDS_TAG_NAME
        );
        while(dependsElements.hasNext()){
            final Element dependsElement = (Element)dependsElements.next();
            final DependsMetaData dependsData = new DependsMetaData(manager == null ? null : manager.getName());
            if(ifdefData != null){
                dependsData.setIfDefMetaData(ifdefData);
                ifdefData.addChild(dependsData);
            }
            dependsData.importXML(dependsElement);
            if(ifdefMatch){
                depends.add(dependsData);
            }
        }
    }
    
    @Override
    public StringBuilder toXML(StringBuilder buf){
        appendComment(buf);
        buf.append('<').append(SERVICE_TAG_NAME);
        if(name != null){
            buf.append(' ').append(NAME_ATTRIBUTE_NAME)
                .append("=\"").append(name).append("\"");
        }
        if(code != null){
            buf.append(' ').append(CODE_ATTRIBUTE_NAME)
                .append("=\"").append(code).append("\"");
        }
        if(initState != null
             && Service.State.STARTED != initStateValue){
            buf.append(' ').append(INIT_STATE_ATTRIBUTE_NAME)
                .append("=\"").append(initState).append("\"");
        }
        if(instance != null && !INSTANCE_TYPE_SINGLETON.equals(instance)){
            buf.append(' ').append(INSTANCE_ATTRIBUTE_NAME)
                .append("=\"").append(instance).append("\"");
        }
        if(isManagement){
            buf.append(' ').append(MANAGEMENT_ATTRIBUTE_NAME)
                .append("=\"").append(isManagement).append("\"");
        }
        if(!isCreateTemplate){
            buf.append(' ').append(CREATE_TEMPLATE_ATTRIBUTE_NAME)
                .append("=\"").append(isCreateTemplate).append("\"");
        }
        
        if(constructor == null && fields.size() == 0
             && attributes.size() == 0 && invokes.size() == 0
             && depends.size() == 0
             && (ifDefMetaDataList == null || ifDefMetaDataList.size() == 0)
        ){
            buf.append("/>");
        }else{
            buf.append('>');
            if(constructor != null && constructor.getIfDefMetaData() == null){
                buf.append(LINE_SEPARATOR);
                buf.append(
                    addIndent(constructor.toXML(new StringBuilder()))
                );
            }
            if(fields.size() != 0){
                buf.append(LINE_SEPARATOR);
                final Iterator<FieldMetaData> datas = fields.values().iterator();
                while(datas.hasNext()){
                    MetaData data = (MetaData)datas.next();
                    if(data.getIfDefMetaData() != null){
                        continue;
                    }
                    buf.append(
                        addIndent(data.toXML(new StringBuilder()))
                    );
                    if(datas.hasNext()){
                        buf.append(LINE_SEPARATOR);
                    }
                }
            }
            if(attributes.size() != 0){
                buf.append(LINE_SEPARATOR);
                final Iterator<AttributeMetaData> datas = attributes.values().iterator();
                while(datas.hasNext()){
                    MetaData data = (MetaData)datas.next();
                    if(data.getIfDefMetaData() != null){
                        continue;
                    }
                    buf.append(
                        addIndent(data.toXML(new StringBuilder()))
                    );
                    if(datas.hasNext()){
                        buf.append(LINE_SEPARATOR);
                    }
                }
            }
            if(invokes.size() != 0){
                buf.append(LINE_SEPARATOR);
                final Iterator<InvokeMetaData> datas = invokes.iterator();
                while(datas.hasNext()){
                    MetaData data = (MetaData)datas.next();
                    if(data.getIfDefMetaData() != null){
                        continue;
                    }
                    buf.append(
                        addIndent(data.toXML(new StringBuilder()))
                    );
                    if(datas.hasNext()){
                        buf.append(LINE_SEPARATOR);
                    }
                }
            }
            if(depends.size() != 0){
                buf.append(LINE_SEPARATOR);
                final Iterator<DependsMetaData> datas = depends.iterator();
                while(datas.hasNext()){
                    DependsMetaData data = datas.next();
                    if(data.getIfDefMetaData() != null){
                        continue;
                    }
                    buf.append(
                        addIndent(data.toXML(new StringBuilder()))
                    );
                    if(datas.hasNext()){
                        buf.append(LINE_SEPARATOR);
                    }
                }
            }
            buf.append(LINE_SEPARATOR);
            if(ifDefMetaDataList != null && ifDefMetaDataList.size() != 0){
                for(int i = 0, imax = ifDefMetaDataList.size(); i < imax; i++){
                    IfDefMetaData ifdefData = (IfDefMetaData)ifDefMetaDataList.get(i);
                    buf.append(
                        addIndent(ifdefData.toXML(new StringBuilder()))
                    );
                    buf.append(LINE_SEPARATOR);
                }
            }
            buf.append("</").append(SERVICE_TAG_NAME).append('>');
        }
        return buf;
    }
    
    /**
     * ̃CX^X̕𐶐B<p>
     *
     * @return ̃CX^X̕
     */
    @Override
    public Object clone(){
        ServiceMetaData clone = (ServiceMetaData)super.clone();
        clone.depends = new ArrayList<DependsMetaData>(depends);
        return clone;
    }
    
    /**
     * ̃CX^X̕\擾B<p>
     *
     * @return \
     */
    @Override
    public String toString(){
        final StringBuilder buf = new StringBuilder();
        buf.append(getClass().getName());
        buf.append('@');
        buf.append(hashCode());
        buf.append('{');
        if(getName() != null){
            buf.append(NAME_ATTRIBUTE_NAME);
            buf.append('=');
            buf.append(getName());
            buf.append(',');
        }
        buf.append(CODE_ATTRIBUTE_NAME);
        buf.append('=');
        buf.append(getCode());
        buf.append('}');
        return buf.toString();
    }
    
    /**
     * ˑ֌W`&lt;depends&gt;vf^f[^B<p>
     * T[rX`t@C&lt;depends&gt;vfɋLqꂽei[郁^f[^ReiłB<p>
     *
     * @author M.Takata
     * @see <a href="nimbus-service_1_0.dtd">T[rX`t@CDTD</a>
     */
    public class DependsMetaData extends ServiceNameMetaData
     implements Serializable{
        
        private static final long serialVersionUID = -2867707582371947233L;
        
        private ServiceMetaData serviceData;
        
        /**
         * w肳ꂽT[rX}l[Wɓo^ꂽT[rXɈˑ鎖\ˑ֌W`̃CX^X𐶐B<p>
         *
         * @param manager ˑT[rXo^T[rX}l[W
         */
        public DependsMetaData(String manager){
            super(ServiceMetaData.this, manager);
        }
        
        /**
         * w肳ꂽT[rXɈˑ鎖\ˑ֌W`̃CX^X𐶐B<p>
         *
         * @param manager ˑT[rXo^T[rX}l[W
         * @param service ˑT[rX
         */
        public DependsMetaData(String manager, String service){
            super(ServiceMetaData.this, DEPENDS_TAG_NAME, manager, service);
        }
        
        /**
         * &lt;depends&gt;vf̎qvf&lt;service&gt;vf\ServiceMetaData擾B<p>
         *
         * @return &lt;service&gt;vf\ServiceMetaData
         */
        public ServiceMetaData getService(){
            return serviceData;
        }
        
        /**
         * &lt;depends&gt;vfElementp[XāAg̏Ayюqvf̃^f[^̐sB<p>
         *
         * @param element &lt;depends&gt;vfElement
         * @exception DeploymentException &lt;depends&gt;vf̉́Ǎʂɂ郁^f[^̐Ɏsꍇ
         */
        @Override
        public void importXML(Element element) throws DeploymentException{
            
            if(!element.getTagName().equals(DEPENDS_TAG_NAME)){
                throw new DeploymentException(
                    "Tag must be " + DEPENDS_TAG_NAME + " : "
                     + element.getTagName()
                );
            }
            tagName = element.getTagName();
            final Element serviceElement = getOptionalChild(
                element,
                SERVICE_TAG_NAME
            );
            final boolean ifdefMatch
                = DependsMetaData.this.getIfDefMetaData() == null ? true : DependsMetaData.this.getIfDefMetaData().isMatch();
            if(serviceElement == null){
                final ServiceMetaData parent = (ServiceMetaData)getParent();
                if(parent != null && parent.getManager() != null){
                    setManagerName(parent.getManager().getName());
                }
                super.importXML(element);
            }else{
                final ServiceMetaData serviceData
                     = new ServiceMetaData(myLoader, this, manager);
                serviceData.importXML(serviceElement);
                serviceData.setIfDefMetaData(DependsMetaData.this.getIfDefMetaData());
                setServiceName(serviceData.getName());
                if(ifdefMatch){
                    this.serviceData = serviceData;
                    setManagerName(manager.getName());
                }
            }
        }
    }
}
