/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.io;

import java.io.*;
import java.util.List;

import jp.ossc.nimbus.util.converter.PaddingStringConverter;

/**
 * FLV`WriterNXB<p>
 * <pre>
 * import java.io.*;
 * import jp.ossc.nimbus.io.FLVWriter;
 *
 * FileOutputStream fos = new FileOutputStream("sample.csv");
 * OutputStreamWriter osw = new OutputStreamWriter(fos, "UTF-8");
 * FLVWriter writer = new FLVWriter(
 *     osw,
 *     new PaddingStringConverter[]{
 *         new PaddingStringConverter(10),
 *         new PaddingStringConverter(12),
 *     }
 * );
 * String[] flv = new String[2];
 * try{
 *     flv[0] = "hoge";
 *     flv[1] = "100";
 *     writer.writeFLV(flv);
 *        :
 * }finally{
 *     writer.close();
 * }
 * </pre>
 * 
 * @author M.Takata
 */
public class FLVWriter extends BufferedWriter{
    
    /**
     * ftHg̉sB<p>
     */
    public static final String DEFAULT_LINE_SEPARATOR
         = System.getProperty("line.separator");
    protected String lineSeparator = DEFAULT_LINE_SEPARATOR;
    protected boolean isAppendElement;
    protected int fieldIndex;
    protected String nullValue;
    protected PaddingStringConverter[] converters;
    
    protected WriterWrapper writerWrapper;
    
    /**
     * ftHg̏݃obt@TCYڑ̃CX^X𐶐B<p>
     */
    public FLVWriter(){
        this(new WriterWrapper());
    }
    
    /**
     * ftHg̏݃obt@TCYCX^X𐶐B<p>
     *
     * @param writer ݐWriter
     */
    public FLVWriter(Writer writer){
        this(writer, (PaddingStringConverter[])null);
    }
    
    /**
     * ftHg̏݃obt@TCYCX^X𐶐B<p>
     *
     * @param writer ݐWriter
     * @param convs tB[hpfBORo[^z
     */
    public FLVWriter(Writer writer, PaddingStringConverter... convs){
        super(writer instanceof WriterWrapper ? writer : new WriterWrapper(writer));
        writerWrapper = (WriterWrapper)lock;
        converters = convs;
    }
    
    /**
     * w肳ꂽ݃obt@TCYڑ̃CX^X𐶐B<p>
     *
     * @param size ݃obt@TCY
     */
    public FLVWriter(int size){
        this(new WriterWrapper(), size);
    }
    
    /**
     * w肳ꂽ݃obt@TCYCX^X𐶐B<p>
     *
     * @param writer ݐWriter
     * @param size ݃obt@TCY
     */
    public FLVWriter(Writer writer, int size){
        this(writer, size, (PaddingStringConverter[])null);
    }
    
    /**
     * w肳ꂽ݃obt@TCYCX^X𐶐B<p>
     *
     * @param writer ݐWriter
     * @param size ݃obt@TCY
     * @param convs tB[hpfBORo[^z
     */
    public FLVWriter(Writer writer, int size, PaddingStringConverter... convs){
        super(writer instanceof WriterWrapper ? writer : new WriterWrapper(writer), size);
        writerWrapper = (WriterWrapper)lock;
        converters = convs;
    }
    
    /**
     * Writerݒ肷B<p>
     *
     * @param writer Writer
     * @exception IOException Writerݒ肳Ăꍇ
     */
    public void setWriter(Writer writer) throws IOException{
        writerWrapper.setWriter(writer);
        isAppendElement = false;
        fieldIndex = 0;
    }
    
    /**
     * etB[h̃pfBOsRo[^ݒ肷B<p>
     *
     * @param convs pfBOsRo[^̔z
     */
    public void setFieldPaddingStringConverter(PaddingStringConverter... convs){
        converters = convs;
    }
    
    /**
     * etB[h̃pfBOsRo[^擾B<p>
     *
     * @return pfBOsRo[^̔z
     */
    public PaddingStringConverter[] getFieldPaddingStringConverter(){
        return converters;
    }
    
    /**
     * sZp[^ݒ肷B<p>
     *
     * @param separator sZp[^
     */
    public void setLineSeparator(String separator){
        this.lineSeparator = separator;
    }
    
    /**
     * sZp[^擾B<p>
     *
     * @return sZp[^
     */
    public String getLineSeparator(){
         return lineSeparator;
    }
    
    /**
     * nullCSVvfƂďƂꍇɁAo͂镶ݒ肷B<p>
     * ݒ肵Ȃꍇ́ANullPointerExceptionB<br>
     *
     * @param value 
     */
    public void setNullValue(String value){
        nullValue = value;
    }
    
    /**
     * nullCSVvfƂďƂꍇɁAo͂镶擾B<p>
     *
     * @return 
     */
    public String getNullValue(){
        return nullValue;
    }
    
    /**
     * s؂蕶ށB<p>
     * s؂蕶́A{@link #getLineSeparator()}gpB<br>
     * 
     * @exception IOException o̓G[ꍇ
     */
    public void newLine() throws IOException{
        super.write(lineSeparator);
        isAppendElement = false;
        fieldIndex = 0;
    }
    
    /**
     * FLVvfށB<p>
     * pfBOōsB<br>
     * 
     * @param element FLVvf
     * @exception IOException o̓G[ꍇ
     */
    public void writeElement(String element) throws IOException{
        if(converters != null && converters.length != 0 && converters[fieldIndex] != null){
            element = converters[fieldIndex].padding(element);
        }
        super.write(element);
        isAppendElement = true;
        fieldIndex++;
    }
    
    /**
     * FLVvfށB<p>
     * 
     * @param element FLVvf
     * @exception IOException o̓G[ꍇ
     * @see #writeElement(String)
     */
    public void writeElement(boolean element) throws IOException{
        writeElement(Boolean.toString(element));
    }
    
    /**
     * FLVvfށB<p>
     * 
     * @param element FLVvf
     * @exception IOException o̓G[ꍇ
     * @see #writeElement(String)
     */
    public void writeElement(byte element) throws IOException{
        writeElement(Byte.toString(element));
    }
    
    /**
     * FLVvfށB<p>
     * 
     * @param element FLVvf
     * @exception IOException o̓G[ꍇ
     * @see #writeElement(String)
     */
    public void writeElement(char element) throws IOException{
        writeElement(Character.toString(element));
    }
    
    /**
     * FLVvfށB<p>
     * 
     * @param element FLVvf
     * @exception IOException o̓G[ꍇ
     * @see #writeElement(String)
     */
    public void writeElement(short element) throws IOException{
        writeElement(Short.toString(element));
    }
    
    /**
     * FLVvfށB<p>
     * 
     * @param element FLVvf
     * @exception IOException o̓G[ꍇ
     * @see #writeElement(String)
     */
    public void writeElement(int element) throws IOException{
        writeElement(Integer.toString(element));
    }
    
    /**
     * FLVvfށB<p>
     * 
     * @param element FLVvf
     * @exception IOException o̓G[ꍇ
     * @see #writeElement(String)
     */
    public void writeElement(long element) throws IOException{
        writeElement(Long.toString(element));
    }
    
    /**
     * FLVvfށB<p>
     * 
     * @param element FLVvf
     * @exception IOException o̓G[ꍇ
     * @see #writeElement(String)
     */
    public void writeElement(float element) throws IOException{
        writeElement(Float.toString(element));
    }
    
    /**
     * FLVvfށB<p>
     * 
     * @param element FLVvf
     * @exception IOException o̓G[ꍇ
     * @see #writeElement(String)
     */
    public void writeElement(double element) throws IOException{
        writeElement(Double.toString(element));
    }
    
    /**
     * FLVvfށB<p>
     * 
     * @param element FLVvf
     * @exception IOException o̓G[ꍇ
     * @see #writeElement(String)
     */
    public void writeElement(Object element) throws IOException{
        writeElement(element == null ? (String)null : element.toString());
    }
    
    /**
     * w肳ꂽzFLVƂďށB<p>
     * s̒ǉAZp[^̒ǉAZp[^܂܂Ăꍇ̃GXP[vA͂ݕł̈͂ݏōsB<br>
     *
     * @param elements FLV`ŏo͂镶z
     * @exception IOException o̓G[ꍇ
     */
    public void writeFLV(String... elements) throws IOException{
        for(int i = 0; i < elements.length; i++){
            writeElement(elements[i]);
        }
        newLine();
    }
    
    /**
     * w肳ꂽzFLVƂďށB<p>
     *
     * @param elements FLV`ŏo͂z
     * @exception IOException o̓G[ꍇ
     * @see #writeFLV(String[])
     */
    public void writeFLV(Object... elements) throws IOException{
        for(int i = 0; i < elements.length; i++){
            writeElement(elements[i]);
        }
        newLine();
    }
    
    /**
     * w肳ꂽXgFLVƂďށB<p>
     * s̒ǉAZp[^̒ǉAZp[^܂܂Ăꍇ̃GXP[vA͂ݕł̈͂ݏōsB<br>
     *
     * @param elements FLV`ŏo͂郊Xg
     * @exception IOException o̓G[ꍇ
     */
    public void writeFLV(List<?> elements) throws IOException{
        for(int i = 0, imax = elements.size(); i < imax; i++){
            writeElement(elements.get(i));
        }
        newLine();
    }
    
    /**
     * ڑ̕𐶐B<p>
     *
     * @return ڑ̕
     */
    public FLVWriter cloneWriter(){
        return cloneWriter(new FLVWriter());
    }
    
    /**
     * ڑ̕𐶐B<p>
     *
     * @param clone ڑ̃CX^X
     * @return ڑ̕
     */
    protected FLVWriter cloneWriter(FLVWriter clone){
        clone.lineSeparator = lineSeparator;
        if(converters != null && converters.length != 0){
            clone.converters = new PaddingStringConverter[converters.length];
            System.arraycopy(converters, 0, clone.converters, 0, converters.length);
        }
        return clone;
    }
    
    private static class WriterWrapper extends Writer{
        
        private Writer realWriter;
        
        public WriterWrapper(){
        }
        
        public WriterWrapper(Writer writer){
            realWriter = writer;
        }
        
        @SuppressWarnings("unused")
        public Writer getWriter(){
            return realWriter;
        }
        
        public void setWriter(Writer writer) throws IOException{
            if(realWriter != null){
                throw new IOException("Writer is already commited.");
            }
            realWriter = writer;
        }
        
        @Override
        public void write(int c) throws IOException{
            if(realWriter == null){
                throw new IOException("Writer is null.");
            }
            realWriter.write(c);
        }
        
        @Override
        public void write(char[] cbuf) throws IOException{
            if(realWriter == null){
                throw new IOException("Writer is null.");
            }
            realWriter.write(cbuf);
        }
        
        @Override
        public void write(char[] cbuf, int off, int len) throws IOException{
            if(realWriter == null){
                throw new IOException("Writer is null.");
            }
            realWriter.write(cbuf, off, len);
        }
        
        @Override
        public void write(String str) throws IOException{
            if(realWriter == null){
                throw new IOException("Writer is null.");
            }
            realWriter.write(str);
        }
        
        @Override
        public void write(String str, int off, int len) throws IOException{
            if(realWriter == null){
                throw new IOException("Writer is null.");
            }
            realWriter.write(str, off, len);
        }
        
        @Override
        public Writer append(CharSequence csq) throws IOException{
            if(realWriter == null){
                throw new IOException("Writer is null.");
            }
            return realWriter.append(csq);
        }
        
        @Override
        public Writer append(CharSequence csq, int off, int len) throws IOException{
            if(realWriter == null){
                throw new IOException("Writer is null.");
            }
            return realWriter.append(csq, off, len);
        }
        
        @Override
        public Writer append(char c) throws IOException{
            if(realWriter == null){
                throw new IOException("Writer is null.");
            }
            return realWriter.append(c);
        }
        
        @Override
        public void flush() throws IOException{
            if(realWriter != null){
                realWriter.flush();
            }
        }
        
        @Override
        public void close() throws IOException{
            if(realWriter != null){
                realWriter.close();
            }
        }
    }
}