/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.service.cache;

import java.util.*;
import java.io.*;

import jp.ossc.nimbus.core.ServiceBase;

/**
 * Least Frequency Used ӂASYT[rXB<p>
 * ȉɁALFUłӂΏۂƂȂLbVIuWFNg肷邠ӂASYT[rX̃T[rX`B<br>
 * <pre>
 * &lt;?xml version="1.0" encoding="Shift_JIS"?&gt;
 * 
 * &lt;nimbus&gt;
 *     
 *     &lt;manager name="Sample"&gt;
 *         
 *         &lt;service name="LFUOverflowAlgorithm"
 *                  code="jp.ossc.nimbus.service.cache.LFUOverflowAlgorithmService"/&gt;
 *         
 *     &lt;/manager&gt;
 *     
 * &lt;/nimbus&gt;
 * </pre>
 *
 * @author M.Takata
 */
public class LFUOverflowAlgorithmService<E> extends ServiceBase
 implements OverflowAlgorithm<E>, CacheRemoveListener<E>, CacheAccessListener<E>,
            java.io.Serializable, LFUOverflowAlgorithmServiceMBean{
    
    private static final long serialVersionUID = -8742917099381213489L;

    private Map<CachedReference<E>, CounterCachedReference<E>> referenceMap;
    private List<CounterCachedReference<E>> referenceList;
    private boolean cachedRatioCompare = false;
    private long ratioUnitTime = 1000;
    
    // LFUOverflowAlgorithmServiceMBeanJavaDoc
    @Override
    public int size(){
        return referenceList == null ? 0 : referenceList.size();
    }
    
    // LFUOverflowAlgorithmServiceMBeanJavaDoc
    @Override
    public int getMaximumReferenceCount(){
        if(referenceMap == null){
            return 0;
        }
        synchronized(referenceMap){
            if(referenceMap.size() == 0){
                return 0;
            }
            Collections.sort(referenceList);
            CounterCachedReference<E> couterRef
                 = referenceList.get(referenceList.size() - 1);
            return couterRef.getCount();
        }
    }
    
    // LFUOverflowAlgorithmServiceMBeanJavaDoc
    @Override
    public int getMinimumReferenceCount(){
        if(referenceMap == null){
            return 0;
        }
        synchronized(referenceMap){
            if(referenceMap.size() == 0){
                return 0;
            }
            Collections.sort(referenceList);
            CounterCachedReference<E> couterRef
                 = referenceList.get(0);
            return couterRef.getCount();
        }
    }
    
    // LFUOverflowAlgorithmServiceMBeanJavaDoc
    @Override
    public String displayReferenceCounts(){
        if(referenceMap == null){
            return "";
        }
        synchronized(referenceMap){
            if(referenceMap.size() == 0){
                return "";
            }
            final StringWriter sw = new StringWriter();
            final PrintWriter pw = new PrintWriter(sw);
            Collections.sort(referenceList);
            CounterCachedReference<E> couterRef
                 = referenceList.get(referenceList.size() - 1);
            final double unitCount = couterRef.getCount() / 100.0d;
            final Iterator<CounterCachedReference<E>> itr = referenceList.iterator();
            while(itr.hasNext()){
                couterRef = itr.next();
                CachedReference<?> ref = couterRef.getCachedReference();
                final int count = couterRef.getCount();
                final int point = (int)(count / unitCount);
                for(int i = 0; i < point; i++){
                    pw.print('*');
                }
                pw.print('@');
                pw.print(Integer.toString(count));
                if(ref instanceof KeyCachedReference<?,?>){
                    pw.print('@');
                    pw.print(((KeyCachedReference<?,?>)ref).getKey());
                }
                pw.println("<br>");
            }
            return sw.toString();
        }
    }
    
    @Override
    public boolean isCachedRatioCompare() {
        return cachedRatioCompare;
    }
    
    @Override
    public void setCachedRatioCompare(boolean cachedRatioCompare) {
        this.cachedRatioCompare = cachedRatioCompare;
    }

    @Override
    public void setRatioUnitTime(long ratioUnitTime) {
        this.ratioUnitTime = ratioUnitTime;
    }
    
    /**
     * T[rX̐sB<p>
     * CX^Xϐ̏sB
     *
     * @exception Exception T[rX̐Ɏsꍇ
     */
    @Override
    public void createService() throws Exception{
        referenceMap = Collections.synchronizedMap(new HashMap<CachedReference<E>, CounterCachedReference<E>>());
        referenceList = Collections.synchronizedList(new ArrayList<CounterCachedReference<E>>());
    }
    
    /**
     * T[rX̔jsB<p>
     * CX^Xϐ̊JsB
     *
     * @exception Exception T[rX̔jɎsꍇ
     */
    @Override
    public void destroyService() throws Exception{
        reset();
        referenceMap = null;
        referenceList = null;
    }
    
    /**
     * LbVQƂǉB<p>
     * œnꂽLbVQƂێBɁA{@link CachedReference#addCacheAccessListener(CacheAccessListener)}ŁA{@link CacheAccessListener}ƂĎgo^B܂A{@link CachedReference#addCacheRemoveListener(CacheRemoveListener)}ŁA{@link CacheRemoveListener}ƂĎgo^B<br>
     *
     * @param ref LbVQ
     */
    @Override
    public void add(CachedReference<E> ref){
        if(referenceMap == null || ref == null){
            return;
        }
        synchronized(referenceMap){
            if(!referenceMap.containsKey(ref)){
                CounterCachedReference<E> counterRef
                     = new CounterCachedReference<E>(ref, cachedRatioCompare, ratioUnitTime);
                referenceMap.put(ref, counterRef);
                referenceList.add(counterRef);
                ref.addCacheAccessListener(this);
                ref.addCacheRemoveListener(this);
            }
        }
    }
    
    /**
     * LbVQƂ폜B<p>
     * œnꂽLbVQƂŕێĂꍇ́AjBɁA{@link CachedReference#removeCacheAccessListener(CacheAccessListener)}ŁA{@link CacheAccessListener}ƂĎgo^B܂A{@link CachedReference#removeCacheRemoveListener(CacheRemoveListener)}ŁA{@link CacheRemoveListener}ƂĎgo^B<br>
     *
     * @param ref LbVQ
     */
    @Override
    public void remove(CachedReference<E> ref){
        if(referenceMap == null || ref == null){
            return;
        }
        synchronized(referenceMap){
            if(referenceMap.containsKey(ref)){
                CounterCachedReference<E> counterRef
                     = referenceMap.remove(ref);
                referenceList.remove(counterRef);
                ref.removeCacheAccessListener(this);
                ref.removeCacheRemoveListener(this);
            }
        }
    }
    
    /**
     * łQƕpxႢLbVQƂӂꂳB<p>
     * {@link #add(CachedReference)}œnꂽLbVQƂ̒AłQƕpxႢLbVQƂAӂLbVQƂƂĕԂB<br>
     *
     * @return łQƕpxႢLbVQ
     */
    @Override
    public CachedReference<E> overflow(){
        if(referenceMap == null){
            return null;
        }
        synchronized(referenceMap){
            if(referenceMap.size() != 0){
                Collections.sort(referenceList);
                CounterCachedReference<E> couterRef
                     = referenceList.get(0);
                referenceList.remove(couterRef);
                referenceMap.remove(couterRef.getCachedReference());
                return couterRef.getCachedReference();
            }
            return null;
        }
    }
    
    /**
     * łQƕpxႢLbVQƂӂꂳB<p>
     * {@link #add(CachedReference)}œnꂽLbVQƂ̒AłQƕpxႢLbVQƂAӂLbVQƂƂĕԂB<br>
     *
     * @param size ӂꐔ
     * @return łQƕpxႢLbVQ
     */
    @SuppressWarnings("unchecked")
    public CachedReference<E>[] overflow(int size){
        if(referenceMap == null || referenceMap.size() == 0){
            return null;
        }
        synchronized(referenceMap){
            if(referenceMap.size() != 0){
                final CachedReference<E>[] result = new CachedReference[Math.min(referenceMap.size(), size)];
                Collections.sort(referenceList);
                for(int i = 0; i < result.length; i++){
                    CounterCachedReference<E> couterRef
                         = referenceList.remove(0);
                    referenceMap.remove(couterRef.getCachedReference());
                    result[i] = couterRef.getCachedReference();
                }
                return result;
            }
            return null;
        }
    }
    
    /**
     * ӂASYs邽߂ɕێĂB<p>
     * {@link #add(CachedReference)}œnꂽLbVQƂSĔjB<br>
     */
    @Override
    public void reset(){
        if(referenceMap != null){
            referenceMap.clear();
            referenceList.clear();
        }
    }
    
    /**
     * LbV폜ꂽLbVQƂ̒ʒm󂯂B<p>
     * {@link #remove(CachedReference)}ĂяoB<br>
     *
     * @param ref LbV폜ꂽLbVQ
     */
    @Override
    public void removed(CachedReference<E> ref){
        remove(ref);
    }
    
    /**
     * QƂꂽLbVQƂ̒ʒm󂯂B<p>
     * ŕێĂLbVQƂ̃XgɊ܂܂ꍇ́AQƉ񐔂𑝉B<br>
     *
     * @param ref QƂꂽLbVQ
     */
    @Override
    public void accessed(CachedReference<E> ref){
        if(referenceMap == null){
            return;
        }
        synchronized(referenceMap){
            if(referenceMap != null && referenceMap.containsKey(ref)){
                CounterCachedReference<E> counterRef
                     = referenceMap.get(ref);
                if(!counterRef.increment()){
                    for(CounterCachedReference<E> cref : referenceMap.values()){
                        cref.offset();
                    }
                    counterRef.increment();
                }
            }
        }
    }
    
    private static class CounterCachedReference<E>
     implements java.io.Serializable, Comparable<CounterCachedReference<E>>{
        
        private static final long serialVersionUID = -5670267780842863519L;
        
        private CachedReference<E> reference;
        private int count = 1;
        private long lastAccessTime;
        private boolean cachedRatioCompare = false;
        private long ratioUnitTime;
        
        public CounterCachedReference(CachedReference<E> ref, boolean cachedRatioCompare, long ratioUnitTime){
            reference = ref;
            lastAccessTime = System.currentTimeMillis();
            this.cachedRatioCompare = cachedRatioCompare;
            this.ratioUnitTime = ratioUnitTime;
        }
        public CachedReference<E> getCachedReference(){
            return reference;
        }
        public void offset(){
            count/=1000;
        }
        public boolean increment(){
            if(count == Integer.MAX_VALUE){
                return false;
            }
            count++;
            lastAccessTime = System.currentTimeMillis();
            return true;
        }
        public int getCount(){
            return count;
        }
        public long getLastAccessTime(){
            return lastAccessTime;
        }

        @Override
        public int compareTo(CounterCachedReference<E> arg0) {
            if(arg0 == null){
                return 1;
            }
            if(arg0 == this){
                return 0;
            }
            CounterCachedReference<E> comp = arg0;
            
            if(cachedRatioCompare){
                long currentTime = System.currentTimeMillis();
                long currentDiffTime = currentTime - getLastAccessTime();
                long compDifftime = currentTime - comp.getLastAccessTime();
                
                // LbVƂ̍ꍇ
                if(currentDiffTime <= 0){
                    // LbVƂ̍ꍇAJEgŔr
                    if(compDifftime <= 0){
                        if(getCount() == comp.getCount()){
                            if(reference.equals(comp.getCachedReference())){
                                return 0;
                            }else{
                                return hashCode() > comp.hashCode() ? -1 : 1;
                            }
                        }else if(getCount() > comp.getCount()){
                            return 1;
                        }else{
                            return -1;
                        }
                    }
                    // JEg݂ꍇ͐
                    else if(getCount() > 0){
                            return 1;
                    }
                    else{
                        return 1;
                    }
                }
                // LbVƂ݂̍ꍇ́AŔr
                else{
                    double ratio = getRatio(this, currentDiffTime);
                    double compRatio = getRatio(comp, compDifftime);
                    
                    if(ratio == compRatio){
                        if(reference.equals(comp.getCachedReference())){
                            return 0;
                        }else{
                            return hashCode() > comp.hashCode() ? -1 : 1;
                        }
                    }else if(ratio > compRatio){
                        return 1;
                    }else{
                        return -1;
                    }
                }
            }else{
                
                if(comp.getCount() == getCount()){
                    if(reference.equals(comp.getCachedReference())){
                        return 0;
                    }else{
                        return hashCode() > comp.hashCode() ? -1 : 1;
                    }
                }else if(comp.getCount() > getCount()){
                    return -1;
                }else{
                    return 1;
                }
            }
        }
        
        private double getRatio(CounterCachedReference<E> target, long diffTime){
            
            return (double)target.getCount() / Math.max(1.0, ((double)diffTime / (double)ratioUnitTime));
        }
     }
}
