/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.service.journal.editor;

import java.util.Enumeration;
import java.util.HashSet;
import java.util.Set;
import java.util.Date;
import java.util.Stack;

import javax.servlet.http.HttpSession;

import jp.ossc.nimbus.service.journal.EditorFinder;

/**
 * {@link HttpSession}JSON`ɕҏWW[iGfB^[B<p>
 *
 * @author M.Takata
 */
public class HttpSessionJSONJournalEditorService
 extends JSONJournalEditorService
 implements HttpSessionJSONJournalEditorServiceMBean{
    
    private static final long serialVersionUID = 8220048416985842970L;
    
    protected String[] secretAttributes;
    protected Set<String> secretAttributeSet;
    protected String[] enabledAttributes;
    protected Set<String> enabledAttributeSet;
    protected String[] disabledAttributes;
    protected Set<String> disabledAttributeSet;
    
    @Override
    public void setSecretAttributes(String[] names){
        secretAttributes = names;
    }
    
    @Override
    public String[] getSecretAttributes(){
        return secretAttributes;
    }
    
    @Override
    public void setEnabledAttributes(String[] names){
        enabledAttributes = names;
    }
    
    @Override
    public String[] getEnabledAttributes(){
        return enabledAttributes;
    }
    
    @Override
    public void setDisabledAttributes(String[] names){
        disabledAttributes = names;
    }
    
    @Override
    public String[] getDisabledAttributes(){
        return disabledAttributes;
    }
    
    @Override
    public void startService() throws Exception{
        if(secretAttributes != null && secretAttributes.length != 0){
            secretAttributeSet = new HashSet<String>(secretAttributes.length);
            for(String attr : secretAttributes){
                secretAttributeSet.add(attr);
            }
        }
        if(enabledAttributes != null && enabledAttributes.length != 0){
            enabledAttributeSet = new HashSet<String>(enabledAttributes.length);
            for(String attr : enabledAttributes){
                enabledAttributeSet.add(attr);
            }
        }
        if(disabledAttributes != null && disabledAttributes.length != 0){
            disabledAttributeSet = new HashSet<String>(disabledAttributes.length);
            for(String attr : disabledAttributes){
                disabledAttributeSet.add(attr);
            }
        }
    }
    
    protected boolean isOutputAttribute(String name){
        if(name != null
            && disabledAttributeSet != null
            && disabledAttributeSet.contains(name)
        ){
            return false;
        }
        if(name != null
            && enabledAttributeSet != null
            && !enabledAttributeSet.contains(name)
        ){
            return false;
        }
        return true;
    }
    
    protected boolean isSecretAttribute(String name){
        return name != null && secretAttributeSet != null && secretAttributeSet.contains(name);
    }
    
    @Override
    protected StringBuilder appendUnknownValue(StringBuilder buf, EditorFinder finder, Class<?> type, Object value, Stack<Object> stack){
        if(!(value instanceof HttpSession)){
            return super.appendUnknownValue(buf, finder, type, value, stack);
        }
        final HttpSession session = (HttpSession)value;
        
        buf.append(OBJECT_ENCLOSURE_START);
        appendHttpSession(buf, finder, session, false, stack);
        buf.append(OBJECT_ENCLOSURE_END);
        return buf;
    }
    
    protected boolean appendHttpSession(StringBuilder buf, EditorFinder finder, HttpSession session, boolean isAppended, Stack<Object> stack){
        isAppended |= appendId(buf, finder, session, isAppended, stack);
        isAppended |= appendCreationTime(buf, finder, session, isAppended, stack);
        isAppended |= appendLastAccessedTime(buf, finder, session, isAppended, stack);
        isAppended |= appendMaxInactiveInterval(buf, finder, session, isAppended, stack);
        isAppended |= appendIsNew(buf, finder, session, isAppended, stack);
        isAppended |= appendAttributes(buf, finder, session, isAppended, stack);
        return isAppended;
    }
    
    protected boolean appendId(StringBuilder buf, EditorFinder finder, HttpSession session, boolean isAppended, Stack<Object> stack){
        if(isOutputProperty(PROPERTY_ID)){
            if(isAppended){
                buf.append(ARRAY_SEPARATOR);
            }
            appendProperty(
                buf,
                finder,
                PROPERTY_ID,
                session.getId(),
                stack
            );
            return true;
        }else{
            return false;
        }
    }
    
    protected boolean appendCreationTime(StringBuilder buf, EditorFinder finder, HttpSession session, boolean isAppended, Stack<Object> stack){
        if(isOutputProperty(PROPERTY_CREATION_TIME)){
            if(isAppended){
                buf.append(ARRAY_SEPARATOR);
            }
            appendProperty(
                buf,
                finder,
                PROPERTY_CREATION_TIME,
                new Date(session.getCreationTime()),
                stack
            );
            return true;
        }else{
            return false;
        }
    }
    
    protected boolean appendLastAccessedTime(StringBuilder buf, EditorFinder finder, HttpSession session, boolean isAppended, Stack<Object> stack){
        if(isOutputProperty(PROPERTY_LAST_ACCESSED_TIME)){
            if(isAppended){
                buf.append(ARRAY_SEPARATOR);
            }
            appendProperty(
                buf,
                finder,
                PROPERTY_LAST_ACCESSED_TIME,
                new Date(session.getLastAccessedTime()),
                stack
            );
            return true;
        }else{
            return false;
        }
    }
    
    protected boolean appendMaxInactiveInterval(StringBuilder buf, EditorFinder finder, HttpSession session, boolean isAppended, Stack<Object> stack){
        if(isOutputProperty(PROPERTY_MAX_INACTIVE_INTERVAL)){
            if(isAppended){
                buf.append(ARRAY_SEPARATOR);
            }
            appendProperty(
                buf,
                finder,
                PROPERTY_MAX_INACTIVE_INTERVAL,
                session.getMaxInactiveInterval(),
                stack
            );
            return true;
        }else{
            return false;
        }
    }
    
    protected boolean appendIsNew(StringBuilder buf, EditorFinder finder, HttpSession session, boolean isAppended, Stack<Object> stack){
        if(isOutputProperty(PROPERTY_IS_NEW)){
            if(isAppended){
                buf.append(ARRAY_SEPARATOR);
            }
            appendProperty(
                buf,
                finder,
                PROPERTY_IS_NEW,
                session.isNew(),
                stack
            );
            return true;
        }else{
            return false;
        }
    }
    
    @SuppressWarnings("unchecked")
    protected boolean appendAttributes(StringBuilder buf, EditorFinder finder, HttpSession session, boolean isAppended, Stack<Object> stack){
        if(isOutputProperty(PROPERTY_ATTRIBUTE)){
            if(isAppended){
                buf.append(ARRAY_SEPARATOR);
            }
            appendName(buf, PROPERTY_ATTRIBUTE);
            buf.append(PROPERTY_SEPARATOR);
            buf.append(OBJECT_ENCLOSURE_START);
            final Enumeration<String> names = (Enumeration<String>)session.getAttributeNames();
            boolean isOutput = false;
            while(names.hasMoreElements()){
                String name = names.nextElement();
                if(!isOutputAttribute(name)){
                    continue;
                }
                if(isOutput){
                    buf.append(ARRAY_SEPARATOR);
                }
                isOutput = true;
                appendName(buf, name);
                buf.append(PROPERTY_SEPARATOR);
                if(isSecretAttribute(name)){
                    appendValue(buf, finder, null, secretString, stack);
                }else{
                    appendValue(buf, finder, null, session.getAttribute(name), stack);
                }
            }
            buf.append(OBJECT_ENCLOSURE_END);
            return true;
        }else{
            return false;
        }
    }
}