/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.core;

import static org.testng.Assert.*;

import java.util.ArrayList;
import java.util.List;

import org.testng.annotations.Test;
import org.testng.annotations.DataProvider;
import org.w3c.dom.Element;

import static jp.ossc.nimbus.TestUtility.*;

/**
 * {@link PropertyEditorsMetaData}eXgP[XB<p>
 * 
 * @author M.Takata
 */
public class PropertyEditorsMetaDataTest{

    @DataProvider(name = "PropertyEditorsMetaData")
    public Object[][] dataPropertyEditorsMetaData(){
        return new Object[][]{
            {null},
            {new NimbusMetaData(null, null)},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.PropertyEditorsMetaData#PropertyEditorsMetaData(jp.ossc.nimbus.core.MetaData)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*"},
          dataProvider = "PropertyEditorsMetaData")
    public void testPropertyEditorsMetaData(MetaData parent){
        PropertyEditorsMetaData metaData = new PropertyEditorsMetaData(parent);
        assertEquals(metaData.getParent(), parent);
    }
    
    @DataProvider(name = "SetPropertyEditor")
    public Object[][] dataSetPropertyEditor(){
        return new Object[][]{
            new Object[]{null},
            new Object[]{new String[][]{{"java.lang.String", "test.StringPropertyEditor"}}},
            new Object[]{new String[][]{{"java.lang.String", "test.StringPropertyEditor"}, {"java.lang.Byte", "test.BytePropertyEditor"}, {"java.lang.Short", "test.ShortPropertyEditor"}, {"java.lang.Character", "test.CharacterPropertyEditor"}, {"java.lang.Integer", "test.IntegerPropertyEditor"}, {"java.lang.Long", "test.LongPropertyEditor"}, {"java.lang.Float", "test.FloatPropertyEditor"}, {"java.lang.Double", "test.DoublePropertyEditor"}}},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.PropertyEditorsMetaData#setPropertyEditor(java.lang.String,java.lang.String)",
                    "jp.ossc.nimbus.core.PropertyEditorsMetaData#getPropertyEditorTypes()",
                    "jp.ossc.nimbus.core.PropertyEditorsMetaData#getPropertyEditor(java.lang.String)",
                    "jp.ossc.nimbus.core.PropertyEditorsMetaData#removePropertyEditor(java.lang.String)",
                    "jp.ossc.nimbus.core.PropertyEditorsMetaData#clearPropertyEditors()",
                    "jp.ossc.nimbus.core.PropertyEditorsMetaData#setPropertyEditorMetaData(jp.ossc.nimbus.core.PropertyEditorMetaData)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.PropertyEditorsMetaData#PropertyEditorsMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)"},
          dataProvider = "SetPropertyEditor")
    public void testSetPropertyEditor(String[][] editors){
        PropertyEditorsMetaData metaData = new PropertyEditorsMetaData(null);
        if(editors == null || editors.length == 0){
            assertTrue(metaData.getPropertyEditorTypes().size() == 0);
        }else{
            for(String[] typeAndEditor : editors){
                metaData.setPropertyEditor(typeAndEditor[0], typeAndEditor[1]);
                assertEquals(metaData.getPropertyEditor(typeAndEditor[0]), typeAndEditor[1]);
            }
            List<String> types = new ArrayList<String>(metaData.getPropertyEditorTypes());
            assertEquals(types.size(), editors.length);
            for(int i = 0, imax = types.size(); i < imax; i+=2){
                metaData.removePropertyEditor(types.get(i));
            }
            types = new ArrayList<String>(metaData.getPropertyEditorTypes());
            int index = 1;
            for(int i = 0; i < types.size(); i++){
                assertEquals(metaData.getPropertyEditor(types.get(i)), editors[index][1]);
                index += 2;
            }
            metaData.clearPropertyEditors();
            assertEquals(metaData.getPropertyEditorTypes().size(), 0);
            for(String[] typeAndEditor : editors){
                PropertyEditorMetaData propData = new PropertyEditorMetaData(metaData);
                propData.setType(typeAndEditor[0]);
                propData.setEditor(typeAndEditor[1]);
                metaData.setPropertyEditorMetaData(propData);
                assertEquals(metaData.getPropertyEditor(typeAndEditor[0]), typeAndEditor[1]);
            }
            assertEquals(metaData.getPropertyEditorTypes().size(), editors.length);
        }
    }

    @DataProvider(name = "ImportXML")
    public Object[][] dataImportXML() throws Exception{
        return new Object[][]{
            {createElement("<property-editors/>"), new String[0][]},
            {
                createElement("<property-editors><property-editor type=\"java.lang.String\">test.StringPropertyEditor</property-editor></property-editors>"),
                new String[][]{{"java.lang.String", "test.StringPropertyEditor"}}
            },
            {
                createElement("<property-editors><property-editor type=\"java.lang.String\">test.StringPropertyEditor</property-editor><property-editor type=\"java.lang.Byte\">test.BytePropertyEditor</property-editor></property-editors>"),
                new String[][]{{"java.lang.String", "test.StringPropertyEditor"}, {"java.lang.Byte", "test.BytePropertyEditor"}}
            },
            {
                createElement("<property-editors><ifdef name=\"test\" value=\"true\"><property-editor type=\"java.lang.String\">test.StringPropertyEditor</property-editor></ifdef></property-editors>"),
                new String[][]{{"java.lang.String", "test.StringPropertyEditor"}}
            },
            {
                createElement("<property-editors><ifdef name=\"test\" value=\"false\"><property-editor type=\"java.lang.String\">test.StringPropertyEditor</property-editor></ifdef></property-editors>"),
                new String[0][]
            },
            {
                createElement("<property-editors><ifdef name=\"test\" value=\"true\"><property-editor type=\"java.lang.String\">test.StringPropertyEditor</property-editor><property-editor type=\"java.lang.Byte\">test.BytePropertyEditor</property-editor></ifdef></property-editors>"),
                new String[][]{{"java.lang.String", "test.StringPropertyEditor"}, {"java.lang.Byte", "test.BytePropertyEditor"}}
            },
            {
                createElement("<property-editors><ifdef name=\"test\" value=\"false\"><property-editor type=\"java.lang.String\">test.StringPropertyEditor</property-editor><property-editor type=\"java.lang.Byte\">test.BytePropertyEditor</property-editor></ifdef></property-editors>"),
                new String[0][]
            },
            {createElement("<property-editors unknown=\"hoge\"/>"), new String[0][]},
            {createElement("<property-editors><unknown/></property-editors>"), new String[0][]},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.PropertyEditorsMetaData#importXML(org.w3c.dom.Element)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.PropertyEditorsMetaData#PropertyEditorsMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.PropertyEditorsMetaData#getPropertyEditorTypes\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.PropertyEditorsMetaData#getPropertyEditor\\(java\\.lang\\.String\\)"},
          dataProvider = "ImportXML")
    public void testImportXML(
        Element element,
        String[][] editors
    ) throws DeploymentException{
        System.setProperty("test", "true");
        try{
            PropertyEditorsMetaData metaData = new PropertyEditorsMetaData(null);
            metaData.importXML(element);
            if(editors == null || editors.length == 0){
                assertEquals(metaData.getPropertyEditorTypes().size(), 0);
            }else{
                assertEquals(metaData.getPropertyEditorTypes().size(), editors.length);
                for(String[] typeAndEditor : editors){
                    assertEquals(metaData.getPropertyEditor(typeAndEditor[0]), typeAndEditor[1]);
                }
            }
        }finally{
            System.clearProperty("test");
        }
    }

    @DataProvider(name = "ImportXMLError")
    public Object[][] dataImportXMLError() throws Exception{
        return new Object[][]{
            {null},
            {createElement("<hoge/>")},
            {createElement("<property-editors><property-editor type=\"java.lang.String\">test.StringPropertyEditor</property-editor><property-editor type=\"java.lang.String\">test.StringPropertyEditor</property-editor></property-editors>")},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.PropertyEditorsMetaData#importXML(org.w3c.dom.Element)",
                    "Error"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.PropertyEditorsMetaData#PropertyEditorsMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)"},
          expectedExceptions = {NullPointerException.class, DeploymentException.class},
          dataProvider = "ImportXMLError")
    public void testImportXMLError(Element element) throws DeploymentException{
        PropertyEditorsMetaData metaData = new PropertyEditorsMetaData(null);
        metaData.importXML(element);
    }

    @DataProvider(name = "ToXML")
    public Object[][] dataToXML() throws Exception{
        return new Object[][]{
            {createElement("<property-editors/>"), "<property-editors>" + System.getProperty("line.separator") + "</property-editors>"},
            {createElement("<property-editors></property-editors>"), "<property-editors>" + System.getProperty("line.separator") + "</property-editors>"},
            {createElement("<property-editors><property-editor type=\"java.lang.String\">test.StringPropertyEditor</property-editor></property-editors>"), "<property-editors>" + System.getProperty("line.separator") + "    <property-editor type=\"java.lang.String\">test.StringPropertyEditor</property-editor>" + System.getProperty("line.separator") + "</property-editors>"},
            {createElement("<property-editors><property-editor type=\"java.lang.String\">test.StringPropertyEditor</property-editor><property-editor type=\"java.lang.Byte\">test.BytePropertyEditor</property-editor></property-editors>"), "<property-editors>" + System.getProperty("line.separator") + "    <property-editor type=\"java.lang.String\">test.StringPropertyEditor</property-editor>" + System.getProperty("line.separator") + "    <property-editor type=\"java.lang.Byte\">test.BytePropertyEditor</property-editor>" + System.getProperty("line.separator") + "</property-editors>"},
            {createElement("<property-editors><ifdef name=\"test\" value=\"true\"><property-editor type=\"java.lang.String\">test.StringPropertyEditor</property-editor></ifdef></property-editors>"), "<property-editors>" + System.getProperty("line.separator") + "    <ifdef name=\"test\" value=\"true\">" + System.getProperty("line.separator") + "        <property-editor type=\"java.lang.String\">test.StringPropertyEditor</property-editor>" + System.getProperty("line.separator") + "    </ifdef>" + System.getProperty("line.separator") + "</property-editors>"},
            {createElement("<property-editors><ifdef name=\"test\" value=\"true\"><property-editor type=\"java.lang.String\">test.StringPropertyEditor</property-editor><property-editor type=\"java.lang.Byte\">test.BytePropertyEditor</property-editor></ifdef></property-editors>"), "<property-editors>" + System.getProperty("line.separator") + "    <ifdef name=\"test\" value=\"true\">" + System.getProperty("line.separator") + "        <property-editor type=\"java.lang.String\">test.StringPropertyEditor</property-editor>" + System.getProperty("line.separator") + "        <property-editor type=\"java.lang.Byte\">test.BytePropertyEditor</property-editor>" + System.getProperty("line.separator") + "    </ifdef>" + System.getProperty("line.separator") + "</property-editors>"},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.PropertyEditorsMetaData#toXML(java.lang.StringBuilder)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.PropertyEditorsMetaData#PropertyEditorsMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.PropertyEditorsMetaData#importXML\\(org\\.w3c\\.dom\\.Element\\)"},
          dataProvider = "ToXML")
    public void testToXML(Element element, String expected) throws DeploymentException{
        PropertyEditorsMetaData metaData = new PropertyEditorsMetaData(null);
        metaData.importXML(element);
        StringBuilder buf = new StringBuilder();
        metaData.toXML(buf);
        assertEquals(buf.toString(), expected);
    }
}
