/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.service.aop.interceptor;

import java.lang.reflect.*;

import jp.ossc.nimbus.core.*;
import jp.ossc.nimbus.service.aop.*;
import jp.ossc.nimbus.service.context.*;
import jp.ossc.nimbus.service.journal.*;
import jp.ossc.nimbus.service.journal.editor.*;

/**
 * \bhW[iC^[Zv^B<p>
 * \bhĂяõW[i擾C^[Zv^łBW[ȉo͂́AʓrW[iT[rX̒`KvłB<br>
 * ̃C^[Zv^ŏo͂W[íA\bhĂяoi{@link MethodCallJournalData}jA\bh߂li{@link MethodReturnJournalData}jA\bhOi{@link MethodThrowJournalData}jłB<br>
 * ȉɁA\bȟĂяoW[iR\[ɏo͂C^[Zv^̃T[rX`B<br>
 * <pre>
 * &lt;?xml version="1.0" encoding="Shift_JIS"?&gt;
 * 
 * &lt;nimbus&gt;
 *     
 *     &lt;manager name="Sample"&gt;
 *         
 *         &lt;service name="MethodJournalInterceptor"
 *                  code="jp.ossc.nimbus.service.aop.interceptor.MethodJournalInterceptorService"&gt;
 *             &lt;attribute name="JournalServiceName"&gt;#Journal&lt;/attribute&gt;
 *             &lt;depends&gt;Journal&lt;/depends&gt;
 *         &lt;/service&gt;
 * &lt;!-- ȉ̓W[iT[rX` --&gt;
 *         &lt;service name="Journal"
 *                  code="jp.ossc.nimbus.service.journal.DefaultJournalService"&gt;
 *             &lt;attribute name="EditorFinderServiceName"&gt;#JournalEditorFinder&lt;/attribute&gt;
 *             &lt;attribute name="WritableElementKey"&gt;Journal for Sample&lt;/attribute&gt;
 *             &lt;attribute name="CategoryServiceNames"&gt;#JournalCategory&lt;/attribute&gt;
 *             &lt;depends&gt;JournalEditorFinder&lt;/depends&gt;
 *             &lt;depends&gt;JournalCategory&lt;/depends&gt;
 *         &lt;/service&gt;
 *         
 *         &lt;service name="JournalCategory"
 *                  code="jp.ossc.nimbus.service.writer.SimpleCategoryService"&gt;
 *             &lt;attribute name="MessageWriterServiceName"&gt;#JournalWriter&lt;/attribute&gt;
 *             &lt;attribute name="WritableRecordFactoryServiceName"&gt;#JournalWritableRecordFactory&lt;/attribute&gt;
 *             &lt;attribute name="CategoryServiceNames"&gt;#JournalCategory&lt;/attribute&gt;
 *             &lt;depends&gt;JournalWriter&lt;/depends&gt;
 *             &lt;depends&gt;JournalWritableRecordFactory&lt;/depends&gt;
 *         &lt;/service&gt;
 *         
 *         &lt;service name="JournalWritableRecordFactory"
 *                  code="jp.ossc.nimbus.service.writer.DefaultWritableRecordFactoryService"&gt;
 *             &lt;attribute name="Format"&gt;%Journal for Sample%&lt;/attribute&gt;
 *         &lt;/service&gt;
 *         
 *         &lt;service name="JournalWriter"
 *                  code="jp.ossc.nimbus.service.writer.ConsoleWriterService"/&gt;
 *         
 *         &lt;service name="JournalEditorFinder"
 *                  code="jp.ossc.nimbus.service.journal.DefaultEditorFinderService"&gt;
 *             &lt;attribute name="EditorProperties"&gt;
 *                 java.lang.Object=#ObjectJournalEditor
 *                 java.lang.Class=#ClassJournalEditor
 *                 java.util.Date=#DateJournalEditor
 *                 jp.ossc.nimbus.service.journal.JournalRecord=#RequestJournalEditor
 *                 jp.ossc.nimbus.service.journal.editor.MethodJournalData=#MethodJournalEditor
 *                 jp.ossc.nimbus.service.journal.editor.MethodCallJournalData=#MethodCallJournalEditor
 *                 jp.ossc.nimbus.service.journal.editor.MethodReturnJournalData=#MethodReturnJournalEditor
 *             &lt;/attribute&gt;
 *             &lt;depends&gt;ObjectJournalEditor&lt;/depends&gt;
 *             &lt;depends&gt;ClassJournalEditor&lt;/depends&gt;
 *             &lt;depends&gt;DateJournalEditor&lt;/depends&gt;
 *             &lt;depends&gt;RequestJournalEditor&lt;/depends&gt;
 *             &lt;depends&gt;MethodJournalEditor&lt;/depends&gt;
 *             &lt;depends&gt;MethodCallJournalEditor&lt;/depends&gt;
 *             &lt;depends&gt;MethodReturnJournalEditor&lt;/depends&gt;
 *         &lt;/service&gt;
 *         
 *         &lt;service name="ObjectJournalEditor"
 *                  code="jp.ossc.nimbus.service.journal.editor.ObjectJournalEditorService"/&gt;
 *         
 *         &lt;service name="ClassJournalEditor"
 *                  code="jp.ossc.nimbus.service.journal.editor.ClassJournalEditorService"&gt;
 *             &lt;attribute name="ShortClassName"&gt;true&lt;/attribute&gt;
 *         &lt;/service&gt;
 *         
 *         &lt;service name="DateJournalEditor"
 *                  code="jp.ossc.nimbus.service.journal.editor.DateJournalEditorService"&gt;
 *             &lt;attribute name="Format"&gt;yyyy/MM/dd HH:mm:ss.SSS&lt;/attribute&gt;
 *         &lt;/service&gt;
 *         
 *         &lt;service name="RequestJournalEditor"
 *                  code="jp.ossc.nimbus.service.journal.editor.JournalRecordJSONJournalEditorService"/&gt;
 *         
 *         &lt;service name="MethodJournalEditor"
 *                  code="jp.ossc.nimbus.service.journal.editor.MethodJournalJSONJournalEditorService"/&gt;
 *         
 *         &lt;service name="MethodCallJournalEditor"
 *                  code="jp.ossc.nimbus.service.journal.editor.MethodCallJournalJSONJournalEditorService"/&gt;
 *         
 *         &lt;service name="MethodReturnJournalEditor"
 *                  code="jp.ossc.nimbus.service.journal.editor.MethodReturnJournalJSONJournalEditorService"/&gt;
 *         
 *     &lt;/manager&gt;
 *     
 * &lt;/nimbus&gt;
 * </pre>
 *
 * @author M.Takata
 * @see Journal
 * @see EditorFinder
 * @see Context
 */
public class MethodJournalInterceptorService extends ServiceBase
 implements Interceptor, MethodJournalInterceptorServiceMBean{
    
    private static final long serialVersionUID = 6121765320688713719L;
    
    private ServiceName threadContextName;
    private Context<Object, Object> threadContext;
    
    private ServiceName journalName;
    private Journal journal;
    
    private ServiceName editorFinderName;
    private EditorFinder editorFinder;
    
    private String requestJournalKey = DEFAULT_REQUEST_JOURNAL_KEY;
    private String methodCallJournalKey = DEFAULT_METHOD_CALL_JOURNAL_KEY;
    private String methodReturnJournalKey = DEFAULT_METHOD_RETURN_JOURNAL_KEY;
    
    private String requestIdKey = ThreadContextKey.REQUEST_ID;
    private boolean isEnabled = true;
    private boolean isBushingCallBlock = false;
    
    protected ThreadLocal<CallStack> callStack;
    
    // MethodJournalInterceptorServiceMBeanJavaDoc
    public void setRequestIdKey(String key){
        requestIdKey = key;
    }
    
    // MethodJournalInterceptorServiceMBeanJavaDoc
    public String getRequestIdKey(){
        return requestIdKey;
    }
    
    // MethodJournalInterceptorServiceMBeanJavaDoc
    public void setThreadContextServiceName(ServiceName name){
        threadContextName = name;
    }
    
    // MethodJournalInterceptorServiceMBeanJavaDoc
    public ServiceName getThreadContextServiceName(){
        return threadContextName;
    }
    
    // MethodJournalInterceptorServiceMBeanJavaDoc
    public void setJournalServiceName(ServiceName name){
        journalName = name;
    }
    
    // MethodJournalInterceptorServiceMBeanJavaDoc
    public ServiceName getJournalServiceName(){
        return journalName;
    }
    
    // MethodJournalInterceptorServiceMBeanJavaDoc
    public void setEditorFinderServiceName(ServiceName name){
        editorFinderName = name;
    }
    
    // MethodJournalInterceptorServiceMBeanJavaDoc
    public ServiceName getEditorFinderServiceName(){
        return editorFinderName;
    }
    
    // MethodJournalInterceptorServiceMBeanJavaDoc
    public void setRequestJournalKey(String key){
        requestJournalKey = key;
    }
    
    // MethodJournalInterceptorServiceMBeanJavaDoc
    public String getRequestJournalKey(){
        return requestJournalKey;
    }
    
    // MethodJournalInterceptorServiceMBeanJavaDoc
    public void setMethodCallJournalKey(String key){
        methodCallJournalKey = key;
    }
    
    // MethodJournalInterceptorServiceMBeanJavaDoc
    public String getMethodCallJournalKey(){
        return methodCallJournalKey;
    }
    
    // MethodJournalInterceptorServiceMBeanJavaDoc
    public void setMethodReturnJournalKey(String key){
        methodReturnJournalKey = key;
    }
    
    // MethodJournalInterceptorServiceMBeanJavaDoc
    public String getMethodReturnJournalKey(){
        return methodReturnJournalKey;
    }
    
    // MethodJournalInterceptorServiceMBeanJavaDoc
    public void setEnabled(boolean enable){
        isEnabled = enable;
    }
    
    // MethodJournalInterceptorServiceMBeanJavaDoc
    public boolean isEnabled(){
        return isEnabled;
    }
    
    // MethodJournalInterceptorServiceMBeanJavaDoc
    public void setBushingCallBlock(boolean isBlock){
        isBushingCallBlock = isBlock;
    }
    
    // MethodJournalInterceptorServiceMBeanJavaDoc
    public boolean isBushingCallBlock(){
        return isBushingCallBlock;
    }
    
    /**
     * W[io͂{@link jp.ossc.nimbus.service.journal.Journal Journal}ݒ肷B<p>
     *
     * @param journal Journal
     */
    public void setJournal(Journal journal) {
        this.journal = journal;
    }
    
    /**
     * W[iJñW[iҏWɎgp{@link jp.ossc.nimbus.service.journal.editorfinder.EditorFinder EditorFinder}ݒ肷B<p>
     *
     * @param editorFinder EditorFinder
     */
    public void setEditorFinder(EditorFinder editorFinder) {
        this.editorFinder = editorFinder;
    }
    
    /**
     * NGXgID擾{@link jp.ossc.nimbus.service.context.Context}ݒ肷B<p>
     *
     * @param context Context
     */
    public void setThreadContext(Context<Object, Object> context) {
        threadContext = context;
    }
    
    /**
     * T[rX̊JnsB<p>
     *
     * @exception Exception w肳ꂽ{@link Journal}Ay{@link EditorFinder}A{@link Context}T[rXȂꍇ
     */
    public void startService() throws Exception{
        if(journalName != null){
            journal = ServiceManagerFactory.getServiceObject(journalName);
        }
        if(editorFinderName != null){
            editorFinder = ServiceManagerFactory
                .getServiceObject(editorFinderName);
        }
        if(threadContextName != null){
            threadContext = ServiceManagerFactory
                .getServiceObject(threadContextName);
        }
        if(isBushingCallBlock){
            callStack = new ThreadLocal<CallStack>(){
                protected CallStack initialValue(){
                    return new CallStack();
                }
            };
        }
    }
    
    /**
     * T[rX̒~sB<p>
     *
     * @exception Exception ~Ɏsꍇ
     */
    public void stopService() throws Exception{
        callStack = null;
    }
    
    /**
     * T[rX̔jsB<p>
     *
     * @exception Exception jɎsꍇ
     */
    public void destroyService(){
        journal = null;
        editorFinder = null;
    }
    
    /**
     * \bhĂяoJñW[io͂āÃC^[Zv^ĂяoA߂ĂƂŁA\bhĂяoĨW[io͂B<p>
     * T[rXJnĂȂꍇ́AW[io͂s킸Ɏ̃C^[Zv^ĂяoB<br>
     *
     * @param context ĂяõReLXg
     * @param chain ̃C^[Zv^Ăяo߂̃`F[
     * @return Ăяoʂ̖߂l
     * @exception Throwable ĂяoŗOꍇA܂͂̃C^[Zv^ŔCӂ̗OꍇBAA{Ăяo鏈throwȂRuntimeExceptionȊO̗OthrowĂAĂяoɂ͓`dȂB
     */
    public Object invoke(
        InvocationContext context,
        InterceptorChain chain
    ) throws Throwable{
        final MethodInvocationContext ctx = (MethodInvocationContext)context;
        
        if(getState() == State.STARTED && isEnabled()
            && (callStack == null
                 || ((CallStack)callStack.get()).stackIndex == 0)){
            Object ret = null;
            try{
                preNext(ctx);
                if(callStack != null){
                    ((CallStack)callStack.get()).stackIndex++;
                }
                ret = chain.invokeNext(ctx);
                postNext(ctx, ret);
            }catch(RuntimeException e){
                throw throwRuntimeException(ctx, e);
            }catch(Exception e){
                throw throwException(ctx, e);
            }catch(Error e){
                throw throwError(ctx, e);
            }finally{
                if(callStack != null){
                    ((CallStack)callStack.get()).stackIndex--;
                }
                finallyNext(ctx, ret);
            }
            return ret;
        }else{
            return chain.invokeNext(ctx);
        }
    }
    
    /**
     * ̃C^[Zv^ĂяoOsB<p>
     * W[iR[hJnB܂A{@link #setThreadContextServiceName(ServiceName)}ŁA{@link Context}T[rXݒ肳Ăꍇ́A{@link #setRequestIdKey(String)}Őݒ肳ꂽL[ContextT[rX烊NGXgID擾āAW[iR[hɐݒi{@link Journal#setRequestId(String)}jB<br>
     * {@link #setMethodCallJournalKey(String)}Őݒ肳ꂽL[ŁA{@link MethodCallJournalData}W[iɏói{@link Journal#addInfo(String, Object)}jB<br>
     *
     * @param context ĂяõReLXg
     * @exception Throwable OɎsꍇ
     */
    protected void preNext(MethodInvocationContext context) throws Throwable{
        if(journal == null){
            return;
        }
        journal.startJournal(requestJournalKey, editorFinder);
        if(threadContext != null && requestIdKey != null){
            journal.setRequestId((String)threadContext.get(requestIdKey));
        }
        final Method method = context.getTargetMethod();
        final MethodCallJournalData data = new MethodCallJournalData(
            context.getTargetObject(),
            method.getDeclaringClass(),
            method.getName(),
            method.getParameterTypes(),
            context.getParameters()
        );
        journal.addInfo(methodCallJournalKey, data);
    }
    
    /**
     * ̃C^[Zv^Ăяo㏈sB<p>
     * {@link #setMethodReturnJournalKey(String)}Őݒ肳ꂽL[ŁA{@link MethodReturnJournalData}W[iɏói{@link Journal#addInfo(String, Object)}jB<br>
     *
     * @param context ĂяõReLXg
     * @param ret Ăяo̖߂l
     * @exception Throwable ㏈Ɏsꍇ
     */
    protected void postNext(MethodInvocationContext context, Object ret)
     throws Throwable{
        if(journal == null){
            return;
        }
        
        final Method method = context.getTargetMethod();
        final MethodReturnJournalData data = new MethodReturnJournalData(
            context.getTargetObject(),
            method.getDeclaringClass(),
            method.getName(),
            method.getParameterTypes(),
            ret
        );
        journal.addInfo(methodReturnJournalKey, data);
    }
    
    /**
     * ̃C^[Zv^ĂяoRuntimeExceptionꍇ̌㏈sB<p>
     * {@link #setMethodReturnJournalKey(String)}Őݒ肳ꂽL[ŁA{@link MethodThrowJournalData}W[iɏói{@link Journal#addInfo(String, Object)}jB<br>
     *
     * @param context ĂяõReLXg
     * @param e ĂяoɔRuntimeException
     * @return Ŏw肳ꂽRuntimeException
     * @exception Throwable ㏈Ɏsꍇ
     */
    protected RuntimeException throwRuntimeException(
        MethodInvocationContext context,
        RuntimeException e
    ) throws Throwable{
        if(journal == null){
            return e;
        }
        final Method method = context.getTargetMethod();
        final MethodThrowJournalData data = new MethodThrowJournalData(
            context.getTargetObject(),
            method.getDeclaringClass(),
            method.getName(),
            method.getParameterTypes(),
            e
        );
        journal.addInfo(methodReturnJournalKey, data);
        return e;
    }
    
    /**
     * ̃C^[Zv^ĂяoRuntimeExceptionȊOExceptionꍇ̌㏈sB<p>
     * {@link #setMethodReturnJournalKey(String)}Őݒ肳ꂽL[ŁA{@link MethodThrowJournalData}W[iɏói{@link Journal#addInfo(String, Object)}jB<br>
     *
     * @param context ĂяõReLXg
     * @param e ĂяoɔException
     * @return Ŏw肳ꂽException
     * @exception Throwable ㏈Ɏsꍇ
     */
    protected Exception throwException(
        MethodInvocationContext context,
        Exception e
    ) throws Throwable{
        if(journal == null){
            return e;
        }
        final Method method = context.getTargetMethod();
        final MethodThrowJournalData data = new MethodThrowJournalData(
            context.getTargetObject(),
            method.getDeclaringClass(),
            method.getName(),
            method.getParameterTypes(),
            e
        );
        journal.addInfo(methodReturnJournalKey, data);
        return e;
    }
    
    /**
     * ̃C^[Zv^ĂяoErrorꍇ̌㏈sB<p>
     * {@link #setMethodReturnJournalKey(String)}Őݒ肳ꂽL[ŁA{@link MethodThrowJournalData}W[iɏói{@link Journal#addInfo(String, Object)}jB<br>
     *
     * @param context ĂяõReLXg
     * @param error ĂяoɔError
     * @return Ŏw肳ꂽError
     * @exception Throwable ㏈Ɏsꍇ
     */
    protected Error throwError(
        MethodInvocationContext context,
        Error error
    ) throws Throwable{
        if(journal == null){
            return error;
        }
        final Method method = context.getTargetMethod();
        final MethodThrowJournalData data = new MethodThrowJournalData(
            context.getTargetObject(),
            method.getDeclaringClass(),
            method.getName(),
            method.getParameterTypes(),
            error
        );
        journal.addInfo(methodReturnJournalKey, data);
        return error;
    }
    
    /**
     * ̃C^[Zv^Ăяofinally߂ł̏sB<p>
     * W[iR[hIB<br>
     *
     * @param context ĂяõReLXg
     * @param ret Ăяo̖߂l
     * @exception Throwable ㏈Ɏsꍇ
     */
    protected void finallyNext(MethodInvocationContext context, Object ret)
     throws Throwable{
        if(journal == null){
            return;
        }
        
        journal.endJournal();
    }
    
    protected static class CallStack{
        public int stackIndex;
    }
}