/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.service.queue;

import jp.ossc.nimbus.core.*;
import jp.ossc.nimbus.daemon.*;
import jp.ossc.nimbus.util.*;

/**
 * QueueHandlerReiT[rXB<p>
 * 
 * @author M.Takata
 */
public class QueueHandlerContainerService<E> extends ServiceBase
 implements QueueHandlerContainer<E>, QueueHandlerContainerServiceMBean{
    
    private static final long serialVersionUID = -6527205946658554031L;
    
    protected ServiceName queueServiceName;
    protected Queue<E> requestQueue;
    protected Daemon[] daemons;
    protected QueueReceiver<E>[] invokers;
    protected int queueHandlerSize = 1;
    protected ServiceName queueHandlerServiceName;
    protected QueueHandler<E> queueHandler;
    protected boolean isDaemonQueueHandler = true;
    
    protected long waitTimeout = -1;
    protected int maxRetryCount = 0;
    protected long retryInterval = 1000;
    protected String handlingErrorMessageId = DEFAULT_HANDLING_ERROR_MESSAGE_ID;
    protected String retryOverErrorMessageId = DEFAULT_RETRY_OVER_ERROR_MESSAGE_ID;
    protected int queueHandlerThreadPriority = -1;
    protected boolean isReleaseQueue = true;
    protected long count = 0;
    protected boolean isQueueHandlerNowaitOnStop;
    protected boolean isSuspend;
    protected SynchronizeMonitor suspendMonitor = new WaitSynchronizeMonitor();
    
    // QueueHandlerContainerServiceMBeanJavaDoc
    public void setQueueServiceName(ServiceName name){
        queueServiceName = name;
    }
    // QueueHandlerContainerServiceMBeanJavaDoc
    public ServiceName getQueueServiceName(){
        return queueServiceName;
    }
    
    // QueueHandlerContainerServiceMBeanJavaDoc
    public void setQueueHandlerServiceName(ServiceName name){
        if(queueHandlerServiceName == null){
            queueHandlerServiceName = name;
            if(daemons != null){
                for(Daemon daemon : daemons){
                    daemon.resume();
                }
            }
        }else{
            queueHandlerServiceName = name;
        }
    }
    // QueueHandlerContainerServiceMBeanJavaDoc
    public ServiceName getQueueHandlerServiceName(){
        return queueHandlerServiceName;
    }
    
    // QueueHandlerContainerServiceMBeanJavaDoc
    public void setQueueHandlerSize(int size){
        queueHandlerSize = size;
    }
    // QueueHandlerContainerServiceMBeanJavaDoc
    public int getQueueHandlerSize(){
        return queueHandlerSize;
    }
    
    // QueueHandlerContainerServiceMBeanJavaDoc
    public void setReleaseQueue(boolean isRelease){
        isReleaseQueue = isRelease;
    }
    // QueueHandlerContainerServiceMBeanJavaDoc
    public boolean isReleaseQueue(){
        return isReleaseQueue;
    }
    
    // QueueHandlerContainerServiceMBeanJavaDoc
    public void setWaitTimeout(long timeout){
        waitTimeout = timeout;
    }
    // QueueHandlerContainerServiceMBeanJavaDoc
    public long getWaitTimeout(){
        return waitTimeout;
    }
    
    // QueueHandlerContainerServiceMBeanJavaDoc
    public void setMaxRetryCount(int count){
        maxRetryCount = count;
    }
    // QueueHandlerContainerServiceMBeanJavaDoc
    public int getMaxRetryCount(){
        return maxRetryCount;
    }
    
    // QueueHandlerContainerServiceMBeanJavaDoc
    public void setRetryInterval(long interval){
        retryInterval = interval;
    }
    // QueueHandlerContainerServiceMBeanJavaDoc
    public long getRetryInterval(){
        return retryInterval;
    }
    
    // QueueHandlerContainerServiceMBeanJavaDoc
    public void setHandlingErrorMessageId(String id){
        handlingErrorMessageId = id;
    }
    // QueueHandlerContainerServiceMBeanJavaDoc
    public String getHandlingErrorMessageId(){
        return handlingErrorMessageId;
    }
    
    // QueueHandlerContainerServiceMBeanJavaDoc
    public void setRetryOverErrorMessageId(String id){
        retryOverErrorMessageId = id;
    }
    // QueueHandlerContainerServiceMBeanJavaDoc
    public String getRetryOverErrorMessageId(){
        return retryOverErrorMessageId;
    }
    
    // QueueHandlerContainerJavaDoc
    public int getActiveQueueHandlerSize(){
        if(invokers == null){
            return 0;
        }
        int count = 0;
        for(QueueReceiver<E> invoker : invokers){
            if(invoker.isActive){
                count++;
            }
        }
        return count;
    }
    
    // QueueHandlerContainerJavaDoc
    public int getStandbyQueueHandlerSize(){
        if(invokers == null){
            return 0;
        }
        int count = 0;
        for(QueueReceiver<E> invoker : invokers){
            if(!invoker.isActive){
                count++;
            }
        }
        return count;
    }
    
    // QueueHandlerContainerServiceMBeanJavaDoc
    public void setDaemonQueueHandler(boolean isDaemon){
        isDaemonQueueHandler = isDaemon;
    }
    
    // QueueHandlerContainerServiceMBeanJavaDoc
    public boolean isDaemonQueueHandler(){
        return isDaemonQueueHandler;
    }
    
    // QueueHandlerContainerServiceMBeanJavaDoc
    public void setQueueHandlerThreadPriority(int newPriority){
        queueHandlerThreadPriority = newPriority;
    }
    // QueueHandlerContainerServiceMBeanJavaDoc
    public int getQueueHandlerThreadPriority(){
        return queueHandlerThreadPriority;
    }
    
    // QueueHandlerContainerServiceMBeanJavaDoc
    public void setQueueHandlerNowaitOnStop(boolean isNowait){
        isQueueHandlerNowaitOnStop = isNowait;
    }
    
    // QueueHandlerContainerServiceMBeanJavaDoc
    public boolean isQueueHandlerNowaitOnStop(){
        return isQueueHandlerNowaitOnStop;
    }
    
    // QueueHandlerContainerServiceMBeanJavaDoc
    public long getAverageHandleProcessTime(){
        if(invokers == null){
            return 0;
        }
        int time = 0;
        if(invokers.length != 0){
            for(QueueReceiver<E> invoker : invokers){
                time += invoker.getAverageReceiveProcessTime();
            }
            time /= invokers.length;
        }
        return time;
    }
    
    /**
     * T[rX̊JnsB<p>
     *
     * @exception Exception QueueT[rX̎擾Ɏsꍇ
     */
    @SuppressWarnings("unchecked")
    public void startService() throws Exception{
        if(getQueueServiceName() != null){
            setQueueService(
                (Queue<E>)ServiceManagerFactory.getServiceObject(queueServiceName)
            );
        }
        if(getQueueService() == null){
            if(getQueueHandler() == null){
                throw new IllegalArgumentException("QueueHandler is null.");
            }
        }else{
            
            // L[tJn
            getQueueService().accept();
            
            // f[N
            if(queueHandlerSize < 0){
                throw new IllegalArgumentException("queueHandlerSize < 0.");
            }
            invokers = new QueueReceiver[queueHandlerSize];
            daemons = new Daemon[queueHandlerSize];
            for(int i = 0; i < queueHandlerSize; i++){
                invokers[i] = new QueueReceiver<E>();
                invokers[i].handler = getQueueHandler();
                
                daemons[i] = new Daemon(invokers[i]);
                daemons[i].setDaemon(isDaemonQueueHandler);
                daemons[i].setName(getServiceNameObject() + " QueueReceiver" + (i + 1));
                if(queueHandlerThreadPriority > 0){
                    daemons[i].setPriority(queueHandlerThreadPriority);
                }
                if(invokers[i].handler == null){
                    daemons[i].suspend();
                }
                daemons[i].start();
            }
        }
    }
    
    /**
     * T[rX̒~sB<p>
     *
     * @exception Exception T[rX̒~Ɏsꍇ
     */
    public void stopService() throws Exception{
        
        if(daemons != null){
            // f[~
            for(int i = 0; i < daemons.length; i++){
                if(isQueueHandlerNowaitOnStop){
                    daemons[i].stopNoWait();
                }else{
                    daemons[i].stop();
                }
                daemons[i] = null;
                invokers[i] = null;
            }
        }
        
        // L[t~
        if(getQueueService() != null && isReleaseQueue){
            getQueueService().release();
        }
        daemons = null;
        invokers = null;
        count = 0;
    }
    
    /**
     * Ăяo񓯊ɂ邽߂{@link Queue}T[rXݒ肷B<p>
     *
     * @param queue QueueT[rX
     */
    public void setQueueService(Queue<E> queue){
        this.requestQueue = queue;
    }
    
    /**
     * Ăяo񓯊ɂ邽߂{@link Queue}T[rX擾B<p>
     *
     * @return QueueT[rX
     */
    public Queue<E> getQueueService(){
        return requestQueue;
    }
    
    /**
     * QueueHandlerݒ肷B<p>
     *
     * @param handler QueueHandler
     */
    public void setQueueHandler(QueueHandler<E> handler){
        if(queueHandler == null){
            queueHandler = handler;
            if(daemons != null){
                for(Daemon daemon : daemons){
                    daemon.resume();
                }
            }
        }else{
            queueHandler = handler;
        }
    }
    
    /**
     * QueueHandler擾B<p>
     *
     * @return QueueHandler
     */
    public QueueHandler<E> getQueueHandler(){
        if(queueHandler != null){
            return queueHandler;
        }
        if(queueHandlerServiceName != null){
            return ServiceManagerFactory
                .<QueueHandler<E>>getServiceObject(queueHandlerServiceName);
        }
        return null;
    }
    
    public void push(E item){
        if(getQueueService() == null){
            count++;
            handleDequeuedObjectWithLock(getQueueHandler(), item, null);
        }else{
            getQueueService().push(item);
        }
    }
    
    /**
     * L[f[^oB<p>
     * T|[g܂B<br>
     * 
     * @return L[擾IuWFNg
     */
    public E get(){
        throw new UnsupportedOperationException();
    }
    
    /**
     * L[f[^oB<p>
     * T|[g܂B<br>
     * 
     * @param timeOutMs ^CAEg[ms]
     * @return L[擾IuWFNg
     */
    public E get(long timeOutMs){
        throw new UnsupportedOperationException();
    }
    
    public E peek(){
        if(getQueueService() == null){
            return null;
        }else{
            return getQueueService().peek();
        }
    }
    
    public E peek(long timeOutMs){
        if(getQueueService() == null){
            return null;
        }else{
            return getQueueService().peek(timeOutMs);
        }
    }
    
    public void remove(Object item){
        if(getQueueService() != null){
            getQueueService().remove(item);
        }
    }
    
    public void clear(){
        if(getQueueService() != null){
            getQueueService().clear();
        }
    }
    
    public int size(){
        if(getQueueService() == null){
            return 0;
        }else{
            return getQueueService().size();
        }
    }
    
    public long getCount(){
        if(getQueueService() == null){
            return count;
        }else{
            return getQueueService().getCount();
        }
    }
    
    public void accept(){
        if(getQueueService() != null){
            getQueueService().accept();
        }
    }
    
    public void release(){
        if(getQueueService() != null){
            getQueueService().release();
        }
        count = 0;
    }
    
    public synchronized void resume(){
        if(!isSuspend){
            return;
        }
        isSuspend = false;
        if(getQueueService() == null){
            suspendMonitor.notifyAllMonitor();
            suspendMonitor.releaseAllMonitor();
        }else{
            if(daemons != null){
                for(Daemon daemon : daemons){
                    daemon.resume();
                }
            }
        }
    }
    
    public synchronized void suspend(){
        if(isSuspend){
            return;
        }
        if(getQueueService() != null){
            if(daemons != null){
                for(Daemon daemon : daemons){
                    daemon.suspend();
                }
            }
        }
        isSuspend = true;
    }
    
    public boolean isSuspend(){
        return isSuspend;
    }
    
    protected <T> void handleDequeuedObjectWithLock(QueueHandler<T> handler, T dequeued, QueueReceiver<?> receiver){
        if(isSuspend){
            try{
                suspendMonitor.initAndWaitMonitor();
            }catch(InterruptedException e){
            }
        }
        handleDequeuedObject(handler, dequeued, receiver);
    }
    
    protected <T> void handleDequeuedObject(QueueHandler<T> handler, T dequeued, QueueReceiver<?> receiver){
        if(handler == null){
            return;
        }
        boolean isRetry = false;
        int retryCount = 0;
        do{
            try{
                if(receiver != null){
                    receiver.isActive = true;
                }
                try{
                    handler.handleDequeuedObject(dequeued);
                    isRetry = false;
                }catch(Throwable th){
                    if(maxRetryCount > 0){
                        if(retryCount >= maxRetryCount){
                            isRetry = false;
                            try{
                                handler.handleRetryOver(dequeued, th);
                            }catch(Throwable th2){
                                getLogger().write(
                                    retryOverErrorMessageId,
                                    th,
                                    dequeued
                                );
                            }
                        }else{
                            isRetry = true;
                            try{
                                isRetry = handler.handleError(dequeued, th);
                            }catch(Throwable th2){
                                isRetry = false;
                                getLogger().write(
                                    handlingErrorMessageId,
                                    th,
                                    dequeued
                                );
                            }
                        }
                    }else{
                        isRetry = false;
                        try{
                            handler.handleRetryOver(dequeued, th);
                        }catch(Throwable th2){
                            getLogger().write(
                                retryOverErrorMessageId,
                                th,
                                dequeued
                            );
                        }
                    }
                }
            }finally{
                if(receiver != null){
                    receiver.isActive = false;
                }
            }
            if(isRetry && retryInterval > 0){
                try{
                    Thread.sleep(retryInterval);
                }catch(InterruptedException e){
                    isRetry = false;
                }
            }
            retryCount++;
        }while(isRetry);
    }
    
    protected class QueueReceiver<T> implements DaemonRunnable<T>{
        
        protected QueueHandler<T> handler;
        
        /**
         * sǂtOB<p>
         */
        public boolean isActive;
        
        protected long receiveCount;
        protected long receiveProcessTime;
        
        public long getReceiveCount(){
            return receiveCount;
        }
        
        public long getAverageReceiveProcessTime(){
            return receiveCount == 0 ? 0 : (receiveProcessTime / receiveCount);
        }
        
        /**
         * f[JnɌĂяoB<p>
         * 
         * @return trueԂ
         */
        public boolean onStart() {
            return true;
        }
        
        /**
         * f[~ɌĂяoB<p>
         * 
         * @return trueԂ
         */
        public boolean onStop() {
            return true;
        }
        
        /**
         * f[fɌĂяoB<p>
         * 
         * @return trueԂ
         */
        public boolean onSuspend() {
            return true;
        }
        
        /**
         * f[ĊJɌĂяoB<p>
         * 
         * @return trueԂ
         */
        public boolean onResume() {
            return true;
        }
        
        /**
         * L[PoĕԂB<p>
         * 
         * @param ctrl DaemonControlIuWFNg
         * @return ̓IuWFNg
         */
        @SuppressWarnings("unchecked")
        public T provide(DaemonControl ctrl){
            if(handler == null){
                handler = (QueueHandler<T>)getQueueHandler();
                if(handler == null){
                    return null;
                }
            }
            return (T)getQueueService().get(waitTimeout);
        }
        
        /**
         * dequeuedœnꂽIuWFNgQueueHandlerĂяoB<p>
         *
         * @param dequeued L[oꂽIuWFNg
         * @param ctrl DaemonControlIuWFNg
         */
        public void consume(T dequeued, DaemonControl ctrl){
            receiveCount++;
            long start = System.currentTimeMillis();
            try{
                handleDequeuedObject(handler, dequeued, this);
            }finally{
                receiveProcessTime += (System.currentTimeMillis() - start);
            }
        }
        
        /**
         * L[̒gfoB<p>
         */
        @SuppressWarnings("unchecked")
        public void garbage(){
            if(getQueueService() != null){
                while(getQueueService().size() > 0){
                    consume((T)getQueueService().get(0), null);
                }
            }
        }
    }
}
