/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.core;

import static org.testng.Assert.*;

import java.util.HashMap;
import java.util.Map;

import org.testng.annotations.Test;
import org.testng.annotations.DataProvider;
import org.w3c.dom.Element;

import static jp.ossc.nimbus.TestUtility.*;

/**
 * {@link ManagerMetaData}eXgP[XB<p>
 * 
 * @author M.Takata
 */
public class ManagerMetaDataTest{

    @Test(groups = {"jp.ossc.nimbus.core.ManagerMetaData#ManagerMetaData()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*"})
    public void testManagerMetaData(){
        ManagerMetaData metaData = new ManagerMetaData();
        assertNull(metaData.getParent());
        assertNull(metaData.getServiceLoader());
        assertEquals(metaData.getName(), ServiceManager.DEFAULT_NAME);
        assertNull(metaData.getCode());
        assertFalse(metaData.isExistShutdownHook());
        assertNull(metaData.getRepository());
        assertNull(metaData.getLog());
        assertNull(metaData.getMessage());
        assertEquals(metaData.getServices().size(), 0);
        assertEquals(metaData.getPropertyNameSet().size(), 0);
        assertEquals(metaData.getProperties().size(), 0);
    }

    @DataProvider(name = "ManagerMetaData")
    public Object[][] dataManagerMetaData2(){
        return new Object[][]{
            {null, null},
            {null, new NimbusMetaData(null, null)},
            {new DefaultServiceLoaderService(), null},
            {new DefaultServiceLoaderService(), new NimbusMetaData(null, null)},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ManagerMetaData#ManagerMetaData(jp.ossc.nimbus.core.ServiceLoader,jp.ossc.nimbus.core.MetaData)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*"},
          dataProvider = "ManagerMetaData")
    public void testManagerMetaData2(ServiceLoader loader, MetaData parent){
        ManagerMetaData metaData = new ManagerMetaData(loader, parent);
        assertEquals(metaData.getServiceLoader(), loader);
        assertEquals(metaData.getParent(), parent);
        assertEquals(metaData.getName(), ServiceManager.DEFAULT_NAME);
        assertNull(metaData.getCode());
        assertFalse(metaData.isExistShutdownHook());
        assertNull(metaData.getRepository());
        assertNull(metaData.getLog());
        assertNull(metaData.getMessage());
        assertEquals(metaData.getServices().size(), 0);
        assertEquals(metaData.getPropertyNameSet().size(), 0);
        assertEquals(metaData.getProperties().size(), 0);
    }

    @DataProvider(name = "SetServiceLoader")
    public Object[][] dataSetServiceLoader(){
        return new Object[][]{
            {null},
            {new DefaultServiceLoaderService()}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ManagerMetaData#setServiceLoader(jp.ossc.nimbus.core.ServiceLoader)",
                    "jp.ossc.nimbus.core.ManagerMetaData#getServiceLoader()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ManagerMetaData#ManagerMetaData\\(\\)"},
          dataProvider = "SetServiceLoader")
    public void testSetServiceLoader(ServiceLoader loader){
        ManagerMetaData metaData = new ManagerMetaData();
        metaData.setServiceLoader(loader);
        assertEquals(metaData.getServiceLoader(), loader);
    }

    @DataProvider(name = "SetName")
    public Object[][] dataSetName(){
        return new Object[][]{
            {null, ServiceManager.DEFAULT_NAME},
            {"manager1", "manager1"}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ManagerMetaData#setName(java.lang.String)",
                    "jp.ossc.nimbus.core.ManagerMetaData#getName()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ManagerMetaData#ManagerMetaData\\(\\)"},
          dataProvider = "SetName")
    public void testSetName(String name, String expected){
        ManagerMetaData metaData = new ManagerMetaData();
        metaData.setName(name);
        assertEquals(metaData.getName(), expected);
    }

    @DataProvider(name = "SetCode")
    public Object[][] dataSetCode(){
        return new Object[][]{
            {null},
            {"test.MyServiceLoader"}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ManagerMetaData#setCode(java.lang.String)",
                    "jp.ossc.nimbus.core.ManagerMetaData#getCode()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ManagerMetaData#ManagerMetaData\\(\\)"},
          dataProvider = "SetCode")
    public void testSetCode(String code){
        ManagerMetaData metaData = new ManagerMetaData();
        metaData.setCode(code);
        assertEquals(metaData.getCode(), code);
    }

    @DataProvider(name = "SetExistShutdownHook")
    public Object[][] dataSetExistShutdownHook(){
        return new Object[][]{
            {true},
            {false}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ManagerMetaData#setExistShutdownHook(boolean)",
                    "jp.ossc.nimbus.core.ManagerMetaData#isExistShutdownHook()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ManagerMetaData#ManagerMetaData\\(\\)"},
          dataProvider = "SetExistShutdownHook")
    public void testSetExistShutdownHook(boolean isExist){
        ManagerMetaData metaData = new ManagerMetaData();
        metaData.setExistShutdownHook(isExist);
        assertEquals(metaData.isExistShutdownHook(), isExist);
    }

    @DataProvider(name = "SetRepository")
    public Object[][] dataSetRepository(){
        return new Object[][]{
            {null},
            {new ServiceNameMetaData(null)}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ManagerMetaData#setRepository(jp.ossc.nimbus.core.ServiceNameMetaData)",
                    "jp.ossc.nimbus.core.ManagerMetaData#getRepository()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ManagerMetaData#ManagerMetaData\\(\\)"},
          dataProvider = "SetRepository")
    public void testSetRepository(ServiceNameMetaData data){
        ManagerMetaData metaData = new ManagerMetaData();
        metaData.setRepository(data);
        assertEquals(metaData.getRepository(), data);
    }

    @DataProvider(name = "SetLog")
    public Object[][] dataSetLog(){
        return new Object[][]{
            {null},
            {new ServiceNameMetaData(null)}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ManagerMetaData#setLog(jp.ossc.nimbus.core.ServiceNameMetaData)",
                    "jp.ossc.nimbus.core.ManagerMetaData#getLog()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ManagerMetaData#ManagerMetaData\\(\\)"},
          dataProvider = "SetLog")
    public void testSetLog(ServiceNameMetaData data){
        ManagerMetaData metaData = new ManagerMetaData();
        metaData.setLog(data);
        assertEquals(metaData.getLog(), data);
    }

    @DataProvider(name = "SetMessage")
    public Object[][] dataSetMessage(){
        return new Object[][]{
            {null},
            {new ServiceNameMetaData(null)}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ManagerMetaData#setMessage(jp.ossc.nimbus.core.ServiceNameMetaData)",
                    "jp.ossc.nimbus.core.ManagerMetaData#getMessage()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ManagerMetaData#ManagerMetaData\\(\\)"},
          dataProvider = "SetMessage")
    public void testSetMessage(ServiceNameMetaData data){
        ManagerMetaData metaData = new ManagerMetaData();
        metaData.setMessage(data);
        assertEquals(metaData.getMessage(), data);
    }

    @DataProvider(name = "AddService")
    public Object[][] dataAddService(){
        return new Object[][]{
            {null},
            {new ServiceMetaData[]{new ServiceMetaData("service1", null)}},
            {new ServiceMetaData[]{new ServiceMetaData("service1", null), new ServiceMetaData("service2", null), new ServiceMetaData("service3", null), new ServiceMetaData("service4", null), new ServiceMetaData("service5", null), new ServiceMetaData("service6", null)}},
            {new ServiceMetaData[]{null, new ServiceMetaData("service1", null)}}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ManagerMetaData#addService(jp.ossc.nimbus.core.ServiceMetaData)",
                    "jp.ossc.nimbus.core.ManagerMetaData#getServiceNameSet()",
                    "jp.ossc.nimbus.core.ManagerMetaData#getService(java.lang.String)",
                    "jp.ossc.nimbus.core.ManagerMetaData#getServices()",
                    "jp.ossc.nimbus.core.ManagerMetaData#removeService(java.lang.String)",
                    "jp.ossc.nimbus.core.ManagerMetaData#clearServices()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ManagerMetaData#ManagerMetaData\\(\\)"},
          dataProvider = "AddService")
    public void testAddService(ServiceMetaData[] services){
        ManagerMetaData metaData = new ManagerMetaData();
        if(services == null || services.length == 0){
            assertEquals(metaData.getServices().size(), 0);
        }else{
            Map<String, ServiceMetaData> serviceMap = new HashMap<String, ServiceMetaData>();
            for(ServiceMetaData service : services){
                metaData.addService(service);
                if(service != null){
                    serviceMap.put(service.getName(), service);
                }
            }
            for(String name : metaData.getServiceNameSet()){
                assertEquals(metaData.getService(name), serviceMap.get(name));
                assertEquals(metaData.getService(name).getManager(), metaData);
            }
            
            Map<String, ServiceMetaData> map = metaData.getServices();
            assertEquals(map, serviceMap);
            
            for(int i = 0, imax = services.length; i < imax; i+=2){
                if(services[i] != null){
                    metaData.removeService(services[i].getName());
                    assertNull(services[i].getManager());
                    serviceMap.remove(services[i].getName());
                }
            }
            assertEquals(map, serviceMap);
            metaData.clearServices();
            assertEquals(map.size(), 0);
            for(int i = 0, imax = services.length; i < imax; i++){
                if(services[i] != null){
                    assertNull(services[i].getManager());
                }
            }
        }
    }

    @DataProvider(name = "SetProperty")
    public Object[][] dataSetProperty(){
        return new Object[][]{
            {null},
            {new String[][]{{"prop1", "1"}}},
            {new String[][]{{"prop1", "1"}, {"prop2", "2"}, {"prop3", "3"}, {"prop4", "4"}, {"prop5", "5"}, {"prop6", "6"}, {"prop7", "7"}, {"prop8", "8"}, {"prop9", "9"}, {"prop10", "10"}}},
            {new String[][]{{"prop1", null}, {null, "2"}, {null, null}}},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ManagerMetaData#setProperty(java.lang.String,java.lang.String)",
                    "jp.ossc.nimbus.core.ManagerMetaData#getPropertyNameSet()",
                    "jp.ossc.nimbus.core.ManagerMetaData#getProperty(java.lang.String)",
                    "jp.ossc.nimbus.core.ManagerMetaData#removeProperty(java.lang.String)",
                    "jp.ossc.nimbus.core.ManagerMetaData#getProperties()",
                    "jp.ossc.nimbus.core.ManagerMetaData#clearProperties()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ManagerMetaData#ManagerMetaData\\(\\)"},
          dataProvider = "SetProperty")
    public void testSetProperty(String[][] propertySet){
        ManagerMetaData metaData = new ManagerMetaData();
        if(propertySet == null || propertySet.length == 0){
            assertEquals(metaData.getProperties().size(), 0);
        }else{
            Map<String, String> properties = new HashMap<String, String>();
            for(String[] property : propertySet){
                metaData.setProperty(property[0], property[1]);
                properties.put(property[0], property[1]);
            }
            for(String name : metaData.getPropertyNameSet()){
                assertEquals(metaData.getProperty(name), properties.get(name));
            }
            for(int i = 0, imax = propertySet.length; i < imax; i+=2){
                metaData.removeProperty(propertySet[i][0]);
                properties.remove(propertySet[i][0]);
            }
            Map<String, String> props = metaData.getProperties();
            assertEquals(props, properties);
            metaData.clearProperties();
            props = metaData.getProperties();
            assertEquals(props.size(), 0);
        }
    }

    @DataProvider(name = "ImportXML")
    public Object[][] dataImportXML() throws Exception{
        return new Object[][]{
            {createElement("<manager/>"), new Object[]{ServiceManager.DEFAULT_NAME, false, null, null, null, null, new MetaData[0]}},
            {createElement("<manager name=\"manager1\"/>"), new Object[]{"manager1", false, null, null, null, null, new MetaData[0]}},
            {createElement("<manager shutdown-hook=\"true\"/>"), new Object[]{ServiceManager.DEFAULT_NAME, true, null, null, null, null, new MetaData[0]}},
            {createElement("<manager code=\"test.MyServiceManager\"/>"), new Object[]{ServiceManager.DEFAULT_NAME, false, "test.MyServiceManager", null, null, null, new MetaData[0]}},
            {createElement("<manager><repository>Repository</repository></manager>"), new Object[]{ServiceManager.DEFAULT_NAME, false, null, importXML(new ServiceNameMetaData(null), "<repository>Repository</repository>"), null, null, new MetaData[0]}},
            {createElement("<manager><repository>Repository</repository><ifdef name=\"test\" value=\"false\"><repository>Repository</repository></ifdef></manager>"), new Object[]{ServiceManager.DEFAULT_NAME, false, null, importXML(new ServiceNameMetaData(null), "<repository>Repository</repository>"), null, null, new MetaData[0]}},
            {createElement("<manager name=\"manager1\"><repository>Repository</repository></manager>"), new Object[]{"manager1", false, null, importXML(new ServiceNameMetaData(null), "<repository>manager1#Repository</repository>"), null, null, new MetaData[0]}},
            {createElement("<manager name=\"manager1\"><repository>#Repository</repository></manager>"), new Object[]{"manager1", false, null, importXML(new ServiceNameMetaData(null), "<repository>manager1#Repository</repository>"), null, null, new MetaData[0]}},
            {createElement("<manager><log>Log</log></manager>"), new Object[]{ServiceManager.DEFAULT_NAME, false, null, null, importXML(new ServiceNameMetaData(null), "<log>Log</log>"), null, new MetaData[0]}},
            {createElement("<manager><log>Log</log><ifdef name=\"test\" value=\"false\"><log>Log</log></ifdef></manager>"), new Object[]{ServiceManager.DEFAULT_NAME, false, null, null, importXML(new ServiceNameMetaData(null), "<log>Log</log>"), null, new MetaData[0]}},
            {createElement("<manager name=\"manager1\"><log>Log</log></manager>"), new Object[]{"manager1", false, null, null, importXML(new ServiceNameMetaData(null), "<log>manager1#Log</log>"), null, new MetaData[0]}},
            {createElement("<manager name=\"manager1\"><log>#Log</log></manager>"), new Object[]{"manager1", false, null, null, importXML(new ServiceNameMetaData(null), "<log>manager1#Log</log>"), null, new MetaData[0]}},
            {createElement("<manager><message>Message</message></manager>"), new Object[]{ServiceManager.DEFAULT_NAME, false, null, null, null, importXML(new ServiceNameMetaData(null), "<message>Message</message>"), new MetaData[0]}},
            {createElement("<manager><message>Message</message><ifdef name=\"test\" value=\"false\"><message>Message</message></ifdef></manager>"), new Object[]{ServiceManager.DEFAULT_NAME, false, null, null, null, importXML(new ServiceNameMetaData(null), "<message>Message</message>"), new MetaData[0]}},
            {createElement("<manager name=\"manager1\"><message>Message</message></manager>"), new Object[]{"manager1", false, null, null, null, importXML(new ServiceNameMetaData(null), "<message>manager1#Message</message>"), new MetaData[0]}},
            {createElement("<manager name=\"manager1\"><message>#Message</message></manager>"), new Object[]{"manager1", false, null, null, null, importXML(new ServiceNameMetaData(null), "<message>manager1#Message</message>"), new MetaData[0]}},
            {createElement("<manager><service name=\"Hoge\" code=\"test.Hoge\"/></manager>"), new Object[]{ServiceManager.DEFAULT_NAME, false, null, null, null, null, new MetaData[]{importXML(new ServiceMetaData(), "<service name=\"Hoge\" code=\"test.Hoge\"/>")}}},
            {createElement("<manager><service name=\"Hoge\" code=\"test.Hoge\"><depends><service name=\"Fuga\" code=\"test.Fuga\"/></depends></service></manager>"), new Object[]{ServiceManager.DEFAULT_NAME, false, null, null, null, null, new MetaData[]{importXML(new ServiceMetaData(), "<service name=\"Hoge\" code=\"test.Hoge\"><depends>Nimbus#Fuga</depends></service>"), importXML(new ServiceMetaData(), "<service name=\"Fuga\" code=\"test.Fuga\"/>")}}},
            {createElement("<manager><service name=\"Hoge\" code=\"test.Hoge\"/><ifdef name=\"test\" value=\"false\"><service name=\"Hoge\" code=\"test.Hoge\"/></ifdef></manager>"), new Object[]{ServiceManager.DEFAULT_NAME, false, null, null, null, null, new MetaData[]{importXML(new ServiceMetaData(), "<service name=\"Hoge\" code=\"test.Hoge\"/>")}}},
            {createElement("<manager><service name=\"Hoge1\" code=\"test.Hoge\"/><service name=\"Hoge2\" code=\"test.Hoge\"/></manager>"), new Object[]{ServiceManager.DEFAULT_NAME, false, null, null, null, null, new MetaData[]{importXML(new ServiceMetaData(), "<service name=\"Hoge1\" code=\"test.Hoge\"/>"), importXML(new ServiceMetaData(), "<service name=\"Hoge2\" code=\"test.Hoge\"/>")}}},
            {createElement("<manager><ifdef name=\"test\" value=\"true\"><repository>Repository</repository></ifdef></manager>"), new Object[]{ServiceManager.DEFAULT_NAME, false, null, importXML(new ServiceNameMetaData(null), "<repository>Repository</repository>"), null, null, new MetaData[0]}},
            {createElement("<manager><ifdef name=\"test\" value=\"false\"><repository>Repository</repository></ifdef></manager>"), new Object[]{ServiceManager.DEFAULT_NAME, false, null, null, null, null, new MetaData[0]}},
            {createElement("<manager><ifdef name=\"test\" value=\"true\"><log>Log</log></ifdef></manager>"), new Object[]{ServiceManager.DEFAULT_NAME, false, null, null, importXML(new ServiceNameMetaData(null), "<log>Log</log>"), null, new MetaData[0]}},
            {createElement("<manager><ifdef name=\"test\" value=\"false\"><log>Log</log></ifdef></manager>"), new Object[]{ServiceManager.DEFAULT_NAME, false, null, null, null, null, new MetaData[0]}},
            {createElement("<manager><ifdef name=\"test\" value=\"true\"><message>Message</message></ifdef></manager>"), new Object[]{ServiceManager.DEFAULT_NAME, false, null, null, null, importXML(new ServiceNameMetaData(null), "<message>Message</message>"), new MetaData[0]}},
            {createElement("<manager><ifdef name=\"test\" value=\"false\"><message>Message</message></ifdef></manager>"), new Object[]{ServiceManager.DEFAULT_NAME, false, null, null, null, null, new MetaData[0]}},
            {createElement("<manager><ifdef name=\"test\" value=\"true\"><service name=\"Hoge\" code=\"test.Hoge\"/></ifdef></manager>"), new Object[]{ServiceManager.DEFAULT_NAME, false, null, null, null, null, new MetaData[]{importXML(new ServiceMetaData(), "<service name=\"Hoge\" code=\"test.Hoge\"/>")}}},
            {createElement("<manager><ifdef name=\"test\" value=\"false\"><service name=\"Hoge\" code=\"test.Hoge\"/></ifdef></manager>"), new Object[]{ServiceManager.DEFAULT_NAME, false, null, null, null, null, new MetaData[0]}},
            {createElement("<manager><ifdef name=\"test\" value=\"true\"><repository>Repository</repository><log>Log</log><message>Message</message><service name=\"Hoge\" code=\"test.Hoge\"/></ifdef></manager>"), new Object[]{ServiceManager.DEFAULT_NAME, false, null, importXML(new ServiceNameMetaData(null), "<repository>Repository</repository>"), importXML(new ServiceNameMetaData(null), "<log>Log</log>"), importXML(new ServiceNameMetaData(null), "<message>Message</message>"), new MetaData[]{importXML(new ServiceMetaData(), "<service name=\"Hoge\" code=\"test.Hoge\"/>")}}},
            {createElement("<manager><ifdef name=\"test\" value=\"false\"><repository>Repository</repository><log>Log</log><message>Message</message><service name=\"Hoge\" code=\"test.Hoge\"/></ifdef></manager>"), new Object[]{ServiceManager.DEFAULT_NAME, false, null, null, null, null, new MetaData[0]}},
            {createElement("<manager unknown=\"hoge\"><unknown/></manager>"), new Object[]{ServiceManager.DEFAULT_NAME, false, null, null, null, null, new MetaData[0]}},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ManagerMetaData#importXML(org.w3c.dom.Element)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ManagerMetaData#ManagerMetaData\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ManagerMetaData#getName\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ManagerMetaData#isExistShutdownHook\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ManagerMetaData#getCode\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ManagerMetaData#getRepository\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ManagerMetaData#getLog\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ManagerMetaData#getMessage\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ManagerMetaData#getServices\\(\\)"},
          dataProvider = "ImportXML")
    public void testImportXML(
        Element element,
        Object[] assertValues
    ) throws DeploymentException{
        
        System.setProperty("test", "true");
        try{
            ManagerMetaData metaData = new ManagerMetaData();
            metaData.importXML(element);
            assertEquals(metaData.getName(), assertValues[0]);
            assertEquals(metaData.isExistShutdownHook(), assertValues[1]);
            assertEquals(metaData.getCode(), assertValues[2]);
            assertMetaDataEquals(metaData.getRepository(), (MetaData)assertValues[3]);
            assertMetaDataEquals(metaData.getLog(), (MetaData)assertValues[4]);
            assertMetaDataEquals(metaData.getMessage(), (MetaData)assertValues[5]);
            Map<String, ServiceMetaData> services = metaData.getServices();
            MetaData[] srvs = (MetaData[])assertValues[6];
            if(srvs.length == 0){
                assertTrue(services.size() == 0);
            }else{
                assertEquals(services.size(), srvs.length);
                for(int i = 0; i < srvs.length; i++){
                    assertMetaDataEquals(services.get(((ServiceMetaData)srvs[i]).getName()), srvs[i]);
                }
            }
        }finally{
            System.clearProperty("test");
        }
    }

    @DataProvider(name = "ImportXMLError")
    public Object[][] dataImportXMLError() throws Exception{
        return new Object[][]{
            {null},
            {createElement("<hoge/>")},
            {createElement("<manager><repository>#Repository</repository><repository>#Repository</repository></manager>")},
            {createElement("<manager><repository>#Repository</repository><ifdef name=\"test\" value=\"true\"><repository>#Repository</repository></ifdef></manager>")},
            {createElement("<manager><log>#Log</log><log>#Log</log></manager>")},
            {createElement("<manager><log>#Log</log><ifdef name=\"test\" value=\"true\"><log>#Log</log></ifdef></manager>")},
            {createElement("<manager><message>#Message</message><message>#Message</message></manager>")},
            {createElement("<manager><message>#Message</message><ifdef name=\"test\" value=\"true\"><message>#Message</message></ifdef></manager>")},
            {createElement("<manager><service name=\"Hoge\" code=\"test.Hoge\"/><ifdef name=\"test\" value=\"true\"><service name=\"Hoge\" code=\"test.Hoge\"/></ifdef></manager>")},
            {createElement("<manager><service name=\"Hoge\" code=\"test.Hoge\"/><service name=\"Hoge\" code=\"test.Hoge\"/></manager>")},
            {createElement("<manager><service name=\"Hoge\" code=\"test.Hoge\"><depends><service name=\"Hoge\" code=\"test.Hoge\"/></depends></service></manager>")},
            {createElement("<manager><service name=\"Hoge1\" code=\"test.Hoge\"/><service name=\"Hoge2\" code=\"test.Hoge\"><depends><service name=\"Hoge1\" code=\"test.Hoge\"/></depends></service></manager>")},
            {createElement("<manager><service name=\"Hoge1\" code=\"test.Hoge\"/><service name=\"Hoge2\" code=\"test.Hoge\"><ifdef name=\"test\" value=\"true\"><depends><service name=\"Hoge1\" code=\"test.Hoge\"/></depends></ifdef></service></manager>")},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ManagerMetaData#importXML(org.w3c.dom.Element)",
                    "Error"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ManagerMetaData#ManagerMetaData\\(\\)"},
          expectedExceptions = {NullPointerException.class, DeploymentException.class},
          dataProvider = "ImportXMLError")
    public void testImportXMLError(Element element) throws DeploymentException{
        System.setProperty("test", "true");
        try{
            ManagerMetaData metaData = new ManagerMetaData();
            metaData.importXML(element);
        }finally{
            System.clearProperty("test");
        }
    }

    @DataProvider(name = "ToXML")
    public Object[][] dataToXML() throws Exception{
        return new Object[][]{
            {createElement("<manager/>"), "<manager>" + System.getProperty("line.separator") + "</manager>"},
            {createElement("<manager name=\"manager1\"/>"), "<manager name=\"manager1\">" + System.getProperty("line.separator") + "</manager>"},
            {createElement("<manager name=\"manager1\" code=\"test.MyServiceManager\" shutdown-hook=\"true\"/>"), "<manager name=\"manager1\" code=\"test.MyServiceManager\" shutdown-hook=\"true\">" + System.getProperty("line.separator") + "</manager>"},
            {createElement("<manager><repository>Repository</repository></manager>"), "<manager>" + System.getProperty("line.separator") + "    <repository>" + ServiceManager.DEFAULT_NAME + "#Repository</repository>" + System.getProperty("line.separator") + "</manager>"},
            {createElement("<manager name=\"manager1\"><repository>#Repository</repository></manager>"), "<manager name=\"manager1\">" + System.getProperty("line.separator") + "    <repository>manager1#Repository</repository>" + System.getProperty("line.separator") + "</manager>"},
            {createElement("<manager name=\"manager1\"><log>#Log</log></manager>"), "<manager name=\"manager1\">" + System.getProperty("line.separator") + "    <log>manager1#Log</log>" + System.getProperty("line.separator") + "</manager>"},
            {createElement("<manager name=\"manager1\"><message>#Message</message></manager>"), "<manager name=\"manager1\">" + System.getProperty("line.separator") + "    <message>manager1#Message</message>" + System.getProperty("line.separator") + "</manager>"},
            {createElement("<manager name=\"manager1\"><service name=\"Hoge\" code=\"test.Hoge\"/></manager>"), "<manager name=\"manager1\">" + System.getProperty("line.separator") + "    <service name=\"Hoge\" code=\"test.Hoge\"/>" + System.getProperty("line.separator") + "</manager>"},
            {createElement("<manager name=\"manager1\"><service name=\"Hoge1\" code=\"test.Hoge\"/><service name=\"Hoge2\" code=\"test.Hoge\"/></manager>"), "<manager name=\"manager1\">" + System.getProperty("line.separator") + "    <service name=\"Hoge1\" code=\"test.Hoge\"/>" + System.getProperty("line.separator") + "    <service name=\"Hoge2\" code=\"test.Hoge\"/>" + System.getProperty("line.separator") + "</manager>"},
            {createElement("<manager name=\"manager1\"><ifdef name=\"test\" value=\"true\"><repository>#Repository</repository></ifdef></manager>"), "<manager name=\"manager1\">" + System.getProperty("line.separator") + "    <ifdef name=\"test\" value=\"true\">" + System.getProperty("line.separator") + "        <repository>manager1#Repository</repository>" + System.getProperty("line.separator") + "    </ifdef>" + System.getProperty("line.separator") + "</manager>"},
            {createElement("<manager name=\"manager1\"><ifdef name=\"test\" value=\"true\"><log>#Log</log></ifdef></manager>"), "<manager name=\"manager1\">" + System.getProperty("line.separator") + "    <ifdef name=\"test\" value=\"true\">" + System.getProperty("line.separator") + "        <log>manager1#Log</log>" + System.getProperty("line.separator") + "    </ifdef>" + System.getProperty("line.separator") + "</manager>"},
            {createElement("<manager name=\"manager1\"><ifdef name=\"test\" value=\"true\"><message>#Message</message></ifdef></manager>"), "<manager name=\"manager1\">" + System.getProperty("line.separator") + "    <ifdef name=\"test\" value=\"true\">" + System.getProperty("line.separator") + "        <message>manager1#Message</message>" + System.getProperty("line.separator") + "    </ifdef>" + System.getProperty("line.separator") + "</manager>"},
            {createElement("<manager name=\"manager1\"><ifdef name=\"test\" value=\"true\"><service name=\"Hoge\" code=\"test.Hoge\"/></ifdef></manager>"), "<manager name=\"manager1\">" + System.getProperty("line.separator") + "    <ifdef name=\"test\" value=\"true\">" + System.getProperty("line.separator") + "        <service name=\"Hoge\" code=\"test.Hoge\"/>" + System.getProperty("line.separator") + "    </ifdef>" + System.getProperty("line.separator") + "</manager>"},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ManagerMetaData#toXML(java.lang.StringBuilder)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ManagerMetaData#ManagerMetaData\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ManagerMetaData#importXML\\(org\\.w3c\\.dom\\.Element\\)"},
          dataProvider = "ToXML")
    public void testToXML(Element element, String expected) throws DeploymentException{
        ManagerMetaData metaData = new ManagerMetaData();
        metaData.importXML(element);
        StringBuilder buf = new StringBuilder();
        metaData.toXML(buf);
        assertEquals(buf.toString(), expected);
    }
}
