/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.core;

import static org.testng.Assert.*;

import java.util.List;

import org.testng.annotations.Test;
import org.testng.annotations.DataProvider;
import org.w3c.dom.Element;

import static jp.ossc.nimbus.TestUtility.*;

/**
 * {@link ObjectMetaData}eXgP[XB<p>
 * 
 * @author M.Takata
 */
public class ObjectMetaDataTest{
    
    @Test(groups = {"jp.ossc.nimbus.core.ObjectMetaData#ObjectMetaData()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*"})
    public void testObjectMetaData(){
        ObjectMetaData metaData = new ObjectMetaData();
        assertNull(metaData.getCode());
        assertNull(metaData.getServiceLoader());
        assertNull(metaData.getParent());
        assertNull(metaData.getManagerName());
        assertNull(metaData.getConstructor());
        assertEquals(metaData.getFields().size(), 0);
        assertEquals(metaData.getAttributes().size(), 0);
        assertEquals(metaData.getInvokes().size(), 0);
    }

    @DataProvider(name = "ObjectMetaData2")
    public Object[][] dataObjectMetaData2(){
        return new Object[][]{
            {null, null, null},
            {new DefaultServiceLoaderService(), null, null},
            {null, new AttributeMetaData(null), null},
            {null, null, "manager1"},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ObjectMetaData#ObjectMetaData(jp.ossc.nimbus.core.ServiceLoader,jp.ossc.nimbus.core.MetaData,java.lang.String)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*"},
          dataProvider = "ObjectMetaData2")
    public void testObjectMetaData2(ServiceLoader loader, MetaData parent, String manager){
        ObjectMetaData metaData = new ObjectMetaData(loader, parent, manager);
        assertEquals(metaData.getServiceLoader(), loader);
        assertEquals(metaData.getParent(), parent);
        assertEquals(metaData.getManagerName(), manager);
        assertNull(metaData.getConstructor());
        assertEquals(metaData.getFields().size(), 0);
        assertEquals(metaData.getAttributes().size(), 0);
        assertEquals(metaData.getInvokes().size(), 0);
    }

    @DataProvider(name = "SetServiceLoader")
    public Object[][] dataSetServiceLoader(){
        return new Object[][]{
            {null},
            {new DefaultServiceLoaderService()}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ObjectMetaData#setServiceLoader(jp.ossc.nimbus.core.ServiceLoader)",
                    "jp.ossc.nimbus.core.ObjectMetaData#getServiceLoader()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData#ObjectMetaData\\(\\)"},
          dataProvider = "SetServiceLoader")
    public void testSetServiceLoader(ServiceLoader loader){
        ObjectMetaData metaData = new ObjectMetaData();
        metaData.setServiceLoader(loader);
        assertEquals(metaData.getServiceLoader(), loader);
    }

    @DataProvider(name = "SetCode")
    public Object[][] dataSetCode(){
        return new Object[][]{
            {null},
            {"java.lang.String"}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ObjectMetaData#setCode(java.lang.String)",
                    "jp.ossc.nimbus.core.ObjectMetaData#getCode()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData#ObjectMetaData\\(\\)"},
          dataProvider = "SetCode")
    public void testSetCode(String code){
        ObjectMetaData metaData = new ObjectMetaData();
        metaData.setCode(code);
        assertEquals(metaData.getCode(), code);
    }

    @DataProvider(name = "SetConstructor")
    public Object[][] dataSetConstructor(){
        return new Object[][]{
            {null},
            {new ConstructorMetaData(null)}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ObjectMetaData#setConstructor(jp.ossc.nimbus.core.ConstructorMetaData)",
                    "jp.ossc.nimbus.core.ObjectMetaData#getConstructor()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData#ObjectMetaData\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ConstructorMetaData#ConstructorMetaData\\(jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData\\)"},
          dataProvider = "SetConstructor")
    public void testSetRepository(ConstructorMetaData data){
        ObjectMetaData metaData = new ObjectMetaData();
        metaData.setConstructor(data);
        assertEquals(metaData.getConstructor(), data);
    }

    @DataProvider(name = "SetManagerName")
    public Object[][] dataSetManagerName(){
        return new Object[][]{
            {null},
            {"manager1"}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ObjectMetaData#setManagerName(java.lang.String)",
                    "jp.ossc.nimbus.core.ObjectMetaData#getManagerName()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData#ObjectMetaData\\(\\)"},
          dataProvider = "SetManagerName")
    public void testSetManagerName(String name){
        ObjectMetaData metaData = new ObjectMetaData();
        metaData.setManagerName(name);
        assertEquals(metaData.getManagerName(), name);
    }

    @DataProvider(name = "AddField")
    public Object[][] dataAddField() throws Exception{
        return new Object[][]{
            {null},
            {new MetaData[]{importXML(new FieldMetaData(null), "<field name=\"field1\">1</field>")}},
            {
                new MetaData[]{
                    importXML(new FieldMetaData(null), "<field name=\"field1\">1</field>"),
                    importXML(new FieldMetaData(null), "<field name=\"field2\">2</field>"),
                    importXML(new FieldMetaData(null), "<field name=\"field3\">3</field>"),
                    importXML(new FieldMetaData(null), "<field name=\"field4\">4</field>"),
                    importXML(new FieldMetaData(null), "<field name=\"field5\">5</field>"),
                    importXML(new FieldMetaData(null), "<field name=\"field6\">6</field>"),
                    importXML(new FieldMetaData(null), "<field name=\"field7\">7</field>"),
                    importXML(new FieldMetaData(null), "<field name=\"field8\">8</field>"),
                    importXML(new FieldMetaData(null), "<field name=\"field9\">9</field>"),
                    importXML(new FieldMetaData(null), "<field name=\"field10\">10</field>")
                }
            }
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ObjectMetaData#addField(jp.ossc.nimbus.core.FieldMetaData)",
                    "jp.ossc.nimbus.core.ObjectMetaData#getFields()",
                    "jp.ossc.nimbus.core.ObjectMetaData#getFieldNameList()",
                    "jp.ossc.nimbus.core.ObjectMetaData#getField(java.lang.String)",
                    "jp.ossc.nimbus.core.ObjectMetaData#removeField(java.lang.String)",
                    "jp.ossc.nimbus.core.ObjectMetaData#clearFields()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.FieldMetaData#FieldMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.FieldMetaData#importXML\\(org\\.w3c\\.dom\\.Element\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData#ObjectMetaData\\(\\)"},
          dataProvider = "AddField")
    public void testAddField(MetaData[] fields){
        ObjectMetaData metaData = new ObjectMetaData();
        if(fields == null || fields.length == 0){
            assertEquals(metaData.getFields().size(), 0);
        }else{
            for(MetaData field : fields){
                metaData.addField((FieldMetaData)field);
            }
            for(int i = 0; i < fields.length; i++){
                assertEquals(metaData.getField(((FieldMetaData)fields[i]).getName()), fields[i]);
            }
            List<String> names = metaData.getFieldNameList();
            for(int i = 0, imax = names.size(); i < imax; i+=2){
                metaData.removeField(names.get(i));
            }
            List<FieldMetaData> list = metaData.getFields();
            int index = 1;
            for(int i = 0; i < list.size(); i++){
                assertEquals(list.get(i), fields[index]);
                index += 2;
            }
            metaData.clearFields();
            assertEquals(metaData.getFields().size(), 0);
        }
    }

    @DataProvider(name = "AddAttribute")
    public Object[][] dataAddAttribute() throws Exception{
        return new Object[][]{
            {null},
            {new MetaData[]{importXML(new AttributeMetaData(null), "<attribute name=\"Attribute1\">1</attribute>")}},
            {
                new MetaData[]{
                    importXML(new AttributeMetaData(null), "<attribute name=\"Attribute1\">1</attribute>"),
                    importXML(new AttributeMetaData(null), "<attribute name=\"Attribute2\">2</attribute>"),
                    importXML(new AttributeMetaData(null), "<attribute name=\"Attribute3\">3</attribute>"),
                    importXML(new AttributeMetaData(null), "<attribute name=\"Attribute4\">4</attribute>"),
                    importXML(new AttributeMetaData(null), "<attribute name=\"Attribute5\">5</attribute>"),
                    importXML(new AttributeMetaData(null), "<attribute name=\"Attribute6\">6</attribute>"),
                    importXML(new AttributeMetaData(null), "<attribute name=\"Attribute7\">7</attribute>"),
                    importXML(new AttributeMetaData(null), "<attribute name=\"Attribute8\">8</attribute>"),
                    importXML(new AttributeMetaData(null), "<attribute name=\"Attribute9\">9</attribute>"),
                    importXML(new AttributeMetaData(null), "<attribute name=\"Attribute10\">10</attribute>")
                }
            }
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ObjectMetaData#addAttribute(jp.ossc.nimbus.core.AttributeMetaData)",
                    "jp.ossc.nimbus.core.ObjectMetaData#getAttributes()",
                    "jp.ossc.nimbus.core.ObjectMetaData#getAttributeNameList()",
                    "jp.ossc.nimbus.core.ObjectMetaData#getAttribute(java.lang.String)",
                    "jp.ossc.nimbus.core.ObjectMetaData#removeAttribute(java.lang.String)",
                    "jp.ossc.nimbus.core.ObjectMetaData#clearAttributes()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.AttributeMetaData#AttributeMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.AttributeMetaData#importXML\\(org\\.w3c\\.dom\\.Element\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData#ObjectMetaData\\(\\)"},
          dataProvider = "AddAttribute")
    public void testAddAttribute(MetaData[] attributes){
        ObjectMetaData metaData = new ObjectMetaData();
        if(attributes == null || attributes.length == 0){
            assertEquals(metaData.getAttributes().size(), 0);
        }else{
            for(MetaData attribute : attributes){
                metaData.addAttribute((AttributeMetaData)attribute);
            }
            for(int i = 0; i < attributes.length; i++){
                assertEquals(metaData.getAttribute(((AttributeMetaData)attributes[i]).getName()), attributes[i]);
            }
            List<String> names = metaData.getAttributeNameList();
            for(int i = 0, imax = names.size(); i < imax; i+=2){
                metaData.removeAttribute(names.get(i));
            }
            List<AttributeMetaData> list = metaData.getAttributes();
            int index = 1;
            for(int i = 0; i < list.size(); i++){
                assertEquals(list.get(i), attributes[index]);
                index += 2;
            }
            metaData.clearAttributes();
            assertEquals(metaData.getAttributes().size(), 0);
        }
    }

    @DataProvider(name = "AddInvoke")
    public Object[][] dataAddInvoke() throws Exception{
        return new Object[][]{
            {null},
            {new InvokeMetaData[]{new InvokeMetaData(null)}},
            {
                new InvokeMetaData[]{
                    new InvokeMetaData(null),
                    new InvokeMetaData(null),
                    new InvokeMetaData(null),
                    new InvokeMetaData(null),
                    new InvokeMetaData(null),
                    new InvokeMetaData(null),
                    new InvokeMetaData(null),
                    new InvokeMetaData(null),
                    new InvokeMetaData(null),
                    new InvokeMetaData(null)
                }
            },
            {new InvokeMetaData[]{null, new InvokeMetaData(null)}}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ObjectMetaData#addInvoke(jp.ossc.nimbus.core.InvokeMetaData)",
                    "jp.ossc.nimbus.core.ObjectMetaData#getInvokes()",
                    "jp.ossc.nimbus.core.ObjectMetaData#removeInvoke(jp.ossc.nimbus.core.InvokeMetaData)",
                    "jp.ossc.nimbus.core.ObjectMetaData#clearInvokes()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData#ObjectMetaData\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.InvokeMetaData#InvokeMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)"},
          dataProvider = "AddInvoke")
    public void testAddInvoke(InvokeMetaData[] invokes){
        ObjectMetaData metaData = new ObjectMetaData();
        if(invokes == null || invokes.length == 0){
            assertEquals(metaData.getInvokes().size(), 0);
        }else{
            for(InvokeMetaData invoke : invokes){
                metaData.addInvoke(invoke);
            }
            List<InvokeMetaData> list = metaData.getInvokes();
            for(int i = 0; i < list.size(); i++){
                assertEquals(list.get(i), invokes[i]);
            }
            for(int i = 0, imax = list.size(); i < imax; i+=2){
                metaData.removeInvoke(invokes[i]);
            }
            list = metaData.getInvokes();
            int index = 1;
            for(int i = 0; i < list.size(); i++){
                assertEquals(list.get(i), invokes[index]);
                index += 2;
            }
            metaData.clearInvokes();
            assertEquals(metaData.getInvokes().size(), 0);
        }
    }

    @DataProvider(name = "ImportXML")
    public Object[][] dataImportXML() throws Exception{
        return new Object[][]{
            {
                createElement("<object code=\"test.Hoge\"/>"),
                new Object[]{"test.Hoge", null, new MetaData[0], new MetaData[0], new MetaData[0]}
            },
            {
                createElement("<object code=\"test.Hoge\"><constructor><argument>fuga</argument></constructor></object>"),
                new Object[]{"test.Hoge", importXML(new ConstructorMetaData(null), "<constructor><argument>fuga</argument></constructor>"), new MetaData[0], new MetaData[0], new MetaData[0]}
            },
            {
                createElement("<object code=\"test.Hoge\"><field name=\"field1\">1</field></object>"),
                new Object[]{"test.Hoge", null, new MetaData[]{importXML(new FieldMetaData(null), "<field name=\"field1\">1</field>")}, new MetaData[0], new MetaData[0]}
            },
            {
                createElement("<object code=\"test.Hoge\"><field name=\"field1\">1</field><field name=\"field2\">2</field></object>"),
                new Object[]{"test.Hoge", null, new MetaData[]{importXML(new FieldMetaData(null), "<field name=\"field1\">1</field>"), importXML(new FieldMetaData(null), "<field name=\"field2\">2</field>")}, new MetaData[0], new MetaData[0]}
            },
            {
                createElement("<object code=\"test.Hoge\"><attribute name=\"Attribute1\">1</attribute></object>"),
                new Object[]{"test.Hoge", null, new MetaData[0], new MetaData[]{importXML(new AttributeMetaData(null), "<attribute name=\"Attribute1\">1</attribute>")}, new MetaData[0]}
            },
            {
                createElement("<object code=\"test.Hoge\"><attribute name=\"Attribute1\">1</attribute><attribute name=\"Attribute2\">2</attribute></object>"),
                new Object[]{"test.Hoge", null, new MetaData[0], new MetaData[]{importXML(new AttributeMetaData(null), "<attribute name=\"Attribute1\">1</attribute>"), importXML(new AttributeMetaData(null), "<attribute name=\"Attribute2\">2</attribute>")}, new MetaData[0]}
            },
            {
                createElement("<object code=\"test.Hoge\"><invoke name=\"method1\"/></object>"),
                new Object[]{"test.Hoge", null, new MetaData[0], new MetaData[0], new MetaData[]{importXML(new InvokeMetaData(null), "<invoke name=\"method1\"/>")}}
            },
            {
                createElement("<object code=\"test.Hoge\"><invoke name=\"method1\"/><invoke name=\"method2\"/></object>"),
                new Object[]{"test.Hoge", null, new MetaData[0], new MetaData[0], new MetaData[]{importXML(new InvokeMetaData(null), "<invoke name=\"method1\"/>"), importXML(new InvokeMetaData(null), "<invoke name=\"method2\"/>")}}
            },
            {
                createElement("<object code=\"test.Hoge\"><ifdef name=\"test\" value=\"true\"><constructor><argument>fuga</argument></constructor></ifdef></object>"),
                new Object[]{"test.Hoge", importXML(new ConstructorMetaData(null), "<constructor><argument>fuga</argument></constructor>"), new MetaData[0], new MetaData[0], new MetaData[0]}
            },
            {
                createElement("<object code=\"test.Hoge\"><ifdef name=\"test\" value=\"false\"><constructor><argument>fuga</argument></constructor></ifdef></object>"),
                new Object[]{"test.Hoge", null, new MetaData[0], new MetaData[0], new MetaData[0]}
            },
            {
                createElement("<object code=\"test.Hoge\"><ifdef name=\"test\" value=\"true\"><field name=\"field1\">1</field></ifdef></object>"),
                new Object[]{"test.Hoge", null, new MetaData[]{importXML(new FieldMetaData(null), "<field name=\"field1\">1</field>")}, new MetaData[0], new MetaData[0]}
            },
            {
                createElement("<object code=\"test.Hoge\"><ifdef name=\"test\" value=\"false\"><field name=\"field1\">1</field></ifdef></object>"),
                new Object[]{"test.Hoge", null, new MetaData[0], new MetaData[0], new MetaData[0]}
            },
            {
                createElement("<object code=\"test.Hoge\"><ifdef name=\"test\" value=\"true\"><field name=\"field1\">1</field><field name=\"field2\">2</field></ifdef></object>"),
                new Object[]{"test.Hoge", null, new MetaData[]{importXML(new FieldMetaData(null), "<field name=\"field1\">1</field>"), importXML(new FieldMetaData(null), "<field name=\"field2\">2</field>")}, new MetaData[0], new MetaData[0]}
            },
            {
                createElement("<object code=\"test.Hoge\"><ifdef name=\"test\" value=\"false\"><field name=\"field1\">1</field><field name=\"field2\">2</field></ifdef></object>"),
                new Object[]{"test.Hoge", null, new MetaData[0], new MetaData[0], new MetaData[0]}
            },
            {
                createElement("<object code=\"test.Hoge\"><ifdef name=\"test\" value=\"true\"><attribute name=\"Attribute1\">1</attribute></ifdef></object>"),
                new Object[]{"test.Hoge", null, new MetaData[0], new MetaData[]{importXML(new AttributeMetaData(null), "<attribute name=\"Attribute1\">1</attribute>")}, new MetaData[0]}
            },
            {
                createElement("<object code=\"test.Hoge\"><ifdef name=\"test\" value=\"false\"><attribute name=\"Attribute1\">1</attribute></ifdef></object>"),
                new Object[]{"test.Hoge", null, new MetaData[0], new MetaData[0], new MetaData[0]}
            },
            {
                createElement("<object code=\"test.Hoge\"><ifdef name=\"test\" value=\"true\"><attribute name=\"Attribute1\">1</attribute><attribute name=\"Attribute2\">2</attribute></ifdef></object>"),
                new Object[]{"test.Hoge", null, new MetaData[0], new MetaData[]{importXML(new AttributeMetaData(null), "<attribute name=\"Attribute1\">1</attribute>"), importXML(new AttributeMetaData(null), "<attribute name=\"Attribute2\">2</attribute>")}, new MetaData[0]}
            },
            {
                createElement("<object code=\"test.Hoge\"><ifdef name=\"test\" value=\"false\"><attribute name=\"Attribute1\">1</attribute><attribute name=\"Attribute2\">2</attribute></ifdef></object>"),
                new Object[]{"test.Hoge", null, new MetaData[0], new MetaData[0], new MetaData[0]}
            },
            {
                createElement("<object code=\"test.Hoge\"><ifdef name=\"test\" value=\"true\"><invoke name=\"method1\"/></ifdef></object>"),
                new Object[]{"test.Hoge", null, new MetaData[0], new MetaData[0], new MetaData[]{importXML(new InvokeMetaData(null), "<invoke name=\"method1\"/>")}}
            },
            {
                createElement("<object code=\"test.Hoge\"><ifdef name=\"test\" value=\"false\"><invoke name=\"method1\"/></ifdef></object>"),
                new Object[]{"test.Hoge", null, new MetaData[0], new MetaData[0], new MetaData[0]}
            },
            {
                createElement("<object code=\"test.Hoge\"><ifdef name=\"test\" value=\"true\"><invoke name=\"method1\"/><invoke name=\"method2\"/></ifdef></object>"),
                new Object[]{"test.Hoge", null, new MetaData[0], new MetaData[0], new MetaData[]{importXML(new InvokeMetaData(null), "<invoke name=\"method1\"/>"), importXML(new InvokeMetaData(null), "<invoke name=\"method2\"/>")}}
            },
            {
                createElement("<object code=\"test.Hoge\"><ifdef name=\"test\" value=\"false\"><invoke name=\"method1\"/><invoke name=\"method2\"/></ifdef></object>"),
                new Object[]{"test.Hoge", null, new MetaData[0], new MetaData[0], new MetaData[0]}
            },
            {
                createElement("<object code=\"test.Hoge\"><constructor><argument>fuga</argument></constructor><ifdef name=\"test\" value=\"false\"><constructor><argument>fuga</argument></constructor></ifdef></object>"),
                new Object[]{"test.Hoge", importXML(new ConstructorMetaData(null), "<constructor><argument>fuga</argument></constructor>"), new MetaData[0], new MetaData[0], new MetaData[0]}
            },
            {
                createElement("<object code=\"test.Hoge\"><field name=\"field1\">1</field><ifdef name=\"test\" value=\"false\"><field name=\"field1\">1</field></ifdef></object>"),
                new Object[]{"test.Hoge", null, new MetaData[]{importXML(new FieldMetaData(null), "<field name=\"field1\">1</field>")}, new MetaData[0], new MetaData[0]}
            },
            {
                createElement("<object code=\"test.Hoge\"><attribute name=\"Attribute1\">1</attribute><ifdef name=\"test\" value=\"false\"><attribute name=\"Attribute1\">1</attribute></ifdef></object>"),
                new Object[]{"test.Hoge", null, new MetaData[0], new MetaData[]{importXML(new AttributeMetaData(null), "<attribute name=\"Attribute1\">1</attribute>")}, new MetaData[0]}
            },
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ObjectMetaData#importXML(org.w3c.dom.Element)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ConstructorMetaData#ConstructorMetaData\\(jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ConstructorMetaData#importXML\\(org\\.w3c\\.dom\\.Element\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ConstructorMetaData#toXML\\(java\\.lang\\.StringBuilder\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.FieldMetaData#FieldMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.FieldMetaData#importXML\\(org\\.w3c\\.dom\\.Element\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.FieldMetaData#toXML\\(java\\.lang\\.StringBuilder\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.AttributeMetaData#AttributeMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.AttributeMetaData#importXML\\(org\\.w3c\\.dom\\.Element\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.AttributeMetaData#toXML\\(java\\.lang\\.StringBuilder\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.InvokeMetaData#InvokeMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.InvokeMetaData#importXML\\(org\\.w3c\\.dom\\.Element\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.InvokeMetaData#toXML\\(java\\.lang\\.StringBuilder\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData#ObjectMetaData\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData#getCode\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData#getConstructor\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData#getFields\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData#getAttributes\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData#getInvokes\\(\\)"},
          dataProvider = "ImportXML")
    public void testImportXML(
        Element element,
        Object[] assertValues
    ) throws DeploymentException{
        System.setProperty("test", "true");
        try{
            ObjectMetaData metaData = new ObjectMetaData();
            metaData.importXML(element);
            assertEquals(metaData.getCode(), assertValues[0]);
            assertMetaDataEquals(metaData.getConstructor(), (MetaData)assertValues[1]);
            List<FieldMetaData> fieldList = metaData.getFields();
            MetaData[] fields = (MetaData[])assertValues[2];
            if(fields.length == 0){
                assertTrue(fieldList.size() == 0);
            }else{
                assertEquals(fieldList.size(), fields.length);
                for(int i = 0; i < fieldList.size(); i++){
                    assertMetaDataEquals(fieldList.get(i), fields[i]);
                }
            }
            List<AttributeMetaData> attributeList = metaData.getAttributes();
            MetaData[] attributes = (MetaData[])assertValues[3];
            if(attributes.length == 0){
                assertTrue(attributeList.size() == 0);
            }else{
                assertEquals(attributeList.size(), attributes.length);
                for(int i = 0; i < attributeList.size(); i++){
                    assertMetaDataEquals(attributeList.get(i), attributes[i]);
                }
            }
            List<InvokeMetaData> invokeList = metaData.getInvokes();
            MetaData[] invokes = (MetaData[])assertValues[4];
            if(invokes.length == 0){
                assertTrue(invokeList.size() == 0);
            }else{
                assertEquals(invokeList.size(), invokes.length);
                for(int i = 0; i < invokeList.size(); i++){
                    assertMetaDataEquals(invokeList.get(i), invokes[i]);
                }
            }
        }finally{
            System.clearProperty("test");
        }
    }

    @DataProvider(name = "ImportXMLError")
    public Object[][] dataImportXMLError() throws Exception{
        return new Object[][]{
            {null},
            {createElement("<hoge/>")},
            {createElement("<object/>")},
            {createElement("<object code=\"\"/>")},
            {createElement("<object code=\"test.Hoge\"><constructor><argument>fuga</argument></constructor><constructor><argument>fuga</argument></constructor></object>")},
            {createElement("<object code=\"test.Hoge\"><field name=\"field1\">1</field><field name=\"field1\">1</field></object>")},
            {createElement("<object code=\"test.Hoge\"><attribute name=\"Attribute1\">1</attribute><attribute name=\"Attribute1\">1</attribute></object>")},
            {createElement("<object code=\"test.Hoge\"><constructor><argument>fuga</argument></constructor><ifdef name=\"test\" value=\"true\"><constructor><argument>fuga</argument></constructor></ifdef></object>")},
            {createElement("<object code=\"test.Hoge\"><field name=\"field1\">1</field><ifdef name=\"test\" value=\"true\"><field name=\"field1\">1</field></ifdef></object>")},
            {createElement("<object code=\"test.Hoge\"><attribute name=\"Attribute1\">1</attribute><ifdef name=\"test\" value=\"true\"><attribute name=\"Attribute1\">1</attribute></ifdef></object>")},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ObjectMetaData#importXML(org.w3c.dom.Element)",
                    "Error"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData#ObjectMetaData\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ConstructorMetaData#ConstructorMetaData\\(jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ConstructorMetaData#importXML\\(org\\.w3c\\.dom\\.Element\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.FieldMetaData#FieldMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.FieldMetaData#importXML\\(org\\.w3c\\.dom\\.Element\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.AttributeMetaData#AttributeMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.AttributeMetaData#importXML\\(org\\.w3c\\.dom\\.Element\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.InvokeMetaData#InvokeMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.InvokeMetaData#importXML\\(org\\.w3c\\.dom\\.Element\\)"},
          expectedExceptions = {NullPointerException.class, DeploymentException.class},
          dataProvider = "ImportXMLError")
    public void testImportXMLError(Element element) throws DeploymentException{
        System.setProperty("test", "true");
        try{
            ObjectMetaData metaData = new ObjectMetaData();
            metaData.importXML(element);
        }finally{
            System.clearProperty("test");
        }
    }

    @DataProvider(name = "ToXML")
    public Object[][] dataToXML() throws Exception{
        return new Object[][]{
            {createElement("<object code=\"test.Hoge\"/>"), "<object code=\"test.Hoge\"/>"},
            {createElement("<object code=\"test.Hoge\"></object>"), "<object code=\"test.Hoge\"/>"},
            {createElement("<object code=\"test.Hoge\"><constructor><argument>fuga</argument></constructor></object>"), "<object code=\"test.Hoge\">" + System.getProperty("line.separator") + "    <constructor>" + System.getProperty("line.separator") + "        <argument>fuga</argument>" + System.getProperty("line.separator") + "    </constructor>" + System.getProperty("line.separator") + "</object>"},
            {createElement("<object code=\"test.Hoge\"><field name=\"field1\">1</field></object>"), "<object code=\"test.Hoge\">" + System.getProperty("line.separator") + "    <field name=\"field1\">1</field>" + System.getProperty("line.separator") + "</object>"},
            {createElement("<object code=\"test.Hoge\"><field name=\"field1\">1</field><field name=\"field2\">2</field></object>"), "<object code=\"test.Hoge\">" + System.getProperty("line.separator") + "    <field name=\"field1\">1</field>" + System.getProperty("line.separator") + "    <field name=\"field2\">2</field>" + System.getProperty("line.separator") + "</object>"},
            {createElement("<object code=\"test.Hoge\"><attribute name=\"Attribute1\">1</attribute></object>"), "<object code=\"test.Hoge\">" + System.getProperty("line.separator") + "    <attribute name=\"Attribute1\">1</attribute>" + System.getProperty("line.separator") + "</object>"},
            {createElement("<object code=\"test.Hoge\"><attribute name=\"Attribute1\">1</attribute><attribute name=\"Attribute2\">2</attribute></object>"), "<object code=\"test.Hoge\">" + System.getProperty("line.separator") + "    <attribute name=\"Attribute1\">1</attribute>" + System.getProperty("line.separator") + "    <attribute name=\"Attribute2\">2</attribute>" + System.getProperty("line.separator") + "</object>"},
            {createElement("<object code=\"test.Hoge\"><invoke name=\"method1\"/></object>"), "<object code=\"test.Hoge\">" + System.getProperty("line.separator") + "    <invoke name=\"method1\"/>" + System.getProperty("line.separator") + "</object>"},
            {createElement("<object code=\"test.Hoge\"><invoke name=\"method1\"/><invoke name=\"method2\"/></object>"), "<object code=\"test.Hoge\">" + System.getProperty("line.separator") + "    <invoke name=\"method1\"/>" + System.getProperty("line.separator") + "    <invoke name=\"method2\"/>" + System.getProperty("line.separator") + "</object>"},
            {createElement("<object code=\"test.Hoge\"><ifdef name=\"test\" value=\"true\"><constructor><argument>fuga</argument></constructor></ifdef></object>"), "<object code=\"test.Hoge\">" + System.getProperty("line.separator") + "    <ifdef name=\"test\" value=\"true\">" + System.getProperty("line.separator") + "        <constructor>" + System.getProperty("line.separator") + "            <argument>fuga</argument>" + System.getProperty("line.separator") + "        </constructor>" + System.getProperty("line.separator") + "    </ifdef>" + System.getProperty("line.separator") + "</object>"},
            {createElement("<object code=\"test.Hoge\"><ifdef name=\"test\" value=\"true\"><field name=\"field1\">1</field></ifdef></object>"), "<object code=\"test.Hoge\">" + System.getProperty("line.separator") + "    <ifdef name=\"test\" value=\"true\">" + System.getProperty("line.separator") + "        <field name=\"field1\">1</field>" + System.getProperty("line.separator") + "    </ifdef>" + System.getProperty("line.separator") + "</object>"},
            {createElement("<object code=\"test.Hoge\"><ifdef name=\"test\" value=\"true\"><field name=\"field1\">1</field><field name=\"field2\">2</field></ifdef></object>"), "<object code=\"test.Hoge\">" + System.getProperty("line.separator") + "    <ifdef name=\"test\" value=\"true\">" + System.getProperty("line.separator") + "        <field name=\"field1\">1</field>" + System.getProperty("line.separator") + "        <field name=\"field2\">2</field>" + System.getProperty("line.separator") + "    </ifdef>" + System.getProperty("line.separator") + "</object>"},
            {createElement("<object code=\"test.Hoge\"><ifdef name=\"test\" value=\"true\"><attribute name=\"Attribute1\">1</attribute></ifdef></object>"), "<object code=\"test.Hoge\">" + System.getProperty("line.separator") + "    <ifdef name=\"test\" value=\"true\">" + System.getProperty("line.separator") + "        <attribute name=\"Attribute1\">1</attribute>" + System.getProperty("line.separator") + "    </ifdef>" + System.getProperty("line.separator") + "</object>"},
            {createElement("<object code=\"test.Hoge\"><ifdef name=\"test\" value=\"true\"><attribute name=\"Attribute1\">1</attribute><attribute name=\"Attribute2\">2</attribute></ifdef></object>"), "<object code=\"test.Hoge\">" + System.getProperty("line.separator") + "    <ifdef name=\"test\" value=\"true\">" + System.getProperty("line.separator") + "        <attribute name=\"Attribute1\">1</attribute>" + System.getProperty("line.separator") + "        <attribute name=\"Attribute2\">2</attribute>" + System.getProperty("line.separator") + "    </ifdef>" + System.getProperty("line.separator") + "</object>"},
            {createElement("<object code=\"test.Hoge\"><ifdef name=\"test\" value=\"true\"><invoke name=\"method1\"/></ifdef></object>"), "<object code=\"test.Hoge\">" + System.getProperty("line.separator") + "    <ifdef name=\"test\" value=\"true\">" + System.getProperty("line.separator") + "        <invoke name=\"method1\"/>" + System.getProperty("line.separator") + "    </ifdef>" + System.getProperty("line.separator") + "</object>"},
            {createElement("<object code=\"test.Hoge\"><ifdef name=\"test\" value=\"true\"><invoke name=\"method1\"/><invoke name=\"method2\"/></ifdef></object>"), "<object code=\"test.Hoge\">" + System.getProperty("line.separator") + "    <ifdef name=\"test\" value=\"true\">" + System.getProperty("line.separator") + "        <invoke name=\"method1\"/>" + System.getProperty("line.separator") + "        <invoke name=\"method2\"/>" + System.getProperty("line.separator") + "    </ifdef>" + System.getProperty("line.separator") + "</object>"},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ObjectMetaData#toXML(java.lang.StringBuilder)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData#ObjectMetaData\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData#importXML\\(org\\.w3c\\.dom\\.Element\\)"},
          dataProvider = "ToXML")
    public void testToXML(Element element, String expected) throws DeploymentException{
        ObjectMetaData metaData = new ObjectMetaData();
        metaData.importXML(element);
        StringBuilder buf = new StringBuilder();
        metaData.toXML(buf);
        assertEquals(buf.toString(), expected);
    }
}
