/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.core;

import static org.testng.Assert.*;

import java.util.List;

import org.testng.annotations.Test;
import org.testng.annotations.DataProvider;
import org.w3c.dom.Element;

import static jp.ossc.nimbus.TestUtility.*;

/**
 * {@link ServiceMetaData}eXgP[XB<p>
 * 
 * @author M.Takata
 */
public class ServiceMetaDataTest{
    
    @Test(groups = {"jp.ossc.nimbus.core.ServiceMetaData#ServiceMetaData()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData#.*"})
    public void testServiceMetaData(){
        ServiceMetaData metaData = new ServiceMetaData();
        
        assertNull(metaData.getCode());
        assertNull(metaData.getServiceLoader());
        assertNull(metaData.getParent());
        assertNull(metaData.getManagerName());
        assertNull(metaData.getConstructor());
        assertEquals(metaData.getFields().size(), 0);
        assertEquals(metaData.getAttributes().size(), 0);
        assertEquals(metaData.getInvokes().size(), 0);
        
        assertNull(metaData.getName());
        assertEquals(metaData.getInitState(), Service.State.STARTED.toString());
        assertEquals(metaData.getInitStateValue(), Service.State.STARTED);
        assertEquals(metaData.getInstance(), ServiceMetaData.INSTANCE_TYPE_SINGLETON);
        assertFalse(metaData.isFactory());
        assertFalse(metaData.isManagement());
        assertTrue(metaData.isCreateTemplate());
        assertNull(metaData.getOptionalConfig());
        assertEquals(metaData.getDepends().size(), 0);
        assertNull(metaData.getManager());
    }

    @DataProvider(name = "ServiceMetaData2")
    public Object[][] dataServiceMetaData2(){
        return new Object[][]{
            {null, null},
            {null, "test.Hoge"},
            {"Hoge", null},
            {"Hoge", "test.Hoge"},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceMetaData#ServiceMetaData(java.lang.String,java.lang.String)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData#.*"},
          dataProvider = "ServiceMetaData2")
    public void testServiceMetaData2(String name, String code){
        ServiceMetaData metaData = new ServiceMetaData(name, code);
        
        assertNull(metaData.getServiceLoader());
        assertNull(metaData.getParent());
        assertNull(metaData.getManagerName());
        assertNull(metaData.getConstructor());
        assertEquals(metaData.getFields().size(), 0);
        assertEquals(metaData.getAttributes().size(), 0);
        assertEquals(metaData.getInvokes().size(), 0);
        
        assertEquals(metaData.getName(), name);
        assertEquals(metaData.getCode(), code);
        assertEquals(metaData.getInitState(), Service.State.STARTED.toString());
        assertEquals(metaData.getInitStateValue(), Service.State.STARTED);
        assertEquals(metaData.getInstance(), ServiceMetaData.INSTANCE_TYPE_SINGLETON);
        assertFalse(metaData.isFactory());
        assertFalse(metaData.isManagement());
        assertTrue(metaData.isCreateTemplate());
        assertNull(metaData.getOptionalConfig());
        assertEquals(metaData.getDepends().size(), 0);
        assertNull(metaData.getManager());
    }

    @DataProvider(name = "ServiceMetaData3")
    public Object[][] dataServiceMetaData3() throws Exception{
        return new Object[][]{
            {null, null, null},
            {new DefaultServiceLoaderService(), null, null},
            {new DefaultServiceLoaderService(), new ManagerMetaData(), null},
            {null, new ManagerMetaData(), null},
            {null, new ManagerMetaData(), new ManagerMetaData()},
            {null, null, new ManagerMetaData()},
            {new DefaultServiceLoaderService(), null, new ManagerMetaData()},
            {new DefaultServiceLoaderService(), new ManagerMetaData(), new ManagerMetaData()},
            {new DefaultServiceLoaderService(), new ManagerMetaData(), importXML(new ManagerMetaData(), "<manager name=\"manager\"/>")},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceMetaData#ServiceMetaData(jp.ossc.nimbus.core.ServiceLoader,jp.ossc.nimbus.core.MetaData,,jp.ossc.nimbus.core.ManagerMetaData)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData#.*"},
          dataProvider = "ServiceMetaData3")
    public void testServiceMetaData3(ServiceLoader loader, MetaData parent, ManagerMetaData manager){
        ServiceMetaData metaData = new ServiceMetaData(loader, parent, manager);
        
        assertNull(metaData.getCode());
        assertEquals(metaData.getServiceLoader(), loader);
        assertEquals(metaData.getParent(), parent);
        assertEquals(metaData.getManagerName(), manager == null ? null : manager.getName());
        assertNull(metaData.getConstructor());
        assertEquals(metaData.getFields().size(), 0);
        assertEquals(metaData.getAttributes().size(), 0);
        assertEquals(metaData.getInvokes().size(), 0);
        
        assertNull(metaData.getName());
        assertEquals(metaData.getInitState(), Service.State.STARTED.toString());
        assertEquals(metaData.getInitStateValue(), Service.State.STARTED);
        assertEquals(metaData.getInstance(), ServiceMetaData.INSTANCE_TYPE_SINGLETON);
        assertFalse(metaData.isFactory());
        assertFalse(metaData.isManagement());
        assertTrue(metaData.isCreateTemplate());
        assertNull(metaData.getOptionalConfig());
        assertEquals(metaData.getDepends().size(), 0);
        assertEquals(metaData.getManager(), manager);
    }

    @DataProvider(name = "SetName")
    public Object[][] dataSetName(){
        return new Object[][]{
            {null},
            {"Hoge"}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceMetaData#setName(java.lang.String)",
                    "jp.ossc.nimbus.core.ServiceMetaData#getName()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceMetaData#ServiceMetaData\\(\\)"},
          dataProvider = "SetName")
    public void testSetName(String name){
        ServiceMetaData metaData = new ServiceMetaData();
        metaData.setName(name);
        assertEquals(metaData.getName(), name);
    }

    @DataProvider(name = "SetManager")
    public Object[][] dataSetManager(){
        return new Object[][]{
            {null},
            {new ManagerMetaData()}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceMetaData#setManager(jp.ossc.nimbus.core.ManaerMetaData)",
                    "jp.ossc.nimbus.core.ServiceMetaData#getManager()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceMetaData#ServiceMetaData\\(\\)"},
          dataProvider = "SetManager")
    public void testSetManager(ManagerMetaData manager){
        ServiceMetaData metaData = new ServiceMetaData();
        metaData.setManager(manager);
        assertEquals(metaData.getManager(), manager);
    }

    @DataProvider(name = "SetInitState")
    public Object[][] dataSetInitState(){
        return new Object[][]{
            {Service.State.CREATED.toString()},
            {Service.State.STARTED.toString()},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceMetaData#setInitState(java.lang.String)",
                    "jp.ossc.nimbus.core.ServiceMetaData#getInitState()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceMetaData#ServiceMetaData\\(\\)"},
          dataProvider = "SetInitState")
    public void testSetInitState(String state){
        ServiceMetaData metaData = new ServiceMetaData();
        metaData.setInitState(state);
        assertEquals(metaData.getInitState(), state);
    }

    @DataProvider(name = "SetInitStateError")
    public Object[][] dataSetInitStateError(){
        return new Object[][]{
            {null},
            {Service.State.CREATING.toString()},
            {Service.State.STARTING.toString()},
            {Service.State.STOPPING.toString()},
            {Service.State.STOPPED.toString()},
            {Service.State.DESTROYING.toString()},
            {Service.State.DESTROYED.toString()},
            {Service.State.UNKNOWN.toString()},
            {"hoge"}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceMetaData#setInitState(java.lang.String)",
                    "Error"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceMetaData#ServiceMetaData\\(\\)"},
          expectedExceptions = IllegalArgumentException.class,
          dataProvider = "SetInitStateError")
    public void testSetInitStateError(String state){
        ServiceMetaData metaData = new ServiceMetaData();
        metaData.setInitState(state);
    }

    @DataProvider(name = "SetInitStateValue")
    public Object[][] dataSetInitStateValue(){
        return new Object[][]{
            {Service.State.CREATED},
            {Service.State.STARTED},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceMetaData#setInitStateValue(jp.ossc.nimbus.core.Service.State)",
                    "jp.ossc.nimbus.core.ServiceMetaData#getInitStateValue()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceMetaData#ServiceMetaData\\(\\)"},
          dataProvider = "SetInitStateValue")
    public void testSetInitStateValue(Service.State state){
        ServiceMetaData metaData = new ServiceMetaData();
        metaData.setInitStateValue(state);
        assertEquals(metaData.getInitStateValue(), state);
    }

    @DataProvider(name = "SetInitStateValueError")
    public Object[][] dataSetInitStateValueError(){
        return new Object[][]{
            {null},
            {Service.State.CREATING},
            {Service.State.STARTING},
            {Service.State.STOPPING},
            {Service.State.STOPPED},
            {Service.State.DESTROYING},
            {Service.State.DESTROYED},
            {Service.State.UNKNOWN}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceMetaData#setInitStateValue(jp.ossc.nimbus.core.Service.State)",
                    "Error"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceMetaData#ServiceMetaData\\(\\)"},
          expectedExceptions = IllegalArgumentException.class,
          dataProvider = "SetInitStateValueError")
    public void testSetInitStateValueError(Service.State state){
        ServiceMetaData metaData = new ServiceMetaData();
        metaData.setInitStateValue(state);
    }

    @DataProvider(name = "SetInstance")
    public Object[][] dataSetInstance(){
        return new Object[][]{
            {ServiceMetaData.INSTANCE_TYPE_FACTORY},
            {ServiceMetaData.INSTANCE_TYPE_SINGLETON},
            {ServiceMetaData.INSTANCE_TYPE_THREADLOCAL}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceMetaData#setInstance(java.lang.String)",
                    "jp.ossc.nimbus.core.ServiceMetaData#getInstance()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceMetaData#ServiceMetaData\\(\\)"},
          dataProvider = "SetInstance")
    public void testSetInstance(String instance){
        ServiceMetaData metaData = new ServiceMetaData();
        metaData.setInstance(instance);
        assertEquals(metaData.getInstance(), instance);
    }

    @DataProvider(name = "SetInstanceError")
    public Object[][] dataSetInstanceError(){
        return new Object[][]{
            {null},
            {"hoge"}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceMetaData#setInstance(java.lang.String)",
                    "Error"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceMetaData#ServiceMetaData\\(\\)"},
          expectedExceptions = IllegalArgumentException.class,
          dataProvider = "SetInstanceError")
    public void testSetInstanceError(String instance){
        ServiceMetaData metaData = new ServiceMetaData();
        metaData.setInstance(instance);
    }

    @DataProvider(name = "SetFactory")
    public Object[][] dataSetFactory(){
        return new Object[][]{
            {true},
            {false},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceMetaData#setFactory(boolean)",
                    "jp.ossc.nimbus.core.ServiceMetaData#isFactory()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceMetaData#ServiceMetaData\\(\\)"},
          dataProvider = "SetFactory")
    public void testSetFactory(boolean isFactory){
        ServiceMetaData metaData = new ServiceMetaData();
        metaData.setFactory(isFactory);
        assertEquals(metaData.isFactory(), isFactory);
    }

    @DataProvider(name = "SetManagement")
    public Object[][] dataSetManagement(){
        return new Object[][]{
            {true},
            {false},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceMetaData#setManagement(boolean)",
                    "jp.ossc.nimbus.core.ServiceMetaData#isManagement()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceMetaData#ServiceMetaData\\(\\)"},
          dataProvider = "SetManagement")
    public void testSetManagement(boolean isManagement){
        ServiceMetaData metaData = new ServiceMetaData();
        metaData.setManagement(isManagement);
        assertEquals(metaData.isManagement(), isManagement);
    }

    @DataProvider(name = "SetCreateTemplate")
    public Object[][] dataSetCreateTemplate(){
        return new Object[][]{
            {true},
            {false},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceMetaData#setCreateTemplate(boolean)",
                    "jp.ossc.nimbus.core.ServiceMetaData#isCreateTemplate()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceMetaData#ServiceMetaData\\(\\)"},
          dataProvider = "SetCreateTemplate")
    public void testSetCreateTemplate(boolean isCreateTemplate){
        ServiceMetaData metaData = new ServiceMetaData();
        metaData.setCreateTemplate(isCreateTemplate);
        assertEquals(metaData.isCreateTemplate(), isCreateTemplate);
    }

    @DataProvider(name = "SetOptionalConfig")
    public Object[][] dataSetOptionalConfig() throws Exception{
        return new Object[][]{
            {null},
            {createElement("<test/>")},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceMetaData#setOptionalConfig(org.w3c.dom.Element)",
                    "jp.ossc.nimbus.core.ServiceMetaData#isOptionalConfig()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceMetaData#ServiceMetaData\\(\\)"},
          dataProvider = "SetOptionalConfig")
    public void testSetOptionalConfig(Element element){
        ServiceMetaData metaData = new ServiceMetaData();
        metaData.setOptionalConfig(element);
        assertEquals(metaData.getOptionalConfig(), element);
    }

    @DataProvider(name = "AddDepends")
    public Object[][] dataAddDepends(){
        ServiceMetaData metaData = new ServiceMetaData();
        return new Object[][]{
            {null},
            {new ServiceMetaData.DependsMetaData[]{metaData.new DependsMetaData(null)}},
            {new ServiceMetaData.DependsMetaData[]{metaData.new DependsMetaData(null), metaData.new DependsMetaData(null), metaData.new DependsMetaData(null), metaData.new DependsMetaData(null), metaData.new DependsMetaData(null), metaData.new DependsMetaData(null), metaData.new DependsMetaData(null), metaData.new DependsMetaData(null), metaData.new DependsMetaData(null), metaData.new DependsMetaData(null), metaData.new DependsMetaData(null)}},
            {new ServiceMetaData.DependsMetaData[]{null, metaData.new DependsMetaData(null)}}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceMetaData#addDepends(jp.ossc.nimbus.core.ServiceMetaData.DependsMetaData)",
                    "jp.ossc.nimbus.core.ServiceMetaData#getDepends()",
                    "jp.ossc.nimbus.core.ServiceMetaData#removeDepends(jp.ossc.nimbus.core.ServiceMetaData.DependsMetaData)",
                    "jp.ossc.nimbus.core.ServiceMetaData#clearDepends()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceMetaData#ServiceMetaData\\(.*\\)"},
          dataProvider = "AddDepends")
    public void testAddDepends(ServiceMetaData.DependsMetaData[] depends){
        ServiceMetaData metaData = new ServiceMetaData();
        if(depends == null || depends.length == 0){
            assertEquals(metaData.getDepends().size(), 0);
        }else{
            for(ServiceMetaData.DependsMetaData depend : depends){
                metaData.addDepends(depend);
            }
            List<ServiceMetaData.DependsMetaData> list = metaData.getDepends();
            for(int i = 0; i < list.size(); i++){
                assertEquals(list.get(i), depends[i]);
            }
            for(int i = 0, imax = list.size(); i < imax; i+=2){
                metaData.removeDepends(depends[i]);
            }
            list = metaData.getDepends();
            int index = 1;
            for(int i = 0; i < list.size(); i++){
                assertEquals(list.get(i), depends[index]);
                index += 2;
            }
            metaData.clearDepends();
            assertEquals(metaData.getDepends().size(), 0);
        }
    }

    @DataProvider(name = "CreateDependsMetaData")
    public Object[][] dataCreateDependsMetaData(){
        return new Object[][]{
            {null, null, ServiceManager.DEFAULT_NAME},
            {null, "service", ServiceManager.DEFAULT_NAME},
            {"manager", null, "manager"},
            {"manager", "service", "manager"},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceMetaData#createDependsMetaData(java.lang.String,java.lang.String)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceMetaData#ServiceMetaData\\(\\)"},
          dataProvider = "CreateDependsMetaData")
    public void testCreateDependsMetaData(String manager, String service, String expectedManager){
        ServiceMetaData metaData = new ServiceMetaData();
        ServiceNameMetaData nameData = metaData.createDependsMetaData(manager, service);
        assertNotNull(nameData);
        assertEquals(nameData.tagName, "depends");
        assertEquals(nameData.getParent(), metaData);
        assertEquals(nameData.getManagerName(), expectedManager);
        assertEquals(nameData.getServiceName(), service);
    }

    @DataProvider(name = "ImportXML")
    public Object[][] dataImportXML() throws Exception{
        ServiceMetaData metaData = new ServiceMetaData();
        return new Object[][]{
            {
                createElement("<service name=\"Hoge\" code=\"test.Hoge\"/>"),
                new Object[]{
                    "Hoge",
                    "test.Hoge",
                    Service.State.STARTED,
                    ServiceMetaData.INSTANCE_TYPE_SINGLETON,
                    false,
                    false,
                    true,
                    null,
                    null,
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[0],
                }
            },
            {
                createElement("<service name=\"Hoge\" code=\"test.Hoge\" initState=\"Created\"/>"),
                new Object[]{
                    "Hoge",
                    "test.Hoge",
                    Service.State.CREATED,
                    ServiceMetaData.INSTANCE_TYPE_SINGLETON,
                    false,
                    false,
                    true,
                    null,
                    null,
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[0],
                }
            },
            {
                createElement("<service name=\"Hoge\" code=\"test.Hoge\" initState=\"Started\"/>"),
                new Object[]{
                    "Hoge",
                    "test.Hoge",
                    Service.State.STARTED,
                    ServiceMetaData.INSTANCE_TYPE_SINGLETON,
                    false,
                    false,
                    true,
                    null,
                    null,
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[0],
                }
            },
            {
                createElement("<service name=\"Hoge\" code=\"test.Hoge\" instance=\"singleton\"/>"),
                new Object[]{
                    "Hoge",
                    "test.Hoge",
                    Service.State.STARTED,
                    ServiceMetaData.INSTANCE_TYPE_SINGLETON,
                    false,
                    false,
                    true,
                    null,
                    null,
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[0],
                }
            },
            {
                createElement("<service name=\"Hoge\" code=\"test.Hoge\" instance=\"factory\"/>"),
                new Object[]{
                    "Hoge",
                    "test.Hoge",
                    Service.State.STARTED,
                    ServiceMetaData.INSTANCE_TYPE_FACTORY,
                    true,
                    false,
                    true,
                    null,
                    null,
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[0],
                }
            },
            {
                createElement("<service name=\"Hoge\" code=\"test.Hoge\" instance=\"threadlocal\"/>"),
                new Object[]{
                    "Hoge",
                    "test.Hoge",
                    Service.State.STARTED,
                    ServiceMetaData.INSTANCE_TYPE_THREADLOCAL,
                    true,
                    false,
                    true,
                    null,
                    null,
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[0],
                }
            },
            {
                createElement("<service name=\"Hoge\" code=\"test.Hoge\" instance=\"factory\" management=\"true\"/>"),
                new Object[]{
                    "Hoge",
                    "test.Hoge",
                    Service.State.STARTED,
                    ServiceMetaData.INSTANCE_TYPE_FACTORY,
                    true,
                    true,
                    true,
                    null,
                    null,
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[0],
                }
            },
            {
                createElement("<service name=\"Hoge\" code=\"test.Hoge\" instance=\"factory\" management=\"false\"/>"),
                new Object[]{
                    "Hoge",
                    "test.Hoge",
                    Service.State.STARTED,
                    ServiceMetaData.INSTANCE_TYPE_FACTORY,
                    true,
                    false,
                    true,
                    null,
                    null,
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[0],
                }
            },
            {
                createElement("<service name=\"Hoge\" code=\"test.Hoge\" instance=\"factory\" createTemplate=\"true\"/>"),
                new Object[]{
                    "Hoge",
                    "test.Hoge",
                    Service.State.STARTED,
                    ServiceMetaData.INSTANCE_TYPE_FACTORY,
                    true,
                    false,
                    true,
                    null,
                    null,
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[0],
                }
            },
            {
                createElement("<service name=\"Hoge\" code=\"test.Hoge\" instance=\"factory\" createTemplate=\"false\"/>"),
                new Object[]{
                    "Hoge",
                    "test.Hoge",
                    Service.State.STARTED,
                    ServiceMetaData.INSTANCE_TYPE_FACTORY,
                    true,
                    false,
                    false,
                    null,
                    null,
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[0],
                }
            },
            {
                createElement("<service name=\"Hoge\" code=\"test.Hoge\"><optional-config><hoge>fuga</hoge></optional-config></service>"),
                new Object[]{
                    "Hoge",
                    "test.Hoge",
                    Service.State.STARTED,
                    ServiceMetaData.INSTANCE_TYPE_SINGLETON,
                    false,
                    false,
                    true,
                    createElement("<hoge>fuga</hoge>"),
                    null,
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[0],
                }
            },
            {
                createElement("<service name=\"Hoge\" code=\"test.Hoge\"><ifdef name=\"test\" value=\"true\"><optional-config><hoge>fuga</hoge></optional-config></ifdef></service>"),
                new Object[]{
                    "Hoge",
                    "test.Hoge",
                    Service.State.STARTED,
                    ServiceMetaData.INSTANCE_TYPE_SINGLETON,
                    false,
                    false,
                    true,
                    createElement("<hoge>fuga</hoge>"),
                    null,
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[0],
                }
            },
            {
                createElement("<service name=\"Hoge\" code=\"test.Hoge\"><ifdef name=\"test\" value=\"false\"><optional-config><hoge>fuga</hoge></optional-config></ifdef></service>"),
                new Object[]{
                    "Hoge",
                    "test.Hoge",
                    Service.State.STARTED,
                    ServiceMetaData.INSTANCE_TYPE_SINGLETON,
                    false,
                    false,
                    true,
                    null,
                    null,
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[0],
                }
            },
            {
                createElement("<service name=\"Hoge\" code=\"test.Hoge\"><constructor><argument>fuga</argument></constructor></service>"),
                new Object[]{
                    "Hoge",
                    "test.Hoge",
                    Service.State.STARTED,
                    ServiceMetaData.INSTANCE_TYPE_SINGLETON,
                    false,
                    false,
                    true,
                    null,
                    importXML(new ConstructorMetaData(null), "<constructor><argument>fuga</argument></constructor>"),
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[0],
                }
            },
            {
                createElement("<service name=\"Hoge\" code=\"test.Hoge\"><ifdef name=\"test\" value=\"true\"><constructor><argument>fuga</argument></constructor></ifdef></service>"),
                new Object[]{
                    "Hoge",
                    "test.Hoge",
                    Service.State.STARTED,
                    ServiceMetaData.INSTANCE_TYPE_SINGLETON,
                    false,
                    false,
                    true,
                    null,
                    importXML(new ConstructorMetaData(null), "<constructor><argument>fuga</argument></constructor>"),
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[0],
                }
            },
            {
                createElement("<service name=\"Hoge\" code=\"test.Hoge\"><ifdef name=\"test\" value=\"false\"><constructor><argument>fuga</argument></constructor></ifdef></service>"),
                new Object[]{
                    "Hoge",
                    "test.Hoge",
                    Service.State.STARTED,
                    ServiceMetaData.INSTANCE_TYPE_SINGLETON,
                    false,
                    false,
                    true,
                    null,
                    null,
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[0],
                }
            },
            {
                createElement("<service name=\"Hoge\" code=\"test.Hoge\"><field name=\"field1\">1</field></service>"),
                new Object[]{
                    "Hoge",
                    "test.Hoge",
                    Service.State.STARTED,
                    ServiceMetaData.INSTANCE_TYPE_SINGLETON,
                    false,
                    false,
                    true,
                    null,
                    null,
                    new MetaData[]{importXML(new FieldMetaData(null), "<field name=\"field1\">1</field>")},
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[0],
                }
            },
            {
                createElement("<service name=\"Hoge\" code=\"test.Hoge\"><ifdef name=\"test\" value=\"true\"><field name=\"field1\">1</field></ifdef></service>"),
                new Object[]{
                    "Hoge",
                    "test.Hoge",
                    Service.State.STARTED,
                    ServiceMetaData.INSTANCE_TYPE_SINGLETON,
                    false,
                    false,
                    true,
                    null,
                    null,
                    new MetaData[]{importXML(new FieldMetaData(null), "<field name=\"field1\">1</field>")},
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[0],
                }
            },
            {
                createElement("<service name=\"Hoge\" code=\"test.Hoge\"><ifdef name=\"test\" value=\"false\"><field name=\"field1\">1</field></ifdef></service>"),
                new Object[]{
                    "Hoge",
                    "test.Hoge",
                    Service.State.STARTED,
                    ServiceMetaData.INSTANCE_TYPE_SINGLETON,
                    false,
                    false,
                    true,
                    null,
                    null,
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[0],
                }
            },
            {
                createElement("<service name=\"Hoge\" code=\"test.Hoge\"><field name=\"field1\">1</field><field name=\"field2\">2</field></service>"),
                new Object[]{
                    "Hoge",
                    "test.Hoge",
                    Service.State.STARTED,
                    ServiceMetaData.INSTANCE_TYPE_SINGLETON,
                    false,
                    false,
                    true,
                    null,
                    null,
                    new MetaData[]{importXML(new FieldMetaData(null), "<field name=\"field1\">1</field>"), importXML(new FieldMetaData(null), "<field name=\"field2\">2</field>")},
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[0],
                }
            },
            {
                createElement("<service name=\"Hoge\" code=\"test.Hoge\"><attribute name=\"Attribute1\">1</attribute></service>"),
                new Object[]{
                    "Hoge",
                    "test.Hoge",
                    Service.State.STARTED,
                    ServiceMetaData.INSTANCE_TYPE_SINGLETON,
                    false,
                    false,
                    true,
                    null,
                    null,
                    new MetaData[0],
                    new MetaData[]{importXML(new AttributeMetaData(null), "<attribute name=\"Attribute1\">1</attribute>")},
                    new MetaData[0],
                    new MetaData[0],
                }
            },
            {
                createElement("<service name=\"Hoge\" code=\"test.Hoge\"><ifdef name=\"test\" value=\"true\"><attribute name=\"Attribute1\">1</attribute></ifdef></service>"),
                new Object[]{
                    "Hoge",
                    "test.Hoge",
                    Service.State.STARTED,
                    ServiceMetaData.INSTANCE_TYPE_SINGLETON,
                    false,
                    false,
                    true,
                    null,
                    null,
                    new MetaData[0],
                    new MetaData[]{importXML(new AttributeMetaData(null), "<attribute name=\"Attribute1\">1</attribute>")},
                    new MetaData[0],
                    new MetaData[0],
                }
            },
            {
                createElement("<service name=\"Hoge\" code=\"test.Hoge\"><ifdef name=\"test\" value=\"false\"><attribute name=\"Attribute1\">1</attribute></ifdef></service>"),
                new Object[]{
                    "Hoge",
                    "test.Hoge",
                    Service.State.STARTED,
                    ServiceMetaData.INSTANCE_TYPE_SINGLETON,
                    false,
                    false,
                    true,
                    null,
                    null,
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[0],
                }
            },
            {
                createElement("<service name=\"Hoge\" code=\"test.Hoge\"><attribute name=\"Attribute1\">1</attribute><attribute name=\"Attribute2\">2</attribute></service>"),
                new Object[]{
                    "Hoge",
                    "test.Hoge",
                    Service.State.STARTED,
                    ServiceMetaData.INSTANCE_TYPE_SINGLETON,
                    false,
                    false,
                    true,
                    null,
                    null,
                    new MetaData[0],
                    new MetaData[]{importXML(new AttributeMetaData(null), "<attribute name=\"Attribute1\">1</attribute>"), importXML(new AttributeMetaData(null), "<attribute name=\"Attribute2\">2</attribute>")},
                    new MetaData[0],
                    new MetaData[0],
                }
            },
            {
                createElement("<service name=\"Hoge\" code=\"test.Hoge\"><invoke name=\"method1\"/></service>"),
                new Object[]{
                    "Hoge",
                    "test.Hoge",
                    Service.State.STARTED,
                    ServiceMetaData.INSTANCE_TYPE_SINGLETON,
                    false,
                    false,
                    true,
                    null,
                    null,
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[]{importXML(new InvokeMetaData(metaData), "<invoke name=\"method1\"/>")},
                    new MetaData[0],
                }
            },
            {
                createElement("<service name=\"Hoge\" code=\"test.Hoge\"><ifdef name=\"test\" value=\"true\"><invoke name=\"method1\"/></ifdef></service>"),
                new Object[]{
                    "Hoge",
                    "test.Hoge",
                    Service.State.STARTED,
                    ServiceMetaData.INSTANCE_TYPE_SINGLETON,
                    false,
                    false,
                    true,
                    null,
                    null,
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[]{importXML(new InvokeMetaData(metaData), "<invoke name=\"method1\"/>")},
                    new MetaData[0],
                }
            },
            {
                createElement("<service name=\"Hoge\" code=\"test.Hoge\"><ifdef name=\"test\" value=\"false\"><invoke name=\"method1\"/></ifdef></service>"),
                new Object[]{
                    "Hoge",
                    "test.Hoge",
                    Service.State.STARTED,
                    ServiceMetaData.INSTANCE_TYPE_SINGLETON,
                    false,
                    false,
                    true,
                    null,
                    null,
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[0],
                }
            },
            {
                createElement("<service name=\"Hoge\" code=\"test.Hoge\"><invoke name=\"method1\" callState=\"Started\"/></service>"),
                new Object[]{
                    "Hoge",
                    "test.Hoge",
                    Service.State.STARTED,
                    ServiceMetaData.INSTANCE_TYPE_SINGLETON,
                    false,
                    false,
                    true,
                    null,
                    null,
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[]{importXML(new InvokeMetaData(metaData), "<invoke name=\"method1\" callState=\"Started\"/>")},
                    new MetaData[0],
                }
            },
            {
                createElement("<service name=\"Hoge\" code=\"test.Hoge\"><invoke name=\"method1\"/><invoke name=\"method2\"/></service>"),
                new Object[]{
                    "Hoge",
                    "test.Hoge",
                    Service.State.STARTED,
                    ServiceMetaData.INSTANCE_TYPE_SINGLETON,
                    false,
                    false,
                    true,
                    null,
                    null,
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[]{importXML(new InvokeMetaData(metaData), "<invoke name=\"method1\"/>"), importXML(new InvokeMetaData(metaData), "<invoke name=\"method2\"/>")},
                    new MetaData[0],
                }
            },
            {
                createElement("<service name=\"Hoge\" code=\"test.Hoge\"><depends>Fuga</depends></service>"),
                new Object[]{
                    "Hoge",
                    "test.Hoge",
                    Service.State.STARTED,
                    ServiceMetaData.INSTANCE_TYPE_SINGLETON,
                    false,
                    false,
                    true,
                    null,
                    null,
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[]{metaData.createDependsMetaData("Nimbus", "Fuga")},
                }
            },
            {
                createElement("<service name=\"Hoge\" code=\"test.Hoge\"><depends><service name=\"Fuga\" code=\"test.Fuga\"/></depends></service>"),
                new Object[]{
                    "Hoge",
                    "test.Hoge",
                    Service.State.STARTED,
                    ServiceMetaData.INSTANCE_TYPE_SINGLETON,
                    false,
                    false,
                    true,
                    null,
                    null,
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[]{metaData.createDependsMetaData("Nimbus", "Fuga")},
                }
            },
            {
                createElement("<service name=\"Hoge\" code=\"test.Hoge\"><ifdef name=\"test\" value=\"true\"><depends>Fuga</depends></ifdef></service>"),
                new Object[]{
                    "Hoge",
                    "test.Hoge",
                    Service.State.STARTED,
                    ServiceMetaData.INSTANCE_TYPE_SINGLETON,
                    false,
                    false,
                    true,
                    null,
                    null,
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[]{metaData.createDependsMetaData("Nimbus", "Fuga")},
                }
            },
            {
                createElement("<service name=\"Hoge\" code=\"test.Hoge\"><ifdef name=\"test\" value=\"false\"><depends>Fuga</depends></ifdef></service>"),
                new Object[]{
                    "Hoge",
                    "test.Hoge",
                    Service.State.STARTED,
                    ServiceMetaData.INSTANCE_TYPE_SINGLETON,
                    false,
                    false,
                    true,
                    null,
                    null,
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[0],
                }
            },
            {
                createElement("<service name=\"Hoge\" code=\"test.Hoge\"><depends>Fuga1</depends><depends>Fuga2</depends></service>"),
                new Object[]{
                    "Hoge",
                    "test.Hoge",
                    Service.State.STARTED,
                    ServiceMetaData.INSTANCE_TYPE_SINGLETON,
                    false,
                    false,
                    true,
                    null,
                    null,
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[0],
                    new MetaData[]{metaData.createDependsMetaData("Nimbus", "Fuga1"), metaData.createDependsMetaData("Nimbus", "Fuga2")},
                }
            },
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceMetaData#importXML(org.w3c.dom.Element)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ConstructorMetaData#ConstructorMetaData\\(jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ConstructorMetaData#importXML\\(org\\.w3c\\.dom\\.Element\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ConstructorMetaData#toXML\\(java\\.lang\\.StringBuilder\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.FieldMetaData#FieldMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.FieldMetaData#importXML\\(org\\.w3c\\.dom\\.Element\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.FieldMetaData#toXML\\(java\\.lang\\.StringBuilder\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.AttributeMetaData#AttributeMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.AttributeMetaData#importXML\\(org\\.w3c\\.dom\\.Element\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.AttributeMetaData#toXML\\(java\\.lang\\.StringBuilder\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.InvokeMetaData#InvokeMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.InvokeMetaData#importXML\\(org\\.w3c\\.dom\\.Element\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.InvokeMetaData#toXML\\(java\\.lang\\.StringBuilder\\)"},
          dataProvider = "ImportXML")
    public void testImportXML(
        Element element,
        Object[] assertValues
    ) throws DeploymentException{
        System.setProperty("test", "true");
        try{
            ServiceMetaData metaData = new ServiceMetaData();
            metaData.importXML(element);
            assertEquals(metaData.getName(), assertValues[0]);
            assertEquals(metaData.getCode(), assertValues[1]);
            assertEquals(metaData.getInitStateValue(), assertValues[2]);
            assertEquals(metaData.getInstance(), assertValues[3]);
            assertEquals(metaData.isFactory(), assertValues[4]);
            assertEquals(metaData.isManagement(), assertValues[5]);
            assertEquals(metaData.isCreateTemplate(), assertValues[6]);
            if(assertValues[7] == null){
                assertNull(metaData.getOptionalConfig());
            }else{
                assertNotNull(metaData.getOptionalConfig());
            }
            assertMetaDataEquals(metaData.getConstructor(), (MetaData)assertValues[8]);
            List<FieldMetaData> fieldList = metaData.getFields();
            MetaData[] fields = (MetaData[])assertValues[9];
            if(fields.length == 0){
                assertTrue(fieldList.size() == 0);
            }else{
                assertEquals(fieldList.size(), fields.length);
                for(int i = 0; i < fieldList.size(); i++){
                    assertMetaDataEquals(fieldList.get(i), fields[i]);
                }
            }
            List<AttributeMetaData> attributeList = metaData.getAttributes();
            MetaData[] attributes = (MetaData[])assertValues[10];
            if(attributes.length == 0){
                assertTrue(attributeList.size() == 0);
            }else{
                assertEquals(attributeList.size(), attributes.length);
                for(int i = 0; i < attributeList.size(); i++){
                    assertMetaDataEquals(attributeList.get(i), attributes[i]);
                }
            }
            List<InvokeMetaData> invokeList = metaData.getInvokes();
            MetaData[] invokes = (MetaData[])assertValues[11];
            if(invokes.length == 0){
                assertTrue(invokeList.size() == 0);
            }else{
                assertEquals(invokeList.size(), invokes.length);
                for(int i = 0; i < invokeList.size(); i++){
                    assertMetaDataEquals(invokeList.get(i), invokes[i]);
                }
            }
            List<ServiceMetaData.DependsMetaData> dependsList = metaData.getDepends();
            MetaData[] depends = (MetaData[])assertValues[12];
            if(depends.length == 0){
                assertTrue(dependsList.size() == 0);
            }else{
                assertEquals(dependsList.size(), depends.length);
                for(int i = 0; i < dependsList.size(); i++){
                    assertMetaDataEquals(dependsList.get(i), depends[i]);
                }
            }
        }finally{
            System.clearProperty("test");
        }
    }

    @DataProvider(name = "ImportXMLError")
    public Object[][] dataImportXMLError() throws Exception{
        return new Object[][]{
            {null},
            {createElement("<hoge/>")},
            {createElement("<object/>")},
            {createElement("<service/>")},
            {createElement("<service code=\"test.Hoge\"/>")},
            {createElement("<service name=\"Hoge\"/>")},
            {createElement("<service name=\"Hoge\" code=\"\"/>")},
            {createElement("<service name=\"Hoge\" code=\"test.Hoge\" initState=\"" + Service.State.CREATING + "\"/>")},
            {createElement("<service name=\"Hoge\" code=\"test.Hoge\" initState=\"" + Service.State.STARTING + "\"/>")},
            {createElement("<service name=\"Hoge\" code=\"test.Hoge\" initState=\"" + Service.State.STOPPING + "\"/>")},
            {createElement("<service name=\"Hoge\" code=\"test.Hoge\" initState=\"" + Service.State.STOPPED + "\"/>")},
            {createElement("<service name=\"Hoge\" code=\"test.Hoge\" initState=\"" + Service.State.DESTROYING + "\"/>")},
            {createElement("<service name=\"Hoge\" code=\"test.Hoge\" initState=\"" + Service.State.DESTROYED + "\"/>")},
            {createElement("<service name=\"Hoge\" code=\"test.Hoge\" initState=\"" + Service.State.FAILED + "\"/>")},
            {createElement("<service name=\"Hoge\" code=\"test.Hoge\" initState=\"" + Service.State.UNKNOWN + "\"/>")},
            {createElement("<service name=\"Hoge\" code=\"test.Hoge\" initState=\"hoge\"/>")},
            {createElement("<service name=\"Hoge\" code=\"test.Hoge\" instance=\"hoge\"/>")},
            {createElement("<service name=\"Hoge\" code=\"test.Hoge\"><depends><service name=\"Fuga\" code=\"test.Fuga\"><depends><service name=\"Fuga\" code=\"test.Fuga\"/></depends></service></depends></service>")},
            {createElement("<service name=\"Hoge\" code=\"test.Hoge\"><depends><service name=\"Fuga\" code=\"test.Fuga\"/></depends><depends><service name=\"Fuga\" code=\"test.Fuga\"/></depends></service>")},
            {createElement("<service name=\"Hoge\" code=\"test.Hoge\"><constructor><argument>fuga</argument></constructor><constructor><argument>fuga</argument></constructor></service>")},
            {createElement("<service name=\"Hoge\" code=\"test.Hoge\"><field name=\"field1\">1</field><field name=\"field1\">1</field></service>")},
            {createElement("<service name=\"Hoge\" code=\"test.Hoge\"><attribute name=\"Attribute1\">1</attribute><attribute name=\"Attribute1\">1</attribute></service>")},
            {createElement("<service name=\"Hoge\" code=\"test.Hoge\"><constructor><argument>fuga</argument></constructor><ifdef name=\"test\" value=\"true\"><constructor><argument>fuga</argument></constructor></ifdef></service>")},
            {createElement("<service name=\"Hoge\" code=\"test.Hoge\"><field name=\"field1\">1</field><ifdef name=\"test\" value=\"true\"><field name=\"field1\">1</field></ifdef></service>")},
            {createElement("<service name=\"Hoge\" code=\"test.Hoge\"><attribute name=\"Attribute1\">1</attribute><ifdef name=\"test\" value=\"true\"><attribute name=\"Attribute1\">1</attribute></ifdef></service>")},
            {createElement("<service name=\"Hoge\" code=\"test.Hoge\"><depends><service name=\"Fuga\" code=\"test.Fuga\"/></depends><ifdef name=\"test\" value=\"true\"><depends><service name=\"Fuga\" code=\"test.Fuga\"/></depends></ifdef></service>")},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceMetaData#importXML(org.w3c.dom.Element)",
                    "Error"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceMetaData#ServiceMetaData\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ConstructorMetaData#ConstructorMetaData\\(jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ConstructorMetaData#importXML\\(org\\.w3c\\.dom\\.Element\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.FieldMetaData#FieldMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.FieldMetaData#importXML\\(org\\.w3c\\.dom\\.Element\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.AttributeMetaData#AttributeMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.AttributeMetaData#importXML\\(org\\.w3c\\.dom\\.Element\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.InvokeMetaData#InvokeMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.InvokeMetaData#importXML\\(org\\.w3c\\.dom\\.Element\\)"},
          expectedExceptions = {NullPointerException.class, DeploymentException.class},
          dataProvider = "ImportXMLError")
    public void testImportXMLError(Element element) throws DeploymentException{
        System.setProperty("test", "true");
        try{
            ServiceMetaData metaData = new ServiceMetaData();
            metaData.setManager(new ManagerMetaData());
            metaData.importXML(element);
        }finally{
            System.clearProperty("test");
        }
    }

    @DataProvider(name = "ToXML")
    public Object[][] dataToXML() throws Exception{
        return new Object[][]{
            {createElement("<service name=\"Hoge\" code=\"test.Hoge\"/>"), "<service name=\"Hoge\" code=\"test.Hoge\"/>"},
            {createElement("<service name=\"Hoge\" code=\"test.Hoge\" initState=\"Created\"/>"), "<service name=\"Hoge\" code=\"test.Hoge\" initState=\"Created\"/>"},
            {createElement("<service name=\"Hoge\" code=\"test.Hoge\" instance=\"factory\"/>"), "<service name=\"Hoge\" code=\"test.Hoge\" instance=\"factory\"/>"},
            {createElement("<service name=\"Hoge\" code=\"test.Hoge\" instance=\"threadlocal\"/>"), "<service name=\"Hoge\" code=\"test.Hoge\" instance=\"threadlocal\"/>"},
            {createElement("<service name=\"Hoge\" code=\"test.Hoge\" instance=\"singleton\"/>"), "<service name=\"Hoge\" code=\"test.Hoge\"/>"},
            {createElement("<service name=\"Hoge\" code=\"test.Hoge\" instance=\"factory\" management=\"true\"/>"), "<service name=\"Hoge\" code=\"test.Hoge\" instance=\"factory\" management=\"true\"/>"},
            {createElement("<service name=\"Hoge\" code=\"test.Hoge\" instance=\"factory\" createTemplate=\"false\"/>"), "<service name=\"Hoge\" code=\"test.Hoge\" instance=\"factory\" createTemplate=\"false\"/>"},
            {createElement("<service name=\"Hoge\" code=\"test.Hoge\"></service>"), "<service name=\"Hoge\" code=\"test.Hoge\"/>"},
            {createElement("<service name=\"Hoge\" code=\"test.Hoge\"><constructor><argument>fuga</argument></constructor></service>"), "<service name=\"Hoge\" code=\"test.Hoge\">" + System.getProperty("line.separator") + "    <constructor>" + System.getProperty("line.separator") + "        <argument>fuga</argument>" + System.getProperty("line.separator") + "    </constructor>" + System.getProperty("line.separator") + "</service>"},
            {createElement("<service name=\"Hoge\" code=\"test.Hoge\"><field name=\"field1\">1</field></service>"), "<service name=\"Hoge\" code=\"test.Hoge\">" + System.getProperty("line.separator") + "    <field name=\"field1\">1</field>" + System.getProperty("line.separator") + "</service>"},
            {createElement("<service name=\"Hoge\" code=\"test.Hoge\"><field name=\"field1\">1</field><field name=\"field2\">2</field></service>"), "<service name=\"Hoge\" code=\"test.Hoge\">" + System.getProperty("line.separator") + "    <field name=\"field1\">1</field>" + System.getProperty("line.separator") + "    <field name=\"field2\">2</field>" + System.getProperty("line.separator") + "</service>"},
            {createElement("<service name=\"Hoge\" code=\"test.Hoge\"><attribute name=\"Attribute1\">1</attribute></service>"), "<service name=\"Hoge\" code=\"test.Hoge\">" + System.getProperty("line.separator") + "    <attribute name=\"Attribute1\">1</attribute>" + System.getProperty("line.separator") + "</service>"},
            {createElement("<service name=\"Hoge\" code=\"test.Hoge\"><attribute name=\"Attribute1\">1</attribute><attribute name=\"Attribute2\">2</attribute></service>"), "<service name=\"Hoge\" code=\"test.Hoge\">" + System.getProperty("line.separator") + "    <attribute name=\"Attribute1\">1</attribute>" + System.getProperty("line.separator") + "    <attribute name=\"Attribute2\">2</attribute>" + System.getProperty("line.separator") + "</service>"},
            {createElement("<service name=\"Hoge\" code=\"test.Hoge\"><invoke name=\"method1\"/></service>"), "<service name=\"Hoge\" code=\"test.Hoge\">" + System.getProperty("line.separator") + "    <invoke name=\"method1\" callState=\"Created\"/>" + System.getProperty("line.separator") + "</service>"},
            {createElement("<service name=\"Hoge\" code=\"test.Hoge\"><invoke name=\"method1\" callState=\"Creating\"/></service>"), "<service name=\"Hoge\" code=\"test.Hoge\">" + System.getProperty("line.separator") + "    <invoke name=\"method1\" callState=\"Creating\"/>" + System.getProperty("line.separator") + "</service>"},
            {createElement("<service name=\"Hoge\" code=\"test.Hoge\"><invoke name=\"method1\"/><invoke name=\"method2\"/></service>"), "<service name=\"Hoge\" code=\"test.Hoge\">" + System.getProperty("line.separator") + "    <invoke name=\"method1\" callState=\"Created\"/>" + System.getProperty("line.separator") + "    <invoke name=\"method2\" callState=\"Created\"/>" + System.getProperty("line.separator") + "</service>"},
            {createElement("<service name=\"Hoge\" code=\"test.Hoge\"><depends>Fuga</depends></service>"), "<service name=\"Hoge\" code=\"test.Hoge\">" + System.getProperty("line.separator") + "    <depends>Nimbus#Fuga</depends>" + System.getProperty("line.separator") + "</service>"},
            {createElement("<service name=\"Hoge\" code=\"test.Hoge\"><depends>Fuga1</depends><depends>Fuga2</depends></service>"), "<service name=\"Hoge\" code=\"test.Hoge\">" + System.getProperty("line.separator") + "    <depends>Nimbus#Fuga1</depends>" + System.getProperty("line.separator") + "    <depends>Nimbus#Fuga2</depends>" + System.getProperty("line.separator") + "</service>"},
            {createElement("<service name=\"Hoge\" code=\"test.Hoge\"><depends><service name=\"Fuga\" code=\"test.Fuga\"/></depends></service>"), "<service name=\"Hoge\" code=\"test.Hoge\">" + System.getProperty("line.separator") + "    <depends>Nimbus#Fuga</depends>" + System.getProperty("line.separator") + "</service>"},
            {createElement("<service name=\"Hoge\" code=\"test.Hoge\"><depends><service name=\"Fuga1\" code=\"test.Fuga\"><depends><service name=\"Fuga2\" code=\"test.Fuga\"/></depends></service></depends></service>"), "<service name=\"Hoge\" code=\"test.Hoge\">" + System.getProperty("line.separator") + "    <depends>Nimbus#Fuga1</depends>" + System.getProperty("line.separator") + "</service>"},
            {createElement("<service name=\"Hoge\" code=\"test.Hoge\"><ifdef name=\"test\" value=\"true\"><constructor><argument>fuga</argument></constructor></ifdef></service>"), "<service name=\"Hoge\" code=\"test.Hoge\">" + System.getProperty("line.separator") + "    <ifdef name=\"test\" value=\"true\">" + System.getProperty("line.separator") + "        <constructor>" + System.getProperty("line.separator") + "            <argument>fuga</argument>" + System.getProperty("line.separator") + "        </constructor>" + System.getProperty("line.separator") + "    </ifdef>" + System.getProperty("line.separator") + "</service>"},
            {createElement("<service name=\"Hoge\" code=\"test.Hoge\"><ifdef name=\"test\" value=\"true\"><field name=\"field1\">1</field></ifdef></service>"), "<service name=\"Hoge\" code=\"test.Hoge\">" + System.getProperty("line.separator") + "    <ifdef name=\"test\" value=\"true\">" + System.getProperty("line.separator") + "        <field name=\"field1\">1</field>" + System.getProperty("line.separator") + "    </ifdef>" + System.getProperty("line.separator") + "</service>"},
            {createElement("<service name=\"Hoge\" code=\"test.Hoge\"><ifdef name=\"test\" value=\"true\"><field name=\"field1\">1</field><field name=\"field2\">2</field></ifdef></service>"), "<service name=\"Hoge\" code=\"test.Hoge\">" + System.getProperty("line.separator") + "    <ifdef name=\"test\" value=\"true\">" + System.getProperty("line.separator") + "        <field name=\"field1\">1</field>" + System.getProperty("line.separator") + "        <field name=\"field2\">2</field>" + System.getProperty("line.separator") + "    </ifdef>" + System.getProperty("line.separator") + "</service>"},
            {createElement("<service name=\"Hoge\" code=\"test.Hoge\"><ifdef name=\"test\" value=\"true\"><attribute name=\"Attribute1\">1</attribute></ifdef></service>"), "<service name=\"Hoge\" code=\"test.Hoge\">" + System.getProperty("line.separator") + "    <ifdef name=\"test\" value=\"true\">" + System.getProperty("line.separator") + "        <attribute name=\"Attribute1\">1</attribute>" + System.getProperty("line.separator") + "    </ifdef>" + System.getProperty("line.separator") + "</service>"},
            {createElement("<service name=\"Hoge\" code=\"test.Hoge\"><ifdef name=\"test\" value=\"true\"><attribute name=\"Attribute1\">1</attribute><attribute name=\"Attribute2\">2</attribute></ifdef></service>"), "<service name=\"Hoge\" code=\"test.Hoge\">" + System.getProperty("line.separator") + "    <ifdef name=\"test\" value=\"true\">" + System.getProperty("line.separator") + "        <attribute name=\"Attribute1\">1</attribute>" + System.getProperty("line.separator") + "        <attribute name=\"Attribute2\">2</attribute>" + System.getProperty("line.separator") + "    </ifdef>" + System.getProperty("line.separator") + "</service>"},
            {createElement("<service name=\"Hoge\" code=\"test.Hoge\"><ifdef name=\"test\" value=\"true\"><invoke name=\"method1\"/></ifdef></service>"), "<service name=\"Hoge\" code=\"test.Hoge\">" + System.getProperty("line.separator") + "    <ifdef name=\"test\" value=\"true\">" + System.getProperty("line.separator") + "        <invoke name=\"method1\" callState=\"Created\"/>" + System.getProperty("line.separator") + "    </ifdef>" + System.getProperty("line.separator") + "</service>"},
            {createElement("<service name=\"Hoge\" code=\"test.Hoge\"><ifdef name=\"test\" value=\"true\"><invoke name=\"method1\"/><invoke name=\"method2\"/></ifdef></service>"), "<service name=\"Hoge\" code=\"test.Hoge\">" + System.getProperty("line.separator") + "    <ifdef name=\"test\" value=\"true\">" + System.getProperty("line.separator") + "        <invoke name=\"method1\" callState=\"Created\"/>" + System.getProperty("line.separator") + "        <invoke name=\"method2\" callState=\"Created\"/>" + System.getProperty("line.separator") + "    </ifdef>" + System.getProperty("line.separator") + "</service>"},
            {createElement("<service name=\"Hoge\" code=\"test.Hoge\"><ifdef name=\"test\" value=\"true\"><depends>Fuga1</depends><depends>Fuga2</depends></ifdef></service>"), "<service name=\"Hoge\" code=\"test.Hoge\">" + System.getProperty("line.separator") + "    <ifdef name=\"test\" value=\"true\">" + System.getProperty("line.separator") + "        <depends>Nimbus#Fuga1</depends>" + System.getProperty("line.separator") + "        <depends>Nimbus#Fuga2</depends>" + System.getProperty("line.separator") + "    </ifdef>" + System.getProperty("line.separator") + "</service>"},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceMetaData#toXML(java.lang.StringBuilder)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceMetaData#ServiceMetaData\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceMetaData#importXML\\(org\\.w3c\\.dom\\.Element\\)"},
          dataProvider = "ToXML")
    public void testToXML(Element element, String expected) throws DeploymentException{
        ServiceMetaData metaData = new ServiceMetaData();
        metaData.importXML(element);
        StringBuilder buf = new StringBuilder();
        metaData.toXML(buf);
        assertEquals(buf.toString(), expected);
    }
}
