# -*- coding: utf-8 -*-
#
#  Copyright (C) 2001-2004 by MATSUMURA Namihiko <nie@counterghost.net>
#  Copyright (C) 2004-2014 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#

import codecs
import io
import logging
import os
import re
import time
import urllib.request
import urllib.parse
import webbrowser

from xml.dom import pulldom

from gi.repository import Gtk
from gi.repository import Gdk
from gi.repository import Pango
from gi.repository import GObject
import cairo

import ninix.home
import ninix.pix
import ninix.install

# 注意:
# - このURLを本ゴーストマネージャクローン以外の用途で使う場合は,
#   「できるだけ事前に」AQRS氏に連絡をお願いします.
#   (「何かゴーストマネージャ」のページ: http://www.aqrs.jp/ngm/)
# - アクセスには帯域を多く使用しますので,
#   必ず日時指定の差分アクセスをし余分な負荷をかけないようお願いします.
#   (差分アクセスの方法については本プログラムの実装が参考になると思います.)
MASTERLIST_URL = 'http://www.aqrs.jp/cgi-bin/ghostdb/request2.cgi'

# 10000以上のIDを持つデータは仮登録
ID_LIMIT = 10000

ELEMENTS = ['Name', 'SakuraName', 'KeroName', 'GhostType',
            'HPUrl', 'HPTitle', 'Author', 'PublicUrl', 'ArchiveUrl',
            'ArchiveSize', 'ArchiveTime', 'Version', 'AIName',
            'SurfaceSetFlg', 'KisekaeFlg', 'AliasName',
            'SakuraSurfaceList', 'KeroSurfaceList', 'DeleteFlg',
            'NetworkUpdateTime', 'AcceptName', 'NetworkUpdateFlg',
            'SakuraPreviewMD5', 'KeroPreviewMD5', 'ArchiveMD5',
            'InstallDir', 'ArchiveName', 'UpdateUrl',
            'AnalysisError', 'InstallCount']


class Catalog_xml:

    #public methods
    def __init__(self, datadir):
        self.data = {}
        self.url = {}
        self.cgi = {}
        self.datadir = datadir
        if not os.path.exists(self.datadir):
            os.makedirs(self.datadir)
        self.last_update = '1970-01-01 00:00:00'
        self.load_MasterList()

    # data handling functions
    def get(self, entry, key):
        return entry.get(key) # XXX

    # updates etc
    def network_update(self, updatehook):
        last_update = self.last_update
        self.last_update = time.strftime('%Y-%m-%d %H:%M:%S')
        if self.cgi:
            priority = sorted(self.cgi.keys())
            url = self.cgi[priority[-1]][-1]
        else:
            url = MASTERLIST_URL
        try:
            f = urllib.request.urlopen(url, bytes(urllib.parse.urlencode(
                {'time': '"{0}"'.format(last_update), 'charset': 'UTF-8'}),
                'ascii'))
        except:
            return ## FIXME
        for _ in self.import_from_fileobj(f):
            updatehook()
        self.save_MasterList()
        f.close()

    # private methods
    def load_MasterList(self):
        try:
            with open(os.path.join(self.datadir, b'MasterList.xml'), 'rb') as f:
                for _ in self.import_from_fileobj(f):
                    pass
        except IOError:
            return

    def save_MasterList(self):
        with open(os.path.join(self.datadir, b'MasterList.xml'), 'w') as f:
            self.export_to_fileobj(f)

    def get_encoding(self, line):
        m = re.compile('<\?xml version="1.0" encoding="(.+)" \?>').search(line)
        return m.group(1)

    def create_entry(self, node):
        entry = {}
        for key, text in node:
            assert key in ELEMENTS
            entry[key] = text
        return entry

    def import_from_fileobj(self, fileobj):
        line0 = fileobj.readline().decode('ascii', 'ignore')
        encoding = self.get_encoding(line0)
        try:
            codecs.lookup(encoding)
        except:
            raise SystemExit('Unsupported encoding {0}'.format(repr(encoding)))
        nest = 0
        new_entry = {}
        set_id = None
        node = []
        re_list = re.compile('<GhostList>')
        re_setid = re.compile('<FileSet ID="?([0-9]+)"?>')
        re_set = re.compile('</FileSet>')
        re_priority = re.compile('<RequestCgiUrl Priority="?([0-9]+)"?>(.+)</RequestCgiUrl>')
        re_misc = re.compile('<(.+)>(.+)</(.+)>')
        for line in fileobj:
            yield
            assert nest >= 0
            if not line:
                continue
            line = line.decode(encoding, 'ignore')
            m = re_list.search(line)
            if m:
                nest += 1
                continue
            m = re_setid.search(line)
            if m:
                nest += 1
                set_id = int(m.group(1))
                continue
            m = re_set.search(line)
            if m:
                nest -= 1
                new_entry[set_id] = self.create_entry(node)
                node = []
                continue
            m = re_priority.search(line)
            if m:
                g = m.groups()
                priority = int(g[0])
                url = g[1]
                if priority in self.cgi:
                    self.cgi[priority].append(url)
                else:
                    self.cgi[priority] = [url]
                continue
            m = re_misc.search(line)
            if m:
                g = m.groups()
                if set_id is not None:
                    key = g[0]
                    text = g[1]
                    text = text.replace('&apos;', '\'')                
                    text = text.replace('&quot;', '"')
                    text = text.replace('&gt;', '>')
                    text = text.replace('&lt;', '<')
                    text = text.replace('&amp;', '&')
                    node.append([key, text])
                else:
                    key = g[0]
                    text = g[1]
                    assert key in ['LastUpdate', 'NGMVersion',
                                   'SakuraPreviewBaseUrl',
                                   'KeroPreviewBaseUrl',
                                   'ArcMD5BaseUrl', 'NGMUpdateBaseUrl']
                    if key == 'LastUpdate':
                        self.last_update = text
                    elif key == 'NGMVersion':
                        version = float(text)
                        if version < 0.51:
                            return
                        else:
                            self.version = version
                    elif key in ['SakuraPreviewBaseUrl', 'KeroPreviewBaseUrl',
                                  'ArcMD5BaseUrl', 'NGMUpdateBaseUrl']:
                        self.url[key] = text
        self.data.update(new_entry)        

    def dump_entry(self, entry, fileobj):
        for key in ELEMENTS:
            if key in entry and entry[key] is not None:
                text = entry[key]
                text = text.replace('&', '&amp;')
                text = text.replace('<', '&lt;')
                text = text.replace('>', '&gt;')
                text = text.replace('"', '&quot;')
                text = text.replace('\'', '&apos;')
            else:
                text = '' # fileobj.write('  <{0}/>\n'.format(key))
            fileobj.write('  <{0}>{1}</{0}>\n'.format(key, text))

    def export_to_fileobj(self, fileobj):
        fileobj.write('<?xml version="1.0" encoding="UTF-8" ?>\n')
        fileobj.write('<GhostList>\n')
        fileobj.write('<LastUpdate>{0}</LastUpdate>\n'.format(self.last_update))
        for key in ['SakuraPreviewBaseUrl', 'KeroPreviewBaseUrl',
                    'ArcMD5BaseUrl', 'NGMUpdateBaseUrl']:
            if key in self.url:
                fileobj.write('<{0}>{1}</{0}>\n'.format(key, self.url[key]))
            else:
                fileobj.write('<{0}></{0}>\n'.format(key)) # fileobj.write('<{0}/>\n'.format(key))
        fileobj.write('<NGMVersion>{0}</NGMVersion>\n'.format(self.version))
        key_list = sorted(self.cgi.keys())
        key_list.reverse()
        for priority in key_list:
            for url in self.cgi[priority]:
                fileobj.write(
                    '<RequestCgiUrl Priority="{0:d}">{1}</RequestCgiUrl>\n'.format(priority, url))
        ids = sorted(self.data.keys())
        for set_id in ids:
            fileobj.write('<FileSet ID="{0}">\n'.format(set_id))
            entry = self.data[set_id]
            self.dump_entry(entry, fileobj)
            fileobj.write('</FileSet>\n')
        fileobj.write('</GhostList>\n')


class Catalog(Catalog_xml):

    TYPE = ['Sakura', 'Kero']

    def image_filename(self, set_id, side):
        p = '{0}_{1:d}.png'.format(self.TYPE[side], set_id)
        d = os.path.join(self.datadir, os.fsencode(p))
        return d if os.path.exists(d) else None

    def retrieve_image(self, set_id, side, updatehook):
        p = '{0}_{1:d}.png'.format(self.TYPE[side], set_id)
        d = os.path.join(self.datadir, os.fsencode(p))
        if not os.path.exists(d):
            url = self.url['SakuraPreviewBaseUrl'] if side == 0 else \
                self.url['KeroPreviewBaseUrl']
            try:
                urllib.request.urlretrieve(''.join((url, p)), d, updatehook)
            except:
                return ## FIXME


class SearchDialog:

    def __init__(self):
        self.request_parent = lambda *a: None # dummy
        self.dialog = Gtk.Dialog()
        self.dialog.connect('delete_event', lambda *a: True) # XXX
        self.dialog.set_modal(True)
        self.dialog.set_position(Gtk.WindowPosition.CENTER)
        label = Gtk.Label(label=_('Search for'))
        content_area = self.dialog.get_content_area()
        content_area.add(label)
        self.pattern_entry = Gtk.Entry()
        self.pattern_entry.set_size_request(300, -1)
        content_area.add(self.pattern_entry)
        content_area.show_all()
        self.dialog.add_button(Gtk.STOCK_OK, Gtk.ResponseType.OK)
        self.dialog.add_button(Gtk.STOCK_CANCEL, Gtk.ResponseType.CANCEL)
        self.dialog.connect('response', self.response)

    def set_responsible(self, request_method):
        self.request_parent = request_method

    def set_pattern(self, text):
        self.pattern_entry.set_text(text)
                
    def get_pattern(self):
        return self.pattern_entry.get_text()
    
    def hide(self):
        self.dialog.hide()

    def show(self, default=None):
        if default:
            self.set_pattern(default)
        self.dialog.show()

    def ok(self):
        word = self.get_pattern()
        self.request_parent('NOTIFY', 'search', word)
        self.hide()

    def cancel(self):
        self.hide()

    def response(self, widget, response):
        func = {Gtk.ResponseType.OK: self.ok,
                Gtk.ResponseType.CANCEL: self.cancel,
                Gtk.ResponseType.DELETE_EVENT: self.cancel,
                }
        func[response]()
        return True


class UI:

    def __init__(self):
        self.request_parent = lambda *a: None # dummy
        self.ui_info = '''
        <ui>
          <menubar name='MenuBar'>
            <menu action='FileMenu'>
              <menuitem action='Search'/>
              <menuitem action='Search Forward'/>
              <separator/>
              <menuitem action='Settings'/>
              <separator/>
              <menuitem action='DB Network Update'/>
              <separator/>
              <menuitem action='Close'/>
            </menu>
            <menu action='ViewMenu'>
              <menuitem action='Mask'/>
              <menuitem action='Reset to Default'/>
              <menuitem action='Show All'/>
            </menu>
            <menu action='ArchiveMenu'>
            </menu>
            <menu action='HelpMenu'>
            </menu>
          </menubar>
        </ui>'''
        self.entries = (
            ( 'FileMenu', None, _('_File') ),
            ( 'ViewMenu', None, _('_View') ),
            ( 'ArchiveMenu', None, _('_Archive') ),
            ( 'HelpMenu', None, _('_Help') ),
            ( 'Search', None,                  # name, stock id
              _('Search(_F)'), '<control>F',   # label, accelerator
              'Search',                        # tooltip
              lambda *a: self.open_search_dialog() ),
            ( 'Search Forward', None,
              _('Search Forward(_S)'), 'F3',
              None,
              lambda *a: self.search_forward() ),
            ( 'Settings', None,
              _('Settings(_O)'), None,
              None,
              lambda *a: self.request_parent(
                    'NOTIFY', 'open_preference_dialog') ),
            ( 'DB Network Update', None,
              _('DB Network Update(_N)'), None,
              None,
              lambda *a: self.network_update() ),
            ( 'Close', None,
              _('Close(_X)'), None,
              None,
              lambda *a: self.close() ),
            ( 'Mask', None,
              _('Mask(_M)'), None,
              None,
              lambda *a: self.request_parent(
                    'NOTIFY', 'open_mask_dialog') ),
            ( 'Reset to Default', None,
              _('Reset to Default(_Y)'), None,
              None,
              lambda *a: self.request_parent(
                    'NOTIFY', 'reset_to_default') ),
            ( 'Show All', None,
              _('Show All(_Z)'), None,
              None,
              lambda *a: self.request_parent(
                    'NOTIFY', 'show_all') ),
            )
        self.opened = 0
        self.textview = [None, None]
        self.darea = [None, None]
        self.surface = [None, None]
        self.info = None
        self.button = {}
        self.url = {}
        self.search_word = ''
        self.search_dialog = SearchDialog()
        self.search_dialog.set_responsible(self.handle_request)
        self.create_dialog()

    def set_responsible(self, request_method):
        self.request_parent = request_method

    def handle_request(self, event_type, event, *arglist, **argdict):
        assert event_type in ['GET', 'NOTIFY']
        handlers = {
            }
        handler = handlers.get(event, getattr(self, event, None))
        if handler is None:
            result = self.request_parent(
                event_type, event, *arglist, **argdict)
        else:
            result = handler(*arglist, **argdict)
        if event_type == 'GET':
            return result

    def create_dialog(self):
        self.window = Gtk.Window()
        self.window.set_title(_('Ghost Manager'))
        self.window.set_resizable(False)
        self.window.connect('delete_event', lambda *a: self.close())
        self.window.set_position(Gtk.WindowPosition.CENTER)
        self.window.set_gravity(Gdk.Gravity.CENTER)
        actions = Gtk.ActionGroup('Actions')
        actions.add_actions(self.entries)
        ui = Gtk.UIManager()
        ui.insert_action_group(actions, 0)
        self.window.add_accel_group(ui.get_accel_group())
        try:
            mergeid = ui.add_ui_from_string(self.ui_info)
        except GObject.GError as msg:
            logging.error('building menus failed: {0}'.format(msg))
        vbox = Gtk.Box(orientation=Gtk.Orientation.VERTICAL)
        self.window.add(vbox)
        vbox.show()
        vbox.pack_start(ui.get_widget('/MenuBar'), False, False, 0)
        separator = Gtk.HSeparator()
        vbox.pack_start(separator, False, True, 0)
        separator.show()
        hbox = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL)
        vbox.pack_start(hbox, False, True, 10)
        hbox.show()
        self.surface_area_sakura = self.create_surface_area(0)
        hbox.pack_start(self.surface_area_sakura, False, True, 10)
        self.surface_area_kero = self.create_surface_area(1)
        hbox.pack_start(self.surface_area_kero, False, True, 10)
        self.info_area = self.create_info_area()
        hbox.pack_start(self.info_area, False, True, 10)
        box = Gtk.ButtonBox(orientation=Gtk.Orientation.HORIZONTAL)
        box.set_layout(Gtk.ButtonBoxStyle.SPREAD)
        vbox.pack_start(box, False, True, 4)
        box.show()
        button = Gtk.Button(_('Previous'))
        button.connect('clicked', lambda b, w=self: w.show_previous())
        box.add(button)
        button.show()
        self.button['previous'] = button
        button = Gtk.Button(_('Next'))
        button.connect('clicked', lambda b, w=self: w.show_next())
        box.add(button)
        button.show()
        self.button['next'] = button
        self.statusbar = Gtk.Statusbar()
        vbox.pack_start(self.statusbar, False, True, 0)
        self.statusbar.show()

    def network_update(self):
        self.window.set_sensitive(False)
        def updatehook(*args):
            while Gtk.events_pending():
                Gtk.main_iteration()
        self.request_parent('NOTIFY', 'network_update', updatehook)
        self.update()
        self.window.set_sensitive(True)

    def open_search_dialog(self):
        self.search_dialog.show(default=self.search_word)

    def search(self, word):
        if word:
            self.search_word = word
            if self.request_parent('GET', 'search', word):
                self.update()
            else:
                pass ## FIXME

    def search_forward(self):
        if self.search_word:
            if self.request_parent('GET', 'search_forward', self.search_word):
                self.update()
            else:
                pass ## FIXME

    def show_next(self):
        self.request_parent('NOTIFY', 'go_next')
        self.update()

    def show_previous(self):
        self.request_parent('NOTIFY', 'previous')
        self.update()

    def create_surface_area(self, side):
        assert side in [0, 1]
        vbox = Gtk.Box(orientation=Gtk.Orientation.VERTICAL)
        vbox.show()
        textview = Gtk.TextView()
        textview.set_editable(False)
        textview.set_size_request(128, 16)
        vbox.pack_start(textview, False, True, 0)
        textview.show()
        self.textview[side] = textview
        darea = Gtk.DrawingArea()
        vbox.pack_start(darea, False, True, 0)
        darea.set_events(Gdk.EventMask.EXPOSURE_MASK)
        darea.connect('draw', self.redraw, side)
        darea.show()
        self.darea[side] = darea
        return vbox

    def redraw(self, widget, cr, side):
        if self.surface[side] is not None:
            cr.set_source_surface(self.surface[side], 0, 0)
            cr.set_operator(cairo.OPERATOR_OVER)
            cr.paint()
        else:
            cr.set_operator(cairo.OPERATOR_CLEAR)
            cr.paint()

    def update_surface_area(self):
        for side in [0, 1]:
            target = 'SakuraName' if side == 0 else 'KeroName'
            name = self.request_parent('GET', 'get', target)
            textbuffer = self.textview[side].get_buffer()
            textbuffer.set_text(name)
            filename = self.request_parent('GET', 'get_image_filename', side)
            darea = self.darea[side]
            darea.realize()
            if filename is not None:
                try:
                    surface = ninix.pix.create_surface_from_file(filename)
                except:
                    surface = None
                else:
                    w = surface.get_width()
                    h = surface.get_height()
                    darea.set_size_request(w, h)
            else:
                surface = None
            self.surface[side] = surface
            darea.queue_draw()

    def create_info_area(self):
        vbox = Gtk.Box(orientation=Gtk.Orientation.VERTICAL)
        vbox.show()
        hbox = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL)
        box = Gtk.ButtonBox(orientation=Gtk.Orientation.HORIZONTAL)
        box.set_layout(Gtk.ButtonBoxStyle.SPREAD)
        box.show()
        button = Gtk.Button(_('Install'))
        button.connect(
            'clicked',
            lambda b, w=self: w.request_parent('NOTIFY', 'install_current'))
        box.add(button)
        button.show()
        self.button['install'] = button
        button = Gtk.Button(_('Update'))
        button.connect(
            'clicked',
            lambda b, w=self: w.request_parent('NOTIFY', 'update_current'))
        box.add(button)
        button.show()
        self.button['update'] = button
        hbox.pack_start(box, True, True, 10)
        vbox2 = Gtk.Box(orientation=Gtk.Orientation.VERTICAL)
        hbox.pack_start(vbox2, False, True, 0)
        vbox2.show()
        button = Gtk.Button('') # with GtkLabel
        button.set_relief(Gtk.ReliefStyle.NONE)
        self.url['HP'] = [None, button.get_child()]
        vbox2.pack_start(button, False, True, 0)
        button.connect(
            'clicked', lambda b: webbrowser.open(self.url['HP'][0]))
        button.show()
        button = Gtk.Button('')
        button.set_relief(Gtk.ReliefStyle.NONE)
        button.set_use_underline(True)
        self.url['Public'] = [None, button.get_child()]
        vbox2.pack_start(button, False, True, 0)
        button.connect(
            'clicked', lambda b: webbrowser.open(self.url['Public'][0]))
        button.show()
        vbox.pack_start(hbox, False, True, 0)
        hbox.show()
        textview = Gtk.TextView()
        textview.set_editable(False)
        textview.set_size_request(256, 144)
        vbox.pack_start(textview, False, True, 0)
        textview.show()
        self.info = textview
        return vbox

    def update_info_area(self):
        info_list = [(_('Author:'), 'Author'),
                     (_('ArchiveTime:'), 'ArchiveTime'),
                     (_('ArchiveSize:'), 'ArchiveSize'),
                     (_('NetworkUpdateTime:'), 'NetworkUpdateTime'),
                     (_('Version:'), 'Version'),
                     (_('AIName:'), 'AIName')]
        text = ''
        text = ''.join((text, self.request_parent('GET', 'get', 'Name'), '\n'))
        for item in info_list:
            text = ''.join((text, item[0],
                            self.request_parent('GET', 'get', item[1]), '\n'))
        text = ''.join((text,
                        self.request_parent('GET', 'get', 'SakuraName'),
                        _('SurfaceList:'),
                        self.request_parent('GET', 'get', 'SakuraSurfaceList'),
                        '\n'))
        text = ''.join((text,
                        self.request_parent('GET', 'get', 'KeroName'),
                        _('SurfaceList:'),
                        self.request_parent('GET', 'get', 'KeroSurfaceList'),
                        '\n'))
        textbuffer = self.info.get_buffer()
        textbuffer.set_text(text)
        url = self.request_parent('GET', 'get', 'HPUrl')
        text = self.request_parent('GET', 'get', 'HPTitle')
        self.url['HP'][0] = url
        label = self.url['HP'][1]
        label.set_markup('<span foreground="blue">{0}</span>'.format(text))
        url = self.request_parent('GET', 'get', 'PublicUrl')
        text = ''.join((self.request_parent('GET', 'get', 'Name'),
                        _(' Web Page')))
        self.url['Public'][0] = url
        label = self.url['Public'][1]
        label.set_markup('<span foreground="blue">{0}</span>'.format(text))
        target_dir = os.path.join(
            os.fsencode(self.request_parent('GET', 'get_home_dir')),
            b'ghost',
            os.fsencode(self.request_parent('GET', 'get', 'InstallDir')))
        self.button['install'].set_sensitive(
            bool(not os.path.isdir(target_dir) and
                 self.request_parent('GET', 'get', 'ArchiveUrl') != 'No data'))
        self.button['update'].set_sensitive(
            bool(os.path.isdir(target_dir) and
                 self.request_parent('GET', 'get', 'GhostType') == 'ゴースト' and
                 self.request_parent('GET', 'get', 'UpdateUrl') != 'No data'))

    def update(self):
        self.update_surface_area()
        self.update_info_area()
        self.button['next'].set_sensitive(
            bool(self.request_parent('GET', 'exist_next')))
        self.button['previous'].set_sensitive(
            bool(self.request_parent('GET', 'exist_previous')))

    def show(self):
        if self.opened:
            return
        self.update()
        self.window.show()
        self.opened = 1

    def close(self):
        self.window.hide()
        self.opened = 0
        return True


class NGM:

    def __init__(self):
        self.request_parent = lambda *a: None # dummy
        self.current = 0
        self.opened = 0
        self.home_dir = ninix.home.get_ninix_home()
        self.catalog = Catalog(os.path.join(self.home_dir, b'ngm/data'))
        self.installer = ninix.install.Installer()
        self.ui = UI()
        self.ui.set_responsible(self.handle_request)

    def set_responsible(self, request_method):
        self.request_parent = request_method

    def handle_request(self, event_type, event, *arglist, **argdict):
        assert event_type in ['GET', 'NOTIFY']
        handlers = {
            'get_home_dir': lambda *a: self.home_dir,
            }
        handler = handlers.get(event, getattr(self, event, None))
        if handler is None:
            result = self.request_parent(
                event_type, event, *arglist, **argdict)
        else:
            result = handler(*arglist, **argdict)
        if event_type == 'GET':
            return result

    def get(self, element):
        if self.current in self.catalog.data:
            entry = self.catalog.data[self.current]
            text = self.catalog.get(entry, element)
            if text is not None:
                return text
        return 'No data'

    def get_image_filename(self, side):
        return self.catalog.image_filename(self.current, side)

    def search(self, word, set_id=0):
        while set_id < ID_LIMIT:
            if set_id in self.catalog.data:
                entry = self.catalog.data[set_id]
                for element in ['Name', 'SakuraName', 'KeroName',
                                'Author', 'HPTitle']:
                    text = self.catalog.get(entry, element)
                    if not text:
                        continue
                    if word in text:
                        self.current = set_id
                        return True
            set_id += 1
        return False

    def search_forward(self, word):
        return self.search(word, set_id=self.current + 1)

    def open_preference_dialog(self):
        pass

    def network_update(self, updatehook):
        self.catalog.network_update(updatehook)
        for set_id in self.catalog.data:
            for side in [0, 1]:
                self.catalog.retrieve_image(set_id, side, updatehook)

    def open_mask_dialog(self):
        pass

    def reset_to_default(self):
        pass

    def show_all(self):
        pass

    def go_next(self):
        next_index = self.current + 1
        if next_index < ID_LIMIT and next_index in self.catalog.data:
            self.current = next_index

    def exist_next(self):
        next_index = self.current + 1
        return bool(next_index < ID_LIMIT and next_index in self.catalog.data)

    def previous(self):
        previous = self.current - 1
        assert previous < ID_LIMIT
        if previous in self.catalog.data:
            self.current = previous

    def exist_previous(self):
        previous = self.current - 1
        assert previous < ID_LIMIT
        return bool(previous in self.catalog.data)

    def show_dialog(self):
        self.ui.show()

    def install_current(self):
        try:
            filetype, target_dir, names, errno = self.installer.install(
                self.get('ArchiveUrl'), ninix.home.get_ninix_home())
        except:
            target_dir = None
        if target_dir is not None:
            assert filetype == 'ghost'
            self.request_parent('NOTIFY', 'add_sakura', target_dir)

    def update_current(self): ## FIXME
        self.request_parent('NOTIFY', 'update_sakura', self.get('Name'), 'NGM')


if __name__ == '__main__':
    pass
