﻿using System;
using System.Collections.Generic;
using System.Drawing;
using System.IO;
using System.Text;
using System.Text.RegularExpressions;
using System.Windows.Forms;
using Microsoft.Scripting.Hosting;
using NotepadNeueExtension;
using Sgry.Azuki;
using WeifenLuo.WinFormsUI.Docking;

namespace NotepadNeue
{
    public partial class MainForm : Form, NotepadNeueExtension.IExtensionHost
    {
        public event EventHandler MainFormClosed;
        public event ParseEventHandler ParseStarted;
        public event CheckWordEventHandleer CheckWordStarted;

        public NotepadNeueSkin Skin
        {
            get
            {
                return skin;
            }
        }
        public string CurrentDirectory
        {
            get
            {
                return currentDirectory;
            }
        }
        public bool IsDocumentEmpty
        {
            get
            {
                return CurrentDocument == null ? true : false;
            }
        }
        public string SelectedTextInDocument
        {
            get
            {
                ExAzukiControl exac = CurrentDocument;
                if (exac == null) return null;
                return exac.GetSelectedText();
            }
        }
        public Range SelectionInDocument
        {
            get
            {
                ExAzukiControl exac = CurrentDocument;
                if (exac == null) return Range.InvalidRange;
                int start = 0, end = 0;
                exac.GetSelection(out start, out end);
                return new Range(start, end);
            }
            set
            {
                if (value.IsInvalid) return;
                ExAzukiControl exac = CurrentDocument;
                if (exac == null) return;
                try
                {
                    exac.SetSelection(value.Start, value.End);
                }
                catch (Exception e)
                {
                    MessageBox.Show(e.Message);
                }
            }
        }
        public string DocumentText
        {
            get
            {
                ExAzukiControl exac = CurrentDocument;
                if (exac == null) return null;
                return exac.Text;
            }
        }
        /// <summary>
        /// 現在のドキュメントのテキストを改行コードで分割した配列で返します
        /// 終了行を含みます。
        /// </summary>
        /// <param name="startlineindex">開始行</param>
        /// <param name="endlineindex">終了行</param>
        /// <returns>行ごとのテキスト</returns>
        public string[] DocumentTextAsArray(int startlineindex, int endlineindex)
        {
            ExAzukiControl exac = CurrentDocument;
            if (exac == null) return new string[0];
            if (startlineindex < 0) startlineindex = 0;
            if (startlineindex >= exac.LineCount) startlineindex = exac.LineCount;
            if (endlineindex < 0) endlineindex = 0;
            if (endlineindex >= exac.LineCount) endlineindex = exac.LineCount;
            List<string> pool = new List<string>(endlineindex - startlineindex);
            for (int i = startlineindex; i < endlineindex; i++)
            {
                pool.Add(exac.Document.GetLineContent(i));
            }
            return pool.ToArray();
        }
        public int DocumentTextLength
        {
            get
            {
                ExAzukiControl exac = CurrentDocument;
                if (exac == null) return -1;
                return exac.Document.Length;
            }
        }
        public int DocumentLineNum
        {
            get
            {
                ExAzukiControl exac = CurrentDocument;
                if (exac == null) return -1;
                return exac.Document.LineCount;
            }
        }
        public bool DocumentSaved
        {
            get
            {
                TextEditWindow tew = CurrentWindow;
                if (tew == null) return false;
                return tew.Saved;
            }
        }
        public string DocumentPath
        {
            get
            {
                TextEditWindow tew = CurrentWindow;
                if (tew == null) return null;
                return tew.Filename;
            }
        }

        public string[] AllDocumentPath
        {
            get
            {
                List<string> rets = new List<string>();
                foreach (IDockContent idc in dockPanel1.Documents)
                {
                    TextEditWindow tew = idc as TextEditWindow;
                    if (tew != null)
                    {
                        rets.Add(tew.Filename);
                    }
                }
                return rets.ToArray();
            }
        }

        public bool QueryAlreadyOpenFile(string before, string next)
        {
            if (before == null || next == null) return false;
            foreach (DockContent df in dockPanel1.Documents)
            {
                TextEditWindow tew = df as TextEditWindow;
                if (tew != null)
                {
                    if (FileUtil.IsSameFile(tew.Filename, before))
                    {
                        string changebefore = Path.GetFileNameWithoutExtension(before);
                        string changenext = Path.GetFileNameWithoutExtension(next);
                        string text = "ドキュメントに含まれる\n" + changebefore + "  を  " + changenext + "\nに変換しますか?";
                        if (MessageBox.Show(text, "置換確認", MessageBoxButtons.YesNo) == DialogResult.Yes)
                        {
                            string s = tew.ExAzukiControl.Document.Text.Replace(changebefore, changenext);
                            tew.ExAzukiControl.Document.Text = s;
                        }
                        tew.Filename = Directory.GetParent(before) + "\\" + next;
                        return true;
                    }
                }
            }
            return false;
        }
        public bool OpenFile(string filename)
        {
            if (CurrentDocument == null || filename == null) return false;
            TextEditWindow temptew = null;
            if (CheckAlreadyOpenFile(filename, out temptew))
            {
                if (temptew != null)
                {
                    temptew.Pane.ActiveContent = temptew;
                    this.CurrentDocument.Focus();
                }
                return false;
            }
            if (this.CurrentDocument.Text != "" && !this.CurrentWindow.Saved)
            {
                DialogResult dr = AskSave();
                if (dr == DialogResult.Cancel) return false;
                else if (dr == DialogResult.Yes)
                {
                    OverSave();
                }
            }
            return OpenFile(CheckAlreadyLoadedExtensionInfo(filename), filename);
        }

        public bool OpenFileToNewDocument(string filename)
        {
            if (filename == null) return false;
            TextEditWindow temptew = null;
            if (CheckAlreadyOpenFile(filename, out temptew))
            {
                if (temptew != null)
                {
                    temptew.Pane.ActiveContent = temptew;
                    this.CurrentDocument.Focus();
                }
                return false;
            }
            ExtensionInfomation exinfo = CheckAlreadyLoadedExtensionInfo(filename);
            TextEditWindow tew = new TextEditWindow();
            tew.Initialize(this, myToolTip1, myDataGridView1, notepadUtility, exinfo);
            if (ActiveDocumentPane == null)
            {
                tew.Show(dockPanel1, DockState.Document);
            }
            else
            {
                DockPane dp = ActiveDocumentPane;
                DockContent temp = (DockContent)dp.ActiveContent;
                tew.Show(dp, temp);
                temp.Show(dp, tew);
                dp.ActiveContent = tew;
            }
            Utility.SetToEditor(CurrentDocument);
            CurrentDocument.Font = new Font(notepadUtility.Fontname, notepadUtility.Fontsize);
            this.myToolTip1.Font = CurrentDocument.Font;

            if (notepadUtility.AutoFormat)
            {
                this.CurrentDocument.AutoIndentHook = Sgry.Azuki.AutoIndentHooks.GenericHook;
            }
            else
            {
                this.CurrentDocument.AutoIndentHook = null;
            }
            return OpenFile(exinfo, filename);
        }

        private bool OpenFile(ExtensionInfomation extensionInfo, string fileName)
        {
            try
            {
                using (StreamReader sr = new StreamReader(fileName, AutoCheckEncoding(fileName, extensionInfo.Encoding)))
                {
                    string text = sr.ReadToEnd();
                    if (notepadUtility.DetectNewLineString)
                    {
                        SetTextToDocument(this.CurrentDocument.Document, text);
                    }
                    else
                    {
                        this.CurrentDocument.Document.Text = text;
                    }
                }
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
            }
            this.CurrentDocument.ExtensionInfo = extensionInfo;
            this.CurrentWindow.Filename = fileName;
            this.CurrentWindow.Save();
            this.CurrentDocument.Focus();
            ChangeSettingState();
            ChangeCustomToolStrip();
            myDataGridView1.Visible = false;
            myToolTip1.Visible = false;
            ManageDocument();
            ChangeFormText();
            return true;
        }

        private Encoding AutoCheckEncoding(string filename, Encoding defaultEncoding)
        {
            var charCode = CharCodeDetector.DetectCharCode(File.ReadAllBytes(filename));
            switch (charCode)
            {
                case CharCodeDetector.CharCode.ASCII:
                    return Encoding.ASCII;
                case CharCodeDetector.CharCode.EUC:
                    return Encoding.GetEncoding("euc-jp");
                case CharCodeDetector.CharCode.SJIS:
                    return Encoding.GetEncoding(932);
                case CharCodeDetector.CharCode.UTF8N:
                    return Encoding.UTF8;
                case CharCodeDetector.CharCode.UTF16LE:
                    return Encoding.Unicode;
                case CharCodeDetector.CharCode.UTF16BE:
                    return Encoding.BigEndianUnicode;
                default:
                    return defaultEncoding;
            }
        }
        public bool ScrollToCaret()
        {
            ExAzukiControl exac = CurrentDocument;
            if (exac == null) return false;
            exac.ScrollToCaret();
            return true;
        }
        public Range FindNext(string text, int startindex, bool ignorecase)
        {
            if (text == null) return Range.InvalidRange;
            ExAzukiControl exac = CurrentDocument;
            if (exac == null) return Range.InvalidRange;
            SearchResult sr = exac.Document.FindNext(text, startindex, !ignorecase);
            if (sr == null) return Range.InvalidRange;
            return new Range(sr.Begin, sr.End);
        }
        public Range FindNext(System.Text.RegularExpressions.Regex regex, int startindex)
        {
            if (regex == null) return Range.InvalidRange;
            ExAzukiControl exac = CurrentDocument;
            if (exac == null) return Range.InvalidRange;
            SearchResult sr = exac.Document.FindNext(regex, startindex);
            if (sr == null) return Range.InvalidRange;
            return new Range(sr.Begin, sr.End);
        }
        public string GetTextInRange(Range range)
        {
            ExAzukiControl exac = CurrentDocument;
            if (exac == null) return null;
            if (range.IsInvalid) return null;
            return exac.GetTextInRange(range.Start, range.End);
        }
        public bool Replace(string text)
        {
            if (text == null) return false;
            ExAzukiControl exac = CurrentDocument;
            if (exac == null) return false;
            exac.Document.Replace(text);
            return true;
        }
        public bool Replace(string text, Range range)
        {
            if (text == null || range.IsInvalid) return false;
            ExAzukiControl exac = CurrentDocument;
            if (exac == null) return false;
            exac.Document.Replace(text, range.Start, range.End);
            return true;
        }
        public bool MakeInvalidChange()
        {
            ExAzukiControl exac = CurrentDocument;
            if (exac == null) return false;
            exac.MakeInvalidChange();
            return true;
        }
        public bool MakeValidChange()
        {
            ExAzukiControl exac = CurrentDocument;
            if (exac == null) return false;
            exac.MakeValidChange();
            return true;
        }
        public object NextDocument()
        {
            return ChangeToNextDocument();
        }
        public object PreviousDocument()
        {
            return ChangeToPreviousDocument();
        }
        public object PresentDocument()
        {
            return CurrentWindow;
        }
        public void RunPythonFromFile(string path, KeyValuePair<string, object>[] items)
        {
            RunPythonFromFile(path, items, new string[0]);
        }
        public KeyValuePair<string, object>[] RunPythonFromFile(string path, KeyValuePair<string, object>[] items, string[] returns)
        {
            List<KeyValuePair<string, object>> ret = new List<KeyValuePair<string, object>>(returns.Length);
            if (path != null && File.Exists(path))
            {
                ScriptSource source = StaticObjects.PythonEngine.CreateScriptSourceFromFile(path);
                ScriptScope scope = StaticObjects.PythonEngine.CreateScope();
                scope = ExecuteScript(source, scope, items);
                if (scope != null && returns != null)
                {
                    foreach (string wantret in returns)
                    {
                        object o = null;
                        if (scope.TryGetVariable(wantret, out o)) ret.Add(new KeyValuePair<string, object>(wantret, o));
                    }
                }
            }
            else
            {
                MessageBox.Show("ファイルが存在しません");
            }
            return ret.ToArray();
        }
        public void RunPythonFromString(string expression, KeyValuePair<string, object>[] items)
        {
            RunPythonFromString(expression, items, new string[0]);
        }

        public KeyValuePair<string, object>[] RunPythonFromString(string expression, KeyValuePair<string, object>[] items, string[] returns)
        {
            List<KeyValuePair<string, object>> ret = new List<KeyValuePair<string, object>>(returns.Length);
            if (expression != null)
            {
                ScriptSource source = StaticObjects.PythonEngine.CreateScriptSourceFromString(expression);
                ScriptScope scope = StaticObjects.PythonEngine.CreateScope();
                scope = ExecuteScript(source, scope, items);
                if (scope != null && returns != null)
                {
                    foreach (string wantret in returns)
                    {
                        object o = null;
                        if (scope.TryGetVariable(wantret, out o)) ret.Add(new KeyValuePair<string, object>(wantret, o));
                    }
                }
            }
            return ret.ToArray();
        }
        private ScriptScope ExecuteScript(ScriptSource source, ScriptScope scope, KeyValuePair<string, object>[] items)
        {
            try
            {
                scope.SetVariable("editor", CurrentDocument);
                foreach (KeyValuePair<string, object> pair in items)
                {
                    scope.SetVariable(pair.Key, pair.Value);
                }
                source.Execute(scope);
                return scope;
            }
            catch (Exception e)
            {
                StringBuilder sb = new StringBuilder();
                if (scope != null)
                {
                    foreach (System.Collections.Generic.KeyValuePair<string, object> data in scope.GetItems())
                    {
                        sb.Append(data.Key + "\n");
                    }
                }
                MessageBox.Show("スクリプトエラーです\n" + e.Message + "\n" + sb.ToString());
            }
            return null;
        }
        public void ShowBallonTip(string title, string text, ToolTipIcon iconkind, int displayinterval)
        {
            notifyIcon1.Visible = true;
            notifyIcon1.BalloonTipTitle = title == null ? "" : title;
            notifyIcon1.BalloonTipText = text == null ? "" : text;
            notifyIcon1.BalloonTipIcon = iconkind;
            notifyIcon1.ShowBalloonTip(displayinterval);
        }

        private bool ChangeDocumentMode(ExAzukiControl exac, string extension)
        {
            if (extension == null || exac == null) return false;
            if (extension.Length >= 1 && extension[0] == '.') extension = extension.Substring(1);
            ExtensionInfomation targetinfo = null;
            if (!ExtensionInfomationManager.ContainsKey(extension))
            {
                if (File.Exists(currentDirectory + "ini\\" + "reserved_" + extension + ".ini"))
                {
                    ExtensionInfomation exinfo = new ExtensionInfomation(currentDirectory, this);
                    exinfo.Extension = extension;
                    exinfo.readini(currentDirectory + "ini\\" + "reserved_" + extension + ".ini");
                    ExtensionInfomationManager.Add(exinfo.Extension, exinfo);
                    targetinfo = exinfo;
                }
                else return true;
            }
            else
            {
                ExtensionInfomationManager.TryGetValue(extension, out targetinfo);
            }
            exac.ExtensionInfo = targetinfo;
            ChangeFormText();
            return true;
        }
        public bool ChangeDocumentMode(string extension)
        {
            return ChangeDocumentMode(CurrentDocument, extension);
        }

        public bool ChangeDocumentMode(string extension, string filename)
        {
            return ChangeDocumentMode(GetExacFromFilename(filename), extension);
        }

        /// <summary>
        /// 新しいドキュメントを開きます。
        /// </summary>
        public void CreateNewDocument()
        {
            TextEditWindow tew = new TextEditWindow();
            tew.Filename = "new.txt";
            tew.Initialize(this, myToolTip1, myDataGridView1, notepadUtility, defaultExInfo);
            if (ActiveDocumentPane == null)
            {
                tew.Show(dockPanel1, DockState.Document);
            }
            else
            {
                DockPane dp = ActiveDocumentPane;
                DockContent temp = (DockContent)dp.ActiveContent;
                tew.Show(dp, temp);
                temp.Show(dp, tew);
                dp.ActiveContent = tew;
            }
            Utility.SetToEditor(CurrentDocument);
            CurrentDocument.Font = new Font(notepadUtility.Fontname, notepadUtility.Fontsize);
            this.myToolTip1.Font = CurrentDocument.Font;
            ManageDocument();
            if (notepadUtility.AutoFormat)
            {
                this.CurrentDocument.AutoIndentHook = Sgry.Azuki.AutoIndentHooks.GenericHook;
            }
            else
            {
                this.CurrentDocument.AutoIndentHook = null;
            }
        }

        private bool CloseDocument(TextEditWindow Tew)
        {
            if (Tew == null) return false;
            Tew.Close();
            return true;
        }

        public bool CloseDocument()
        {
            return CloseDocument(CurrentWindow);
        }

        public bool CloseDocument(string filename)
        {
            return CloseDocument(GetTewFromFilename(filename));
        }

        public bool VerticalSplitDocument()
        {
            TextEditWindow tew = CurrentWindow;
            if (tew == null) return false;
            tew.VerticalSplit();
            return true;
        }

        public bool VerticalSplitDocument(string filename)
        {
            TextEditWindow tew = GetTewFromFilename(filename);
            if (tew == null) return false;
            tew.VerticalSplit();
            return true;
        }

        /// <summary>
        /// 現在のドキュメントを水平分割します。
        /// </summary>
        /// <returns>現在のドキュメントがない場合はfalse</returns>
        public bool HorizontalSplitDocument()
        {
            TextEditWindow tew = CurrentWindow;
            if (tew == null) return false;
            tew.HorizontalSplit();
            return true;
        }

        public bool HorizontalSplitDocument(string filename)
        {
            TextEditWindow tew = GetTewFromFilename(filename);
            if (tew == null) return false;
            tew.HorizontalSplit();
            return true;
        }

        public bool SaveDocument()
        {
            return SaveDocument(CurrentWindow);
        }

        public bool SaveDocument(string filename)
        {
            return SaveDocument(GetTewFromFilename(filename));
        }

        public bool SaveDocument(TextEditWindow Tew)
        {
            if (Tew == null) return false;
            if (Tew.Filename == "new.txt")
            {

                DialogResult dr = AskSave();
                if (dr == DialogResult.Yes)
                {
                    Save(Tew);
                }
            }
            else
            {
                try
                {
                    StreamWriter sw = new StreamWriter(Tew.Filename, false, Tew.ExAzukiControl.ExtensionInfo.Encoding);
                    sw.Write(Tew.ExAzukiControl.Text);
                    sw.Close();
                    this.toolStripStatusLabel2.Text = st3;
                    this.toolStripStatusLabel3.Text = Tew.Filename;
                    Tew.Save();
                }
                catch (Exception ex)
                {
                    MessageBox.Show(ex.Message);
                }
            }
            return true;
        }

        public bool ReloadDocument()
        {
            return ReloadDocument(CurrentWindow);
        }


        public bool ReloadDocument(string filename)
        {
            return ReloadDocument(GetTewFromFilename(filename));
        }

        private bool ReloadDocument(TextEditWindow Tew)
        {
            if (Tew == null) return false;
            if (Tew.Filename == "new.txt") return false;
            Form2 fm = new Form2();
            if (fm.ShowDialog() == DialogResult.OK)
            {
                this.currentEncoding = fm.Encoding;
                try
                {
                    StreamReader sr = new StreamReader(Tew.Filename, this.currentEncoding);
                    Tew.ExAzukiControl.Text = sr.ReadToEnd();
                    sr.Close();
                }
                catch (Exception ex)
                {
                    MessageBox.Show(ex.Message);
                }
                this.CurrentWindow.Save();
            }
            return true;
        }

        public void Exit()
        {
            this.Close();
        }

        public void Compile()
        {
            TextEditWindow Tew = CurrentWindow;
            if (Tew != null) Tew.Compile(null);
        }

        public void Compile(Action<bool> compileEndAction)
        {
            TextEditWindow Tew = CurrentWindow;
            if (Tew != null) Tew.Compile(compileEndAction);
        }

        public void Compile(string filename)
        {
            TextEditWindow Tew = GetTewFromFilename(filename);
            if (Tew != null) Tew.Compile(null);
        }

        public void Compile(string filename, Action<bool> compileEndAction)
        {
            TextEditWindow Tew = GetTewFromFilename(filename);
            if (Tew != null) Tew.Compile(compileEndAction);
        }

        public void Execute()
        {
            TextEditWindow Tew = CurrentWindow;
            if (Tew != null) Tew.Execute();
        }

        public void Execute(string filename)
        {
            TextEditWindow Tew = GetTewFromFilename(filename);
            if (Tew != null) Tew.Execute();
        }

        public void CompileAndExecute()
        {
            TextEditWindow Tew = CurrentWindow;
            if (Tew != null) Tew.CompileAndExecute(null);
        }

        public void CompileAndExecute(Action<bool> compileEndAction)
        {
            TextEditWindow Tew = CurrentWindow;
            if (Tew != null) Tew.CompileAndExecute(compileEndAction);
        }

        public void CompileAndExecute(string filename)
        {
            TextEditWindow Tew = GetTewFromFilename(filename);
            if (Tew != null) Tew.CompileAndExecute(null);
        }

        public void CompileAndExecute(string filename, Action<bool> compileEndAction)
        {
            TextEditWindow Tew = GetTewFromFilename(filename);
            if (Tew != null) Tew.CompileAndExecute(compileEndAction);
        }

        public bool BackUpFile()
        {
            return BackUpFile(CurrentWindow);
        }

        public bool BackUpFile(string filename)
        {
            return BackUpFile(GetTewFromFilename(filename));
        }

        private bool BackUpFile(TextEditWindow Tew)
        {
            if (Tew == null) return false;
            string dir = currentDirectory + "BackUpFiles";
            string fn = Path.GetFileNameWithoutExtension(Tew.Filename);
            if (!Directory.Exists(dir))
            {
                Directory.CreateDirectory(dir);
            }
            for (int i = Utility.BackUpNum; i >= 1; i--)
            {
                if (File.Exists(dir + "\\" + fn + ".bak" + i))
                {
                    if (i == Utility.BackUpNum)
                    {
                        //delete
                        File.Delete(dir + "\\" + fn + ".bak" + i);
                    }
                    else
                    {
                        Microsoft.VisualBasic.FileIO.FileSystem.RenameFile(dir + "\\" + fn + ".bak" + i, fn + ".bak" + (i + 1));
                    }
                }
            }
            try
            {
                StreamWriter swt = new StreamWriter(dir + "\\" + fn + ".bak1", false, Tew.ExAzukiControl.ExtensionInfo.Encoding);
                swt.Write(this.CurrentDocument.Text);
                swt.Close();
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
            }
            return true;
        }

        TextEditWindow GetTewFromFilename(string filename)
        {
            if (filename == null || filename == "") return null;
            foreach (IDockContent idc in dockPanel1.Documents)
            {
                TextEditWindow tew = idc as TextEditWindow;
                if (tew != null)
                {
                    if (FileUtil.IsSameFile(tew.Filename, filename) || Path.GetFileName(tew.Filename) == filename) return tew;
                }
            }
            return null;
        }

        ExAzukiControl GetExacFromFilename(string filename)
        {
            TextEditWindow tew = GetTewFromFilename(filename);
            if (tew == null) return null;
            return tew.ExAzukiControl;
        }

        public void AddError(ErrorType errortype, string text, int charIndex, string filename)
        {
            TextEditWindow tew = GetTewFromFilename(filename);
            if (tew == null)
            {
                return;
            }
            int row, column;
            tew.ExAzukiControl.GetLineColumnIndexFromCharIndex(charIndex, out row, out column);
            AddError(errortype, text, row + 1, column + 1, filename);
        }

        public void AddError(ErrorType errortype, string text, int row, int column, string filename)
        {
            errorWindow.AddCustom((int)errortype, text, row, column, filename);
            TextEditWindow tew = GetTewFromFilename(filename);
            if (tew == null)
            {
                return;
            }
            tew.AddMark(filename, row, column, NotepadUtility.ErrorMarkId);
        }

        public void ClearError(string filename)
        {
            errorWindow.Clear();
            TextEditWindow tew = GetTewFromFilename(filename);
            if (tew == null)
            {
                return;
            }
            tew.ClearErrorMark();
        }

        public void ReadBuffer(string filename, string parsefilename)
        {
            ExAzukiControl exac = GetExacFromFilename(parsefilename);
            if (exac != null)
            {
                exac.ExtensionInfo.ReadBuffer(filename);
            }
        }

        public string GetBufferDirectory(string parseFileName)
        {
            ExAzukiControl exac = GetExacFromFilename(parseFileName);
            if (exac != null)
            {
                return exac.ExtensionInfo.BufferDirectory;
            }
            return "";
        }

        public NameSpaceInfo GetBuffer(string filename, string parsefilename)
        {
            ExAzukiControl exac = GetExacFromFilename(parsefilename);
            if (exac != null)
            {
                NameSpaceInfo nsi;
                exac.ExtensionInfo.BufferData.TryGetValue(filename, out nsi);
                return nsi;
            }
            return null;
        }

        public EditorAssistData[] GetBuffers(ExtensionInfomation extensionInformation)
        {
            List<EditorAssistData> assistDatas = new List<EditorAssistData>();
            foreach (IDockContent idc in dockPanel1.Documents)
            {
                TextEditWindow tew = idc as TextEditWindow;
                if (tew != null)
                {
                    if (tew.ExAzukiControl.ExtensionInfo == extensionInformation &&
                        tew.ExAzukiControl.AssistData != null)
                    {
                        assistDatas.Add(tew.ExAzukiControl.AssistData);
                    }
                }
            }
            return assistDatas.ToArray();
        }

        public bool Parse(ParseEventArgs e)
        {
            if (this.ParseStarted != null)
            {
                this.ParseStarted.Invoke(e);
                return e.Handled;
            }
            return false;
        }

        public bool CheckWord(CheckWordEventArgs e)
        {
            if (this.CheckWordStarted != null)
            {
                this.CheckWordStarted.Invoke(e);
                return e.Handled;
            }
            return false;
        }

        public EditorAssistData GetBuffer(string filename)
        {
            ExAzukiControl exac = GetExacFromFilename(filename);
            if (exac == null) return null;
            return exac.AssistData;
        }

        public bool GetLineAndColumnIndexFromIndex(string filename, int index, out int lineindex, out int columnindex)
        {
            lineindex = 0;
            columnindex = 0;
            ExAzukiControl exac = GetExacFromFilename(filename);
            if (exac == null) return false;
            if (index >= exac.TextLength)
            {
                lineindex = exac.LineCount - 1;
                if (lineindex < 0) lineindex = 0;
                columnindex = exac.GetLineLength(lineindex) - 1;
                if (columnindex < 0) columnindex = 0;
            }
            else
            {
                exac.GetLineColumnIndexFromCharIndex(index, out lineindex, out columnindex);
            }
            return true;
        }
        public bool FindAndFocus(DefineLocation dl)
        {
            if (dl == null || dl.Editor == null) return false;
            foreach (IDockContent idc in dockPanel1.Documents)
            {
                TextEditWindow tew = idc as TextEditWindow;
                if (tew != null)
                {
                    if (tew.ExAzukiControl == dl.Editor)
                    {
                        if (dl.DefineIndex >= 0 && dl.DefineIndex < tew.ExAzukiControl.TextLength)
                        {
                            tew.ExAzukiControl.SetSelection(dl.DefineIndex, dl.DefineIndex);
                            tew.ExAzukiControl.ScrollToCaret();
                            if (tew.Pane != null) tew.Pane.ActiveContent = tew;
                            if (CurrentDocument != null) CurrentDocument.Focus();
                            return true;
                        }
                        else return false;
                    }
                }
            }
            return false;
        }
        public void FindAndFocus(AssistInformation[] assistInformations)
        {
            if (assistInformations == null || assistInformations.Length == 0)
            {
                return;
            }

            if (assistInformations.Length == 1)
            {
                FindAndFocus(assistInformations[0].DefineLocation);
                return;
            }

            defineLocationWindow.SetDefineLocations(assistInformations);
            defineLocationWindow.Show(dockPanel1, DockState.DockBottom);
        }
        public void GotoIndex(int index)
        {
            TextEditWindow tew = CurrentWindow;
            if (tew == null)
            {
                return;
            }

            tew.ExAzukiControl.GotoIndex(index);
        }
        /// <summary>
        /// 設定を設定します。
        /// </summary>
        /// <param name="extensionName">拡張名。</param>
        /// <param name="key">キー。</param>
        /// <param name="value">値。</param>
        public void SetSetting(string extensionName, string key, string value)
        {
            extensionSettingManager.SetSetting(extensionName, key, value);
        }

        /// <summary>
        /// 設定を取得します。
        /// </summary>
        /// <param name="extensionName">拡張名。</param>
        /// <param name="key">キー。</param>
        /// <returns>設定。存在しない場合は<c>null</c></returns>
        public string GetSetting(string extensionName, string key)
        {
            return extensionSettingManager.GetSetting(extensionName, key);
        }

        /// <summary>
        /// インクリメンタルサーチを行います。
        /// </summary>
        /// <param name="regex">対象の正規表現。</param>
        public void SetIncrementalSearch(Regex regex)
        {
            currentWatchPattern = new WatchPattern(NotepadUtility.SearchHighlightId, regex);
            ActivateIncrementalSearch();
        }

        private void ActivateIncrementalSearch()
        {
            var currentDocument = CurrentDocument;
            if (CurrentDocument == null)
            {
                return;
            }

            currentDocument.Document.WatchPatterns.Register(currentWatchPattern);
            currentDocument.Invalidate();
        }

        /// <summary>
        /// インクリメンタルサーチを解除します。
        /// </summary>
        public void ClearIncrementalSearch()
        {
            currentWatchPattern = new WatchPattern(NotepadUtility.SearchHighlightId, new Regex(""));
            ActivateIncrementalSearch();
        }

        /// <summary>
        /// Invokeします。
        /// </summary>
        /// <param name="action">アクション。</param>
        public void InvokeInMainForm(Action action)
        {
            if (InvokeRequired)
            {
                this.Invoke(action);
                return;
            }
            action();
        }
    }
}
