﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Text;
using System.Windows.Forms;

using NotepadNeueExtension;

namespace NotepadNeue
{
    public partial class MyListBox : UserControl
    {
        public event KeyEventHandler ListKeyDown;
        public event EventHandler ListDoubleClick;
        public event EventHandler ListSelectionChanged;
        public event ScrollEventHandler Scrolled;

        private BufferedGraphics grafx;
        private BufferedGraphicsContext context;
        HScrollBar Hsc;
        VScrollBar Vsc;
        List<AssistInformation> items;
        const int imagesize = 16;
        int maxwidth;
        int margin = 10;
        int maxsize = 6;
        int selectedindex = -1;
        int focusedindex = -1;

        public int MaxDisplayNum
        {
            get
            {
                return maxsize;
            }
            set
            {
                maxsize = value;
            }
        }
        public MyListBox()
        {
            InitializeComponent();
            items = new List<AssistInformation>();
            ItemHeight = 15;
            Hsc = new HScrollBar();
            Vsc = new VScrollBar();
            this.Controls.Add(Hsc);
            this.Controls.Add(Vsc);
            Hsc.Visible = false;
            Vsc.Visible = false;
            Vsc.SmallChange = 1;
            Vsc.LargeChange = 1;
            Vsc.ValueChanged += new EventHandler(Vsc_ValueChanged);
            Hsc.ValueChanged += new EventHandler(Hsc_ValueChanged);
            MouseWheel += new MouseEventHandler(MyTreeView_MouseWheel);
            LostFocus += new EventHandler(MyTreeView_LostFocus);
            this.SetStyle(ControlStyles.AllPaintingInWmPaint | ControlStyles.UserPaint, true);
            context = BufferedGraphicsManager.Current;
            context.MaximumBuffer = new Size(this.Width + 1, this.Height + 1);
            this.SetStyle(ControlStyles.OptimizedDoubleBuffer, true);
            grafx = context.Allocate(this.CreateGraphics(), new Rectangle(0, 0, this.Width, this.Height));
            DrawToBuffer(grafx.Graphics);
        }
        void MyTreeView_LostFocus(object sender, EventArgs e)
        {
        }

        void Hsc_ValueChanged(object sender, EventArgs e)
        {
            if (Scrolled != null)
            {
                Scrolled.Invoke(this, new ScrollEventArgs(ScrollEventType.First, 0));
            }
            DrawandReflash();
        }

        void MyTreeView_MouseWheel(object sender, MouseEventArgs e)
        {
            if ((ModifierKeys & Keys.Control) == Keys.Control)
            {
                if (Hsc.Visible)
                {
                    int result = Hsc.Value - e.Delta / 10;
                    if (result < 0) result = 0;
                    if (result > Hsc.Maximum) result = Hsc.Maximum;
                    Hsc.Value = result;
                }
            }
            else
            {
                if (Vsc.Visible)
                {
                    int result = Vsc.Value - e.Delta / 60;
                    if (result < 0) result = 0;
                    if (result > Vsc.Maximum) result = Vsc.Maximum;
                    Vsc.Value = result;
                }
            }
        }
        void Vsc_ValueChanged(object sender, EventArgs e)
        {
            if (Scrolled != null)
            {
                Scrolled.Invoke(this, new ScrollEventArgs(ScrollEventType.First, 0));
            }
            DrawandReflash();
        }
        private void MyTreeView_Paint(object sender, PaintEventArgs e)
        {
            grafx.Render(e.Graphics);
        }
        public void Clear()
        {
            items.Clear();
            maxwidth = 0;
        }
        public new void Show()
        {
            if (items.Count > 0)
            {
                Visible = true;
            }
            else
            {
                Visible = false;
            }
        }
        public void Add(AssistInformation info)
        {
            items.Add(info);
            Graphics g = Graphics.FromHwnd(this.Handle);
            maxwidth = (int)Math.Max((g.MeasureString(info.Text, this.Font)).Width + margin, maxwidth);
        }
        public void AddRange(AssistInformation[] infos)
        {
            for (int i = 0; i < infos.Length; i++)
            {
                Add(infos[i]);
            }
        }
        public void Sort()
        {
            items.Sort(new AssistInfomationComparer());
        }
        public void SelectAndFinalize(int num)
        {
            if (items.Count > maxsize)
            {
                this.Size = new Size(maxwidth + Vsc.Width + imagesize, imagesize * maxsize + 2);
            }
            else
            {
                this.Size = new Size(maxwidth + imagesize, imagesize * items.Count + 2);
            }
            AdjustSize();
            if (0 <= num && num < items.Count)
            {
                SelectedIndex = num;
            }
            if (0 <= num && num < items.Count)
            {
                Topindex = (num - maxsize / 2 < 0 ? 0 : num - maxsize / 2);
            }
            if (ListSelectionChanged != null)
            {
                ListSelectionChanged.Invoke(this, EventArgs.Empty);
            }
        }
        public void RemoveSameInData()
        {
            List<AssistInformation> shouldRemove = new List<AssistInformation>();
            AssistInformation before = null;
            foreach (AssistInformation current in items)
            {
                if (before != null && current.Equals(before))
                {
                    shouldRemove.Add(current);
                }
                before = current;
            }
            foreach (AssistInformation info in shouldRemove)
            {
                items.Remove(info);
            }
        }
        private void AdjustSize()
        {
            if (items.Count > maxsize)
            {
                Vsc.Location = new Point(this.Width - Vsc.Width - 1, 0);
                Vsc.Height = this.Height;
                Vsc.Maximum = items.Count - maxsize;
                Vsc.Visible = true;
            }
            else
            {
                Vsc.Visible = false;
                Vsc.Value = 0;
            }
        }
        public void EndUpdate()
        {
            AdjustSize();
            DrawandReflash();
        }
        public int Topindex
        {
            get
            {
                return Vsc.Value;
            }
            set
            {
                if (Vsc.Visible)
                {
                    int setnum = value;
                    if (setnum < Vsc.Minimum) setnum = Vsc.Minimum;
                    if (setnum > Vsc.Maximum) setnum = Vsc.Maximum;
                    Vsc.Value = setnum;
                }
            }
        }
        public int SelectedIndex
        {
            get
            {
                return selectedindex;
            }
            set
            {
                if (value < 0)
                {
                    focusedindex = selectedindex;
                }
                else
                {
                    focusedindex = value;
                }
                selectedindex = value;
                int dvalue = selectedindex;
                if (selectedindex < 0)
                {
                    if (focusedindex >= 0)
                    {
                        dvalue = focusedindex;
                    }
                }
                if (dvalue < Vsc.Value)
                {
                    int gain = Vsc.Value - dvalue;
                    if (Vsc.Value - gain < 0)
                    {
                        gain = Vsc.Value;
                    }
                    Vsc.Value -= gain;
                }
                else if (Vsc.Value + maxsize <= dvalue)
                {
                    int gain = dvalue - Vsc.Value - maxsize + 1;
                    if (Vsc.Value + gain > Vsc.Maximum)
                    {
                        gain = Vsc.Maximum - Vsc.Value;
                    }
                    Vsc.Value += gain;
                }
                if (ListSelectionChanged != null)
                {
                    ListSelectionChanged.Invoke(this, EventArgs.Empty);
                }
                DrawandReflash();
            }
        }
        public int FocusedIndex
        {
            get
            {
                return focusedindex;
            }
        }
        public int Count
        {
            get
            {
                return items.Count;
            }
        }
        public string this[int i]
        {
            get
            {
                return items[i].Text;
            }
        }
        public string SelectedText
        {
            get
            {
                if (selectedindex >= 0 && selectedindex < items.Count)
                {
                    return items[selectedindex].Text;
                }
                return "";
            }
            set
            {
            }
        }
        public string SelectedHintText
        {
            get
            {
                if (selectedindex >= 0 && selectedindex < items.Count)
                {
                    return items[selectedindex].HintText;
                }
                return "";
            }
        }
        public string SelectedSummary
        {
            get
            {
                if (selectedindex >= 0 && selectedindex < items.Count &&
                   !String.IsNullOrEmpty(items[selectedindex].Summary))
                {
                    return items[selectedindex].Summary;
                }
                return "";
            }
        }
        private void DrawToBuffer(Graphics g)
        {
            SolidBrush sb = new SolidBrush(this.BackColor);
            g.FillRectangle(sb, new Rectangle(0, 0, this.Width, this.Height));
            sb.Dispose();
            int start = Vsc.Value;
            for (int i = 0; i < maxsize; i++)
            {
                Rectangle rec = new Rectangle(0, i * ItemHeight, this.Width, ItemHeight);
                if (i + start == selectedindex)
                {
                    g.FillRectangle(Brushes.DarkGray, rec);
                }
                else
                {
                    g.FillRectangle(Brushes.White, rec);
                }
                if (i + start == focusedindex)
                {
                    Pen blackPen = new Pen(Color.Black, 1);
                    blackPen.DashStyle = System.Drawing.Drawing2D.DashStyle.Dash;
                    g.DrawRectangle(blackPen, new Rectangle(rec.X + imagesize, rec.Y, rec.Width - imagesize - 1, rec.Height - 1));
                    blackPen.Dispose();
                }
                if (i + start >= 0 && i + start < items.Count)
                {
                    AssistInformation ai = items[i + start] as AssistInformation;
                    if (ai != null)
                    {
                        g.FillRectangle(Brushes.White, new Rectangle(rec.X, rec.Y, 16, 16));
                        if (ai.Image != null) g.DrawImage(ai.Image, new Rectangle(rec.X, rec.Y, 16, 16));
                        g.DrawString(ai.Text, Font, Brushes.Black, new PointF(imagesize + rec.X, rec.Y + 2));
                    }
                }
            }
        }
        public void DrawandReflash()
        {
            DrawToBuffer(grafx.Graphics);
            Refresh();
        }
        public int ItemHeight
        {
            get;
            set;
        }

        private void MyListBox_SizeChanged(object sender, EventArgs e)
        {
            if (this.Width > 0 && this.Height > 0)
            {
                context = BufferedGraphicsManager.Current;
                context.MaximumBuffer = new Size(this.Width + 1, this.Height + 1);
                grafx = context.Allocate(this.CreateGraphics(), new Rectangle(0, 0, this.Width, this.Height));
            }
        }

        private void MyListBox_Paint(object sender, PaintEventArgs e)
        {
            grafx.Render(e.Graphics);
        }

        private void MyListBox_KeyDown(object sender, KeyEventArgs e)
        {
            if (ListKeyDown != null)
            {
                ListKeyDown.Invoke(this, e);
            }
        }

        private void MyListBox_MouseDown(object sender, MouseEventArgs e)
        {
            int num = Vsc.Value + e.Y / ItemHeight;
            SelectedIndex = num;
        }

        private void MyListBox_MouseDoubleClick(object sender, MouseEventArgs e)
        {
            if (ListDoubleClick != null)
            {
                ListDoubleClick.Invoke(this, e);
            }
        }
    }
}
