﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Text.RegularExpressions;
using System.Globalization;
using System.Linq;

namespace NotepadNeue
{
    /// <summary>
    /// 設定読み取りクラス
    /// </summary>
    public class SettingAnalyzer
    {
        Dictionary<string, string> dictionary;

        /// <summary>
        /// iniフォーマットかどうか
        /// </summary>
        public bool IsIniFormat
        {
            get;
            private set;
        }

        /// <summary>
        /// キーを元に文字列を取得します
        /// </summary>
        /// <param name="key"></param>
        /// <returns></returns>
        public string this[string key]
        {
            get
            {
                return ReadString(key);
            }
            set
            {
                ReplaceOrAdd(key, value);
            }
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="data">解析文字列</param>
        public SettingAnalyzer(string data)
        {
            data = data.Replace("\r\n", "\r").Replace("\r", "\n");
            dictionary = new Dictionary<string, string>(10);
            if (data.Contains("[formattype]"))
            {
                IsIniFormat = true;
                ReadAsIni(data);
            }
            else
            {
                ReadAsNormal(data);
            }
        }

        private void ReadAsIni(string data)
        {
            Regex sectionregex = new Regex("^\\[\\w*\\]");
            Regex pairregex = new Regex("^(?<key>[^=]+?) *= *(?<value>.*)$");
            foreach (string line in data.Split('\n'))
            {
                if (line.StartsWith("#")) continue;
                if (sectionregex.Match(line).Success) continue;
                Match m = pairregex.Match(line);
                if (m.Success)
                {
#if DEBUG
                    if (dictionary.ContainsKey(m.Groups["key"].Value))
                    {
                        Console.WriteLine("{0} is multiple defined", m.Groups["key"].Value);
                    }
#endif
                    ReplaceOrAdd(m.Groups["key"].Value, m.Groups["value"].Value);
                }
            }
        }

        private void ReadAsNormal(string data)
        {
            Regex keyRegex = new Regex(@"^\[([^\[\]]+?)\](.*)$");
            string lastKey = null;
            List<string> values = new List<string>();
            foreach (string line in data.Split('\n'))
            {
                Match match = keyRegex.Match(line);
                if (!match.Success)
                {
                    values.Add(line);
                }
                else
                {
                    if (lastKey != null)
                    {
                        ReplaceOrAdd(lastKey, String.Join("\n", values.ToArray()));
                    }
                    lastKey = match.Groups[1].Value;
                    values.Clear();
                    values.Add(match.Groups[2].Value);
                }
            }
            if (lastKey != null)
            {
                ReplaceOrAdd(lastKey, String.Join("\n", values.ToArray()));
            }
        }

        /// <summary>
        /// ディクショナリー
        /// </summary>
        public Dictionary<string, string> Dictionary
        {
            get
            {
                return dictionary;
            }
        }

        /// <summary>
        /// 書き込むか置き換えます
        /// </summary>
        /// <param name="key">キー</param>
        /// <param name="value">バリュー</param>
        public void ReplaceOrAdd(string key, string value)
        {
            value = value.Replace(@"\[", "[").Replace(@"\]", "]");
            if (dictionary.ContainsKey(key))
            {
                dictionary[key] = value;
            }
            else
            {
                dictionary.Add(key, value);
            }
        }

        /// <summary>
        /// 書き込むか置き換えます
        /// </summary>
        /// <param name="key">キー</param>
        /// <param name="value">バリュー</param>
        public void ReplaceOrAdd(string key, int value)
        {
            ReplaceOrAdd(key, value.ToString(CultureInfo.InvariantCulture));
        }

        /// <summary>
        /// 書き込むか置き換えます
        /// </summary>
        /// <param name="key">キー</param>
        /// <param name="value">バリュー</param>
        public void ReplaceOrAdd(string key, float value)
        {
            ReplaceOrAdd(key, value.ToString(CultureInfo.InvariantCulture));
        }

        /// <summary>
        /// 書き込むか置き換えます
        /// </summary>
        /// <param name="key">キー</param>
        /// <param name="value">バリュー</param>
        public void ReplaceOrAdd(string key, double value)
        {
            ReplaceOrAdd(key, value.ToString(CultureInfo.InvariantCulture));
        }

        /// <summary>
        /// 書き込むか置き換えます
        /// </summary>
        /// <param name="key">キー</param>
        /// <param name="value">バリュー</param>
        public void ReplaceOrAdd(string key, bool value)
        {
            ReplaceOrAdd(key, value ? "1" : "0");
        }

        /// <summary>
        /// 設定を取得する
        /// </summary>
        /// <param name="key">キー</param>
        /// <returns></returns>
        public string ReadString(string key)
        {
            string ret = "";
            if (!dictionary.TryGetValue(key, out ret))
            {
#if DEBUG
                Console.WriteLine("No Key:{0}", key);
#endif
                ret = "";

            }
            return ret;
        }

        /// <summary>
        /// intで設定を取得する
        /// </summary>
        /// <param name="key">キー</param>
        /// <param name="defaultValue">規定値</param>
        /// <returns></returns>
        public int ReadInt(string key, int defaultValue)
        {
            int val;
            if (int.TryParse(ReadString(key), NumberStyles.Integer, CultureInfo.InvariantCulture, out val))
            {
                return val;
            }
            return defaultValue;
        }

        /// <summary>
        /// floatで設定を取得する
        /// </summary>
        /// <param name="key">キー</param>
        /// <param name="defaultValue">規定値</param>
        /// <returns></returns>
        public float ReadFloat(string key, float defaultValue)
        {
            float val;
            if (float.TryParse(ReadString(key), NumberStyles.Float, CultureInfo.InvariantCulture, out val))
            {
                return val;
            }
            return defaultValue;
        }

        /// <summary>
        /// doubleで設定を取得する
        /// </summary>
        /// <param name="key">キー</param>
        /// <param name="defaultValue">規定値</param>
        /// <returns></returns>
        public double ReadDouble(string key, double defaultValue)
        {
            double val;
            if (double.TryParse(ReadString(key), NumberStyles.Float, CultureInfo.InvariantCulture, out val))
            {
                return val;
            }
            return defaultValue;
        }

        /// <summary>
        /// boolで設定を取得する
        /// </summary>
        /// <param name="key">キー</param>
        /// <returns></returns>
        public bool ReadBoolean(string key)
        {
            return ReadString(key) == "1";
        }

        public override string ToString()
        {
            return String.Join("\r\n", dictionary.Select(p => String.Format("[{0}]{1}", p.Key,
                p.Value.Replace("[", @"\[").Replace("]", @"\]").Replace("\n", "\r\n"))).ToArray());
        }
    }
}
