﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using NotepadNeueExtension;

namespace ParseExtension
{
    class FortranParser : IParser
    {
        string[][] variableTypes = new string[][] {
            new string[]{
                "INTEGER", 
            },
            new string[]{
                "DOUBLE","PRECISION",
            },
            new string[]{
                "REAL",
            },
            new string[]{
                "COMPLEX",
            },
            new string[]{
                "LOGICAL",
            },
            new string[]{
                "CHARACTER",
            }
        };

        string[][] funcTypes = new string[][]{
            new string[]{
                "EXTERNAL",
            },
            new string[]{
                "INTRINSIC",
            },
        };

        string[] intrinsicSubroutines = new string[] { 
            "DATE_AND_TIME",
            "SYSTEM_CLOCK", 
            "MVBITS", 
            "RANDOM_NUMBER", 
            "RANDOM_SEED",
        };

        Dictionary<string, string[]> variables;
        Dictionary<string, string[]> funcs;

        public IExtensionHost Host
        {
            get;
            private set;
        }

        public FortranParser(IExtensionHost host)
        {
            Host = host;
            variables = new Dictionary<string, string[]>();
            funcs = new Dictionary<string, string[]>();
        }

        #region IParser メンバー

        public void Parse(ParseEventArgs e)
        {
            string[] currentType = null;
            string[] currentFuncType = null;

            foreach (string line in e.ParseText.Split(new string[] { "\r", "\n" }, StringSplitOptions.RemoveEmptyEntries))
            {
                if (line.StartsWith("*"))
                {
                    continue;
                }

                if (currentType != null)
                {
                    if (!ReadVariable(currentType, line, 0))
                    {
                        currentType = null;
                    }
                }
                else if (currentFuncType != null)
                {
                    if (!ReadFunc(currentFuncType, line, 0))
                    {
                        currentFuncType = null;
                    }
                }
                else
                {
                    int index;
                    currentType = GetMatchedType(variableTypes, line, out index);
                    if (currentType != null && !ReadVariable(currentType, line, index))
                    {
                        currentType = null;
                    }
                    else
                    {
                        currentFuncType = GetMatchedType(funcTypes, line, out index);
                        if (currentFuncType != null && !ReadFunc(currentFuncType, line, index))
                        {
                            currentFuncType = null;
                        }
                    }
                }
            }

            int iter = 0;
            foreach (KeyValuePair<string, string[]> func in funcs)
            {
                string retType = "SUBROUTINE";
                string[] retTypes;
                if (variables.TryGetValue(func.Key, out retTypes))
                {
                    variables.Remove(func.Key);
                    retType = String.Join(" ", retTypes.Concat(new string[] { "FUNCTION" }).ToArray());
                }
                else if (func.Value == funcTypes[1] && intrinsicSubroutines.FirstOrDefault(s => s == func.Key) == null)
                {
                    // intrinsic is function
                    retType = "FUNCTION";
                }
                FunctionInfo functionInfo = new FunctionInfo()
                {
                    Name = func.Key,
                    Type = retType,
                };
                functionInfo.AddContent(String.Format("{0} {1}(...)", retType, func.Key));
                e.AssistData.VariableDatas.AddFunctionData(functionInfo, iter++);
            }
            foreach (KeyValuePair<string, string[]> variable in variables)
            {
                e.AssistData.VariableDatas.AddData(new EitherVariableOrTree()
                {
                    VariableInfo = new VariableInfo()
                    {
                        Name = variable.Key,
                        Type = String.Join(" ", variable.Value),
                    },
                }, iter++);
            }
            e.AssistData.VariableDatas.Start = 0;
            e.AssistData.VariableDatas.End = e.ParseText.Length;
        }

        private string[] GetMatchedType(string[][] types, string line, out int index)
        {
            index = 0;
            foreach (string[] type in types)
            {
                index = 0;
                foreach (string t in type)
                {
                    index = SafeIndexOf(line, t, index);
                    if (index < 0)
                    {
                        break;
                    }
                    index += t.Length;
                }

                if (index >= 0)
                {
                    return type;
                }
            }

            return null;
        }

        private int SafeIndexOf(string str, string search, int startIndex)
        {
            if (startIndex < 0 || startIndex >= str.Length)
            {
                return -1;
            }

            return str.IndexOf(search, startIndex);
        }

        private bool ReadVariable(string[] currentType, string line, int startIndex)
        {
            line = line.Substring(startIndex);
            bool isLastEmpty;
            string[] names = ReadImpl(line, out isLastEmpty);
            foreach (string name in names)
            {
                if (!variables.ContainsKey(name))
                {
                    variables.Add(name, currentType);
                }
            }
            return isLastEmpty;
        }

        private bool ReadFunc(string[] currentType, string line, int startIndex)
        {
            line = line.Substring(startIndex);
            bool isLastEmpty;
            string[] names = ReadImpl(line, out isLastEmpty);
            foreach (string name in names)
            {
                if (!funcs.ContainsKey(name))
                {
                    funcs.Add(name, currentType);
                }
            }
            return isLastEmpty;
        }

        private string[] ReadImpl(string line, out bool isLastEmpty)
        {
            HashSet<string> names = new HashSet<string>();
            string lastName = "";
            foreach (string split in line.Split(new char[] { ',' }))
            {
                StringBuilder builder = new StringBuilder();
                for (int i = 0; i < split.Length; i++)
                {
                    char c = split[i];
                    if (Char.IsWhiteSpace(c) || c == '$')
                    {
                        if (builder.Length > 0)
                        {
                            break;
                        }
                        continue;
                    }

                    if (builder.Length > 0 && Char.IsDigit(c))
                    {
                        builder.Append(c);
                    }
                    else if (Char.IsLetter(c))
                    {
                        builder.Append(c);
                    }
                }

                lastName = builder.ToString();
                if (lastName.Length > 0)
                {
                    names.Add(lastName);
                }
            }
            isLastEmpty = lastName.Length == 0;

            return names.ToArray();
        }

        #endregion
    }
}
