﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Text.RegularExpressions;

namespace ParseExtension
{
    class XmlWordChecker : IWordChecker
    {
        static Regex[] regexes = new Regex[]{
            new Regex(@"< *(?<tag>\w+)[^>]*(?<!/)>", RegexOptions.Singleline),
            new Regex(@"< */ *(?<tag>\w+) *>", RegexOptions.Singleline),
        };

        static Regex closeingTag = new Regex(@"< */ *(?<tag>\w*) *");

        #region IWordChecker メンバ

        public bool CheckWord(NotepadNeueExtension.CheckWordEventArgs e)
        {
            if (e.CaretIndex == e.Range.Start && e.CaretIndex == e.Range.End && !e.Document.ShouldBeIgnoredGrammatically(e.CaretIndex))
            {
                string closeingTagName;

                if (!IsClosingTag(e.Document.GetLineContent(e.Document.GetLineIndexFromCharIndex(e.CaretIndex)), e.CaretIndex, out closeingTagName))
                {
                    return true;
                }

                string data = e.Text;
                Tree<string> root = new Tree<string>();
                Tree<string> current = root;
                int startIndex = 0;
                bool[] notFound = new bool[regexes.Length];
                Match[] matches = new Match[regexes.Length];
                while (true)
                {
                    bool finish = false;
                    int tempStartIndex = startIndex;
                    startIndex = int.MaxValue;
                    for (int i = 0; i < regexes.Length; i++)
                    {
                        if (notFound[i])
                        {
                            notFound[i] = true;
                            continue;
                        }

                        matches[i] = regexes[i].Match(data, tempStartIndex);
                        startIndex = Math.Min(startIndex, matches[i].Success ? matches[i].Index + matches[i].Length : int.MaxValue);
                        notFound[i] = !matches[i].Success;
                    }

                    if (startIndex > e.CaretIndex)
                    {
                        if (current != root)
                        {
                            e.AssistInformations.Add(new NotepadNeueExtension.AssistInformation()
                            {
                                Text = String.Format("</{0}>", current.Value),
                                HintText = String.Format("</{0}>", current.Value),
                                Summary = "",
                                Image = Properties.Resources.tagclose_icon,
                            });
                            e.SelectIndex = 0;
                        }
                        finish = true;
                    }

                    if (matches[0].Index + matches[0].Length == startIndex)
                    {
                        if (!e.Document.ShouldBeIgnoredGrammatically(matches[0].Index))
                        {
                            current.AddChild(matches[0].Groups["tag"].Value);
                            current = current.Last;
                        }
                    }
                    else if (matches[1].Index + matches[1].Length == startIndex)
                    {
                        if (!e.Document.ShouldBeIgnoredGrammatically(matches[1].Index))
                        {
                            if (current.Value != matches[1].Groups["tag"].Value)
                            {
                                while (current.Parent != null)
                                {
                                    if (current.Value == matches[1].Groups["tag"].Value)
                                    {
                                        break;
                                    }
                                    current = current.Parent;
                                }

                                if (current == root)
                                {
                                    return true;
                                }
                            }
                            else
                            {
                                current = current.Parent;
                            }
                        }
                    }

                    finish |= Array.TrueForAll(notFound, b => b);

                    if (finish || startIndex == int.MaxValue)
                    {
                        break;
                    }
                }
            }

            return true;
        }

        private bool IsClosingTag(string text, int startIndex, out string tagName)
        {
            tagName = null;
            bool finish = false;
            bool slashFound = false;
            int i = 0;
            for (i = startIndex >= text.Length ? text.Length - 1 : startIndex; i >= 0; i--)
            {
                if (Char.IsWhiteSpace(text[i]))
                {
                    continue;
                }
                else if (Char.IsLetterOrDigit(text[i]))
                {
                    if (slashFound)
                    {
                        break;
                    }
                }
                else if (text[i] == '/')
                {
                    slashFound = true;
                }
                else if (text[i] == '<')
                {
                    if (slashFound)
                    {
                        finish = true;
                    }
                    break;
                }
                else
                {
                    break;
                }
            }

            if (finish)
            {
                Match m = closeingTag.Match(text, i);
                if (m.Success)
                {
                    tagName = m.Groups["tag"].Value;
                    return true;
                }
            }

            return false;
        }

        class Tree<T>
        {
            private Tree<T> parent;
            private List<Tree<T>> children;

            public Tree<T> Parent
            {
                get
                {
                    return parent;
                }
            }

            public Tree<T>[] Children
            {
                get
                {
                    return children.ToArray();
                }
            }

            public Tree<T> Last
            {
                get
                {
                    return children.Count > 0 ? children[children.Count - 1] : null;
                }
            }

            public T Value
            {
                get;
                set;
            }

            public Tree()
            {
                children = new List<Tree<T>>();
            }

            public void AddChild(T value)
            {
                Tree<T> tree = new Tree<T>()
                {
                    Value = value
                };
                tree.parent = this;

                children.Add(tree);
            }
        }

        #endregion
    }
}
