/*	$NetBSD: mips_emul.c,v 1.14.78.11 2010/06/01 19:10:45 matt Exp $ */

/*
 * Copyright (c) 1999 Shuichiro URATA.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/cdefs.h>

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/proc.h>
#include <sys/user.h>
#include <sys/types.h>
#include <sys/signalvar.h>

#include <mips64/mips_opcode.h>

#include <machine/cpu.h>
#include <mips64/reg.h>
#include <mips64/regnum.h>			/* symbolic register indices */
#include <mips64/vmparam.h>			/* for VM_MAX_ADDRESS */
#include <mips64/trap.h>

#define mips_btop(x)          ((paddr_t)(x) >> PGSHIFT)

#if defined(FPEMUL)
void MipsEmulateFP(uint32_t, register_t *, uint32_t);
#ifdef DEBUG_FPEMUL
char* regnames[]={
  "ZERO","AST","V0","V1","A0","A1","A2","A3","T0","T1","T2","T3","T4",
  "T5","T6","T7","S0","S1","S2","S3","S4","S5","S6","S7","T8","T9",
  "K0","K1","GP","SP","S8","RA","SR","MULLO","MULHI","BADVADDR",
  "CAUSE","PC","IC","CPL",
  "F0","F1","F2","F3","F4","F5","F6","F7",
  "F8","F9","F10","F11","F12","F13","F14","F15",
  "F16","F17","F18","F19","F20","F21","F22","F23",
  "F24","F25","F26","F27","F28","F29","F30","F31",
  "FSR",
};
register_t prev_regs[NREGS];

void saveRegs(register_t *regsPtr);
void saveRegs(register_t *regsPtr)
{
  memcpy((void*)prev_regs,(void*)regsPtr,sizeof(prev_regs));
}

void printRegs(register_t *regsPtr);
void printRegs(register_t *regsPtr){
  int i=0;
  int l=0;
  for(i=0;i<NREGS;i++,l++){
    if(prev_regs[i]!=regsPtr[i]){
      printf("%-9s : %016lx  -> %016lx\n",regnames[i],prev_regs[i],regsPtr[i]);
    }
  }
}
#endif
#endif

static inline void	send_sigsegv(vaddr_t, uint32_t, struct trap_frame *,
			    uint32_t);
static inline void	update_pc(struct trap_frame *, uint32_t);

void	MipsEmulateInst(uint32_t, uint32_t, vaddr_t, struct trap_frame *);
u_long	MipsEmulateBranch(struct trap_frame *, long, int, u_int);
void	MipsEmulateLWC0(uint32_t, struct trap_frame *, uint32_t);
void	MipsEmulateSWC0(uint32_t, struct trap_frame *, uint32_t);
void	MipsEmulateSpecial(uint32_t, struct trap_frame *, uint32_t);
void	MipsEmulateSpecial3(uint32_t, struct trap_frame *, uint32_t);
void	MipsEmulateLWC1(uint32_t, struct trap_frame *, uint32_t);
void	MipsEmulateLDC1(uint32_t, struct trap_frame *, uint32_t);
void	MipsEmulateSWC1(uint32_t, struct trap_frame *, uint32_t);
void	MipsEmulateSDC1(uint32_t, struct trap_frame *, uint32_t);

void	bcemul_lb(uint32_t, struct trap_frame *, uint32_t);
void	bcemul_lbu(uint32_t, struct trap_frame *, uint32_t);
void	bcemul_lh(uint32_t, struct trap_frame *, uint32_t);
void	bcemul_lhu(uint32_t, struct trap_frame *, uint32_t);
void	bcemul_lw(uint32_t, struct trap_frame *, uint32_t);
void	bcemul_lwl(uint32_t, struct trap_frame *, uint32_t);
void	bcemul_lwr(uint32_t, struct trap_frame *, uint32_t);
#if defined(__mips_n32) || defined(__mips_n64) || defined(__mips_o64)
void	bcemul_ld(uint32_t, struct trap_frame *, uint32_t);
void	bcemul_ldl(uint32_t, struct trap_frame *, uint32_t);
void	bcemul_ldr(uint32_t, struct trap_frame *, uint32_t);
#endif
void	bcemul_sb(uint32_t, struct trap_frame *, uint32_t);
void	bcemul_sh(uint32_t, struct trap_frame *, uint32_t);
void	bcemul_sw(uint32_t, struct trap_frame *, uint32_t);
void	bcemul_swl(uint32_t, struct trap_frame *, uint32_t);
void	bcemul_swr(uint32_t, struct trap_frame *, uint32_t);
#if defined(__mips_n32) || defined(__mips_n64) || defined(__mips_o64)
void	bcemul_sd(uint32_t, struct trap_frame *, uint32_t);
void	bcemul_sdl(uint32_t, struct trap_frame *, uint32_t);
void	bcemul_sdr(uint32_t, struct trap_frame *, uint32_t);
#endif

/*
 * MIPS2 LL instruction emulation state
 */
struct {
	struct proc *proc;
	vaddr_t addr;
	uint32_t value;
} llstate;

/*
 * Emulate instructions (including floating-point instructions)
 */
void
MipsEmulateInst(uint32_t status, uint32_t cause, vaddr_t opc,
    struct trap_frame *tf)
{
	uint32_t inst;
	int code = ILL_ILLOPC;
	struct proc *p = curcpu()->ci_curproc;
	union sigval sv;
	register_t *regsPtr = (register_t *)tf;

	/*
	 *  Fetch the instruction.
	 */
	if (cause & CR_BR_DELAY)
		guarded_read_4((paddr_t)opc+1, &inst);
	else
		guarded_read_4((paddr_t)opc, &inst);

	//	printf("EmulateInst status:%x cause:%x opc:%x op:%x inst:%x\n", status, cause, opc, ((InstFmt)inst).FRType.op,inst);

	switch (((InstFmt)inst).FRType.op) {
	case OP_LWC0:
		MipsEmulateLWC0(inst, tf, cause);
		break;
	case OP_SWC0:
		MipsEmulateSWC0(inst, tf, cause);
		break;
	case OP_SPECIAL:
		MipsEmulateSpecial(inst, tf, cause);
		break;
	case OP_COP1:
#if defined(FPEMUL)
#if defined(DEBUG_FPEMUL)
	  printf("EmulateInst status:%x cause:%x opc:%x op:%x inst:%x\n", status, cause, opc, ((InstFmt)inst).FRType.op,inst);
	  saveRegs(regsPtr);
#endif
		MipsEmulateFP(inst, regsPtr, cause);
#if defined(DEBUG_FPEMUL)
		printRegs(regsPtr);
#endif
		update_pc(tf, cause);
		break;
#endif
	case OP_LWC1:
#if defined(FPEMUL)
		MipsEmulateLWC1(inst, tf, cause);
		break;
#endif
	case OP_LDC1:
#if defined(FPEMUL)
		MipsEmulateLDC1(inst, tf, cause);
		break;
#endif
	case OP_SWC1:
#if defined(FPEMUL)
		MipsEmulateSWC1(inst, tf, cause);
		break;
#endif
	case OP_SDC1:
	  //	  printf("OP_SDC1 op %0x\n",OP_SDC1);
#if defined(FPEMUL)
		MipsEmulateSDC1(inst, tf, cause);
		break;
#else
		code = ILL_COPROC;
		/* FALLTHROUGH */
#endif
	default:
#ifdef DEBUG
		printf("pid %d (%s): trap: bad insn %p cause %#x insn %#x\n", curproc->p_pid, curproc->p_comm, opc, cause, inst);
#endif
		regsPtr[CAUSE] = cause;
		regsPtr[BADVADDR] = opc;

		p->p_md.md_regs->pc = tf->pc;
		p->p_md.md_regs->cause = tf->cause;
		p->p_md.md_regs->badvaddr = tf->badvaddr;
		sv.sival_ptr = (void *)tf->badvaddr;

		KERNEL_PROC_LOCK(p);
		trapsignal(p, SIGILL, 0, code, sv);
		KERNEL_PROC_UNLOCK(p);
		break;
	}
}

static inline void
send_sigsegv(vaddr_t vaddr, uint32_t exccode, struct trap_frame *tf,
    uint32_t cause)
{
	register_t *regsPtr = (register_t *)tf;
	struct proc *p = curcpu()->ci_curproc;
	union sigval sv;

	cause = (cause & 0xFFFFFF00) | (exccode << CR_EXC_CODE_SHIFT);
	regsPtr[CAUSE] = cause;
	regsPtr[BADVADDR] = (register_t)vaddr;

	p->p_md.md_regs->pc = tf->pc;
	p->p_md.md_regs->cause = tf->cause;
	p->p_md.md_regs->badvaddr = tf->badvaddr;
	sv.sival_ptr = (void *)vaddr;

	KERNEL_PROC_LOCK(p);
	trapsignal(p, SIGSEGV, 0, SEGV_MAPERR, sv);
	KERNEL_PROC_UNLOCK(p);
}

static inline void
update_pc(struct trap_frame *tf, uint32_t cause)
{
	register_t *regsPtr = (register_t *)tf;
	struct proc *p = curcpu()->ci_curproc;

	if (cause & CR_BR_DELAY)
		regsPtr[PC] = 
		    MipsEmulateBranch(tf, regsPtr[PC],
			p->p_md.md_regs->fsr, 0);
	else
		regsPtr[PC] += 4;
}

/*
 * MIPS2 LL instruction
 */
void
MipsEmulateLWC0(uint32_t inst, struct trap_frame *tf, uint32_t cause)
{
	vaddr_t	vaddr;
	int16_t		offset;
	void		*t;
	register_t *regsPtr = (register_t *)tf;

	offset = inst & 0xFFFF;
	vaddr = regsPtr[(inst>>21)&0x1F] + offset;

	/* segment and alignment check */
	if (vaddr < 0 || (vaddr & 3)) {
		send_sigsegv(vaddr, T_ADDR_ERR_LD, tf, cause);
		return;
	}

	t = &(regsPtr[(inst>>16)&0x1F]);

	if (copyin((void *)vaddr, t, 4) != 0) {
		send_sigsegv(vaddr, T_TLB_LD_MISS, tf, cause);
		return;
	}

	llstate.proc = curproc;
	llstate.addr = vaddr;
	llstate.value = *((uint32_t *)t);

	update_pc(tf, cause);
}

/*
 * MIPS2 SC instruction
 */
void
MipsEmulateSWC0(uint32_t inst, struct trap_frame *tf, uint32_t cause)
{
	vaddr_t	vaddr;
	uint32_t	value;
	int16_t		offset;
	register_t	*t;
	register_t *regsPtr = (register_t *)tf;

	offset = inst & 0xFFFF;
	vaddr = regsPtr[(inst>>21)&0x1F] + offset;

	/* segment and alignment check */
	if (vaddr < 0 || (vaddr & 3)) {
		send_sigsegv(vaddr, T_ADDR_ERR_LD, tf, cause);
		return;
	}

	t = (register_t *)&(regsPtr[(inst>>16)&0x1F]);

	/*
	 * Check that the process and address match the last
	 * LL instruction.
	 */
	if (curproc == llstate.proc && vaddr == llstate.addr) {
		llstate.proc = NULL;
		/*
		 * Check that the data at the address hasn't changed
		 * since the LL instruction.
		 */
		if (copyin((void *)vaddr, &value, 4) != 0) {
			send_sigsegv(vaddr, T_TLB_LD_MISS, tf, cause);
			return;
		}
		if (value == llstate.value) {
			/* SC successful */
			if (copyout(t, (void *)vaddr, 4) != 0) {
				send_sigsegv(vaddr, T_TLB_ST_MISS,
				    tf, cause);
				return;
			}
			*t = 1;
			update_pc(tf, cause);
			return;
		}
	}

	/* SC failed */
	*t = 0;
	update_pc(tf, cause);
}

void
MipsEmulateSpecial(uint32_t inst, struct trap_frame *tf, uint32_t cause)
{
	struct proc *p = curcpu()->ci_curproc;
	union sigval sv;
	register_t *regsPtr = (register_t *)tf;

	switch (((InstFmt)inst).RType.func) {
	case OP_SYNC:
		/* nothing */
		break;
	default:
		regsPtr[CAUSE] = cause;
		regsPtr[BADVADDR] = regsPtr[PC];

		p->p_md.md_regs->pc = tf->pc;
		p->p_md.md_regs->cause = tf->cause;
		p->p_md.md_regs->badvaddr = tf->badvaddr;
		sv.sival_ptr = (void *)regsPtr[PC];

		KERNEL_PROC_LOCK(p);
		trapsignal(p, SIGILL, 0, ILL_ILLOPC, sv);
		KERNEL_PROC_UNLOCK(p);
		break;
	}

	update_pc(tf, cause);
}

#if defined(FPEMUL)

#define LWSWC1_MAXLOOP	12

void
MipsEmulateLWC1(uint32_t inst, struct trap_frame *tf, uint32_t cause)
{
	vaddr_t	vaddr;
	int16_t		offset;
	void		*t;
	register_t	pc;
	int		i;
	register_t *regsPtr = (register_t *)tf;
	struct proc *p = curcpu()->ci_curproc;
	register_t *regsPtr2 = (register_t *)p->p_md.md_regs;

	offset = inst & 0xFFFF;
	vaddr = regsPtr[(inst>>21)&0x1F] + offset;

	/* segment and alignment check */
	if (vaddr < 0 || (vaddr & 3)) {
		send_sigsegv(vaddr, T_ADDR_ERR_LD, tf, cause);
		return;
	}

	/* NewABI FIXME */
	t = &(regsPtr2[FPBASE+((inst>>16)&0x1F)]);

	if (copyin((void *)vaddr, t, 4) != 0) {
		send_sigsegv(vaddr, T_TLB_LD_MISS, tf, cause);
		return;
	}

	pc = regsPtr[PC];
	update_pc(tf, cause);

	if (cause & CR_BR_DELAY)
		return;

	for (i = 1; i < LWSWC1_MAXLOOP; i++) {
		if (mips_btop(regsPtr[PC]) != mips_btop(pc))
			return;

		vaddr = regsPtr[PC];	/* XXX truncates to 32 bits */
		guarded_read_4((paddr_t)vaddr, &inst);
		if (((InstFmt)inst).FRType.op != OP_LWC1)
			return;

		offset = inst & 0xFFFF;
		vaddr = regsPtr[(inst>>21)&0x1F] + offset;

		/* segment and alignment check */
		if (vaddr < 0 || (vaddr & 3)) {
			send_sigsegv(vaddr, T_ADDR_ERR_LD, tf, cause);
			return;
		}

		/* NewABI FIXME */
		t = &(regsPtr2[FPBASE+((inst>>16)&0x1F)]);

		if (copyin((void *)vaddr, t, 4) != 0) {
			send_sigsegv(vaddr, T_TLB_LD_MISS, tf, cause);
			return;
		}

		pc = regsPtr[PC];
		update_pc(tf, cause);
	}
}

void
MipsEmulateLDC1(uint32_t inst, struct trap_frame *tf, uint32_t cause)
{
	vaddr_t	vaddr;
	int16_t		offset;
	void		*t;
	register_t *regsPtr = (register_t *)tf;
	struct proc *p = curcpu()->ci_curproc;
	register_t *regsPtr2 = (register_t *)p->p_md.md_regs;
	
	offset = inst & 0xFFFF;
	vaddr = regsPtr[(inst>>21)&0x1F] + offset;

	/* segment and alignment check */
	if (vaddr < 0  || (vaddr & 7)) {
		send_sigsegv(vaddr, T_ADDR_ERR_LD, tf, cause);
		return;
	}

	/* NewABI FIXME */
	t = &(regsPtr2[FPBASE+((inst>>16)&0x1E)]);

	if (copyin((void *)vaddr, t, 8) != 0) {
		send_sigsegv(vaddr, T_TLB_LD_MISS, tf, cause);
		return;
	}

	update_pc(tf, cause);
}

void
MipsEmulateSWC1(uint32_t inst, struct trap_frame *tf, uint32_t cause)
{
	vaddr_t	vaddr;
	int16_t		offset;
	void		*t;
	register_t	pc;
	int		i;
	register_t *regsPtr = (register_t *)tf;
	struct proc *p = curcpu()->ci_curproc;
	register_t *regsPtr2 = (register_t *)p->p_md.md_regs;

	offset = inst & 0xFFFF;
	vaddr = regsPtr[(inst>>21)&0x1F] + offset;

	/* segment and alignment check */
	if (vaddr < 0 || (vaddr & 3)) {
		send_sigsegv(vaddr, T_ADDR_ERR_ST, tf, cause);
		return;
	}

	/* NewABI FIXME */
	t = &(regsPtr2[FPBASE+((inst>>16)&0x1F)]);

	if (copyout(t, (void *)vaddr, 4) != 0) {
		send_sigsegv(vaddr, T_TLB_ST_MISS, tf, cause);
		return;
	}

	pc = regsPtr[PC];
	update_pc(tf, cause);

	if (cause & CR_BR_DELAY)
		return;

	for (i = 1; i < LWSWC1_MAXLOOP; i++) {
		if (mips_btop(regsPtr[PC]) != mips_btop(pc))
			return;

		vaddr = regsPtr[PC];	/* XXX truncates to 32 bits */
		guarded_read_4((paddr_t)vaddr, &inst);
		if (((InstFmt)inst).FRType.op != OP_SWC1)
			return;

		offset = inst & 0xFFFF;
		vaddr = regsPtr[(inst>>21)&0x1F] + offset;

		/* segment and alignment check */
		if (vaddr < 0 || (vaddr & 3)) {
			send_sigsegv(vaddr, T_ADDR_ERR_ST, tf, cause);
			return;
		}

		/* NewABI FIXME */
		t = (register_t *)&(regsPtr[(inst>>16)&0x1F]);

		if (copyout(t, (void *)vaddr, 4) != 0) {
			send_sigsegv(vaddr, T_TLB_ST_MISS, tf, cause);
			return;
		}

		pc = regsPtr[PC];
		update_pc(tf, cause);
	}
}

void
MipsEmulateSDC1(uint32_t inst, struct trap_frame *tf, uint32_t cause)
{
	vaddr_t	vaddr;
	int16_t		offset;
	void		*t;
	register_t *regsPtr = (register_t *)tf;
	struct proc *p = curcpu()->ci_curproc;
	register_t *regsPtr2 = (register_t *)p->p_md.md_regs;

	offset = inst & 0xFFFF;
	vaddr = regsPtr[(inst>>21)&0x1F] + offset;

	/* segment and alignment check */
	if (vaddr < 0 || (vaddr & 7)) {
		send_sigsegv(vaddr, T_ADDR_ERR_ST, tf, cause);
		return;
	}

	/* NewABI FIXME */
	t = &(regsPtr2[FPBASE+((inst>>16)&0x1E)]);

	if (copyout(t, (void *)vaddr, 8) != 0) {
		send_sigsegv(vaddr, T_TLB_ST_MISS, tf, cause);
		return;
	}

	update_pc(tf, cause);
}

void
bcemul_lb(uint32_t inst, struct trap_frame *tf, uint32_t cause)
{
	vaddr_t	vaddr;
	int16_t		offset;
	int8_t		x;
	register_t *regsPtr = (register_t *)tf;

	offset = inst & 0xFFFF;
	vaddr = regsPtr[(inst>>21)&0x1F] + offset;

	/* segment check */
	if (vaddr < 0) {
		send_sigsegv(vaddr, T_ADDR_ERR_LD, tf, cause);
		return;
	}

	if (copyin((void *)vaddr, &x, 1) != 0) {
		send_sigsegv(vaddr, T_TLB_LD_MISS, tf, cause);
		return;
	}

	regsPtr[(inst>>16)&0x1F] = (int32_t)x;

	update_pc(tf, cause);
}

void
bcemul_lbu(uint32_t inst, struct trap_frame *tf, uint32_t cause)
{
	vaddr_t	vaddr;
	int16_t		offset;
	uint8_t	x;
	register_t *regsPtr = (register_t *)tf;

	offset = inst & 0xFFFF;
	vaddr = regsPtr[(inst>>21)&0x1F] + offset;

	/* segment check */
	if (vaddr < 0) {
		send_sigsegv(vaddr, T_ADDR_ERR_LD, tf, cause);
		return;
	}

	if (copyin((void *)vaddr, &x, 1) != 0) {
		send_sigsegv(vaddr, T_TLB_LD_MISS, tf, cause);
		return;
	}

	regsPtr[(inst>>16)&0x1F] = (register_t)x;

	update_pc(tf, cause);
}

void
bcemul_lh(uint32_t inst, struct trap_frame *tf, uint32_t cause)
{
	vaddr_t	vaddr;
	int16_t		offset;
	int16_t		x;
	register_t *regsPtr = (register_t *)tf;

	offset = inst & 0xFFFF;
	vaddr = regsPtr[(inst>>21)&0x1F] + offset;

	/* segment and alignment check */
	if (vaddr < 0 || (vaddr & 1)) {
		send_sigsegv(vaddr, T_ADDR_ERR_LD, tf, cause);
		return;
	}

	if (copyin((void *)vaddr, &x, 2) != 0) {
		send_sigsegv(vaddr, T_TLB_LD_MISS, tf, cause);
		return;
	}

	regsPtr[(inst>>16)&0x1F] = x;

	update_pc(tf, cause);
}

void
bcemul_lhu(uint32_t inst, struct trap_frame *tf, uint32_t cause)
{
	vaddr_t	vaddr;
	int16_t		offset;
	u_int16_t	x;
	register_t *regsPtr = (register_t *)tf;

	offset = inst & 0xFFFF;
	vaddr = regsPtr[(inst>>21)&0x1F] + offset;

	/* segment and alignment check */
	if (vaddr < 0 || (vaddr & 1)) {
		send_sigsegv(vaddr, T_ADDR_ERR_LD, tf, cause);
		return;
	}

	if (copyin((void *)vaddr, &x, 2) != 0) {
		send_sigsegv(vaddr, T_TLB_LD_MISS, tf, cause);
		return;
	}

	regsPtr[(inst>>16)&0x1F] = (register_t)x;

	update_pc(tf, cause);
}

void
bcemul_lw(uint32_t inst, struct trap_frame *tf, uint32_t cause)
{
	vaddr_t	vaddr;
	int16_t		offset;
	register_t *regsPtr = (register_t *)tf;

	offset = inst & 0xFFFF;
	vaddr = regsPtr[(inst>>21)&0x1F] + offset;

	/* segment and alignment check */
	if (vaddr < 0 || (vaddr & 3)) {
		send_sigsegv(vaddr, T_ADDR_ERR_LD, tf, cause);
		return;
	}

	if (copyin((void *)vaddr, &(regsPtr[(inst>>16)&0x1F]), 4) != 0) {
		send_sigsegv(vaddr, T_TLB_LD_MISS, tf, cause);
		return;
	}

	update_pc(tf, cause);
}

void
bcemul_lwl(uint32_t inst, struct trap_frame *tf, uint32_t cause)
{
	vaddr_t	vaddr;
	uint32_t	a, x, shift;
	int16_t		offset;
	register_t *regsPtr = (register_t *)tf;

	offset = inst & 0xFFFF;
	vaddr = regsPtr[(inst>>21)&0x1F] + offset;

	/* segment check */
	if (vaddr < 0) {
		send_sigsegv(vaddr, T_ADDR_ERR_LD, tf, cause);
		return;
	}

	if (copyin((void *)(vaddr & ~3), &a, 4) != 0) {
		send_sigsegv(vaddr, T_TLB_LD_MISS, tf, cause);
		return;
	}

	x = regsPtr[(inst>>16)&0x1F];

	shift = (3 - (vaddr & 0x00000003)) * 8;
	a <<= shift;
	x &= ~(0xFFFFFFFFUL << shift);
	x |= a;

	regsPtr[(inst>>16)&0x1F] = x;

	update_pc(tf, cause);
}

void
bcemul_lwr(uint32_t inst, struct trap_frame *tf, uint32_t cause)
{
	vaddr_t		vaddr;
	uint32_t	a, x, shift;
	int16_t		offset;
	register_t *regsPtr = (register_t *)tf;

	offset = inst & 0xFFFF;
	vaddr = regsPtr[(inst>>21)&0x1F] + offset;

	/* segment check */
	if (vaddr & 0x80000000) {
		send_sigsegv(vaddr, T_ADDR_ERR_LD, tf, cause);
		return;
	}

	if (copyin((void *)(vaddr & ~3), &a, 4) != 0) {
		send_sigsegv(vaddr, T_TLB_LD_MISS, tf, cause);
		return;
	}

	x = regsPtr[(inst>>16)&0x1F];

	shift = (vaddr & 0x00000003) * 8;
	a >>= shift;
	x &= ~(0xFFFFFFFFUL >> shift);
	x |= a;

	regsPtr[(inst>>16)&0x1F] = x;

	update_pc(tf, cause);
}

#if defined(__mips_n32) || defined(__mips_n64) || defined(__mips_o64)
void
bcemul_ld(uint32_t inst, struct trap_frame *tf, uint32_t cause)
{
	vaddr_t	vaddr;
	int16_t		offset;
	register_t *regsPtr = (register_t *)tf;

	offset = inst & 0xFFFF;
	vaddr = regsPtr[(inst>>21)&0x1F] + offset;

	/* segment and alignment check */
	if (vaddr > VM_MAX_ADDRESS || vaddr & 0x7) {
		send_sigsegv(vaddr, T_ADDR_ERR_LD, tf, cause);
		return;
	}

	if (copyin((void *)vaddr, &(regsPtr[(inst>>16)&0x1F]), 8) != 0) {
		send_sigsegv(vaddr, T_TLB_LD_MISS, tf, cause);
		return;
	}

	update_pc(tf, cause);
}

void
bcemul_ldl(uint32_t inst, struct trap_frame *tf, uint32_t cause)
{
	vaddr_t	vaddr;
	uint64_t	a, x;
	uint32_t	shift;
	int16_t		offset;
	register_t *regsPtr = (register_t *)tf;

	offset = inst & 0xFFFF;
	vaddr = regsPtr[(inst>>21)&0x1F] + offset;

	/* segment check */
	if (vaddr & 0x80000000) {
		send_sigsegv(vaddr, T_ADDR_ERR_LD, tf, cause);
		return;
	}

	if (copyin((void *)(vaddr & ~0x7), &a, 8) != 0) {
		send_sigsegv(vaddr, T_TLB_LD_MISS, tf, cause);
		return;
	}

	x = regsPtr[(inst>>16)&0x1F];

	shift = (7 - (vaddr & 0x7)) * 8;
	a <<= shift;
	x &= ~(~(uint64_t)0UL << shift);
	x |= a;

	regsPtr[(inst>>16)&0x1F] = x;

	update_pc(tf, cause);
}

void
bcemul_ldr(uint32_t inst, struct trap_frame *tf, uint32_t cause)
{
	vaddr_t	vaddr;
	uint64_t	a, x;
	uint32_t	shift;
	int16_t		offset;
	register_t *regsPtr = (register_t *)tf;

	offset = inst & 0xFFFF;
	vaddr = regsPtr[(inst>>21)&0x1F] + offset;

	/* segment check */
	if (vaddr < 0) {
		send_sigsegv(vaddr, T_ADDR_ERR_LD, tf, cause);
		return;
	}

	if (copyin((void *)(vaddr & ~0x7), &a, 8) != 0) {
		send_sigsegv(vaddr, T_TLB_LD_MISS, tf, cause);
		return;
	}

	x = regsPtr[(inst>>16)&0x1F];

	shift = (vaddr & 0x7) * 8;
	a >>= shift;
	x &= ~(~(uint64_t)0UL >> shift);
	x |= a;

	regsPtr[(inst>>16)&0x1F] = x;

	update_pc(tf, cause);
}
#endif /* defined(__mips_n32) || defined(__mips_n64) || defined(__mips_o64) */

void
bcemul_sb(uint32_t inst, struct trap_frame *tf, uint32_t cause)
{
	vaddr_t	vaddr;
	int16_t		offset;
	register_t *regsPtr = (register_t *)tf;

	offset = inst & 0xFFFF;
	vaddr = regsPtr[(inst>>21)&0x1F] + offset;

	/* segment check */
	if (vaddr < 0) {
		send_sigsegv(vaddr, T_ADDR_ERR_ST, tf, cause);
		return;
	}

	if (guarded_write_1((paddr_t)vaddr, regsPtr[(inst>>16)&0x1F]) < 0) {
		send_sigsegv(vaddr, T_TLB_ST_MISS, tf, cause);
		return;
	}

	update_pc(tf, cause);
}

void
bcemul_sh(uint32_t inst, struct trap_frame *tf, uint32_t cause)
{
	vaddr_t	vaddr;
	int16_t		offset;
	register_t *regsPtr = (register_t *)tf;

	offset = inst & 0xFFFF;
	vaddr = regsPtr[(inst>>21)&0x1F] + offset;

	/* segment and alignment check */
	if (vaddr < 0 || vaddr & 1) {
		send_sigsegv(vaddr, T_ADDR_ERR_ST, tf, cause);
		return;
	}

	if (guarded_write_2((paddr_t)vaddr, regsPtr[(inst>>16)&0x1F]) < 0) {
		send_sigsegv(vaddr, T_TLB_ST_MISS, tf, cause);
		return;
	}

	update_pc(tf, cause);
}

void
bcemul_sw(uint32_t inst, struct trap_frame *tf, uint32_t cause)
{
	vaddr_t	vaddr;
	int16_t		offset;
	register_t *regsPtr = (register_t *)tf;

	offset = inst & 0xFFFF;
	vaddr = regsPtr[(inst>>21)&0x1F] + offset;

	/* segment and alignment check */
	if (vaddr < 0 || (vaddr & 3)) {
		send_sigsegv(vaddr, T_ADDR_ERR_ST, tf, cause);
		return;
	}

	if (guarded_write_4((paddr_t)vaddr, regsPtr[(inst>>16)&0x1F]) < 0) {
		send_sigsegv(vaddr, T_TLB_ST_MISS, tf, cause);
		return;
	}

	update_pc(tf, cause);
}

void
bcemul_swl(uint32_t inst, struct trap_frame *tf, uint32_t cause)
{
	vaddr_t	vaddr;
	uint32_t	a, x, shift;
	int16_t		offset;
	register_t *regsPtr = (register_t *)tf;

	offset = inst & 0xFFFF;
	vaddr = regsPtr[(inst>>21)&0x1F] + offset;

	/* segment check */
	if (vaddr < 0) {
		send_sigsegv(vaddr, T_ADDR_ERR_ST, tf, cause);
		return;
	}

	if (copyin((void *)(vaddr & ~3), &a, 4) != 0) {
		send_sigsegv(vaddr, T_TLB_ST_MISS, tf, cause);
		return;
	}

	x = regsPtr[(inst>>16)&0x1F];

	shift = (3 - (vaddr & 3)) * 8;
	x >>= shift;
	a &= ~(0xFFFFFFFFUL >> shift);
	a |= x;

	if (guarded_write_4((paddr_t)vaddr, a) < 0) {
		send_sigsegv(vaddr, T_TLB_ST_MISS, tf, cause);
		return;
	}

	update_pc(tf, cause);
}

void
bcemul_swr(uint32_t inst, struct trap_frame *tf, uint32_t cause)
{
	vaddr_t	vaddr;
	uint32_t	a, x, shift;
	int16_t		offset;
	register_t *regsPtr = (register_t *)tf;

	offset = inst & 0xFFFF;
	vaddr = regsPtr[(inst>>21)&0x1F] + offset;

	/* segment check */
	if (vaddr < 0) {
		send_sigsegv(vaddr, T_ADDR_ERR_ST, tf, cause);
		return;
	}

	if (copyin((void *)(vaddr & ~3), &a, 4) != 0) {
		send_sigsegv(vaddr, T_TLB_ST_MISS, tf, cause);
		return;
	}

	x = regsPtr[(inst>>16)&0x1F];

	shift = (vaddr & 3) * 8;
	x <<= shift;
	a &= ~(0xFFFFFFFFUL << shift);
	a |= x;

	if (guarded_write_4((paddr_t)vaddr, a) < 0) {
		send_sigsegv(vaddr, T_TLB_ST_MISS, tf, cause);
		return;
	}

	update_pc(tf, cause);
}

#if defined(__mips_n32) || defined(__mips_n64) || defined(__mips_o64)
void
bcemul_sd(uint32_t inst, struct trap_frame *tf, uint32_t cause)
{
	vaddr_t	vaddr;
	int16_t		offset;
	register_t *regsPtr = (register_t *)tf;

	offset = inst & 0xFFFF;
	vaddr = regsPtr[(inst>>21)&0x1F] + offset;

	/* segment and alignment check */
	if (vaddr < 0 || vaddr & 0x7) {
		send_sigsegv(vaddr, T_ADDR_ERR_ST, tf, cause);
		return;
	}

	if (copyout((void *)vaddr, &regsPtr[(inst>>16)&0x1F], 8) < 0) {
		send_sigsegv(vaddr, T_TLB_ST_MISS, tf, cause);
		return;
	}

	update_pc(tf, cause);
}

void
bcemul_sdl(uint32_t inst, struct trap_frame *tf, uint32_t cause)
{
	vaddr_t	vaddr;
	uint64_t	a, x;
	uint32_t	shift;
	int16_t		offset;
	register_t *regsPtr = (register_t *)tf;

	offset = inst & 0xFFFF;
	vaddr = regsPtr[(inst>>21)&0x1F] + offset;

	/* segment check */
	if (vaddr < 0) {
		send_sigsegv(vaddr, T_ADDR_ERR_ST, tf, cause);
		return;
	}

	if (copyin((void *)(vaddr & ~0x7), &a, 8) != 0) {
		send_sigsegv(vaddr, T_TLB_ST_MISS, tf, cause);
		return;
	}

	x = regsPtr[(inst>>16)&0x1F];

	shift = (7 - (vaddr & 7)) * 8;
	x >>= shift;
	a &= ~(~(uint64_t)0U >> shift);
	a |= x;

	if (copyout((void *)(vaddr & ~0x7), &a, 8) != 0) {
		send_sigsegv(vaddr, T_TLB_ST_MISS, tf, cause);
		return;
	}

	update_pc(tf, cause);
}

void
bcemul_sdr(uint32_t inst, struct trap_frame *tf, uint32_t cause)
{
	vaddr_t	vaddr;
	uint64_t	a, x;
	uint32_t	shift;
	int16_t		offset;
	register_t *regsPtr = (register_t *)tf;

	offset = inst & 0xFFFF;
	vaddr = regsPtr[(inst>>21)&0x1F] + offset;

	/* segment check */
	if (vaddr < 0) {
		send_sigsegv(vaddr, T_ADDR_ERR_ST, tf, cause);
		return;
	}

	if (copyin((void *)(vaddr & ~0x7), &a, 8) != 0) {
		send_sigsegv(vaddr, T_TLB_ST_MISS, tf, cause);
		return;
	}

	x = regsPtr[(inst>>16)&0x1F];

	shift = (vaddr & 7) * 8;
	x <<= shift;
	a &= ~(~(uint64_t)0U << shift);
	a |= x;

	if (copyout((void *)(vaddr & ~0x7), &a, 8) != 0) {
		send_sigsegv(vaddr, T_TLB_ST_MISS, tf, cause);
		return;
	}

	update_pc(tf, cause);
}
#endif /* defined(__mips_n32) || defined(__mips_n64) || defined(__mips_o64) */
#endif /* defined(FPEMUL) */
