/*	$OpenBSD: machdep.c,v 1.19 2010/05/08 21:59:56 miod Exp $ */

/*
 * Copyright (c) 2009, 2010 Miodrag Vallat.
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/*
 * Copyright (c) 2003-2004 Opsycon AB  (www.opsycon.se / www.opsycon.com)
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */
#include <sys/param.h>
#include <sys/systm.h>
#include <sys/kernel.h>
#include <sys/proc.h>
#include <sys/buf.h>
#include <sys/reboot.h>
#include <sys/conf.h>
#include <sys/file.h>
#include <sys/msgbuf.h>
#include <sys/tty.h>
#include <sys/user.h>
#include <sys/exec.h>
#include <sys/sysctl.h>
#include <sys/mount.h>
#include <sys/syscallargs.h>
#include <sys/exec_elf.h>
#ifdef SYSVSHM
#include <sys/shm.h>
#endif
#ifdef SYSVSEM
#include <sys/sem.h>
#endif

#include <uvm/uvm.h>
#include <uvm/uvm_extern.h>

#include <machine/db_machdep.h>
#include <ddb/db_interface.h>

#include <machine/autoconf.h>
#include <machine/cpu.h>
#include <machine/memconf.h>

#include <dev/cons.h>

#include <mips64/archtype.h>

#include <machine/octeon_pcmap_regs.h>

#include <octeon/dev/obiovar.h>
#include <octeon/dev/octeonreg.h>

#define btoc(x) (((vm_offset_t)(x)+PAGE_MASK)>>PAGE_SHIFT)
#define MIPS_PHYS_MASK                  (0x1fffffff)
#define   MIPS_KSEG0_TO_PHYS(x)           ((__uintptr_t)(x) & MIPS_PHYS_MASK)
typedef u_int64_t vm_offset_t;

/* The following is used externally (sysctl_hw) */
char	machine[] = MACHINE;		/* Machine "architecture" */
char	cpu_model[30];
char	pmon_bootp[80];

/*
 * Declare these as initialized data so we can patch them.
 */
#ifndef	BUFCACHEPERCENT
#define	BUFCACHEPERCENT	5	/* Can be changed in config. */
#endif
#ifndef	BUFPAGES
#define BUFPAGES 0		/* Can be changed in config. */
#endif
int	bufpages = BUFPAGES;
int	bufcachepercent = BUFCACHEPERCENT;

/*
 * Even though the system is 64bit, the hardware is constrained to up
 * to 2G of contigous physical memory (direct 2GB DMA area), so there
 * is no particular constraint. paddr_t is long so: 
 */
struct uvm_constraint_range  dma_constraint = { 0x0, 0xffffffffUL };
struct uvm_constraint_range *uvm_md_constraints[] = { NULL };

vm_map_t exec_map;
vm_map_t phys_map;

/*
 * safepri is a safe priority for sleep to set for a spin-wait
 * during autoconfiguration or after a panic.
 */
int   safepri = 0;

caddr_t	msgbufbase;
vaddr_t	uncached_base;

int	physmem;		/* Max supported memory, changes to actual. */
int	ncpu = 1;		/* At least one CPU in the system. */
struct	user *proc0paddr;
int	kbd_reset;

void	generic_device_register(struct device *, void *);
void	generic_reset(void);
void	generic_setup(void);

const struct platform generic_platform = {
	.system_type = 0,
	.vendor = "Generic",
	.product = "OCTEON",

	.setup = generic_setup,
	.device_register = generic_device_register,

	.powerdown = NULL,
	.reset = generic_reset
};

const struct platform *sys_platform = &generic_platform;
struct cpu_hwinfo bootcpu_hwinfo;

/* Pointers to the start and end of the symbol table. */
caddr_t	ssym;
caddr_t	esym;
caddr_t	ekern;

struct phys_mem_desc mem_layout[MAXMEMSEGS];

#if 0
static u_long atoi(const char *, uint);
#endif

void	build_trampoline(vaddr_t, vaddr_t);
void	dumpsys(void);
void	dumpconf(void);
extern	void parsepmonbp(void);
vaddr_t	mips_init(__register_t, __register_t, __register_t, __register_t);
boolean_t is_memory_range(paddr_t, psize_t, psize_t);

cons_decl(oct);

/*
struct consdev octcons = {
	NULL,
	NULL,
	octcngetc,
	octcnputc,
	nullcnpollc,
	NULL,
	makedev(0, 0),
	CN_DEAD
};
*/
struct consdev octcons = cons_init(oct);

#if defined(__mips_n64) 
#define MAX_APP_DESC_ADDR     0xffffffffafffffff
#else
#define MAX_APP_DESC_ADDR     0xafffffff
#endif

uint64_t ciu_get_en_reg_addr_new(int corenum, int intx, int enx, int ciu_ip);
void ciu_dump_interrutps_enabled(int core_num, int intx, int enx, int ciu_ip);

static void octeon_boot_params_init(register_t ptr);
static uint64_t ciu_get_intr_sum_reg_addr(int core_num, int intx, int enx);
static uint64_t ciu_get_intr_en_reg_addr(int core_num, int intx, int enx);

void
generic_reset()
{
}

void
generic_setup(void)
{
}

void
generic_device_register(struct device *dev, void *aux)
{
	const char *drvrname = dev->dv_cfdata->cf_driver->cd_name;
	const char *name = dev->dv_xname;

	if (dev->dv_class != bootdev_class)
		return;	

	/* 
	 * The device numbering must match. There's no way
	 * pmon tells us more info. Depending on the usb slot
	 * and hubs used you may be lucky. Also, assume umass/sd for usb
	 * attached devices.
	 */
	switch (bootdev_class) {
	case DV_DISK:
		if (strcmp(drvrname, "wd") == 0 && strcmp(name, bootdev) == 0)
			bootdv = dev;
		else {
			/* XXX this really only works safely for usb0... */
		    	if ((strcmp(drvrname, "sd") == 0 ||
			    strcmp(drvrname, "cd") == 0) &&
			    strncmp(bootdev, "usb", 3) == 0 &&
			    strcmp(name + 2, bootdev + 3) == 0)
				bootdv = dev;
		}
		break;
	case DV_IFNET:
		/*
		 * This relies on the onboard Ethernet interface being
		 * attached before any other (usb) interface.
		 */
		bootdv = dev;
		break;
	default:
		break;
	}
}

void
octeon_led_write_char(int char_position, char val)
{
	uint64_t ptr = (OCTEON_CHAR_LED_BASE_ADDR | 0xf8);

	if (!octeon_board_real())
		return;

	char_position &= 0x7;  /* only 8 chars */
	ptr += char_position;
	oct_write8_x8(ptr, val);
}

void
octeon_led_write_char0(char val)
{
	uint64_t ptr = (OCTEON_CHAR_LED_BASE_ADDR | 0xf8);

	if (!octeon_board_real())
		return;
	oct_write8_x8(ptr, val);
}

void
octeon_led_write_hexchar(int char_position, char hexval)
{
	uint64_t ptr = (OCTEON_CHAR_LED_BASE_ADDR | 0xf8);
	char char1, char2;

	if (!octeon_board_real())
		return;

	char1 = (hexval >> 4) & 0x0f; char1 = (char1 < 10)?char1+'0':char1+'7';
	char2 = (hexval  & 0x0f); char2 = (char2 < 10)?char2+'0':char2+'7';
	char_position &= 0x7;  /* only 8 chars */
	if (char_position > 6)
		char_position = 6;
	ptr += char_position;
	oct_write8_x8(ptr, char1);
	ptr++;
	oct_write8_x8(ptr, char2);
}

void
octeon_led_write_string(const char *str)
{
	uint64_t ptr = (OCTEON_CHAR_LED_BASE_ADDR | 0xf8);
	int i;

	if (!octeon_board_real())
		return;

	for (i=0; i<8; i++, ptr++) {
		if (str && *str)
			oct_write8_x8(ptr, *str++);
		else
			oct_write8_x8(ptr, ' ');
		oct_read64(OCTEON_MIO_BOOT_BIST_STAT);
	}
}

static char progress[8] = { '-', '/', '|', '\\', '-', '/', '|', '\\'};

void
octeon_led_run_wheel(int *prog_count, int led_position)
{
	if (!octeon_board_real())
		return;
	octeon_led_write_char(led_position, progress[*prog_count]);
	*prog_count += 1;
	*prog_count &= 0x7;
}

void
octeon_led_write_hex(uint32_t wl)
{
	char nbuf[80];

	snprintf(nbuf, 80, "%X", wl);
	octeon_led_write_string(nbuf);
}

void
octeon_ciu_stop_gtimer(int timer)
{
	oct_write64(OCTEON_CIU_GENTIMER_ADDR(timer), 0ll);
}

void
octeon_ciu_start_gtimer(int timer, u_int one_shot, uint64_t time_cycles)
{
    	octeon_ciu_gentimer gentimer;

        gentimer.word64 = 0;
        gentimer.bits.one_shot = one_shot;
        gentimer.bits.len = time_cycles - 1;
        oct_write64(OCTEON_CIU_GENTIMER_ADDR(timer), gentimer.word64);
}

/*
 * octeon_ciu_reset
 *
 * Shutdown all CIU to IP2, IP3 mappings
 */
void
octeon_ciu_reset(void)
{

	octeon_ciu_stop_gtimer(CIU_GENTIMER_NUM_0);
	octeon_ciu_stop_gtimer(CIU_GENTIMER_NUM_1);
	octeon_ciu_stop_gtimer(CIU_GENTIMER_NUM_2);
	octeon_ciu_stop_gtimer(CIU_GENTIMER_NUM_3);

	ciu_disable_intr(CIU_THIS_CORE, CIU_INT_0, CIU_EN_0);
	ciu_disable_intr(CIU_THIS_CORE, CIU_INT_0, CIU_EN_1);
	ciu_disable_intr(CIU_THIS_CORE, CIU_INT_1, CIU_EN_0);
	ciu_disable_intr(CIU_THIS_CORE, CIU_INT_1, CIU_EN_1);

	ciu_clear_int_summary(CIU_THIS_CORE, CIU_INT_0, CIU_EN_0, 0ll);
	ciu_clear_int_summary(CIU_THIS_CORE, CIU_INT_1, CIU_EN_0, 0ll);
	ciu_clear_int_summary(CIU_THIS_CORE, CIU_INT_1, CIU_EN_1, 0ll);
}

/*
 * mips_disable_interrupt_controllers
 *
 * Disable interrupts in the CPU controller
 */
void
mips_disable_interrupt_controls(void)
{
	/*
	 * Disable interrupts in CIU.
	 */
	octeon_ciu_reset();
}

/*
 * ciu_get_intr_sum_reg_addr
 */
static uint64_t
ciu_get_intr_sum_reg_addr(int core_num, int intx, int enx)
{
	uint64_t ciu_intr_sum_reg_addr;

    	if (enx == CIU_EN_0)
            	ciu_intr_sum_reg_addr = OCTEON_CIU_SUMMARY_BASE_ADDR +
		    (core_num * 0x10) + (intx * 0x8);
	else
            	ciu_intr_sum_reg_addr = OCTEON_CIU_SUMMARY_INT1_ADDR;

        return (ciu_intr_sum_reg_addr);
}


/*
 * ciu_get_intr_en_reg_addr
 */
static uint64_t
ciu_get_intr_en_reg_addr(int core_num, int intx, int enx)
{
	uint64_t ciu_intr_reg_addr;

    	ciu_intr_reg_addr = OCTEON_CIU_ENABLE_BASE_ADDR + 
	    ((enx == 0) ? 0x0 : 0x8) + (intx * 0x10) +  (core_num * 0x20);
        return (ciu_intr_reg_addr);
}




/*
 * ciu_get_intr_reg_addr
 *
 * 200 ---int0,en0 ip2
 * 208 ---int0,en1 ip2 ----> this is wrong... this is watchdog
 * 
 * 210 ---int0,en0 ip3 --
 * 218 ---int0,en1 ip3 ----> same here.. .this is watchdog... right?
 * 
 * 220 ---int1,en0 ip2
 * 228 ---int1,en1 ip2
 * 230 ---int1,en0 ip3 --
 * 238 ---int1,en1 ip3
 *
 */
uint64_t
ciu_get_en_reg_addr_new(int corenum, int intx, int enx, int ciu_ip)
{
	uint64_t ciu_intr_reg_addr = OCTEON_CIU_ENABLE_BASE_ADDR;

	/* XXX kasserts? */
	if (enx < CIU_EN_0 || enx > CIU_EN_1) {
		printf("%s: invalid enx value %d, should be %d or %d\n",
		    __func__, enx, CIU_EN_0, CIU_EN_1);
		return 0;
	}
	if (intx < CIU_INT_0 || intx > CIU_INT_1) {
		printf("%s: invalid intx value %d, should be %d or %d\n",
		    __func__, enx, CIU_INT_0, CIU_INT_1);
		return 0;
	}
	if (ciu_ip < CIU_MIPS_IP2 || ciu_ip > CIU_MIPS_IP3) {
		printf("%s: invalid ciu_ip value %d, should be %d or %d\n",
		    __func__, ciu_ip, CIU_MIPS_IP2, CIU_MIPS_IP3);
		return 0;
	}

	ciu_intr_reg_addr += (enx    * 0x8);
	ciu_intr_reg_addr += (ciu_ip * 0x10);
	ciu_intr_reg_addr += (intx   * 0x20);
	return (ciu_intr_reg_addr);
}

/*
 * ciu_get_int_summary
 */
uint64_t
ciu_get_int_summary(int core_num, int intx, int enx)
{
	uint64_t ciu_intr_sum_reg_addr;

	if (core_num == CIU_THIS_CORE)
        	core_num = octeon_get_core_num();
	ciu_intr_sum_reg_addr = ciu_get_intr_sum_reg_addr(core_num, intx, enx);
	return (oct_read64(ciu_intr_sum_reg_addr));
}

//#define DEBUG_CIU 1

#ifdef DEBUG_CIU
#define DEBUG_CIU_SUM 1
#define DEBUG_CIU_EN 1
#endif

/*
 * ciu_clear_int_summary
 */
void
ciu_clear_int_summary(int core_num, int intx, int enx, uint64_t write_bits)
{
	uint32_t cpu_status_bits;
	uint64_t ciu_intr_sum_reg_addr;

#ifdef DEBUG_CIU_SUM
	uint64_t ciu_intr_sum_bits;
#endif


	if (core_num == CIU_THIS_CORE) {
        	core_num = octeon_get_core_num();
	}

#ifdef DEBUG_CIU_SUM
        printf(" CIU: core %u clear sum IntX %u  Enx %u  Bits: 0x%llX\n",
	    core_num, intx, enx, write_bits);
#endif

	cpu_status_bits = disableintr();

	ciu_intr_sum_reg_addr = ciu_get_intr_sum_reg_addr(core_num, intx, enx);

#ifdef DEBUG_CIU_SUM
    	ciu_intr_sum_bits =  oct_read64(ciu_intr_sum_reg_addr);	/* unneeded dummy read */
        printf(" CIU: status: 0x%X  reg_addr: 0x%llX   Val: 0x%llX   ->  0x%llX",
	    cpu_status_bits, ciu_intr_sum_reg_addr, ciu_intr_sum_bits,
	    ciu_intr_sum_bits | write_bits);
#endif

	oct_write64(ciu_intr_sum_reg_addr, write_bits);
	oct_read64(OCTEON_MIO_BOOT_BIST_STAT);	/* Bus Barrier */

#ifdef DEBUG_CIU_SUM
        printf(" Readback: 0x%llX\n\n   ", (uint64_t) oct_read64(ciu_intr_sum_reg_addr));
#endif
    
	setsr(cpu_status_bits);
}

/*
 * ciu_disable_intr
 */
void
ciu_disable_intr(int core_num, int intx, int enx)
{
	uint32_t cpu_status_bits;
	uint64_t ciu_intr_reg_addr;

	if (core_num == CIU_THIS_CORE)
        	core_num = octeon_get_core_num();

	cpu_status_bits = disableintr();
    
	ciu_intr_reg_addr = ciu_get_intr_en_reg_addr(core_num, intx, enx);

	oct_read64(ciu_intr_reg_addr);	/* Dummy read */

	oct_write64(ciu_intr_reg_addr, 0LL);
	oct_read64(OCTEON_MIO_BOOT_BIST_STAT);	/* Bus Barrier */

	setsr(cpu_status_bits);
}

#if 0
void
ciu_dump_interrupts_enabled(int core_num, int intx, int enx, int ciu_ip)
{

	uint64_t ciu_intr_reg_addr;
	uint64_t ciu_intr_bits;

        if (core_num == CIU_THIS_CORE) {
            	core_num = octeon_get_core_num();
        }

#ifndef OCTEON_SMP_1
	ciu_intr_reg_addr = ciu_get_intr_en_reg_addr(core_num, intx, enx);
#else
	ciu_intr_reg_addr = ciu_get_en_reg_addr_new(core_num, intx, enx, ciu_ip);
#endif

        if (!ciu_intr_reg_addr) {
            printf("Bad call to %s\n", __func__);
            while(1);
            return;
        }

	ciu_intr_bits =  oct_read64(ciu_intr_reg_addr);
        printf(" CIU core %d  int: %d  en: %d  ip: %d  Add: %#llx  enabled: %#llx  SR: %x\n",
	    core_num, intx, enx, ciu_ip, (unsigned long long)ciu_intr_reg_addr,
	    (unsigned long long)ciu_intr_bits, mips_rd_status());
}
#endif

/*
 * ciu_enable_interrupts
 */
void ciu_enable_interrupts(int core_num, int intx, int enx,
    uint64_t set_these_interrupt_bits, int ciu_ip)
{
	uint32_t cpu_status_bits;
	uint64_t ciu_intr_reg_addr;
	uint64_t ciu_intr_bits;

        if (core_num == CIU_THIS_CORE)
            	core_num = octeon_get_core_num();

#ifdef DEBUG_CIU_EN
        printf(" CIU: core %u enabling Intx %u  Enx %u IP %d  Bits: 0x%llX\n",
	    core_num, intx, enx, ciu_ip, set_these_interrupt_bits);
#endif

	cpu_status_bits = disableintr();

#ifndef OCTEON_SMP_1
	ciu_intr_reg_addr = ciu_get_intr_en_reg_addr(core_num, intx, enx);
#else
	ciu_intr_reg_addr = ciu_get_en_reg_addr_new(core_num, intx, enx, ciu_ip);
#endif

        if (!ciu_intr_reg_addr) {
		printf("Bad call to %s\n", __func__);
		while(1);
		return;	/* XXX */
        }

	ciu_intr_bits =  oct_read64(ciu_intr_reg_addr);

#ifdef DEBUG_CIU_EN
        printf(" CIU: status: 0x%X  reg_addr: 0x%llX   Val: 0x%llX   ->  0x%llX",
	    cpu_status_bits, ciu_intr_reg_addr, ciu_intr_bits, ciu_intr_bits | set_these_interrupt_bits);
#endif
	ciu_intr_bits |=  set_these_interrupt_bits;
	oct_write64(ciu_intr_reg_addr, ciu_intr_bits);
#ifdef SMP
	mips_wbflush();
#endif
	oct_read64(OCTEON_MIO_BOOT_BIST_STAT);	/* Bus Barrier */

#ifdef DEBUG_CIU_EN
        printf(" Readback: 0x%llX\n\n   ",
	    (uint64_t)oct_read64(ciu_intr_reg_addr));
#endif

	setsr(cpu_status_bits);
}

unsigned long
octeon_get_clock_rate(void)
{
	return octeon_cpu_clock;
}

static void
octeon_memory_init(void)
{
	uint64_t phys_avail[10 + 2];
	uint64_t startpfn, endpfn;
	uint32_t realmem;
	extern char end[];
	int i;
	startpfn = atop(CKSEG0_TO_PHYS((vaddr_t)&end) + PAGE_SIZE);
	endpfn = atop(96 << 20);
	mem_layout[0].mem_first_page = startpfn;
	mem_layout[0].mem_last_page = endpfn;
	mem_layout[0].mem_freelist = VM_FREELIST_DEFAULT;

	physmem = endpfn - startpfn;
	uint32_t realmem_bytes;

	if (octeon_board_real()) {
		realmem_bytes = (octeon_dram - PAGE_SIZE);
		realmem_bytes &= ~(PAGE_SIZE - 1);
	} else {
		/* Simulator we limit to 96 meg */
		realmem_bytes = (96 << 20);
	}
	/* phys_avail regions are in bytes */
	memset((void*)phys_avail,0,sizeof(phys_avail));
	phys_avail[0] = (MIPS_KSEG0_TO_PHYS((vm_offset_t)&end) + 
			 PAGE_SIZE ) & ~(PAGE_SIZE - 1);
	if (octeon_board_real()) {
		if (realmem_bytes > OCTEON_DRAM_FIRST_256_END)
			phys_avail[1] = OCTEON_DRAM_FIRST_256_END;
		else
			phys_avail[1] = realmem_bytes;
		realmem_bytes -= OCTEON_DRAM_FIRST_256_END;
		realmem_bytes &= ~(PAGE_SIZE - 1);
		mem_layout[0].mem_last_page = atop(phys_avail[1]);
	} else {
		/* Simulator gets 96Meg period. */
		phys_avail[1] = (96 << 20);
	}
	/*-
	 * Octeon Memory looks as follows:
         *   PA
	 * First 256 MB DR0
	 * 0000 0000 0000 0000     to  0000 0000 0000 0000
	 * 0000 0000 0FFF FFFF     to  0000 0000 0FFF FFFF
	 * Second 256 MB DR1 
	 * 0000 0004 1000 0000     to  0000 0004 1000 0000
	 * 0000 0004 1FFF FFFF     to  0000 0004 1FFF FFFF
	 * Over 512MB Memory DR2  15.5GB
	 * 0000 0000 2000 0000     to  0000 0000 2000 0000
	 * 0000 0003 FFFF FFFF     to  0000 0003 FFFF FFFF
	 *
	 */
	physmem = btoc(phys_avail[1] - phys_avail[0]);

	if (octeon_board_real()){
		if(realmem_bytes > OCTEON_DRAM_FIRST_256_END){
			/* take out the upper non-cached 1/2 */
			phys_avail[2] = 0x410000000ULL;
			phys_avail[3] = (0x410000000ULL 
					 + OCTEON_DRAM_FIRST_256_END);
			physmem += btoc(phys_avail[3] - phys_avail[2]);
			mem_layout[1].mem_first_page = atop(phys_avail[2]);
			mem_layout[1].mem_last_page = atop(phys_avail[3]-1);
			mem_layout[1].mem_freelist = VM_FREELIST_DEFAULT;
			realmem_bytes -= OCTEON_DRAM_FIRST_256_END;
			/* Now map the rest of the memory */
			phys_avail[4] = 0x20000000ULL;
			phys_avail[5] = (0x20000000ULL + realmem_bytes);
			physmem += btoc(phys_avail[5] - phys_avail[4]);
			mem_layout[2].mem_first_page = atop(phys_avail[4]);
			mem_layout[2].mem_last_page = atop(phys_avail[5]-1);
			mem_layout[2].mem_freelist = VM_FREELIST_DEFAULT;
			realmem_bytes=0;
		}else{
			/* Now map the rest of the memory */
			phys_avail[2] = 0x410000000ULL;
			phys_avail[3] = (0x410000000ULL + realmem_bytes);
			physmem += btoc(phys_avail[3] - phys_avail[2]);
			mem_layout[1].mem_first_page = atop(phys_avail[2]);
			mem_layout[1].mem_last_page = atop(phys_avail[3]-1);
			mem_layout[1].mem_freelist = VM_FREELIST_DEFAULT;
			realmem_bytes=0;
		}
 	}
 	realmem = physmem;
	printf("Total DRAM Size 0x%016X\n", (uint32_t) octeon_dram);
	for(i=0;phys_avail[i];i+=2){
		printf("Bank %d = 0x%016lX   ->  0x%016lX\n",i>>1, 
		       (long)phys_avail[i], (long)phys_avail[i+1]);
	}
	for( i=0;mem_layout[i].mem_last_page;i++){
		printf("mem_layout[%d] page 0x%016lX -> 0x%016lX\n",i,
		       mem_layout[i].mem_first_page,
		       mem_layout[i].mem_last_page);
	}
}

/*
 * Register a memory region.
 */
int
memrange_register(uint64_t startpfn, uint64_t endpfn, uint64_t bmask,
    unsigned int freelist)
{
	struct phys_mem_desc *cur, *m = NULL;
	int i;

#ifdef DEBUG
	printf("%s: memory from %p(%x) to %p(%x)\n",
		__func__, ptoa(startpfn), startpfn, ptoa(endpfn), endpfn);
#endif
	physmem += endpfn - startpfn;

	/*
	 * Prevent use of memory above 16GB physical, until pmap can support
	 * this.
	 */
	if (startpfn >= atop(16UL * 1024 * 1024 * 1024))
		return 0;
	if (endpfn >= atop(16UL * 1024 * 1024 * 1024))
		endpfn = atop(16UL * 1024 * 1024 * 1024);
	
	for (i = 0, cur = mem_layout; i < MAXMEMSEGS; i++, cur++) {
		if (cur->mem_last_page == 0) {
			if (m == NULL)
				m = cur;	/* first free segment */
			continue;
		}
		/* merge contiguous areas if on the same freelist */
		if (cur->mem_freelist == freelist) {
			if (cur->mem_first_page == endpfn &&
			    ((cur->mem_last_page ^ startpfn) & bmask) == 0) {
				cur->mem_first_page = startpfn;
				return 0;
			}
			if (cur->mem_last_page == startpfn &&
			    ((cur->mem_first_page ^ endpfn) & bmask) == 0) {
				cur->mem_last_page = endpfn;
				return 0;
			}
		}
	}

	if (m == NULL)
		return ENOMEM;
	
	m->mem_first_page = startpfn;
	m->mem_last_page = endpfn;
	m->mem_freelist = freelist;
	return 0;
}

uint64_t get_cvmctl(void);
uint64_t get_cvmmemctl(void);

/*
 * Do all the stuff that locore normally does before calling main().
 * Reset mapping and set up mapping to hardware and init "wired" reg.
 */
vaddr_t
mips_init(__register_t a0, __register_t a1, __register_t a2 __unused,
	__register_t a3)
{
	uint prid;
	u_long cpuspeed;
	vaddr_t xtlb_handler;
	int i;

	extern char start[], edata[], end[];
	extern char exception[], e_exception[];
	extern void xtlb_miss;

#ifdef MULTIPROCESSOR
	/*
	 * Set curcpu address on primary processor.
	 */
	setcurcpu(&cpu_info_primary);
#endif
	/*
	 * Clear the compiled BSS segment in OpenBSD code.
	 */

	bzero(edata, end - edata);

	/*
	 * Set up early console output.
	 */
	cn_tab = &octcons;

	/*
	 * Reserve space for the symbol table, if it exists.
	 */
	ssym = (char *)(vaddr_t)*(int32_t *)end;
	if (((long)ssym - (long)end) >= 0 &&
	    ((long)ssym - (long)end) <= 0x1000 &&
	    ssym[0] == ELFMAG0 && ssym[1] == ELFMAG1 &&
	    ssym[2] == ELFMAG2 && ssym[3] == ELFMAG3) {
		/* Pointers exist directly after kernel. */
		esym = (char *)(vaddr_t)*((int32_t *)end + 1);
		ekern = esym;
	} else {
		/* Pointers aren't setup either... */
		ssym = NULL;
		esym = NULL;
		ekern = end;
	}

	octeon_ciu_reset();
	octeon_boot_params_init(a3);

	prid = cp0_get_prid();
	/*
	 * Figure out processor clock speed.
	 * Hopefully the processor speed, in Hertz, will not overflow
	 * uint32_t...
	 */

	cpuspeed = octeon_get_clock_rate();
	bootcpu_hwinfo.clock = cpuspeed;

	/*
	 * Look at arguments passed to us and compute boothowto.
	 */
	boothowto = RB_AUTOBOOT;

	uncached_base = PHYS_TO_XKPHYS(0, CCA_NC);

	octeon_memory_init();

	/*
	 * Set pagesize to enable use of page macros and functions.
	 * Commit available memory to UVM system.
	 */

	uvmexp.pagesize = PAGE_SIZE;
	uvm_setpagesize();

	for (i = 0; i < MAXMEMSEGS && mem_layout[i].mem_last_page != 0; i++) {
		uint64_t fp, lp;
		uint64_t firstkernpage, lastkernpage;
		unsigned int freelist;
		paddr_t firstkernpa, lastkernpa;

		/* kernel is linked in CKSEG0 */
		firstkernpa = CKSEG0_TO_PHYS((vaddr_t)start);
		lastkernpa = CKSEG0_TO_PHYS((vaddr_t)ekern);

		firstkernpage = atop(trunc_page(firstkernpa));
		lastkernpage = atop(round_page(lastkernpa));

		fp = mem_layout[i].mem_first_page;
		lp = mem_layout[i].mem_last_page;
		freelist = mem_layout[i].mem_freelist;

		/* Account for kernel and kernel symbol table. */
		if (fp >= firstkernpage && lp < lastkernpage)
			continue;	/* In kernel. */

		if (lp < firstkernpage || fp > lastkernpage) {
			uvm_page_physload(fp, lp, fp, lp, freelist);
			continue;	/* Outside kernel. */
		}

		if (fp >= firstkernpage)
			fp = lastkernpage;
		else if (lp < lastkernpage)
			lp = firstkernpage;
		else { /* Need to split! */
			uint64_t xp = firstkernpage;
			uvm_page_physload(fp, xp, fp, xp, freelist);
			fp = lastkernpage;
		}
		if (lp > fp) {
			uvm_page_physload(fp, lp, fp, lp, freelist);
		}
	}

	bootcpu_hwinfo.c0prid = prid;
	bootcpu_hwinfo.type = (prid >> 8) & 0xff;
	/* FPU reports itself as type 5, version 0.1... */
	bootcpu_hwinfo.c1prid = bootcpu_hwinfo.c0prid;
	bootcpu_hwinfo.tlbsize = 64;
	bcopy(&bootcpu_hwinfo, &curcpu()->ci_hw, sizeof(struct cpu_hwinfo));

	/*
	 * Configure cache.
	 */

	Octeon_ConfigCache(curcpu());
	printf("l1icache size:%d line:%d set:%d\n",
		curcpu()->ci_l1instcachesize,
		curcpu()->ci_l1instcacheline,
		curcpu()->ci_l1instcacheset);
	printf("l1dcache size:%d line:%d set:%d\n",
		curcpu()->ci_l1datacachesize,
		curcpu()->ci_l1datacacheline,
		curcpu()->ci_l1datacacheset);
	printf("l2size:%d l3size:%d\n",
		curcpu()->ci_l2size,
		curcpu()->ci_l3size);
	Octeon_SyncCache(curcpu());

	tlb_set_page_mask(TLB_PAGE_MASK);
	tlb_set_wired(0);
	tlb_flush(bootcpu_hwinfo.tlbsize);
	tlb_set_wired(UPAGES / 2);

	/*
	 * Get a console, very early but after initial mapping setup.
	 */

//	consinit();
	printf("Initial setup done, switching console.\n");

	/*
	 * Init message buffer.
	 */
	msgbufbase = (caddr_t)pmap_steal_memory(MSGBUFSIZE, NULL,NULL);
	initmsgbuf(msgbufbase, MSGBUFSIZE);

	/*
	 * Allocate U page(s) for proc[0], pm_tlbpid 1.
	 */

	proc0.p_addr = proc0paddr = curcpu()->ci_curprocpaddr =
	    (struct user *)pmap_steal_memory(USPACE, NULL, NULL);
	proc0.p_md.md_regs = (struct trap_frame *)&proc0paddr->u_pcb.pcb_regs;
	tlb_set_pid(1);

	/*
	 * Bootstrap VM system.
	 */

	pmap_bootstrap();

	/*
	 * Copy down exception vector code.
	 */

	bcopy(exception, (char *)CACHE_ERR_EXC_VEC, e_exception - exception);
	bcopy(exception, (char *)GEN_EXC_VEC, e_exception - exception);

	/*
	 * Build proper TLB refill handler trampolines.
	 */

	xtlb_handler = (vaddr_t)&xtlb_miss;
	build_trampoline(TLB_MISS_EXC_VEC, xtlb_handler);
	build_trampoline(XTLB_MISS_EXC_VEC, xtlb_handler);

	/*
	 * Turn off bootstrap exception vectors.
	 * (this is done by PMON already, but it doesn't hurt to be safe)
	 */

	setsr(getsr() & ~SR_BOOT_EXC_VEC);
	proc0.p_md.md_regs->sr = getsr();

#ifdef DDB
	db_machine_init();
	if (boothowto & RB_KDB)
		Debugger();
#endif

	/*
	 * Return the new kernel stack pointer.
	 */
	return ((vaddr_t)proc0paddr + USPACE - 64);
}

/*
 * Build a tlb trampoline
 */
void
build_trampoline(vaddr_t addr, vaddr_t dest)
{
	const uint32_t insns[] = {
		0x3c1a0000,	/* lui k0, imm16 */
		0x675a0000,	/* daddiu k0, k0, imm16 */
		0x001ad438,	/* dsll k0, k0, 0x10 */
		0x675a0000,	/* daddiu k0, k0, imm16 */
		0x001ad438,	/* dsll k0, k0, 0x10 */
		0x675a0000,	/* daddiu k0, k0, imm16 */
		0x03400008,	/* jr k0 */
		0x00000000	/* nop */
	};
	uint32_t *dst = (uint32_t *)addr;
	const uint32_t *src = insns;
	uint32_t a, b, c, d;

	/*
	 * Decompose the handler address in the four components which,
	 * added with sign extension, will produce the correct address.
	 */
	d = dest & 0xffff;
	dest >>= 16;
	if (d & 0x8000)
		dest++;
	c = dest & 0xffff;
	dest >>= 16;
	if (c & 0x8000)
		dest++;
	b = dest & 0xffff;
	dest >>= 16;
	if (b & 0x8000)
		dest++;
	a = dest & 0xffff;

	/*
	 * Build the trampoline, skipping noop computations.
	 */
	*dst++ = *src++ | a;
	if (b != 0)
		*dst++ = *src++ | b;
	else
		src++;
	*dst++ = *src++;
	if (c != 0)
		*dst++ = *src++ | c;
	else
		src++;
	*dst++ = *src++;
	if (d != 0)
		*dst++ = *src++ | d;
	else
		src++;
	*dst++ = *src++;
	*dst++ = *src++;

	/*
	 * Note that we keep the delay slot instruction a nop, instead
	 * of branching to the second instruction of the handler and
	 * having its first instruction in the delay slot, so that the
	 * tlb handler is free to use k0 immediately.
	 */
}

/*
 * Console initialization: called early on from main, before vm init or startup.
 * Do enough configuration to choose and initialize a console.
 */
void
consinit()
{
	static int console_ok = 0;

	if (console_ok == 0) {
		cninit();
		console_ok = 1;
	}
}

/*
 * cpu_startup: allocate memory for variable-sized tables, initialize CPU, and 
 * do auto-configuration.
 */
void
cpu_startup()
{
	vaddr_t minaddr, maxaddr;

	/*
	 * Good {morning,afternoon,evening,night}.
	 */
	printf(version);
	printf("real mem = %u (%uMB)\n", ptoa((psize_t)physmem),
	    ptoa((psize_t)physmem)/1024/1024);

	/*
	 * Allocate a submap for exec arguments. This map effectively
	 * limits the number of processes exec'ing at any time.
	 */
	minaddr = vm_map_min(kernel_map);
	exec_map = uvm_km_suballoc(kernel_map, &minaddr, &maxaddr,
	    16 * NCARGS, VM_MAP_PAGEABLE, FALSE, NULL);
	/* Allocate a submap for physio. */
	phys_map = uvm_km_suballoc(kernel_map, &minaddr, &maxaddr,
	    VM_PHYS_SIZE, 0, FALSE, NULL);

	printf("avail mem = %u (%uMB)\n", ptoa(uvmexp.free),
	    ptoa(uvmexp.free)/1024/1024);

	/*
	 * Set up buffers, so they can be used to read disk labels.
	 */
	bufinit();

	/*
	 * Configure the system.
	 */
	if (boothowto & RB_CONFIG) {
#ifdef BOOT_CONFIG
		user_config();
#else
		printf("kernel does not support -c; continuing..\n");
#endif
	}
}

/*
 * Machine dependent system variables.
 */
int
cpu_sysctl(name, namelen, oldp, oldlenp, newp, newlen, p)
	int *name;
	u_int namelen;
	void *oldp;
	size_t *oldlenp;
	void *newp;
	size_t newlen;
	struct proc *p;
{
	/* All sysctl names at this level are terminal. */
	if (namelen != 1)
		return ENOTDIR;		/* Overloaded */

	switch (name[0]) {
	case CPU_KBDRESET:
		if (securelevel > 0)
			return (sysctl_rdint(oldp, oldlenp, newp, kbd_reset));
		return (sysctl_int(oldp, oldlenp, newp, newlen, &kbd_reset));
	default:
		return EOPNOTSUPP;
	}
}

/*
 * Set registers on exec for native exec format. For o64/64.
 */
void
setregs(p, pack, stack, retval)
	struct proc *p;
	struct exec_package *pack;
	u_long stack;
	register_t *retval;
{
	struct cpu_info *ci = curcpu();

	bzero((caddr_t)p->p_md.md_regs, sizeof(struct trap_frame));
	p->p_md.md_regs->sp = stack;
	p->p_md.md_regs->pc = pack->ep_entry & ~3;
	p->p_md.md_regs->t9 = pack->ep_entry & ~3; /* abicall req */
	p->p_md.md_regs->sr = SR_FR_32 | SR_XX | SR_KSU_USER | SR_KX | SR_UX |
	    SR_EXL | SR_INT_ENAB;
	p->p_md.md_regs->sr |= idle_mask & SR_INT_MASK;
	p->p_md.md_regs->ic = (idle_mask << 8) & IC_INT_MASK;
	p->p_md.md_flags &= ~MDP_FPUSED;
	if (ci->ci_fpuproc == p)
		ci->ci_fpuproc = NULL;
	p->p_md.md_ss_addr = 0;
	p->p_md.md_pc_ctrl = 0;
	p->p_md.md_watch_1 = 0;
	p->p_md.md_watch_2 = 0;

	retval[1] = 0;
}


int	waittime = -1;

void
boot(int howto)
{

	/* Take a snapshot before clobbering any registers. */
	if (curproc)
		savectx(curproc->p_addr, 0);

	if (cold) {
		/*
		 * If the system is cold, just halt, unless the user
		 * explicitly asked for reboot.
		 */
		if ((howto & RB_USERREQ) == 0)
			howto |= RB_HALT;
		goto haltsys;
	}

	boothowto = howto;
	if ((howto & RB_NOSYNC) == 0 && waittime < 0) {
		extern struct proc proc0;
		/* fill curproc with live object */
		if (curproc == NULL)
			curproc = &proc0;
		/*
		 * Synchronize the disks...
		 */
		waittime = 0;
		vfs_shutdown();

		/*
		 * If we've been adjusting the clock, the todr will be out of
		 * sync; adjust it now.
		 */
		if ((howto & RB_TIMEBAD) == 0) {
			resettodr();
		} else {
			printf("WARNING: not updating battery clock\n");
		}
	}

	uvm_shutdown();
	(void) splhigh();		/* Extreme priority. */

	if (howto & RB_DUMP)
		dumpsys();

haltsys:
	doshutdownhooks();

	if (howto & RB_HALT) {
		if (howto & RB_POWERDOWN) {
			if (sys_platform->powerdown != NULL) {
				printf("System Power Down.\n");
				(*(sys_platform->powerdown))();
			} else {
				printf("System Power Down not supported,"
				    " halting system.\n");
			}
		} else
			printf("System Halt.\n");
	} else {
		void (*__reset)(void) = (void (*)(void))RESET_EXC_VEC;
		printf("System restart.\n");
		if (sys_platform->reset != NULL)
			(*(sys_platform->reset))();
		(void)disableintr();
		tlb_set_wired(0);
		tlb_flush(bootcpu_hwinfo.tlbsize);
		__reset();
	}

	for (;;) ;
	/*NOTREACHED*/
}

u_long	dumpmag = 0x8fca0101;	/* Magic number for savecore. */
int	dumpsize = 0;			/* Also for savecore. */
long	dumplo = 0;

void
dumpconf(void)
{
	int nblks;

	if (dumpdev == NODEV ||
	    (nblks = (bdevsw[major(dumpdev)].d_psize)(dumpdev)) == 0)
		return;
	if (nblks <= ctod(1))
		return;

	dumpsize = ptoa(physmem);
	if (dumpsize > atop(round_page(dbtob(nblks - dumplo))))
		dumpsize = atop(round_page(dbtob(nblks - dumplo)));
	else if (dumplo == 0)
		dumplo = nblks - btodb(ptoa(physmem));

	/*
	 * Don't dump on the first page in case the dump device includes a 
	 * disk label.
	 */
	if (dumplo < btodb(PAGE_SIZE))
		dumplo = btodb(PAGE_SIZE);
}

void
dumpsys()
{
	/* XXX TBD */
}

#if 0
/*
 * Convert an ASCII string into an integer.
 */
static u_long
atoi(const char *s, uint b)
{
	int c;
	uint base = b, d;
	int neg = 0;
	u_long val = 0;

	if (s == NULL || *s == '\0')
		return 0;

	/* Skip spaces if any. */
	do {
		c = *s++;
	} while (c == ' ' || c == '\t');

	/* Parse sign, allow more than one (compat). */
	while (c == '-') {
		neg = !neg;
		c = *s++;
	}

	/* Parse base specification, if any. */
	if (base == 0 && c == '0') {
		c = *s++;
		switch (c) {
		case 'X':
		case 'x':
			base = 16;
			c = *s++;
			break;
		case 'B':
		case 'b':
			base = 2;
			c = *s++;
			break;
		default:
			base = 8;
			break;
		}
	}

	/* Parse number proper. */
	for (;;) {
		if (c >= '0' && c <= '9')
			d = c - '0';
		else if (c >= 'a' && c <= 'z')
			d = c - 'a' + 10;
		else if (c >= 'A' && c <= 'Z')
			d = c - 'A' + 10;
		else
			break;
		if (d >= base)
			break;
		val *= base;
		val += d;
		c = *s++;
	}

	return neg ? -val : val;
}
#endif

#include <dev/ic/comreg.h>
/*
 * Early console through pmon routines.
 */
#define  OCTEON_MIO_UART0               0x8001180000000800ull
#define  OCTEON_MIO_UART0_THR           0x8001180000000840ull
#define  OCTEON_MIO_UART0_LSR           0x8001180000000828ull
#define  OCTEON_MIO_UART0_RBR           0x8001180000000800ull
#define  OCTEON_MIO_UART0_USR           0x8001180000000938ull
#define  OCTEON_MIO_UART0_IER		0x8001180000000808ull
#define  OCTEON_MIO_UART0_FCR		0x8001180000000850ull
#define  OCTEON_MIO_UART0_MCR		0x8001180000000820ull
#define  OCTEON_MIO_UART0_LCR           0x8001180000000818ull
#define  OCTEON_MIO_UART0_DLL           0x8001180000000880ull
#define  OCTEON_MIO_UART0_DLH           0x8001180000000888ull
#define  OCTEON_MIO_ADDR_HI24           0x800118
#define  OCTEON_MIO_UART_SIZE           0x400ull
#define  USR_TXFIFO_NOTFULL		2

static int delay_changed = 1;
static int oct16550_delay (void);
static int oct16550_delay (void)
{
	int divisor;
	u_char lcr;
        static int d = 0;

        if (!delay_changed) return d;
        delay_changed = 0;
	lcr = (u_char)*(uint64_t*)OCTEON_MIO_UART0_LCR;
	*(uint64_t*)OCTEON_MIO_UART0_LCR = lcr | LCR_DLAB;
	divisor = (int)(*(uint64_t*)OCTEON_MIO_UART0_DLL | *(uint64_t*)OCTEON_MIO_UART0_DLH << 8);
	*(uint64_t*)OCTEON_MIO_UART0_LCR = lcr;
	
	return 10; /* return an approx delay value */
}

static inline void oct16550_wait_txhr_empty (int d)
{
	while (((*(uint64_t*)OCTEON_MIO_UART0_LSR & LSR_TXRDY) == 0) &&
        	((*(uint64_t*)OCTEON_MIO_UART0_USR & USR_TXFIFO_NOTFULL) == 0))
		delay(d);
}

void
octcninit(struct consdev *consdev)
{
}

void
octcnprobe(struct consdev *consdev)
{
}

void
octcnpollc(dev_t dev, int c)
{
}

void
octcnputc (dev_t dev, int c)
{
	int d;

	/* 1/10th the time to transmit 1 character (estimate). */
	d = oct16550_delay();
        oct16550_wait_txhr_empty(d);
	*(uint64_t*)OCTEON_MIO_UART0_RBR = (uint8_t)c;
        oct16550_wait_txhr_empty(d);
}

int
octcngetc (dev_t dev)
{
	int c, d;

	/* 1/10th the time to transmit 1 character (estimate). */
	d = oct16550_delay();

	while ((*(uint64_t*)OCTEON_MIO_UART0_LSR & LSR_RXRDY) == 0)
		delay(d);

	c = (uint8_t)*(uint64_t*)OCTEON_MIO_UART0_RBR;

	return (c);
}

/* impSTART: This stuff should move back into the Cavium SDK */
/*
 ****************************************************************************************
 *
 * APP/BOOT  DESCRIPTOR  STUFF
 *
 ****************************************************************************************
 */

/* Define the struct that is initialized by the bootloader used by the 
 * startup code.
 *
 * Copyright (c) 2004, 2005, 2006 Cavium Networks.
 *
 * The authors hereby grant permission to use, copy, modify, distribute,
 * and license this software and its documentation for any purpose, provided
 * that existing copyright notices are retained in all copies and that this
 * notice is included verbatim in any distributions. No written agreement,
 * license, or royalty fee is required for any of the authorized uses.
 * Modifications to this software may be copyrighted by their authors
 * and need not follow the licensing terms described here, provided that
 * the new terms are clearly indicated on the first page of each file where
 * they apply.
 */

#define OCTEON_CURRENT_DESC_VERSION     6
#define OCTEON_ARGV_MAX_ARGS            (64)
#define OCTOEN_SERIAL_LEN 20


typedef struct {
	/* Start of block referenced by assembly code - do not change! */
	uint32_t desc_version;
	uint32_t desc_size;

	uint64_t stack_top;
	uint64_t heap_base;
	uint64_t heap_end;
	uint64_t entry_point;   /* Only used by bootloader */
	uint64_t desc_vaddr;
	/* End of This block referenced by assembly code - do not change! */

	uint32_t exception_base_addr;
	uint32_t stack_size;
	uint32_t heap_size;
	uint32_t argc;  /* Argc count for application */
	uint32_t argv[OCTEON_ARGV_MAX_ARGS];
	uint32_t flags;
	uint32_t core_mask;
	uint32_t dram_size;  /**< DRAM size in megabyes */
	uint32_t phy_mem_desc_addr;  /**< physical address of free memory descriptor block*/
	uint32_t debugger_flags_base_addr;  /**< used to pass flags from app to debugger */
	uint32_t eclock_hz;  /**< CPU clock speed, in hz */
	uint32_t dclock_hz;  /**< DRAM clock speed, in hz */
	uint32_t spi_clock_hz;  /**< SPI4 clock in hz */
	uint16_t board_type;
	uint8_t board_rev_major;
	uint8_t board_rev_minor;
	uint16_t chip_type;
	uint8_t chip_rev_major;
	uint8_t chip_rev_minor;
	char board_serial_number[OCTOEN_SERIAL_LEN];
	uint8_t mac_addr_base[6];
	uint8_t mac_addr_count;
	uint64_t cvmx_desc_vaddr;
} octeon_boot_descriptor_t;


typedef struct {
	uint32_t major_version;
	uint32_t minor_version;

	uint64_t stack_top;
	uint64_t heap_base;
	uint64_t heap_end;
	uint64_t desc_vaddr;

	uint32_t exception_base_addr;
	uint32_t stack_size;
	uint32_t flags;
	uint32_t core_mask;
	uint32_t dram_size;  /**< DRAM size in megabyes */
	uint32_t phy_mem_desc_addr;  /**< physical address of free memory descriptor block*/
	uint32_t debugger_flags_base_addr;  /**< used to pass flags from app to debugger */
	uint32_t eclock_hz;  /**< CPU clock speed, in hz */
	uint32_t dclock_hz;  /**< DRAM clock speed, in hz */
	uint32_t spi_clock_hz;  /**< SPI4 clock in hz */
	uint16_t board_type;
	uint8_t board_rev_major;
	uint8_t board_rev_minor;
	uint16_t chip_type;
	uint8_t chip_rev_major;
	uint8_t chip_rev_minor;
	char board_serial_number[OCTOEN_SERIAL_LEN];
	uint8_t mac_addr_base[6];
	uint8_t mac_addr_count;
} cvmx_bootinfo_t;

uint32_t octeon_cpu_clock;
uint64_t octeon_dram;
uint32_t octeon_bd_ver = 0, octeon_cvmx_bd_ver = 0, octeon_board_rev_major, octeon_board_rev_minor, octeon_board_type;
uint8_t octeon_mac_addr[6] = { 0 };
int octeon_core_mask, octeon_mac_addr_count;
int octeon_chip_rev_major = 0, octeon_chip_rev_minor = 0, octeon_chip_type = 0;

static octeon_boot_descriptor_t *app_desc_ptr;
static cvmx_bootinfo_t *cvmx_desc_ptr;

#define OCTEON_BOARD_TYPE_NONE 			0
#define OCTEON_BOARD_TYPE_SIM  			1
#define	OCTEON_BOARD_TYPE_CN3010_EVB_HS5	11

#define OCTEON_CLOCK_MIN     (100 * 1000 * 1000)
#define OCTEON_CLOCK_MAX     (800 * 1000 * 1000)
#define OCTEON_DRAM_DEFAULT  (256 * 1024 * 1024)
#define OCTEON_DRAM_MIN	     30
#define OCTEON_DRAM_MAX	     3000


int
octeon_board_real(void)
{
	switch (octeon_board_type) {
	case OCTEON_BOARD_TYPE_NONE:
	case OCTEON_BOARD_TYPE_SIM:
		return 0;
	case OCTEON_BOARD_TYPE_CN3010_EVB_HS5:
		/*
		 * XXX
		 * The CAM-0100 identifies itself as type 11, revision 0.0,
		 * despite its being rather real.  Disable the revision check
		 * for type 11.
		 */
		return 1;
	default:
		if (octeon_board_rev_major == 0)
			return 0;
		return 1;
	}
}

static void
octeon_process_app_desc_ver_unknown(void)
{
    	printf(" Unknown Boot-Descriptor: Using Defaults\n");

    	octeon_cpu_clock = OCTEON_CLOCK_DEFAULT;
        octeon_dram = OCTEON_DRAM_DEFAULT;
        octeon_board_rev_major = octeon_board_rev_minor = octeon_board_type = 0;
        octeon_core_mask = 1;
        octeon_chip_type = octeon_chip_rev_major = octeon_chip_rev_minor = 0;
        octeon_mac_addr[0] = 0x00; octeon_mac_addr[1] = 0x0f;
        octeon_mac_addr[2] = 0xb7; octeon_mac_addr[3] = 0x10;
        octeon_mac_addr[4] = 0x09; octeon_mac_addr[5] = 0x06;
        octeon_mac_addr_count = 1;
}

static int
octeon_process_app_desc_ver_6(void)
{
	/* XXX Why is 0x00000000ffffffffULL a bad value?  */
	if (app_desc_ptr->cvmx_desc_vaddr == 0 ||
	    app_desc_ptr->cvmx_desc_vaddr == 0xfffffffful) {
            	printf ("Bad cvmx_desc_ptr %p\n", cvmx_desc_ptr);
                return 1;
	}
    	cvmx_desc_ptr =
	    (cvmx_bootinfo_t *)(__intptr_t)app_desc_ptr->cvmx_desc_vaddr;
        cvmx_desc_ptr =
	    (cvmx_bootinfo_t *) ((__intptr_t)cvmx_desc_ptr | CKSEG0_BASE);
        octeon_cvmx_bd_ver = (cvmx_desc_ptr->major_version * 100) +
	    cvmx_desc_ptr->minor_version;
        if (cvmx_desc_ptr->major_version != 1) {
            	panic("Incompatible CVMX descriptor from bootloader: %d.%d %p\n",
                       (int) cvmx_desc_ptr->major_version,
                       (int) cvmx_desc_ptr->minor_version, cvmx_desc_ptr);
        }

        octeon_core_mask = cvmx_desc_ptr->core_mask;
        octeon_cpu_clock  = cvmx_desc_ptr->eclock_hz;
        octeon_board_type = cvmx_desc_ptr->board_type;
        octeon_board_rev_major = cvmx_desc_ptr->board_rev_major;
        octeon_board_rev_minor = cvmx_desc_ptr->board_rev_minor;
        octeon_chip_type = cvmx_desc_ptr->chip_type;
        octeon_chip_rev_major = cvmx_desc_ptr->chip_rev_major;
        octeon_chip_rev_minor = cvmx_desc_ptr->chip_rev_minor;
        octeon_mac_addr[0] = cvmx_desc_ptr->mac_addr_base[0];
        octeon_mac_addr[1] = cvmx_desc_ptr->mac_addr_base[1];
        octeon_mac_addr[2] = cvmx_desc_ptr->mac_addr_base[2];
        octeon_mac_addr[3] = cvmx_desc_ptr->mac_addr_base[3];
        octeon_mac_addr[4] = cvmx_desc_ptr->mac_addr_base[4];
        octeon_mac_addr[5] = cvmx_desc_ptr->mac_addr_base[5];
        octeon_mac_addr_count = cvmx_desc_ptr->mac_addr_count;

        if (app_desc_ptr->dram_size > 16*1024*1024)
            	octeon_dram = (uint64_t)app_desc_ptr->dram_size;
	else
            	octeon_dram = (uint64_t)app_desc_ptr->dram_size << 20;
        return 0;
}

static void
octeon_boot_params_init(register_t ptr)
{
	int bad_desc = 1;
	
    	if (ptr != 0 && ptr < MAX_APP_DESC_ADDR) {
	        app_desc_ptr = (octeon_boot_descriptor_t *)(__intptr_t)ptr;
		octeon_bd_ver = app_desc_ptr->desc_version;
		if (app_desc_ptr->desc_version < 6)
			panic("Your boot code is too old to be supported.\n");
		if (app_desc_ptr->desc_version >= 6)
			bad_desc = octeon_process_app_desc_ver_6();
        }
        if (bad_desc)
        	octeon_process_app_desc_ver_unknown();

        printf("Boot Descriptor Ver: %u -> %u/%u",
               octeon_bd_ver, octeon_cvmx_bd_ver/100, octeon_cvmx_bd_ver%100);
        printf("  CPU clock: %uMHz  Core Mask: %#x\n", octeon_cpu_clock/1000000, octeon_core_mask);
        printf("  Dram: %u MB", (uint32_t)(octeon_dram >> 20));
        printf("  Board Type: %u  Revision: %u/%u\n",
               octeon_board_type, octeon_board_rev_major, octeon_board_rev_minor);
        printf("  Octeon Chip: %u  Rev %u/%u",
               octeon_chip_type, octeon_chip_rev_major, octeon_chip_rev_minor);

        printf("  Mac Address %02X.%02X.%02X.%02X.%02X.%02X (%d)\n",
	    octeon_mac_addr[0], octeon_mac_addr[1], octeon_mac_addr[2],
	    octeon_mac_addr[3], octeon_mac_addr[4], octeon_mac_addr[5],
	    octeon_mac_addr_count);
}
/* impEND: This stuff should move back into the Cavium SDK */

boolean_t
is_memory_range(paddr_t pa, psize_t len, psize_t limit)
{
	struct phys_mem_desc *seg;
	uint64_t fp, lp;
	int i;

	fp = atop(pa);
	lp = atop(round_page(pa + len));

	if (limit != 0 && lp > atop(limit))
		return FALSE;

	for (i = 0, seg = mem_layout; i < MAXMEMSEGS; i++, seg++)
		if (fp >= seg->mem_first_page && lp <= seg->mem_last_page)
			return TRUE;

	return FALSE;
}

#ifdef MULTIPROCESSOR
unsigned octeon_ap_boot = ~0;
struct cpu_info *cpu_info_boot_secondary = NULL;
char _kstack[USPACE];
void
hw_cpu_boot_secondary(struct cpu_info *ci)
{
	vaddr_t kstack;

	kstack = (vaddr_t)_kstack;
#if 0
	kstack = alloc_contiguous_pages(USPACE);
	if (kstack == NULL)
		panic("unable to allocate idle stack\n");
#endif
	ci->ci_curprocpaddr = (void *)kstack;

	cpu_info_boot_secondary = ci;

	printf("spinup ci:%p cpuid:%d\n", ci, ci->ci_cpuid);
	
	octeon_ap_boot = ci->ci_cpuid;

	while (!cpuset_isset(&cpus_running, ci))
		;
}

void
hw_cpu_hatch(struct cpu_info *ci)
{
	int s;
	
	/*
	 * Set curcpu address on this processor.
	 */
	setcurcpu(ci);

	/*
	 * Make sure we can access the extended address space.
	 * Note that r10k and later do not allow XUSEG accesses
	 * from kernel mode unless SR_UX is set.
	 */
	setsr(getsr() | SR_KX | SR_UX);

	tlb_set_page_mask(TLB_PAGE_MASK);
	tlb_set_wired(0);
	tlb_flush(64);
	tlb_set_wired(UPAGES / 2);

	tlb_set_pid(0);

	/*
	 * Turn off bootstrap exception vectors.
	 */
	setsr(getsr() & ~SR_BOOT_EXC_VEC);

	/*
	 * Clear out the I and D caches.
	 */
	Octeon_ConfigCache(ci);
	Mips_SyncCache(ci);

	cpu_startclock(ci);

	printf("%d %s:%s:%d\n", cpu_number(), __FILE__, __func__, __LINE__);
	
	ncpus++;
	cpuset_add(&cpus_running, ci);

	printf("%d %s:%s:%d\n", cpu_number(), __FILE__, __func__, __LINE__);
	mips64_ipi_init();
	obio_setintrmask(0);

	printf("%d %s:%s:%d\n", cpu_number(), __FILE__, __func__, __LINE__);
	spl0();
	(void)updateimask(0);

	printf("%d %s:%s:%d\n", cpu_number(), __FILE__, __func__, __LINE__);
	SCHED_LOCK(s);
	cpu_switchto(NULL, sched_chooseproc());
	printf("%d %s:%s:%d\n", cpu_number(), __FILE__, __func__, __LINE__);
}

int
hw_ipi_intr_establish(int (*func)(void *), u_long cpuid)
{
	printf("hw_ipi_intr_establish(%d)\n", cpuid);
	obio_intr_establish(CIU_INT_MBOX(cpuid), IPL_IPI, func,
		(void *)cpuid, NULL);
	return 0;
};

void
hw_ipi_intr_set(u_long cpuid)
{
	printf("hw_ipi_intr_set(%d)\n", cpuid);
	*(uint64_t *)OCTEON_CIU_MBOX_SETX(cpuid) = 1;
}

void
hw_ipi_intr_clear(u_long cpuid)
{
	printf("hw_ipi_intr_clear(%d)\n", cpuid);
	*(uint64_t *)OCTEON_CIU_MBOX_CLRX(cpuid) = 1;
}

#endif
