/***********************************************************************//**
	@file
	$Revision: 73 $
	$Author: yatsuhashi $
	$Date:: 2009-03-20 03:07:24 +0900#$
***************************************************************************/
#include <assert.h>
#include "libmahjong.h"

namespace mahjong {
/***********************************************************************//**
	コンストラクタ.
	@param	hais	牌の配列
	@param	isMenzen	面前のとき真
***************************************************************************/
Mentsu::Mentsu(const HaiArray& hais, bool isMenzen)
    : haiArray_(hais), 
      fu_(0), 
      machiFu_(0)
{
    haiArray_.sort();
    if(isMenzen) {
        flag_.set(FLAG_MENZEN);
    }
    switch(haiArray_.getSize()) {
    case 2:
        assert(haiArray_[0]->isSame(haiArray_[1]));
        type_ = TYPE_TOITSU;
        break;
    case 3:
        if(haiArray_[0]->isSame(haiArray_[1]) &&
           haiArray_[0]->isSame(haiArray_[2])) {
            type_ = TYPE_KOTSU;
            fu_ = 2;
        }
        else {
            assert(haiArray_[0]->succ()->isSame(haiArray_[1]) &&
                   haiArray_[1]->succ()->isSame(haiArray_[2]));
            type_ = TYPE_SHUNTSU;
        }
        break;
    case 4:
        assert(haiArray_[0]->isSame(haiArray_[1]) &&
               haiArray_[0]->isSame(haiArray_[2]) &&
               haiArray_[0]->isSame(haiArray_[3]));
        type_ = TYPE_KANTSU;
        fu_ = 8;
        break;
    default:
        assert(0);
        break;
    }
    if(fu_ > 0 && haiArray_[0]->isYaochu()) {
        fu_ *= 2;
    }
    clearAgariHai();
}
/***********************************************************************//**
	符を返す.
	@return	符
***************************************************************************/
int Mentsu::getFu() const {
    return (isAnko() ? fu_ * 2 : fu_ + machiFu_);
}
/***********************************************************************//**
	牌の数を返す
	@return	牌の数
***************************************************************************/
int Mentsu::getSize() const {
    return haiArray_.getSize();
}
/***********************************************************************//**
	面子を構成する牌を返す.
	@param	index	インデックス
	@return		牌
***************************************************************************/
const Hai* Mentsu::getHai(int index) const {
    return haiArray_[index];
}
/***********************************************************************//**
	牌を数える
***************************************************************************/
int Mentsu::countHai(const Hai* hai) const {
    return haiArray_.count(hai);
}
/***********************************************************************//**
	
***************************************************************************/
/** 面前のとき真を返す */
bool Mentsu::isMenzen() const {
    return flag_.test(FLAG_MENZEN);
}
/** 対子のとき真を返す */
bool Mentsu::isToitsu() const {
    return (type_ == TYPE_TOITSU);
}
/** 順子のとき真を返す */
bool Mentsu::isShuntsu() const {
    return (type_ == TYPE_SHUNTSU);
}
/** 刻子(カンツ)のとき真を返す */
bool Mentsu::isKotsu() const {
    return (type_ == TYPE_KOTSU || isKantsu());
}
/** カンツのとき真を返す */
bool Mentsu::isKantsu() const {
    return (type_ == TYPE_KANTSU);
}
/** 暗刻のとき真を返す */
bool Mentsu::isAnko() const {
    return (isMenzen() && isKotsu() && !flag_.test(FLAG_DENY_ANKO));
}
/** 平和として許される面子のとき真を返す */
bool Mentsu::isPinfu() const {
    return (getFu() == 0);
}
/***********************************************************************//**
	牌が面子に含まれているか調べる.
	@param	hai	調べる牌.
	@return		含まれているとき真.
***************************************************************************/
bool Mentsu::isInclude(const Hai* hai) const {
    return haiArray_.isInclude(hai);
}
/***********************************************************************//**
	同じタイプか調べる.
	@param	other	比較する面子
	@return		同じタイプのとき真
***************************************************************************/
bool Mentsu::isSameType(const Mentsu& other) const {
    return ((isToitsu() && other.isToitsu()) ||
            (isShuntsu() && other.isShuntsu()) ||
            (isKotsu() && other.isKotsu()));
}
/***********************************************************************//**
	和了牌をクリアする.
***************************************************************************/
void Mentsu::clearAgariHai() {
    flag_.reset(FLAG_DENY_ANKO);
    machiFu_ = 0;
}
/***********************************************************************//**
	和了牌をセットする.
	@param	hai	和了牌
	@param	isRon	ロン和了のとき真
***************************************************************************/
void Mentsu::setAgariHai(const Hai* hai, bool isRon) {
    clearAgariHai();
    if(isRon) {
        flag_.set(FLAG_DENY_ANKO);
    }
    if(isToitsu() ||
       (isShuntsu() &&
        !((hai->isSame(haiArray_[0]) && hai->number < 7) ||
          (hai->isSame(haiArray_[2]) && hai->number > 3)))) {
        machiFu_ = 2;
    }
}
/***********************************************************************//**
	文字列に変換する.
	@return	変換した文字列.
***************************************************************************/
std::string Mentsu::toString() const {
    std::string str;
    if(isMenzen()) {
        str.append("(");
        str.append(haiArray_.toString());
        str.append(")");
    }
    else {
        str.append("<");
        str.append(haiArray_.toString());
        str.append(">");
    }
    return str;
}
/***********************************************************************//**
	$Id: Mentsu.cpp 73 2009-03-19 18:07:24Z yatsuhashi $
***************************************************************************/
}	/* namespace mahjong */
