/***********************************************************************//**
	@file
	$Revision$
	$Author$
	$Date::                           $
***************************************************************************/
#include "common/Command.h"
#include "client/Client.h"
#include "MainFrame.h"
#include "Taku.h"
#include "Client.h"
#include "ClientThread.h"
#include "ConsoleFrame.h"
#include "Choice.h"
#include "Preference/Dialog.h"

namespace openmj {
namespace gui {
/***********************************************************************//**
	コンストラクタ.
***************************************************************************/
MainFrame::MainFrame()
    : super(0), 
      preferenceDialog(new preference::Dialog(this)), 
      consoleFrame(new ConsoleFrame(this)), 
      client(this), 
      choice_(0)
{
    sizer->RecalcSizes();
    sizer->Fit(this);
    SetMaxSize(GetSize());
    Connect(EVENT_RECV_COMMAND, 
            wxCommandEventHandler(MainFrame::onRecvCommand));
}
/***********************************************************************//**
	デストラクタ.
***************************************************************************/
MainFrame::~MainFrame() {
    if(choice_) {
        delete choice_;
    }
    Disconnect(EVENT_RECV_COMMAND, 
               wxCommandEventHandler(MainFrame::onRecvCommand));
}
/***********************************************************************//**
	牌選択を開始する
***************************************************************************/
void MainFrame::beginChoice(Choice* choice) {
    endChoice();
    choice_ = choice;
    taku->beginChoice(choice);
}
/***********************************************************************//**
	牌選択を終了する
***************************************************************************/
void MainFrame::endChoice() {
    if(choice_) {
        taku->endChoice();
        delete choice_;
        choice_ = 0;
    }
}
/***********************************************************************//**
	
***************************************************************************/
Client& MainFrame::getClient() {
    return client;
}
/***********************************************************************//**
	接続.
***************************************************************************/
void MainFrame::selectConnect(wxCommandEvent& event) {
    if(!client.isConnect()) {
        if(client.connect("127.0.0.1", 1326)) {
            ClientThread::Start(client);
        }
    }
}
/***********************************************************************//**
	
***************************************************************************/
void MainFrame::update() {
    taku->draw();
    consoleFrame->console->AppendText(logBuff);
    logBuff.Clear();
}
/***********************************************************************//**
	
***************************************************************************/
void MainFrame::enableButton(int mask, const wxChar** labelTable) {
    wxWindow* buttonTable[5] = {
        button0, 
        button1, 
        button2, 
        button3, 
        button4
    };
    buttonPanel->Enable();
    button0->SetValue(false);
    button1->SetValue(false);
    button2->SetValue(false);
    button3->SetValue(false);
    for(int i = 0; i < 5; i++) {
        buttonTable[i]->SetLabel(labelTable[i]);
        if((mask & (1 << i)) == 0) {
            buttonTable[i]->Disable();
        }
        else {
            buttonTable[i]->Enable();
        }
    }
    Refresh();
}
/***********************************************************************//**
	
***************************************************************************/
void MainFrame::enableTsumoButton(int mask) {
    const wxChar* labelTable[5] = {
        wxT("カン"), 
        wxT("-"), 
        wxT("リーチ"), 
        wxT("ツモ"), 
        wxT("OK")
    };
    if(mask & (BUTTON_RICHI | BUTTON_TSUMO)) {
        const Player* player = client.getPlayer();
        canRichiHais = player->canRichi();
        if(canRichiHais.isEmpty()) {
            mask &= ~(BUTTON_RICHI | BUTTON_TSUMO);
        }
        else if(!player->isAgari()) {
            mask &= ~BUTTON_TSUMO;
        }
        beginChoice(new Choice(player->getMenzen()));
    }
    enableButton(mask, labelTable);
    buttonMode = BUTTON_MODE_TSUMO;
}
/***********************************************************************//**
	鳴きメニュー.
***************************************************************************/
void MainFrame::enableNakiButton(const Command* command) {
    static const wxChar* labelTable[5] = {
        wxT("カン"), 
        wxT("ポン"), 
        wxT("チー"), 
        wxT("ロン"), 
        wxT("パス")
    };
    static const char* commandTable[] = {
        "kan", "pon", "chi", "ron"
    };
    int mask = BUTTON_CANCEL;
    for(int i = 0; i < command->getArgNum(); i++) {
        const std::string& cmd = command->getArg(i);
        for(int j = 0; j < 4; j++) {
            if(cmd == commandTable[j]) {
                mask |= 1 << j;
                break;
            }
        }
    }
    enableButton(mask, labelTable);
    buttonMode = BUTTON_MODE_NAKI;
}
/***********************************************************************//**
	メニュー.
***************************************************************************/
/** 設定 */
void MainFrame::selectPreference(wxCommandEvent& event) {
    preferenceDialog->Show();
}
/** 終了 */
void MainFrame::selectQuit(wxCommandEvent& event) {
    Close(true);
}
/** UMPログ */
void MainFrame::showConsole(wxCommandEvent& event) {
    consoleFrame->Show();
}
/***********************************************************************//**
	カン
***************************************************************************/
void MainFrame::onButton0(wxCommandEvent&) {
}
/***********************************************************************//**
	ポン
***************************************************************************/
void MainFrame::onButton1(wxCommandEvent&) {
}
/***********************************************************************//**
	リーチ/チー
***************************************************************************/
void MainFrame::onButton2(wxCommandEvent&) {
    switch(buttonMode) {
    case BUTTON_MODE_TSUMO:
        if(choice_) {
            if(button2->GetValue()) {
                choice_->pushMask(canRichiHais);
            }
            else {
                choice_->popMask();
            }
        }
        break;
    default:
        break;
    }
}
/***********************************************************************//**
	ツモ/ロン
***************************************************************************/
void MainFrame::onButton3(wxCommandEvent&) {
    switch(buttonMode) {
    case BUTTON_MODE_TSUMO:
        client.send(Command(serial_, Command::CMD_TSUMO));
        break;
    case BUTTON_MODE_NAKI:
        client.send(Command(serial_, Command::CMD_RON));
        break;
    default:
        break;
    }
}
/***********************************************************************//**
	OK/パス
***************************************************************************/
void MainFrame::onButton4(wxCommandEvent&) {
    switch(buttonMode) {
    case BUTTON_MODE_TSUMO:
        client.send(Command(serial_, Command::CMD_OK));
        break;
    case BUTTON_MODE_NAKI:
        client.send(Command(serial_, Command::CMD_NO));
        break;
    default:
        break;
    }
}
/***********************************************************************//**
	イベント.
***************************************************************************/
void MainFrame::onPaintTaku(wxPaintEvent& event) {
    taku->update();
}
void MainFrame::onMouseTaku(wxMouseEvent& event) {
    if(!choice_) {
        return;
    }
    bool doDraw = false;
    int index = taku->getChoiceIndex(event.GetX(), event.GetY());
    if(choice_->setCurrent(index)) {
        doDraw = true;
    }
    if(event.LeftDown()) {
        int mode = choice_->append();
        if(mode != Choice::MODE_NULL) {
            const mahjong::HaiArray hais = choice_->get();
            Command command(serial_);
            switch(mode) {
            case Choice::MODE_SUTEHAI:
                command.setCommand(button2->GetValue()
                                   ? Command::CMD_RICHI
                                   : Command::CMD_SUTEHAI);
                command.append(hais[0]->toString());
                /*
                if(isTsumogiri) {
                    command.push("tsumogiri");
                }
                */
                break;
            case Choice::MODE_PON:
                /*
                command.push("pon");
                command.push(hais[0]->toString());
                command.push(hais[1]->toString());
                */
                break;
            case Choice::MODE_CHI:
                /*
                command.push("chi");
                command.push(hais[0]->toString());
                command.push(hais[1]->toString());
                */
                break;
            default:
                assert(0);
                break;
            }
            client.send(command);
            endChoice();
        }
    }
    /*
    else if(event.RightDown()) {
        if(choice_->cancel()) {
            client.send("no");
            endChoice();
        }
    }
    */
    if(doDraw) {
        taku->draw();
    }
}
/***********************************************************************//**
	コマンド受信.
***************************************************************************/
void MainFrame::recvCommand(const Command& command) {
    wxCommandEvent event(EVENT_RECV_COMMAND);
    event.SetClientData(new Command(command));
    wxPostEvent(this, event);
}
void MainFrame::onRecvCommand(wxCommandEvent& event) {
    const Command* command = 
        static_cast<Command*>(event.GetClientData());
    buttonPanel->Disable();
    serial_ = command->getSerial();
    switch(command->getCommand()) {
    case Command::CMD_GAMESTART:
        taku->setPlayers(client.getPlayers());
        break;
    case Command::CMD_GAMEEND:
        taku->clearPlayers();
        break;
    case Command::CMD_KYOKUSTART:
        execKyokuStart();
        break;
    case Command::CMD_DORA:
        taku->appendDora(mahjong::Hai::Get(command->getArg(0)));
        break;
    case Command::CMD_SUTEHAI_Q:
        enableTsumoButton(BUTTON_RICHI | BUTTON_TSUMO);
        break;
    case Command::CMD_NAKU_Q:
        enableNakiButton(command);
        break;
    case Command::CMD_READY_Q:
        enableTsumoButton(BUTTON_OK);
        break;
    default:
        client.replyCommand(*command);
        break;
    }
    update();
}
/***********************************************************************//**
	
***************************************************************************/
void MainFrame::execKyokuStart() {
    static const wxString kaze[] = {
        wxT("東"), wxT("南"), wxT("西"), wxT("北")
    };
    const wxString& bakaze = kaze[client.getBakaze()->number - 1];
    wxString label = wxString::Format(wxT("%s%d局"), bakaze.GetData(), 
                                      client.getOya() + 1);
    if(client.getTsumibou() > 0) {
        label << wxString::Format(wxT(" %d本場"), client.getTsumibou());
    }
    info_->SetLabel(label);
    taku->startKyoku();
}
/***********************************************************************//**
	
***************************************************************************/
void MainFrame::log(const char* message) {
    logBuff.Append(wxString::FromUTF8(message));
}
/***********************************************************************//**
	$Id$
***************************************************************************/
}	/* namespace gui */
}	/* namespace openmj */
