/** @ingroup backend
 * @file  bidiparallel.c
 * @brief Parallel port backend with bidi status read for the Common UNIX Printing System (CUPS).
 *
 * Parallel backend program send print data to printer
 * and fetch printer status from bidi object. 
 *
 * @date $Date: 2004-10-22 17:32:53 +0900 (Fri, 22 Oct 2004) $
 * @version $Revision: 3 $
 *
 * Copyright (C) 2004 by Turbolinux,Inc.
 */

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <signal.h>
#include <unistd.h>
#include <fcntl.h>
#include <termios.h>
#include <sys/ioctl.h>
#include <linux/lp.h>

#include <cups/cups.h>

#include "bidiEntry.h"
#include "bidiParser.h"
#include "bidiEntryCUPS.h"
#include "backendCommon.h"

#ifndef HAVE_STRLCPY
extern size_t cups_strlcpy(char *, const char *, size_t);
#define strlcpy cups_strlcpy
#endif /* !HAVE_STRLCPY */

#ifndef HAVE_STRLCAT
extern size_t cups_strlcat(char *, const char *, size_t);
#define strlcat cups_strlcat
#endif /* !HAVE_STRLCAT */

#ifndef PARALLEL_NAME
#define PARALLEL_NAME "bidiparallel"
#endif /* PARALLEL_NAME */

/* Get device_id string */
#ifndef IOCNR_GET_DEVICE_ID
#define IOCNR_GET_DEVICE_ID		1
#endif /* IOCNR_GET_DEVICE_ID */

#ifndef LPIOC_GET_DEVICE_ID
#define LPIOC_GET_DEVICE_ID(len)	_IOC(_IOC_READ, 'P', IOCNR_GET_DEVICE_ID, len)
#endif /* LPIOC_GET_DEVICE_ID */

void	list_devices(void);

/**
 * main function for bidiparallel program
 *
 * @param argc Number of argument.
 * @param argv Pointer to argument.
 */
int main(int  argc, char **argv)
{
  char		method[255];	// Method in URI 
  char		hostname[1024];	// Hostname 
  char          username[255];	// Username info (not used) 
  char		resource[1024];	// Resource info (device and options) 
  char          *options;	// Pointer to options 
  int		port;		// Port number (not used) 
  int		inFD;		// Print file 
  int		copies;		// Number of copies to print 
  int		outFD;		// Device file descriptor
  struct termios opts;	// Parallel port options 
  unsigned char	status;	// Port status (off-line, out-of-paper, etc.) 
  BidiWrapperObj *obj=NULL; // Bidi Plugin object 
  
#if defined(HAVE_SIGACTION) && !defined(HAVE_SIGSET)
  struct sigaction action;	/* Actions for POSIX signals */
#endif /* HAVE_SIGACTION && !HAVE_SIGSET */
  
  // Make sure status messages are not buffered... 
  setbuf(stderr, NULL);
  
  // Ignore SIGPIPE signals, Accept ALRM 
#ifdef HAVE_SIGSET
  sigset(SIGPIPE, SIG_IGN);
#elif defined(HAVE_SIGACTION)
  memset(&action, 0, sizeof(action));
  action.sa_handler = SIG_IGN;
  sigaction(SIGPIPE, &action, NULL);
#else
  signal(SIGPIPE, SIG_IGN);
#endif /* HAVE_SIGSET */

  // Check command-line... 
  if (argc == 1){
    list_devices();
    return (0);
  }else if (argc < 6 || argc > 7){
    printf("Usage: %s [job-id] [user] [title] [copies] [options] <file>\n",PARALLEL_NAME);
    return (1);
  }
  
  if (argc == 6){
    inFD = 0;
    copies = 1;
  }else{
    if ((inFD = open(argv[6], O_RDONLY)) < 0){
      perror("ERROR: unable to open print file");
      return (1);
    }
    copies = atoi(argv[4]);
  }

  // Extract the device name and options from the URI... 
  httpSeparate(argv[0], method, username, hostname, &port, resource);
  
  // See if there are any options...  
  if ((options = strchr(resource, '?')) != NULL){
   /*
    * Yup, terminate the device name string and move to the first
    * character of the options...
    */
    *options++ = '\0';
  }

  // Open the parallel port device... 
#ifdef STANDALONE
  int fd;
  if((fd=open(getenv("DEVICE"),O_RDWR)) < 0){
    perror("device open error");
    fprintf(stderr,"DEVICE=%s\n",getenv("DEVICE"));
    fprintf(stderr,"PPD=%s\n",getenv("PPD"));
    exit(-1);
  }else{
    DEBUGPRINT(("Debug mode: open %s",getenv("DEVICE")));
    return fd;
  }
#endif

  do{
    if ((outFD = open(resource, O_RDWR | O_EXCL)) == -1){
      if (errno == EBUSY){
        fputs("INFO: Parallel port busy; will retry in 30 seconds...\n", stderr);
	sleep(30);
      }else if (errno == ENXIO || errno == EIO || errno == ENOENT){
        fputs("INFO: Printer not connected; will retry in 30 seconds...\n", stderr);
	sleep(30);
      }else{
	fprintf(stderr, "ERROR: Unable to open parallel port device file \"%s\": %s\n",
	        resource, strerror(errno));
	return (1);
      }
    }
  }
  while (outFD < 0);
  
  // Set any options provided... 
  tcgetattr(outFD, &opts);
  opts.c_lflag &= ~(ICANON | ECHO | ISIG);	/* Raw mode */
  //*** No options supported yet
  tcsetattr(outFD, TCSANOW, &opts);
  
  initSignal();
  obj=initBidi(getenv("PPD"),outFD,getLang());
  startBidiJob(obj,atoi(argv[1]));

  // Finally, send the print file... 
  while (copies > 0){
    copies --;
    if (inFD != 0){
      fputs("PAGE: 1 / 1\n", stderr);
      lseek(inFD, 0, SEEK_SET);
    }
    writeData(obj,inFD,outFD);
  }

  endBidiJob(obj);
  deleteBidi(obj);
  deleteBidiObj(obj);
  
  // Close the socket connection and input file and return... 
  close(outFD);
  if (inFD != 0)
    close(inFD);
  
  fputs("INFO: Ready to print.\n", stderr);
  
  return (0);
}

/**
 * Function for List devices coneected with parallel port.
 *
 * @param None.
 * @return None.
 */
void list_devices(void)
{
#if DEBUG
  // show DEBUG info 
  fprintf(stderr,"Warninig: this module was compiled with debuggin mode..\n");
  fprintf(stderr,"PPD=%s\n",getenv("PPD"));
  fprintf(stderr,"DEVICE=%s\n",getenv("DEVICE"));
#endif

#ifdef __linux
  int	i;			/* Looping var */
  int	fd;			/* File descriptor */
  char	device[255];		/* Device filename */
  char  probefile[255];		/* Probe filename */
  char  basedevice[255];	/* Base device filename for ports */
  FILE	*probe;			/* /proc/parport/n/autoprobe file */
  char	line[1024];		/* Line from file */
  char *delim;			/* Delimiter in file */
  char make[IPP_MAX_NAME];	/* Make from file */
  char model[IPP_MAX_NAME];	/* Model from file */

  for (i = 0; i < 4; i ++){
    // First open the device to make sure the driver module is loaded... 
    if ((fd = open("/dev/parallel/0", O_WRONLY)) >= 0){
      close(fd);
      strcpy(basedevice, "/dev/parallel/");
    }else{
      sprintf(device, "/dev/lp%d", i);
      if ((fd = open(device, O_WRONLY)) >= 0){
	close(fd);
	strcpy(basedevice, "/dev/lp");
      }else{
	sprintf(device, "/dev/par%d", i);
	if ((fd = open(device, O_WRONLY)) >= 0){
	  close(fd);
	  strcpy(basedevice, "/dev/par");
	}else{
	  sprintf(device, "/dev/printers/%d", i);
	  if ((fd = open(device, O_WRONLY)) >= 0){
	    close(fd);
	    strcpy(basedevice, "/dev/printers/");
	  }else
	    strcpy(basedevice, "/dev/unknown-parallel");
	}
      }
    }
    
    // Then try looking at the probe file... 
    sprintf(probefile, "/proc/parport/%d/autoprobe", i);
    if ((probe = fopen(probefile, "r")) == NULL){
      // Linux 2.4 kernel has different path... 
      sprintf(probefile, "/proc/sys/dev/parport/parport%d/autoprobe", i);
      probe = fopen(probefile, "r");
    }

    if (probe != NULL){
      // Found a probe file! 

      memset(make, 0, sizeof(make));
      memset(model, 0, sizeof(model));
      strcpy(model, "Unknown");

      while (fgets(line, sizeof(line), probe) != NULL){
	// Strip trailing ; and/or newline.

        if ((delim = strrchr(line, ';')) != NULL){
	  *delim = '\0';
	}else if ((delim = strrchr(line, '\n')) != NULL){
	  *delim = '\0';
	}

	// Look for MODEL and MANUFACTURER lines...
        if (strncmp(line, "MODEL:", 6) == 0 &&
	    strncmp(line, "MODEL:Unknown", 13) != 0){
	  strlcpy(model, line + 6, sizeof(model));
	}else if (strncmp(line, "MANUFACTURER:", 13) == 0 &&
		  strncmp(line, "MANUFACTURER:Unknown", 20) != 0){
	  strlcpy(make, line + 13, sizeof(make));
	}
      }
      fclose(probe);

      if (make[0]){
	printf("direct %s:%s%d \"%s %s\" \"Parallel Port #%d\"\n",
	       PARALLEL_NAME,basedevice, i, make, model, i + 1);
      }else{
	printf("direct %s:%s%d \"%s\" \"Parallel Port #%d\"\n",
	       PARALLEL_NAME,basedevice, i, model, i + 1);
      }
    }else if (fd >= 0){
      // No probe file, but we know the port is there...   
      printf("direct %s:%s \"Unknown\" \"Parallel Port #%d\"\n",PARALLEL_NAME, device, i + 1);
    }
  }

#elif defined(__FreeBSD__) || defined(__OpenBSD__) || defined(__NetBSD__)
  int	i;			/* Looping var */
  int	fd;			/* File descriptor */
  char	device[255];		/* Device filename */

  for (i = 0; i < 3; i ++){
    sprintf(device, "/dev/lpt%d", i);
    if ((fd = open(device, O_WRONLY)) >= 0){
      close(fd);
      printf("direct %s:%s \"Unknown\" \"Parallel Port #%d (interrupt-driven)\"\n",PARALLEL_NAME, device, i + 1);
    }

    sprintf(device, "/dev/lpa%d", i);
    if ((fd = open(device, O_WRONLY)) >= 0){
      close(fd);
      printf("direct %s:%s \"Unknown\" \"Parallel Port #%d (polled)\"\n",PARALLEL_NAME, device, i + 1);
    }
  }
#endif
}
