/** @ingroup backend
 * @file  bidiusb.c
 * @brief USB port backend with bidi status read for the Common UNIX Printing System (CUPS).
 *
 * USB backend program send print data to printer
 * and fetch printer status from bidi object. 
 *
 * @date $Date: 2004-10-22 17:32:53 +0900 (Fri, 22 Oct 2004) $
 * @version $Revision: 3 $
 *
 * Copyright (C) 2004 by Turbolinux,Inc.
 */

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <signal.h>
#include <unistd.h>
#include <fcntl.h>
#include <termios.h>
#include <sys/ioctl.h>
#include <linux/lp.h>

#include <cups/cups.h>

#include "bidiEntry.h"
#include "bidiParser.h"
#include "bidiEntryCUPS.h"
#include "backendCommon.h"

#ifndef HAVE_STRLCPY
extern size_t cups_strlcpy(char *, const char *, size_t);
#define strlcpy cups_strlcpy /**< safe string copy.*/
#endif /* !HAVE_STRLCPY */

#ifndef HAVE_STRLCAT
extern size_t cups_strlcat(char *, const char *, size_t); 
#define strlcat cups_strlcat /**< safe string concatenation.*/
#endif /* !HAVE_STRLCAT */

#ifndef USB_NAME
#define USB_NAME "bidiusb" /**< program name.*/
#endif /* USB_NAME */

#ifndef IOCNR_GET_DEVICE_ID
#define IOCNR_GET_DEVICE_ID		1
#endif /* IOCNR_GET_DEVICE_ID */

#ifndef LPIOC_GET_DEVICE_ID
#define LPIOC_GET_DEVICE_ID(len)	_IOC(_IOC_READ, 'P', IOCNR_GET_DEVICE_ID, len) /**< ioctl macro to get device_id string.*/
#endif /* LPIOC_GET_DEVICE_ID */


void decode_device_id(int port, const char *device_id,
			 char *make_model, int mmsize,
			 char *uri, int urisize);
void list_devices(void);
int open_device(const char *uri);

/** 
 * main function for bidiusb program
 *
 * @param argc Number of argument.
 * @param argv Pointer to argument.
 */
int main(int  argc, char **argv)
{
  int inFD;             //Print file 
  int outFD;            //Device file descriptor
  int copies;           //Number of copies to print 
  struct termios opts;	//Port options 
  BidiWrapperObj *obj=NULL;    //Bidi Plugin object 

#if defined(HAVE_SIGACTION) && !defined(HAVE_SIGSET)
  struct sigaction action;
#endif /* HAVE_SIGACTION && !HAVE_SIGSET */

  // Make sure status messages are not buffered...
  setbuf(stderr, NULL);

  // Ignore SIGPIPE signals...
#ifdef HAVE_SIGSET
  sigset(SIGPIPE, SIG_IGN);
#elif defined(HAVE_SIGACTION)
  memset(&action, 0, sizeof(action));
  action.sa_handler = SIG_IGN;
  sigaction(SIGPIPE, &action, NULL);
#else
  signal(SIGPIPE, SIG_IGN);
#endif /* HAVE_SIGSET */
  
  // Check command-line...
  if (argc == 1){
    list_devices();
    return (0);
  }else if (argc < 6 || argc > 7){
    printf("Usage: %s [job-id] [user] [title] [copies] [options] <file>\n",USB_NAME);
    return (1);
  }
  
  if (argc == 6){
    inFD = 0;
    copies = 1;
  }else{
    if ((inFD = open(argv[6], O_RDONLY)) < 0){
      perror("ERROR: unable to open print file");
      return (1);
    }
    copies = atoi(argv[4]);
  }

  // Open the USB port device...
  do{
    if ((outFD = open_device(argv[0])) == -1){
      if (errno == EBUSY){
	fputs("INFO: USB port busy; will retry in 30 seconds...\n", stderr);
	sleep(30);
      }
      else if (errno == ENXIO || errno == EIO || errno == ENOENT){
	fputs("INFO: Printer not connected; will retry in 30 seconds...\n", stderr);
	sleep(30);
      }
      else{
	fprintf(stderr, "ERROR: Unable to open USB device \"%s\": %s\n",
		argv[0], strerror(errno));
	return (1);
      }
    }
  }while (outFD < 0);

  // Set any options provided...
  tcgetattr(outFD, &opts);
  opts.c_lflag &= ~(ICANON | ECHO | ISIG);	// Raw mode
  //  No options supported yet
  tcsetattr(outFD, TCSANOW, &opts);

  initSignal();
  obj=initBidi(getenv("PPD"),outFD,getLang());
  startBidiJob(obj,atoi(argv[1]));

  // Finally, send the print file...
  while (copies > 0){
    copies --;
    if (inFD != 0){
      fputs("PAGE: 1 / 1\n", stderr);
      lseek(inFD, 0, SEEK_SET);
    }
    writeData(obj,inFD,outFD);
  }

  endBidiJob(obj);
  deleteBidi(obj);
  deleteBidiObj(obj);
  
  // Close the socket connection and input file and return...
  close(outFD);
  if (inFD != 0)
    close(inFD);
  
  fputs("INFO: Ready to print.\n", stderr);
  
  return (0);
}

/**
 * Decode the IEEE-1284 device ID string.
 *
 * @param port Port number.
 * @param device_id 1284 device ID string.
 * @param make_model model name.
 * @param mmsize Size of buffer.
 * @param uri Device URI.
 * @param urisize Size of buffer.
 * @return None.
 */
void decode_device_id(int port,const char *device_id,char *make_model,int mmsize,char*uri,int urisize)
{
  char	*attr;					// 1284 attribute
  char  *delim;					// 1284 delimiter
  char  *uriptr;			        // Pointer into URI 
  char  *mfg;				// Manufacturer string 
  char  *mdl;					// Model string 
  char  serial_number[1024];			// Serial number string 
  
  // Look for the description field... 
  if ((attr = strstr(device_id, "DES:")) != NULL){
    attr += 4;
  }else if ((attr = strstr(device_id, "DESCRIPTION:")) != NULL){
    attr += 12;
  }
  
  if ((mfg = strstr(device_id, "MANUFACTURER:")) != NULL)
    mfg += 13;
  else if ((mfg = strstr(device_id, "MFG:")) != NULL)
    mfg += 4;
  
  if ((mdl = strstr(device_id, "MODEL:")) != NULL)
    mdl += 6;
  else if ((mdl = strstr(device_id, "MDL:")) != NULL)
    mdl += 4;
  
  if (attr){
    if (strncasecmp(attr, "Hewlett-Packard ", 16) == 0){
      strlcpy(make_model, "HP ", mmsize);
      strlcpy(make_model + 3, attr + 16, mmsize - 3);
    }else{
      strlcpy(make_model, attr, mmsize);
    }
    
    if ((delim = strchr(make_model, ';')) != NULL)
      *delim = '\0';
  }else if (mfg && mdl){
    // Build a make-model string from the manufacturer and model attributes... 
    strlcpy(make_model, mfg, mmsize);
    
    if ((delim = strchr(make_model, ';')) != NULL)
      *delim = '\0';
    
    strlcat(make_model, " ", mmsize);
    strlcat(make_model, mdl, mmsize);
    
    if ((delim = strchr(make_model, ';')) != NULL)
      *delim = '\0';
  }else{
    // Use "Unknown" as the printer make and model... 
    strlcpy(make_model, "Unknown", mmsize);
  }
  
  // Look for the serial number field... 
  if ((attr = strstr(device_id, "SERN:")) != NULL){
    attr += 5;
  }else if ((attr = strstr(device_id, "SERIALNUMBER:")) != NULL){
    attr += 13;
  }
  
  if (attr){
    strlcpy(serial_number, attr, sizeof(serial_number));
    if ((delim = strchr(serial_number, ';')) != NULL)
      *delim = '\0';
  }else{
    serial_number[0] = '\0';
  }
  
  // Generate the device URI from the make_model and serial number strings. 
  strlcpy(uri, "bidiusb://", urisize);
  for (uriptr = uri + 6, delim = make_model;
       *delim && uriptr < (uri + urisize - 1);
       delim ++)
    if (*delim == ' '){
      delim ++;
      *uriptr++ = '/';
      break;
    }else{
      *uriptr++ = *delim;
    }
  
  for (; *delim && uriptr < (uri + urisize - 3); delim ++)
    if (*delim == ' '){
      *uriptr++ = '%';
      *uriptr++ = '2';
      *uriptr++ = '0';
    }else{
      *uriptr++ = *delim;
    }
  *uriptr = '\0';
  
  if (serial_number[0]){
    // Add the serial number to the URI...
    strlcat(uri, "?serial=", urisize);
    strlcat(uri, serial_number, urisize);
  }
}


/**
 * Function for List all USB devices.
 *
 * @return None.
 */
void list_devices(void)
{
  int	i;			  // i Looping var.
  int	length;			  // length Length of device ID info. 
  int	fd;			  // fd File descriptor.
  char	format[255];		  // format Format for device filename. 
  char  device[255];		  // device Device filename 
  char  device_id[1024];	  // device_id Device ID string. 
  char  device_uri[1024];	  // device_uri Device URI string. 
  char  make_model[1024];	  // make_model Make and model.

#if DEBUG
  // show DEBUG info 
  fprintf(stderr,"Warninig: this module was compiled with debuggin mode..\n");
  fprintf(stderr,"PPD=%s\n",getenv("PPD"));
  fprintf(stderr,"DEVICE=%s\n",getenv("DEVICE"));
#endif

  // First figure out which USB printer filename to use...
  if (access("/dev/usb/lp0", 0) == 0)
    strcpy(format, "/dev/usb/lp%d");
  else if (access("/dev/usb/usblp0", 0) == 0)
    strcpy(format, "/dev/usb/usblp%d");
  else
    strcpy(format, "/dev/usblp%d");
  
  // Then open each USB device... 
  for (i = 0; i < 16; i ++)
    {
      sprintf(device, format, i);
      
      if ((fd = open(device, O_RDWR | O_EXCL)) >= 0)
	{
	  if (ioctl(fd, LPIOC_GET_DEVICE_ID(sizeof(device_id)), device_id) == 0)
	    {
	      length = (((unsigned)device_id[0] & 255) << 8) +
		((unsigned)device_id[1] & 255);
	      
	      /*
	       * Check to see if the length is larger than our buffer; first
	       * assume that the vendor incorrectly implemented the 1284 spec,
	       * and then limit the length to the size of our buffer...
	       */
	      
	      if (length > (sizeof(device_id) - 2))
		length = (((unsigned)device_id[1] & 255) << 8) +
		  ((unsigned)device_id[0] & 255);
	      
	      if (length > (sizeof(device_id) - 2))
		length = sizeof(device_id) - 2;
	      
	      memmove(device_id, device_id + 2, length);
	      device_id[length] = '\0';
	    }
	  else
	    device_id[0] = '\0';
	  
	  close(fd);
	}
      else
	device_id[0] = '\0';
      
      if (device_id[0])
	{
	  decode_device_id(i, device_id, make_model, sizeof(make_model),
			   device_uri, sizeof(device_uri));
	  printf("direct %s:%s \"%s\" \"USB Printer #%d (Bidi support)\"\n", USB_NAME,device,
		 make_model, i + 1);
	}
      else
	printf("direct %s:%s \"Unknown\" \"USB Printer #%d (Bidi support)\"\n", USB_NAME,device, i + 1);
    }
}

/**
 * Function for Open a USB device...
 *
 * @param uri Pointer to open device string.
 * @retval > 0 Opened device descriptor
 * @retval < 0 Fail
 */
int open_device(const char *uri)
{
#ifdef STANDALONE
  int fd;
  if((fd=open(getenv("DEVICE"),O_RDWR)) < 0){
    perror("device open error");
    fprintf(stderr,"DEVICE=%s\n",getenv("DEVICE"));
    fprintf(stderr,"PPD=%s\n",getenv("PPD"));
    exit(-1);
  }else{
    DEBUGPRINT(("Debug mode: open %s",getenv("DEVICE")));
    return fd;
  }
#endif

  if (strncmp(uri, "bidiusb:/dev/", 13) == 0)
    return (open(uri + 8, O_RDWR | O_EXCL));
  else if (strncmp(uri, "bidiusb://", 10) == 0){
    // For Linux, try looking up the device serial number or model... 
    int		i;			/* Looping var */
    int		length;			/* Length of device ID info */
    int		fd;			/* File descriptor */
    char	format[255];		/* Format for device filename */
    char	device[255];		/* Device filename */
    char	device_id[1024];	/* Device ID string */
    char	make_model[1024];	/* Make and model */
    char	device_uri[1024];	/* Device URI string */
    
    // First figure out which USB printer filename to use... 
    if (access("/dev/usb/lp0", 0) == 0)
      strcpy(format, "/dev/usb/lp%d");
    else if (access("/dev/usb/usblp0", 0) == 0)
      strcpy(format, "/dev/usb/usblp%d");
    else
      strcpy(format, "/dev/usblp%d");
    
    // Then find the correct USB device... 
    for (i = 0; i < 16; i ++){
      sprintf(device, format, i);
      if ((fd = open(device, O_RDWR | O_EXCL)) >= 0){
	if (ioctl(fd, LPIOC_GET_DEVICE_ID(sizeof(device_id)), device_id) == 0){
	  length = (((unsigned)device_id[0] & 255) << 8) +
	    ((unsigned)device_id[1] & 255);
	  memmove(device_id, device_id + 2, length);
	  device_id[length] = '\0';
	}else
	  device_id[0] = '\0';
      }else
	device_id[0] = '\0';
      
      if (device_id[0]){
	// Got the device ID - is this the one? 
	decode_device_id(i, device_id, make_model, sizeof(make_model),
			 device_uri, sizeof(device_uri));
	if (strcmp(uri, device_uri) == 0){
	  // Yes, return this file descriptor... 
	  fprintf(stderr, "DEBUG: Printer using device file \"%s\"...\n", device);
	  return (fd);
	}
      }
      // This wasn't the one... 
      close(fd);
    }
    // Couldn't find the printer, return "no such device or address"... 
    errno = ENODEV;
    return (-1);
  }
  else{
    errno = ENODEV;
    return (-1);
  }
}
