/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation;                            * 
 *                                                                         *
 ***************************************************************************/

/**
 * @ingroup bidiformat
 * @file bidiEntryCUPS.c
 * @brief Functions for infoCUPS object.
 * @date $Date: 2004-10-22 17:32:53 +0900 (Fri, 22 Oct 2004) $
 * @version $Revision: 3 $
   copyright            : (C) 2004 by Turbolinux,Inc.
*/

#include <string.h>
#include "bidiEntryCUPS.h"

/**
  * constructor that creates an empty infoCUPS object.
  * @param None.
  * @return Pointer to inofCUPS object.
  */
static infoCUPS *newInfoCUPS(void)
{
  infoCUPS *self;
  self = (infoCUPS *)malloc(sizeof(infoCUPS));
  if(!self) return NULL;

  memset(self->state_reason,'\0',MESG_SIZE);
  memset(self->state_messages,'\0',MESG_SIZE);
  self->next = NULL;
  return self;
}

/**
  * Function to add IPP messages into infoCUPS object.
  *
  * @param self Pointer to infoCUPS object.
  * @param reason Pointer to new state-reason string.
  * @param message Pointer to new state-message string.
  * @return Pointer to inofCUPS object.
  */
static infoCUPS *addInfoCUPS(infoCUPS *self,const char *reason,const char *message)
{
  if( (strlen(reason)+strlen(message)) == 0 ) return self;
  infoCUPS *newinfo = newInfoCUPS();
  if(!newinfo) return NULL;
  snprintf(newinfo->state_reason,MESG_SIZE,reason);
  snprintf(newinfo->state_messages,MESG_SIZE,message);

  if(self){
    infoCUPS *tmp = self;
    while(tmp->next) tmp = tmp->next;
    tmp->next = newinfo;
  }else{
    self = newinfo;
  }
  return self;
}

/**
  * Function to make IPP message based on Alert info.
  *
  * @param entry Pointer to Entry object.
  * @param info  Pointer to infoCUPS object.
  * @return Pointer to inofCUPS object.
  */
static infoCUPS *makeAlertInfo(Entry *entry,infoCUPS *info)
{
  EntryPrtAlert *entAlt;
  char buf_reason[MESG_SIZE];
  char buf_message[MESG_SIZE];

  entAlt = entry->EntryPrtAlertPtr;
  if(entAlt){
    while(entAlt){
      memset(buf_reason,'\0',MESG_SIZE);
      memset(buf_message,'\0',MESG_SIZE);

      switch(entAlt->PrtAltSevLevel){
      case PRTALT_SEVLEV_CRITICAL:

	switch(entAlt->PrtAltGrp){
	case PRTALT_GRP_MARKER_SUPPLIES:
	  /* ignore marker supplies alert */
	  break;
	case PRTALT_GRP_COVER:
	  if( entAlt->PrtAltCode == PRTALT_CODE_COVER_OPEN){
	    snprintf(buf_reason,MESG_SIZE,"%s","cover-open");
	    snprintf(buf_message,MESG_SIZE,"%s",entAlt->PrtAltDesc);
	  }
	  break;
	case PRTALT_GRP_MARKER:
	  if( entAlt->PrtAltCode == PRTALT_CODE_OTHER){
	    snprintf(buf_reason,MESG_SIZE,"%s","other");
	    snprintf(buf_message,MESG_SIZE,"%s",entAlt->PrtAltDesc);
	  }
	  break;
	case PRTALT_GRP_INPUT:
	  if( entAlt->PrtAltCode == PRTALT_CODE_INPUT_MEDIASUPP_EMPTY){
	    snprintf(buf_reason,MESG_SIZE,"%s","media-empty");
	    snprintf(buf_message,MESG_SIZE,"%s",entAlt->PrtAltDesc);
	  }else if (entAlt->PrtAltCode == PRTALT_CODE_JAM){
	    snprintf(buf_reason,MESG_SIZE,"%s","media-jam");
	    snprintf(buf_message,MESG_SIZE,"%s",entAlt->PrtAltDesc);
	  }
	  break;
	case PRTALT_GRP_OTHER:
	  if( entAlt->PrtAltCode == PRTALT_CODE_OTHER){
	    snprintf(buf_reason,MESG_SIZE,"%s","other");
	    snprintf(buf_message,MESG_SIZE,"%s",entAlt->PrtAltDesc);
	  }
	  break;
	}
	break;

      case PRTALT_SEVLEV_WARNING:
	break;

      }
      info = addInfoCUPS(info,buf_reason,buf_message);
      entAlt = entAlt->next;
    }
  }
  return info;
}

/**
  * Function to make IPP message based on Marker info.
  *
  * @param entry Pointer to Entry object.
  * @param info  Pointer to infoCUPS object.
  * @return Pointer to inofCUPS object.
  */
static infoCUPS *makeMarkerInfo(Entry *entry,infoCUPS *info)
{
  int isMarkSupAlert=0;
  char buf_reason[MESG_SIZE];
  char buf_message[MESG_SIZE];
  EntryPrtMarkerSupplies *entMrk;
  EntryPrtAlert *entAlt;

  entMrk = entry->EntryPrtMarkerSuppliesPtr;
  while(entMrk){
    isMarkSupAlert=0;
    memset(buf_reason,'\0',MESG_SIZE);
    memset(buf_message,'\0',MESG_SIZE);
    entAlt = entry->EntryPrtAlertPtr;

    if(entAlt){
      while(entAlt){
	if( entAlt->PrtAltGrp == PRTALT_GRP_MARKER_SUPPLIES && entMrk->PrtMarkSupIndex == entAlt->PrtAltGrpIndex){
	  isMarkSupAlert=1;
	  switch(entAlt->PrtAltSevLevel){
	  case PRTALT_SEVLEV_CRITICAL:
	    snprintf(buf_reason,MESG_SIZE,"opbidi-ink-%s-0-empty",getIndexName(entMrk->PrtMarkSupIndex));
	    break;
	  case PRTALT_SEVLEV_WARNING:
	    snprintf(buf_reason,MESG_SIZE,"opbidi-ink-%s-%d-low-warning",getIndexName(entMrk->PrtMarkSupIndex),(int)entMrk->PrtMarkSupLevel);
	    break;
	  }
	  snprintf(buf_message,MESG_SIZE,"%s",entAlt->PrtAltDesc);
	  break; /* break while(entAlt) */
	}else{
	  isMarkSupAlert=0;
	}
	entAlt=entAlt->next;
      } /* loop while(entAlt) */
    }

    if(!isMarkSupAlert){
      snprintf(buf_reason,MESG_SIZE,"opbidi-ink-%s-%d-report",getIndexName(entMrk->PrtMarkSupIndex),(int)entMrk->PrtMarkSupLevel);
    }

    info = addInfoCUPS(info,buf_reason,buf_message);
    entMrk=entMrk->next;
  }
  return info;
}

/**
  * Function to make IPP message.
  *
  * @param entry Pointer to Entry object.
  * @return Pointer to inofCUPS object.
  */
infoCUPS *makeIPPInfo(Entry *entry){
  infoCUPS *info = NULL;
  info = makeMarkerInfo(entry,info);
  info = makeAlertInfo(entry,info);
  return info;
}
