/*
 * Copyright (c) 2006-2009 OrangeSignal.com All rights reserved.
 */

package jp.sourceforge.orangesignal.trading.strategy;

import jp.sourceforge.orangesignal.trading.order.OrderTiming;

/**
 * <p>トレーディングストラテジーを実装する為の抽象クラスを提供します。</p>
 * 
 * @author 杉澤 浩二
 */
public abstract class TradingStrategySupport extends AbstractTradingStrategy {

	/**
	 * デフォルトコンストラクタです。
	 */
	public TradingStrategySupport() {}

	// ----------------------------------------------------------------------
	// buy

	/**
	 * <p>デフォルトの注文方法で買い注文を発行します。</p>
	 */
	public final void buy() {
		buy(null, 0, getDefaultOrderTiming(), 0);
	}

	/**
	 * <p>デフォルトの注文方法で買い注文を発行します。</p>
	 * 
	 * @param price 指定価格
	 */
	public final void buy(final double price) {
		buy(null, 0, getDefaultOrderTiming(), price);
	}

	/**
	 * <p>デフォルトの注文方法で買い注文を発行します。</p>
	 * 
	 * @param quantity 数量
	 */
	public final void buy(final int quantity) {
		buy(null, quantity, getDefaultOrderTiming(), 0);
	}

	/**
	 * <p>デフォルトの注文方法で買い注文を発行します。</p>
	 * 
	 * @param quantity 数量
	 * @param price 指定価格
	 */
	public final void buy(final int quantity, final double price) {
		buy(null, quantity, getDefaultOrderTiming(), price);
	}

	// ----------------------------------------------------------------------
	// buy (label)

	/**
	 * <p>デフォルトの注文方法で買い注文を発行します。</p>
	 * 
	 * @param label ラベル
	 */
	public final void buy(final String label) {
		buy(label, 0, getDefaultOrderTiming(), 0);
	}

	/**
	 * <p>デフォルトの注文方法で買い注文を発行します。</p>
	 * 
	 * @param label ラベル
	 * @param price 指定価格
	 */
	public final void buy(final String label, final double price) {
		buy(label, 0, getDefaultOrderTiming(), price);
	}

	/**
	 * <p>デフォルトの注文方法で買い注文を発行します。</p>
	 * 
	 * @param label ラベル
	 * @param quantity 数量
	 */
	public final void buy(final String label, final int quantity) {
		buy(label, quantity, getDefaultOrderTiming(), 0);
	}

	/**
	 * <p>デフォルトの注文方法で買い注文を発行します。</p>
	 * 
	 * @param label ラベル
	 * @param quantity 数量
	 * @param price 指定価格
	 */
	public final void buy(final String label, final int quantity, final double price) {
		buy(label, quantity, getDefaultOrderTiming(), price);
	}

	// ----------------------------------------------------------------------
	// buy (timing)

	/**
	 * <p>指定された注文方法で買い注文を発行します。</p>
	 * 
	 * @param timing 注文方法の種類
	 */
	public final void buy(final OrderTiming timing) {
		buy(null, 0, timing, 0);
	}

	/**
	 * <p>指定された注文方法で買い注文を発行します。</p>
	 * 
	 * @param timing 注文方法の種類
	 * @param price 指定価格
	 */
	public final void buy(final OrderTiming timing, final double price) {
		buy(null, 0, timing, price);
	}

	/**
	 * <p>指定された注文方法で買い注文を発行します。</p>
	 * 
	 * @param quantity 数量
	 * @param timing 注文方法の種類
	 */
	public final void buy(final int quantity, final OrderTiming timing) {
		buy(null, quantity, timing, 0);
	}

	/**
	 * <p>指定された注文方法で買い注文を発行します。</p>
	 * 
	 * @param quantity 数量
	 * @param timing 注文方法の種類
	 * @param price 指定価格
	 */
	public final void buy(final int quantity, final OrderTiming timing, final double price) {
		buy(null, quantity, timing, price);
	}

	// ----------------------------------------------------------------------
	// buy (label, timing)

	/**
	 * <p>指定された注文方法で買い注文を発行します。</p>
	 * 
	 * @param label ラベル
	 * @param timing 注文方法の種類
	 */
	public final void buy(final String label, final OrderTiming timing) {
		buy(label, 0, timing, 0);
	}

	/**
	 * <p>指定された注文方法で買い注文を発行します。</p>
	 * 
	 * @param label ラベル
	 * @param timing 注文方法の種類
	 * @param price 指定価格
	 */
	public final void buy(final String label, final OrderTiming timing, final double price) {
		buy(label, 0, timing, price);
	}

	/**
	 * <p>指定された注文方法で買い注文を発行します。</p>
	 * 
	 * @param label ラベル
	 * @param quantity 数量
	 * @param timing 注文方法の種類
	 */
	public final void buy(final String label, final int quantity, final OrderTiming timing) {
		buy(label, quantity, timing, 0);
	}

	/**
	 * <p>指定された注文方法で買い注文を発行します。</p>
	 * 
	 * @param label ラベル
	 * @param quantity 数量
	 * @param timing 注文方法の種類
	 * @param price 指定価格
	 */
	public final void buy(final String label, final int quantity, final OrderTiming timing, final double price) {
		buy(createOrder(label, quantity, timing, price, null, null));
	}

	// ----------------------------------------------------------------------
	// sell short

	/**
	 * <p>デフォルトの注文方法で空売り注文を発行します。</p>
	 */
	public final void sellShort() {
		sellShort(null, 0, getDefaultOrderTiming(), 0);
	}

	/**
	 * <p>デフォルトの注文方法で空売り注文を発行します。</p>
	 * 
	 * @param price 指定価格
	 */
	public final void sellShort(final double price) {
		sellShort(null, 0, getDefaultOrderTiming(), price);
	}

	/**
	 * <p>デフォルトの注文方法で空売り注文を発行します。</p>
	 * 
	 * @param quantity 数量
	 */
	public final void sellShort(final int quantity) {
		sellShort(null, quantity, getDefaultOrderTiming(), 0);
	}

	/**
	 * <p>デフォルトの注文方法で空売り注文を発行します。</p>
	 * 
	 * @param quantity 数量
	 * @param price 指定価格
	 */
	public final void sellShort(final int quantity, final double price) {
		sellShort(null, quantity, getDefaultOrderTiming(), price);
	}

	// ----------------------------------------------------------------------
	// sell short (label)

	/**
	 * <p>デフォルトの注文方法で空売り注文を発行します。</p>
	 * 
	 * @param label ラベル
	 */
	public final void sellShort(final String label) {
		sellShort(label, 0, getDefaultOrderTiming(), 0);
	}

	/**
	 * <p>デフォルトの注文方法で空売り注文を発行します。</p>
	 * 
	 * @param label ラベル
	 * @param price 指定価格
	 */
	public final void sellShort(final String label, final double price) {
		sellShort(label, 0, getDefaultOrderTiming(), price);
	}

	/**
	 * <p>デフォルトの注文方法で空売り注文を発行します。</p>
	 * 
	 * @param label ラベル
	 * @param quantity 数量
	 */
	public final void sellShort(final String label, final int quantity) {
		sellShort(label, quantity, getDefaultOrderTiming(), 0);
	}

	/**
	 * <p>デフォルトの注文方法で空売り注文を発行します。</p>
	 * 
	 * @param label ラベル
	 * @param quantity 数量
	 * @param price 指定価格
	 */
	public final void sellShort(final String label, final int quantity, final double price) {
		sellShort(label, quantity, getDefaultOrderTiming(), price);
	}

	// ----------------------------------------------------------------------
	// sell short (timing)

	/**
	 * <p>指定された注文方法で空売り注文を発行します。</p>
	 * 
	 * @param timing 注文方法の種類
	 */
	public final void sellShort(final OrderTiming timing) {
		sellShort(null, 0, timing, 0);
	}

	/**
	 * <p>指定された注文方法で空売り注文を発行します。</p>
	 * 
	 * @param timing 注文方法の種類
	 * @param price 指定価格
	 */
	public final void sellShort(final OrderTiming timing, final double price) {
		sellShort(null, 0, timing, price);
	}

	/**
	 * <p>指定された注文方法で空売り注文を発行します。</p>
	 * 
	 * @param quantity 数量
	 * @param timing 注文方法の種類
	 */
	public final void sellShort(final int quantity, final OrderTiming timing) {
		sellShort(null, quantity, timing, 0);
	}

	/**
	 * <p>指定された注文方法で空売り注文を発行します。</p>
	 * 
	 * @param quantity 数量
	 * @param timing 注文方法の種類
	 * @param price 指定価格
	 */
	public final void sellShort(final int quantity, final OrderTiming timing, final double price) {
		sellShort(null, quantity, timing, price);
	}

	// ----------------------------------------------------------------------
	// sell short (label, timing)

	/**
	 * <p>指定された注文方法で空売り注文を発行します。</p>
	 * 
	 * @param label ラベル
	 * @param timing 注文方法の種類
	 */
	public final void sellShort(final String label, final OrderTiming timing) {
		sellShort(label, 0, timing, 0);
	}

	/**
	 * <p>指定された注文方法で空売り注文を発行します。</p>
	 * 
	 * @param label ラベル
	 * @param timing 注文方法の種類
	 * @param price 指定価格
	 */
	public final void sellShort(final String label, final OrderTiming timing, final double price) {
		sellShort(label, 0, timing, price);
	}

	/**
	 * <p>指定された注文方法で空売り注文を発行します。</p>
	 * 
	 * @param label ラベル
	 * @param quantity 数量
	 * @param timing 注文方法の種類
	 */
	public final void sellShort(final String label, final int quantity, final OrderTiming timing) {
		sellShort(label, quantity, timing, 0);
	}

	/**
	 * <p>指定された注文方法で空売り注文を発行します。</p>
	 * 
	 * @param label ラベル
	 * @param quantity 数量
	 * @param timing 注文方法の種類
	 * @param price 指定価格
	 */
	public final void sellShort(final String label, final int quantity, final OrderTiming timing, final double price) {
		sellShort(createOrder(label, quantity, timing, price, null, null));
	}

	// ----------------------------------------------------------------------
	// sell

	/**
	 * <p>デフォルトの注文方法で空り注文を発行します。</p>
	 */
	public final void sell() {
		sell(null, 0, getDefaultOrderTiming(), 0, null, null);
	}

	/**
	 * <p>デフォルトの注文方法で空り注文を発行します。</p>
	 * 
	 * @param price 指定価格
	 */
	public final void sell(final double price) {
		sell(null, 0, getDefaultOrderTiming(), price, null, null);
	}

	/**
	 * <p>デフォルトの注文方法で空り注文を発行します。</p>
	 * 
	 * @param quantity 数量
	 */
	public final void sell(final int quantity) {
		sell(null, quantity, getDefaultOrderTiming(), 0, null, null);
	}

	/**
	 * <p>デフォルトの注文方法で空り注文を発行します。</p>
	 * 
	 * @param price 指定価格
	 * @param quantity 数量
	 */
	public final void sell(final int quantity, final double price) {
		sell(null, quantity, getDefaultOrderTiming(), price, null, null);
	}

	// ----------------------------------------------------------------------
	// sell (label)

	/**
	 * <p>デフォルトの注文方法で空り注文を発行します。</p>
	 * 
	 * @param label ラベル
	 */
	public final void sell(final String label) {
		sell(label, 0, getDefaultOrderTiming(), 0, null, null);
	}

	/**
	 * <p>デフォルトの注文方法で空り注文を発行します。</p>
	 * 
	 * @param label ラベル
	 * @param price 指定価格
	 */
	public final void sell(final String label, final double price) {
		sell(label, 0, getDefaultOrderTiming(), price, null, null);
	}

	/**
	 * <p>デフォルトの注文方法で空り注文を発行します。</p>
	 * 
	 * @param label ラベル
	 * @param quantity 数量
	 */
	public final void sell(final String label, final int quantity) {
		sell(label, quantity, getDefaultOrderTiming(), 0, null, null);
	}

	/**
	 * <p>デフォルトの注文方法で空り注文を発行します。</p>
	 * 
	 * @param label ラベル
	 * @param price 指定価格
	 * @param quantity 数量
	 */
	public final void sell(final String label, final int quantity, final double price) {
		sell(label, quantity, getDefaultOrderTiming(), price, null, null);
	}

	// ----------------------------------------------------------------------
	// sell (label, find)

	/**
	 * <p>デフォルトの注文方法で空り注文を発行します。</p>
	 * 
	 * @param label ラベル
	 * @param findLabel 決済注文ラベル
	 */
	public final void sell(final String label, final String findLabel) {
		sell(label, 0, getDefaultOrderTiming(), 0, null, findLabel);
	}

	/**
	 * <p>デフォルトの注文方法で空り注文を発行します。</p>
	 * 
	 * @param label ラベル
	 * @param price 指定価格
	 * @param findLabel 決済注文ラベル
	 */
	public final void sell(final String label, final double price, final String findLabel) {
		sell(label, 0, getDefaultOrderTiming(), price, null, findLabel);
	}

	/**
	 * <p>デフォルトの注文方法で空り注文を発行します。</p>
	 * 
	 * @param label ラベル
	 * @param quantity 数量
	 * @param findLabel 決済注文ラベル
	 */
	public final void sell(final String label, final int quantity, final String findLabel) {
		sell(label, quantity, getDefaultOrderTiming(), 0, null, findLabel);
	}

	/**
	 * <p>デフォルトの注文方法で空り注文を発行します。</p>
	 * 
	 * @param label ラベル
	 * @param quantity 数量
	 * @param price 指定価格
	 * @param findLabel 決済注文ラベル
	 */
	public final void sell(final String label, final int quantity, final double price, final String findLabel) {
		sell(label, quantity, getDefaultOrderTiming(), price, null, findLabel);
	}

	// ----------------------------------------------------------------------
	// sell (timing)

	/**
	 * <p>指定された注文方法で空り注文を発行します。</p>
	 * 
	 * @param timing 注文方法の種類
	 */
	public final void sell(final OrderTiming timing) {
		sell(null, 0, timing, 0, null, null);
	}

	/**
	 * <p>指定された注文方法で空り注文を発行します。</p>
	 * 
	 * @param timing 注文方法の種類
	 * @param price 指定価格
	 */
	public final void sell(final OrderTiming timing, final double price) {
		sell(null, 0, timing, price, null, null);
	}

	/**
	 * <p>指定された注文方法で空り注文を発行します。</p>
	 * 
	 * @param quantity 数量
	 * @param timing 注文方法の種類
	 */
	public final void sell(final int quantity, final OrderTiming timing) {
		sell(null, quantity, timing, 0, null, null);
	}

	/**
	 * <p>指定された注文方法で空り注文を発行します。</p>
	 * 
	 * @param quantity 数量
	 * @param timing 注文方法の種類
	 * @param price 指定価格
	 */
	public final void sell(final int quantity, final OrderTiming timing, final double price) {
		sell(null, quantity, timing, price, null, null);
	}

	// ----------------------------------------------------------------------
	// sell (label, timing)

	/**
	 * <p>指定された注文方法で空り注文を発行します。</p>
	 * 
	 * @param label ラベル
	 * @param timing 注文方法の種類
	 */
	public final void sell(final String label, final OrderTiming timing) {
		sell(label, 0, timing, 0, null, null);
	}

	/**
	 * <p>指定された注文方法で空り注文を発行します。</p>
	 * 
	 * @param label ラベル
	 * @param timing 注文方法の種類
	 * @param price 指定価格
	 */
	public final void sell(final String label, final OrderTiming timing, final double price) {
		sell(label, 0, timing, price, null, null);
	}

	/**
	 * <p>指定された注文方法で空り注文を発行します。</p>
	 * 
	 * @param label ラベル
	 * @param quantity 数量
	 * @param timing 注文方法の種類
	 */
	public final void sell(final String label, final int quantity, final OrderTiming timing) {
		sell(label, quantity, timing, 0, null, null);
	}

	/**
	 * <p>指定された注文方法で空り注文を発行します。</p>
	 * 
	 * @param label ラベル
	 * @param quantity 数量
	 * @param timing 注文方法の種類
	 * @param price 指定価格
	 */
	public final void sell(final String label, final int quantity, final OrderTiming timing, final double price) {
		sell(label, quantity, timing, price, null, null);
	}

	// ----------------------------------------------------------------------
	// sell (label, timing, find)

	/**
	 * <p>指定された注文方法で空り注文を発行します。</p>
	 * 
	 * @param label ラベル
	 * @param timing 注文方法の種類
	 * @param findLabel 決済注文ラベル
	 */
	public final void sell(final String label, final OrderTiming timing, final String findLabel) {
		sell(label, 0, timing, 0, null, findLabel);
	}

	/**
	 * <p>指定された注文方法で空り注文を発行します。</p>
	 * 
	 * @param label ラベル
	 * @param timing 注文方法の種類
	 * @param price 指定価格
	 * @param findLabel 決済注文ラベル
	 */
	public final void sell(final String label, final OrderTiming timing, final double price, final String findLabel) {
		sell(label, 0, timing, price, null, findLabel);
	}

	/**
	 * <p>指定された注文方法で空り注文を発行します。</p>
	 * 
	 * @param label ラベル
	 * @param quantity 数量
	 * @param timing 注文方法の種類
	 * @param findLabel 決済注文ラベル
	 */
	public final void sell(final String label, final int quantity, final OrderTiming timing, final String findLabel) {
		sell(label, quantity, timing, 0, null, findLabel);
	}

	/**
	 * <p>指定された注文方法で空り注文を発行します。</p>
	 * 
	 * @param label ラベル
	 * @param quantity 数量
	 * @param timing 注文方法の種類
	 * @param price 指定価格
	 * @param findLabel 決済注文ラベル
	 */
	public final void sell(final String label, final int quantity, final OrderTiming timing, final double price, final String findLabel) {
		sell(label, quantity, timing, price, null, findLabel);
	}

	// ----------------------------------------------------------------------
	// sell

	private final void sell(final String label, final int quantity, final OrderTiming timing, final double price, final Integer findId, final String findLabel) {
		sell(createOrder(label, quantity, timing, price, findId, findLabel));
	}

	// ----------------------------------------------------------------------
	// buy to cover

	/**
	 * <p>デフォルトの注文方法で買戻し注文を発行します。</p>
	 */
	public final void buyToCover() {
		buyToCover(null, 0, getDefaultOrderTiming(), 0, null, null);
	}

	/**
	 * <p>デフォルトの注文方法で買戻し注文を発行します。</p>
	 * 
	 * @param price 指定価格
	 */
	public final void buyToCover(final double price) {
		buyToCover(null, 0, getDefaultOrderTiming(), price, null, null);
	}

	/**
	 * <p>デフォルトの注文方法で買戻し注文を発行します。</p>
	 * 
	 * @param quantity 数量
	 */
	public final void buyToCover(final int quantity) {
		buyToCover(null, quantity, getDefaultOrderTiming(), 0, null, null);
	}

	/**
	 * <p>デフォルトの注文方法で買戻し注文を発行します。</p>
	 * 
	 * @param price 指定価格
	 * @param quantity 数量
	 */
	public final void buyToCover(final int quantity, final double price) {
		buyToCover(null, quantity, getDefaultOrderTiming(), price, null, null);
	}

	// ----------------------------------------------------------------------
	// buy to cover (label)

	/**
	 * <p>デフォルトの注文方法で買戻し注文を発行します。</p>
	 * 
	 * @param label ラベル
	 */
	public final void buyToCover(final String label) {
		buyToCover(label, 0, getDefaultOrderTiming(), 0, null, null);
	}

	/**
	 * <p>デフォルトの注文方法で買戻し注文を発行します。</p>
	 * 
	 * @param label ラベル
	 * @param price 指定価格
	 */
	public final void buyToCover(final String label, final double price) {
		buyToCover(label, 0, getDefaultOrderTiming(), price, null, null);
	}

	/**
	 * <p>デフォルトの注文方法で買戻し注文を発行します。</p>
	 * 
	 * @param label ラベル
	 * @param quantity 数量
	 */
	public final void buyToCover(final String label, final int quantity) {
		buyToCover(label, quantity, getDefaultOrderTiming(), 0, null, null);
	}

	/**
	 * <p>デフォルトの注文方法で買戻し注文を発行します。</p>
	 * 
	 * @param label ラベル
	 * @param price 指定価格
	 * @param quantity 数量
	 */
	public final void buyToCover(final String label, final int quantity, final double price) {
		buyToCover(label, quantity, getDefaultOrderTiming(), price, null, null);
	}

	// ----------------------------------------------------------------------
	// buy to cover (label, find)

	/**
	 * <p>デフォルトの注文方法で買戻し注文を発行します。</p>
	 * 
	 * @param label ラベル
	 * @param findLabel 決済注文ラベル
	 */
	public final void buyToCover(final String label, final String findLabel) {
		buyToCover(label, 0, getDefaultOrderTiming(), 0, null, findLabel);
	}

	/**
	 * <p>デフォルトの注文方法で買戻し注文を発行します。</p>
	 * 
	 * @param label ラベル
	 * @param price 指定価格
	 * @param findLabel 決済注文ラベル
	 */
	public final void buyToCover(final String label, final double price, final String findLabel) {
		buyToCover(label, 0, getDefaultOrderTiming(), price, null, findLabel);
	}

	/**
	 * <p>デフォルトの注文方法で買戻し注文を発行します。</p>
	 * 
	 * @param label ラベル
	 * @param quantity 数量
	 * @param findLabel 決済注文ラベル
	 */
	public final void buyToCover(final String label, final int quantity, final String findLabel) {
		buyToCover(label, quantity, getDefaultOrderTiming(), 0, null, findLabel);
	}

	/**
	 * <p>デフォルトの注文方法で買戻し注文を発行します。</p>
	 * 
	 * @param label ラベル
	 * @param quantity 数量
	 * @param price 指定価格
	 * @param findLabel 決済注文ラベル
	 */
	public final void buyToCover(final String label, final int quantity, final double price, final String findLabel) {
		buyToCover(label, quantity, getDefaultOrderTiming(), price, null, findLabel);
	}

	// ----------------------------------------------------------------------
	// buy to cover (timing)

	/**
	 * <p>指定された注文方法で買戻し注文を発行します。</p>
	 * 
	 * @param timing 注文方法の種類
	 */
	public final void buyToCover(final OrderTiming timing) {
		buyToCover(null, 0, timing, 0, null, null);
	}

	/**
	 * <p>指定された注文方法で買戻し注文を発行します。</p>
	 * 
	 * @param timing 注文方法の種類
	 * @param price 指定価格
	 */
	public final void buyToCover(final OrderTiming timing, final double price) {
		buyToCover(null, 0, timing, price, null, null);
	}

	/**
	 * <p>指定された注文方法で買戻し注文を発行します。</p>
	 * 
	 * @param quantity 数量
	 * @param timing 注文方法の種類
	 */
	public final void buyToCover(final int quantity, final OrderTiming timing) {
		buyToCover(null, quantity, timing, 0, null, null);
	}

	/**
	 * <p>指定された注文方法で買戻し注文を発行します。</p>
	 * 
	 * @param quantity 数量
	 * @param timing 注文方法の種類
	 * @param price 指定価格
	 */
	public final void buyToCover(final int quantity, final OrderTiming timing, final double price) {
		buyToCover(null, quantity, timing, price, null, null);
	}

	// ----------------------------------------------------------------------
	// buy to cover (label, timing)

	/**
	 * <p>指定された注文方法で買戻し注文を発行します。</p>
	 * 
	 * @param label ラベル
	 * @param timing 注文方法の種類
	 */
	public final void buyToCover(final String label, final OrderTiming timing) {
		buyToCover(label, 0, timing, 0, null, null);
	}

	/**
	 * <p>指定された注文方法で買戻し注文を発行します。</p>
	 * 
	 * @param label ラベル
	 * @param timing 注文方法の種類
	 * @param price 指定価格
	 */
	public final void buyToCover(final String label, final OrderTiming timing, final double price) {
		buyToCover(label, 0, timing, price, null, null);
	}

	/**
	 * <p>指定された注文方法で買戻し注文を発行します。</p>
	 * 
	 * @param label ラベル
	 * @param quantity 数量
	 * @param timing 注文方法の種類
	 */
	public final void buyToCover(final String label, final int quantity, final OrderTiming timing) {
		buyToCover(label, quantity, timing, 0, null, null);
	}

	/**
	 * <p>指定された注文方法で買戻し注文を発行します。</p>
	 * 
	 * @param label ラベル
	 * @param quantity 数量
	 * @param timing 注文方法の種類
	 * @param price 指定価格
	 */
	public final void buyToCover(final String label, final int quantity, final OrderTiming timing, final double price) {
		buyToCover(label, quantity, timing, price, null, null);
	}

	// ----------------------------------------------------------------------
	// buy to cover (label, timing, find)

	/**
	 * <p>指定された注文方法で買戻し注文を発行します。</p>
	 * 
	 * @param label ラベル
	 * @param timing 注文方法の種類
	 * @param findLabel 決済注文ラベル
	 */
	public final void buyToCover(final String label, final OrderTiming timing, final String findLabel) {
		buyToCover(label, 0, timing, 0, null, findLabel);
	}

	/**
	 * <p>指定された注文方法で買戻し注文を発行します。</p>
	 * 
	 * @param label ラベル
	 * @param timing 注文方法の種類
	 * @param price 指定価格
	 * @param findLabel 決済注文ラベル
	 */
	public final void buyToCover(final String label, final OrderTiming timing, final double price, final String findLabel) {
		buyToCover(label, 0, timing, price, null, findLabel);
	}

	/**
	 * <p>指定された注文方法で買戻し注文を発行します。</p>
	 * 
	 * @param label ラベル
	 * @param quantity 数量
	 * @param timing 注文方法の種類
	 * @param findLabel 決済注文ラベル
	 */
	public final void buyToCover(final String label, final int quantity, final OrderTiming timing, final String findLabel) {
		buyToCover(label, quantity, timing, 0, null, findLabel);
	}

	/**
	 * <p>指定された注文方法で買戻し注文を発行します。</p>
	 * 
	 * @param label ラベル
	 * @param quantity 数量
	 * @param timing 注文方法の種類
	 * @param price 指定価格
	 * @param findLabel 決済注文ラベル
	 */
	public final void buyToCover(final String label, final int quantity, final OrderTiming timing, final double price, final String findLabel) {
		buyToCover(label, quantity, timing, price, null, findLabel);
	}

	// ----------------------------------------------------------------------
	// buy to cover

	private final void buyToCover(final String label, final int quantity, final OrderTiming timing, final double price, final Integer findId, final String findLabel) {
		buyToCover(createOrder(label, quantity, timing, price, findId, findLabel));
	}

}
