/*
 * Copyright (c) 2006-2009 OrangeSignal.com All rights reserved.
 */

package jp.sourceforge.orangesignal.trading;

/**
 * ポジション状態の種類を表す列挙型を提供します。
 * 
 * @author 杉澤 浩二
 */
public enum MarketPositionType {

	/**
	 * フラットポジションである事を表します。
	 */
	FLAT {
		@Override public boolean isFlat() { return true; }
		@Override public boolean isLong() { return false; }
		@Override public boolean isShort() { return false; }
	},

	/**
	 * 買いポジションである事を表します。
	 */
	LONG {
		@Override public boolean isFlat() { return false; }
		@Override public boolean isLong() { return true; }
		@Override public boolean isShort() { return false; }
	},

	/**
	 * 売りポジションである事を表します。
	 */
	SHORT {
		@Override public boolean isFlat() { return false; }
		@Override public boolean isLong() { return false; }
		@Override public boolean isShort() { return true; }
	},

	/**
	 * ヘッジ(両建て)ポジションである事を表します。
	 * 
	 * @deprecated 現在サポートしていません。
	 */
	HEDGE {
		@Override public boolean isFlat() { return false; }
		@Override public boolean isLong() { return true; }
		@Override public boolean isShort() { return true; }
	};

	/**
	 * このポジションの種類がポジションを含んでいないかどうかを返します。
	 * 
	 * @return ポジションを含んでいない場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public abstract boolean isFlat();

	/**
	 * このポジションの種類が買いポジションを含んでいるかどうかを返します。
	 * 
	 * @return 買いポジションを含んでいる場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public abstract boolean isLong();

	/**
	 * このポジションの種類が売りポジションを含んでいるかどうかを返します。
	 * 
	 * @return 売りポジションを含んでいる場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	public abstract boolean isShort();

	/**
	 * 指定されたパラメーターからポジションの種類を返します。
	 * 
	 * @param l 買いポジションを含んでいるかどうか
	 * @param s 売りポジションを含んでいるかどうか
	 * @return ポジションの種類
	 */
	public static MarketPositionType valueOf(final boolean l, final boolean s) {
		if (l && s)
			return HEDGE;
		if (l)
			return LONG;
		if (s)
			return SHORT;
		return FLAT;
	}

}
