/*
 * Copyright (c) 2006-2008 OrangeSignal.com All rights reserved.
 */

package jp.sourceforge.orangesignal.trading.backtest.launcher;

import java.text.MessageFormat;
import java.util.Hashtable;
import java.util.Map;
import java.util.ResourceBundle;

import org.apache.commons.lang.ClassUtils;

/**
 * パッケージ単位の文字列リソース管理クラスを提供します。
 * 
 * @author <a href="mailto:sugisawa@orangesignal.com">杉澤 浩二</a>
 */
final class StringManager {

	/**
	 * 文字列リソースを保持します。
	 */
	private ResourceBundle bundle;

	/**
	 * インスタンス化できない事を強制します。
	 * 
	 * @param packageName パッケージ名
	 */
	private StringManager(final String packageName) {
		this.bundle = ResourceBundle.getBundle(packageName + ".LocalStrings");
	}

	/**
	 * アプリケーションリソースからキーに対するメッセージを取得して返します。
	 * 
	 * @param key キー
	 * @param obj 書式
	 * @return メッセージ
	 */
	public String getString(final String key, final Object... obj) {
		final String s = this.bundle.getString(key);
		if (obj != null && obj.length > 0)
			return new MessageFormat(s).format(obj);
		return s;
	}

	/**
	 * <code>StringManager</code> のマッピングリストを保持します。
	 */
	private static final Map<String, StringManager> managers = new Hashtable<String, StringManager>();

	/**
	 * 指定されたパッケージ単位の <code>StringManager</code> オブジェクトを返します。
	 * 
	 * @param cls パッケージ名
	 * @return <code>StringManager</code> オブジェクト
	 */
	public static StringManager getManager(final Class<?> cls) {
		return getManager(ClassUtils.getPackageName(cls));
	}

	/**
	 * 指定されたパッケージ単位の <code>StringManager</code> オブジェクトを返します。
	 * 
	 * @param packageName パッケージ名
	 * @return <code>StringManager</code> オブジェクト
	 */
	public synchronized static StringManager getManager(final String packageName) {
		StringManager manager = managers.get(packageName);
		if (manager == null) {
			manager = new StringManager(packageName);
			managers.put(packageName, manager);
		}
		return manager;
	}

}
