/*
 * Copyright (c) 2009 OrangeSignal.com All rights reserved.
 * 
 * これは Apache ライセンス Version 2.0 (以下、このライセンスと記述) に
 * 従っています。このライセンスに準拠する場合以外、このファイルを使用
 * してはなりません。このライセンスのコピーは以下から入手できます。
 * 
 * http://www.apache.org/licenses/LICENSE-2.0.txt
 * 
 * 適用可能な法律がある、あるいは文書によって明記されている場合を除き、
 * このライセンスの下で配布されているソフトウェアは、明示的であるか暗黙の
 * うちであるかを問わず、「保証やあらゆる種類の条件を含んでおらず」、
 * 「あるがまま」の状態で提供されるものとします。
 * このライセンスが適用される特定の許諾と制限については、このライセンス
 * を参照してください。
 */

package jp.sf.orangesignal.ta.data;

import static org.hamcrest.core.Is.is;
import static org.junit.Assert.assertThat;

import java.text.ParseException;
import java.text.SimpleDateFormat;

import jp.sf.orangesignal.ta.data.annotation.AnnotationParser;
import jp.sf.orangesignal.ta.data.model.Price;
import jp.sf.orangesignal.ta.data.model.PriceDataset;
import jp.sf.orangesignal.ta.data.sample.SimpleDataset;
import jp.sf.orangesignal.ta.data.sample.StringDataset;

import org.junit.BeforeClass;
import org.junit.Test;
import org.springframework.context.support.ClassPathXmlApplicationContext;

/**
 * {@link DatasetBuilder} クラスの単体テストです。
 * 
 * @author 杉澤 浩二
 */
public class DatasetBuilderTest {

	private static PriceDataset dataset;

	@BeforeClass
	public static void setUpBeforeClass() throws Exception {
		final CsvPriceLoader loader = (CsvPriceLoader) new ClassPathXmlApplicationContext("applicationContext.xml").getBean("loader", CsvPriceLoader.class);
		dataset = PriceDataset.newInstance(loader.load("n225").toArray(new Price[0]));
	}

	@Test
	public void testDatasetBinder() {
		// Act
		new DatasetBuilder<PriceDataset>(AnnotationParser.parse(dataset), new PriceDataset());
	}

	@Test
	public void testIncludes() {
		// Arrange
		final DatasetBuilder<PriceDataset> binder = new DatasetBuilder<PriceDataset>(AnnotationParser.parse(dataset), new PriceDataset());
		// Act
		binder.includes("date", "open", "high", "low", "close", "volume");
	}

	@Test(expected = IllegalArgumentException.class)
	public void testIncludesIllegalArgumentException() {
		// Arrange
		final DatasetBuilder<PriceDataset> binder = new DatasetBuilder<PriceDataset>(AnnotationParser.parse(dataset), new PriceDataset());
		binder.excludes("sold", "bought", "split");
		// Act
		binder.includes("date", "open", "high", "low", "close", "volume");
	}

	@Test
	public void testExcludes() {
		// Arrange
		final DatasetBuilder<PriceDataset> binder = new DatasetBuilder<PriceDataset>(AnnotationParser.parse(dataset), new PriceDataset());
		// Act
		binder.excludes("sold", "bought", "split");
	}

	@Test(expected = IllegalArgumentException.class)
	public void testExcludesIllegalArgumentException() {
		// Arrange
		final DatasetBuilder<PriceDataset> binder = new DatasetBuilder<PriceDataset>(AnnotationParser.parse(dataset), new PriceDataset());
		binder.includes("date", "open", "high", "low", "close", "volume");
		// Act
		binder.excludes("sold", "bought", "split");
	}

	@Test
	public void testExecute() throws ParseException {
		// Arrange
		final PriceDataset arrayDataset = new PriceDataset(); 
		// Act
		final PriceDataset r1 = new DatasetBuilder<PriceDataset>(AnnotationParser.parse(dataset), arrayDataset).execute();
		// Assert
		final SimpleDateFormat ymd = new SimpleDateFormat("yyyy/MM/dd");
		assertThat(r1, is(arrayDataset));
		assertThat(r1.getDate().length, is(2694));
		assertThat(r1.getDate()[0], is(ymd.parse("1998/01/05")));
		assertThat(r1.getOpen()[0].intValue(), is(15268));
		assertThat(r1.getHigh()[0].intValue(), is(15307));
		assertThat(r1.getLow()[0].intValue(), is(14956));
		assertThat(r1.getClose()[0].intValue(), is(14956));
		assertThat(r1.getVolume()[0].intValue(), is(156617100));
		assertThat(r1.getDate()[1], is(ymd.parse("1998/01/06")));
		assertThat(r1.getOpen()[1].intValue(), is(15007));
		assertThat(r1.getHigh()[1].intValue(), is(15066));
		assertThat(r1.getLow()[1].intValue(), is(14714));
		assertThat(r1.getClose()[1].intValue(), is(14896));
		assertThat(r1.getVolume()[1].intValue(), is(346720500));
		assertThat(r1.getDate()[2693], is(ymd.parse("2008/12/15")));
		assertThat(r1.getOpen()[2693].intValue(), is(8349));
		assertThat(r1.getHigh()[2693].intValue(), is(8700));
		assertThat(r1.getLow()[2693].intValue(), is(8349));
		assertThat(r1.getClose()[2693].intValue(), is(8664));
		assertThat(r1.getVolume()[2693].intValue(), is(1866440000));

		final StringDataset r2 = new DatasetBuilder<StringDataset>(AnnotationParser.parse(dataset), new StringDataset()).execute();
		assertThat(r2.date.length, is(2694));
//		assertThat(r2.date[0], is("Mon Jan 05 00:00:00 JST"));

		// Arrange
		final SimpleDataset simpleDataset = new SimpleDataset();
		// Act
		final SimpleDataset r3 = new DatasetBuilder<SimpleDataset>(AnnotationParser.parse(dataset), simpleDataset).execute();
		// Assert
		assertThat(r3, is(simpleDataset));
		assertThat(r3.getDate().size(), is(2694));
		assertThat(r3.getDate().get(0), is(ymd.parse("1998/01/05")));
		assertThat(r3.getOpen().get(0).intValue(), is(15268));
		assertThat(r3.getHigh().get(0).intValue(), is(15307));
		assertThat(r3.getLow().get(0).intValue(), is(14956));
		assertThat(r3.getClose().get(0).intValue(), is(14956));
		assertThat(r3.getVolume().get(0).intValue(), is(156617100));
		assertThat(r3.getDate().get(1), is(ymd.parse("1998/01/06")));
		assertThat(r3.getOpen().get(1).intValue(), is(15007));
		assertThat(r3.getHigh().get(1).intValue(), is(15066));
		assertThat(r3.getLow().get(1).intValue(), is(14714));
		assertThat(r3.getClose().get(1).intValue(), is(14896));
		assertThat(r3.getVolume().get(1).intValue(), is(346720500));
		assertThat(r3.getDate().get(2693), is(ymd.parse("2008/12/15")));
		assertThat(r3.getOpen().get(2693).intValue(), is(8349));
		assertThat(r3.getHigh().get(2693).intValue(), is(8700));
		assertThat(r3.getLow().get(2693).intValue(), is(8349));
		assertThat(r3.getClose().get(2693).intValue(), is(8664));
		assertThat(r3.getVolume().get(2693).intValue(), is(1866440000));
	}

}
