/*
 * Copyright (c) 2009 OrangeSignal.com All rights reserved.
 * 
 * これは Apache ライセンス Version 2.0 (以下、このライセンスと記述) に
 * 従っています。このライセンスに準拠する場合以外、このファイルを使用
 * してはなりません。このライセンスのコピーは以下から入手できます。
 * 
 * http://www.apache.org/licenses/LICENSE-2.0.txt
 * 
 * 適用可能な法律がある、あるいは文書によって明記されている場合を除き、
 * このライセンスの下で配布されているソフトウェアは、明示的であるか暗黙の
 * うちであるかを問わず、「保証やあらゆる種類の条件を含んでおらず」、
 * 「あるがまま」の状態で提供されるものとします。
 * このライセンスが適用される特定の許諾と制限については、このライセンス
 * を参照してください。
 */

package jp.sf.orangesignal.csv.handlers;

import java.io.IOException;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.SortedMap;
import java.util.TreeMap;

import jp.sf.orangesignal.csv.CsvReader;
import jp.sf.orangesignal.csv.CsvWriter;

/**
 * 区切り文字形式データの項目位置を基準として Java プログラム要素のリストと区切り文字形式データアクセスを行うハンドラを提供します。
 * 
 * @author 杉澤 浩二
 */
public class ColumnPositionMappingBeanListHandler<T> extends BeanListHandlerSupport<T> {

	/**
	 * 項目位置と Java プログラム要素のフィールド名のマップを保持します。
	 */
	private SortedMap<Integer, String> positionMap = new TreeMap<Integer, String>();

	// ------------------------------------------------------------------------

	/**
	 * コンストラクタです。
	 * 
	 * @param type Java プログラム要素の型
	 * @throws IllegalArgumentException <code>type</code> が <code>null</code> の場合
	 */
	public ColumnPositionMappingBeanListHandler(final Class<T> type) {
		super(type);
	}

	// ------------------------------------------------------------------------

	/**
	 * 指定された Java プログラム要素のフィールド名を現在の最後の項目位置としてマップへ追加します。
	 * 
	 * @param field Java プログラム要素のフィールド名
	 * @return このオブジェクトへの参照
	 */
	public ColumnPositionMappingBeanListHandler<T> addColumn(final String field) {
		return addColumn(getMaxColumnPosition() + 1, field);
	}

	/**
	 * 指定された項目位置と Java プログラム要素のフィールド名をマップへ追加します。
	 * 
	 * @param position 項目位置
	 * @param field Java プログラム要素のフィールド名
	 * @return このオブジェクトへの参照
	 */
	public ColumnPositionMappingBeanListHandler<T> addColumn(final int position, final String field) {
		this.positionMap.put(position, field);
		return this;
	}

	/**
	 * 項目位置と Java プログラム要素のフィールド名のマップを設定します。
	 * 
	 * @param columnMapping 項目位置と Java プログラム要素のフィールド名のマップ
	 * @return このオブジェクトへの参照
	 * @throws IllegalArgumentException <code>columnMapping</code> が <code>null</code> の場合
	 */
	public ColumnPositionMappingBeanListHandler<T> columnMapping(final Map<Integer, String> columnMapping) {
		if (columnMapping == null) {
			throw new IllegalArgumentException("Column mapping must not be null");
		}
		this.positionMap = new TreeMap<Integer, String>(columnMapping);
		return this;
	}

	// ------------------------------------------------------------------------

	private String getFieldName(final int columnPosition) {
		return positionMap.get(columnPosition);
	}

	private int getColumnPosition(final String fieldName) {
		for (final Map.Entry<Integer, String> e : positionMap.entrySet()) {
			if (fieldName.equals(e.getValue())) {
				return e.getKey();
			}
		}
		return -1;
	}

	private int getMaxColumnPosition() {
		if (positionMap.size() > 0) {
			return positionMap.lastKey();
		}
		return -1;
	}

	// ------------------------------------------------------------------------

	@Override
	public List<T> load(final CsvReader reader) throws IOException {
		// TODO: 必要があればここに入力パラメータを検証するコードを記述します。

		// 項目位置とフィールド名のマップが指定されていない場合は、最初の行をヘッダとして読込んでマップを作成します。
		if (positionMap.size() == 0) {
			final List<String> columnNames = reader.readValues();
			if (columnNames == null) {
				// ヘッダがない場合は例外をスローします。
				throw new IOException("No header is available");
			}
			for (final String name : columnNames) {
				addColumn(name);
			}
		}

		// データ部を処理します。
		final List<T> results = new ArrayList<T>();
		List<String> values;
		while ((values = reader.readValues()) != null) {
			final T obj = createBean();
			final int len = values.size();
			for (int pos = 0; pos < len; pos++) {
				final String fieldName = getFieldName(pos);
				if (fieldName == null) {
					continue;
				}
				final Field f = getField(getType(), fieldName);
				final Object o = getConverter().convert(values.get(pos), f.getType());
				if (o != null) {
					setFieldValue(obj, f, o);
				}
			}
			results.add(obj);
		}
		return results;
	}

	@Override
	public void save(final List<T> list, final CsvWriter writer) throws IOException {
		// TODO: 必要があればここに入力パラメータを検証するコードを記述します。

		// 項目位置とフィールド名のマップが指定されていない場合は、フィールドからマップを作成します。
		if (positionMap.size() == 0) {
			for (final Field f : getType().getDeclaredFields()) {
				addColumn(f.getName());
			}
		}
		final int columnCount = getMaxColumnPosition() + 1;

		// ヘッダ部を処理します。
		String[] names = new String[columnCount];
		for (final Map.Entry<Integer, String> e : positionMap.entrySet()) {
			names[e.getKey().intValue()] = e.getValue();
		}
		writer.writeValues(Arrays.asList(names));	// XXX - ヘッダの出力可否を選択できたら嬉しいかもしれない

		// データ部を処理します。
		for (final T obj : list) {
			// 要素が null の場合は null 出力します。
			if (obj == null) {
				writer.writeValues(null);
				continue;
			}

			final String[] values = new String[columnCount];
			for (final Field f : obj.getClass().getDeclaredFields()) {
				final int pos = getColumnPosition(f.getName());
				if (pos == -1) {
					continue;
				}
				values[pos] = getConverter().convert(getFieldValue(obj, f));
			}
			writer.writeValues(Arrays.asList(values));
		}
	}

}
