/*
 * Copyright (c) 2009 OrangeSignal.com All rights reserved.
 *
 * これは Apache ライセンス Version 2.0 (以下、このライセンスと記述) に
 * 従っています。このライセンスに準拠する場合以外、このファイルを使用
 * してはなりません。このライセンスのコピーは以下から入手できます。
 *
 * http://www.apache.org/licenses/LICENSE-2.0.txt
 *
 * 適用可能な法律がある、あるいは文書によって明記されている場合を除き、
 * このライセンスの下で配布されているソフトウェアは、明示的であるか暗黙の
 * うちであるかを問わず、「保証やあらゆる種類の条件を含んでおらず」、
 * 「あるがまま」の状態で提供されるものとします。
 * このライセンスが適用される特定の許諾と制限については、このライセンス
 * を参照してください。
 */

package jp.sf.orangesignal.csv.handlers;

import static org.hamcrest.core.Is.is;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertThat;

import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import jp.sf.orangesignal.csv.CsvConfig;
import jp.sf.orangesignal.csv.CsvReader;
import jp.sf.orangesignal.csv.CsvWriter;

import org.junit.BeforeClass;
import org.junit.Test;

/**
 * {@link ColumnNameMappingBeanListHandler} クラスの単体テストです。
 *
 * @author 杉澤 浩二
 */
public class ColumnNameMappingBeanListHandlerTest {

	private static CsvConfig cfg;

	@BeforeClass
	public static void setUpBeforeClass() throws Exception {
		cfg = new CsvConfig(',');
		cfg.setNullString("NULL");
		cfg.setIgnoreTrailingWhitespaces(true);
		cfg.setIgnoreLeadingWhitespaces(true);
		cfg.setIgnoreEmptyLines(true);
	}

	@Test(expected = IllegalArgumentException.class)
	public void testColumnNameMappingBeanListHandlerIllegalArgumentException() {
		new ColumnNameMappingBeanListHandler<SampleBean>(null);
	}

	@Test
	public void testConverter() {
		final CsvValueConverter converter = new NoopCsvValueConverter();
		assertThat(new ColumnNameMappingBeanListHandler<SampleBean>(SampleBean.class).converter(converter).getConverter(), is(converter));
	}

	@Test
	public void testGetType() {
		assertThat(new ColumnNameMappingBeanListHandler<SampleBean>(SampleBean.class).getType().getName(), is(SampleBean.class.getName()));
	}

	@Test
	public void testCreateBean() throws IOException {
		assertThat(new ColumnNameMappingBeanListHandler<SampleBean>(SampleBean.class).createBean().getClass().getName(), is(SampleBean.class.getName()));
	}

	@Test
	public void testLoad1() throws IOException {
		final CsvReader reader = new CsvReader(new StringReader("symbol,name,price,volume\r\nAAAA,aaa,10000,10\r\nBBBB,bbb,NULL,0"), cfg);
		try {
			final List<SampleBean> list = new ColumnNameMappingBeanListHandler<SampleBean>(SampleBean.class).load(reader);
			assertThat(list.size(), is(2));
			final SampleBean o1 = list.get(0);
			assertThat(o1.symbol, is("AAAA"));
			assertThat(o1.name, is("aaa"));
			assertThat(o1.price.longValue(), is(10000L));
			assertThat(o1.volume.longValue(), is(10L));
			final SampleBean o2 = list.get(1);
			assertThat(o2.symbol, is("BBBB"));
			assertThat(o2.name, is("bbb"));
			assertNull(o2.price);
			assertThat(o2.volume.longValue(), is(0L));
		} finally {
			reader.close();
		}
	}

	@Test
	public void testLoad2() throws IOException {
		final CsvReader reader = new CsvReader(new StringReader("シンボル,名称,価格,出来高\r\nAAAA,aaa,10000,10\r\nBBBB,bbb,NULL,0"), cfg);
		try {
			final List<SampleBean> list = new ColumnNameMappingBeanListHandler<SampleBean>(SampleBean.class)
				.addColumn("シンボル", "symbol")
				.addColumn("名称",     "name")
				.addColumn("価格",     "price")
				.addColumn("出来高",   "volume")
				.load(reader);

			assertThat(list.size(), is(2));
			final SampleBean o1 = list.get(0);
			assertThat(o1.symbol, is("AAAA"));
			assertThat(o1.name, is("aaa"));
			assertThat(o1.price.longValue(), is(10000L));
			assertThat(o1.volume.longValue(), is(10L));
			final SampleBean o2 = list.get(1);
			assertThat(o2.symbol, is("BBBB"));
			assertThat(o2.name, is("bbb"));
			assertNull(o2.price);
			assertThat(o2.volume.longValue(), is(0L));
		} finally {
			reader.close();
		}
	}

	@Test
	public void testLoad3() throws IOException {
		final CsvReader reader = new CsvReader(new StringReader("シンボル,名称,価格,出来高\r\nAAAA,aaa,10000,10\r\nBBBB,bbb,NULL,0"), cfg);
		try {
			final List<SampleBean> list = new ColumnNameMappingBeanListHandler<SampleBean>(SampleBean.class)
				.addColumn("シンボル", "symbol")
				.addColumn("価格",     "price")
				.addColumn("出来高",   "volume")
				.load(reader);

			assertThat(list.size(), is(2));
			final SampleBean o1 = list.get(0);
			assertThat(o1.symbol, is("AAAA"));
			assertNull(o1.name);
			assertThat(o1.price.longValue(), is(10000L));
			assertThat(o1.volume.longValue(), is(10L));
			final SampleBean o2 = list.get(1);
			assertThat(o2.symbol, is("BBBB"));
			assertNull(o2.name);
			assertNull(o2.price);
			assertThat(o2.volume.longValue(), is(0L));
		} finally {
			reader.close();
		}
	}

	@Test
	public void testLoad4() throws IOException {
		final CsvReader reader = new CsvReader(new StringReader("シンボル,名称,価格,出来高\r\nAAAA,aaa,10000,10\r\nBBBB,bbb,NULL,0"), cfg);
		try {
			final Map<String, String> columnMapping = new HashMap<String, String>();
			columnMapping.put("シンボル", "symbol");
			columnMapping.put("名称",     "name");
			columnMapping.put("価格",     "price");
			columnMapping.put("出来高",   "volume");
			final List<SampleBean> list = new ColumnNameMappingBeanListHandler<SampleBean>(SampleBean.class).columnMapping(columnMapping).load(reader);

			assertThat(list.size(), is(2));
			final SampleBean o1 = list.get(0);
			assertThat(o1.symbol, is("AAAA"));
			assertThat(o1.name, is("aaa"));
			assertThat(o1.price.longValue(), is(10000L));
			assertThat(o1.volume.longValue(), is(10L));
			final SampleBean o2 = list.get(1);
			assertThat(o2.symbol, is("BBBB"));
			assertThat(o2.name, is("bbb"));
			assertNull(o2.price);
			assertThat(o2.volume.longValue(), is(0L));
		} finally {
			reader.close();
		}
	}

	@Test
	public void testLoad5() throws IOException {
		final CsvReader reader = new CsvReader(new StringReader("シンボル,名称,価格,出来高\r\nAAAA,aaa,10000,10\r\nBBBB,bbb,NULL,0"), cfg);
		try {
			final Map<String, String> columnMapping = new HashMap<String, String>();
			columnMapping.put("シンボル", "symbol");
			columnMapping.put("価格",     "price");
			columnMapping.put("出来高",   "volume");
			final List<SampleBean> list = new ColumnNameMappingBeanListHandler<SampleBean>(SampleBean.class).columnMapping(columnMapping).load(reader);

			assertThat(list.size(), is(2));
			final SampleBean o1 = list.get(0);
			assertThat(o1.symbol, is("AAAA"));
			assertNull(o1.name);
			assertThat(o1.price.longValue(), is(10000L));
			assertThat(o1.volume.longValue(), is(10L));
			final SampleBean o2 = list.get(1);
			assertThat(o2.symbol, is("BBBB"));
			assertNull(o2.name);
			assertNull(o2.price);
			assertThat(o2.volume.longValue(), is(0L));
		} finally {
			reader.close();
		}
	}

	@Test
	public void testSave1() throws IOException {
		final List<SampleBean> list = new ArrayList<SampleBean>();
		list.add(new SampleBean("AAAA", "aaa", 10000, 10));
		list.add(new SampleBean("BBBB", "bbb", null, 0));

		final StringWriter sw = new StringWriter();
		final CsvWriter writer = new CsvWriter(sw, cfg);
		try {
			new ColumnNameMappingBeanListHandler<SampleBean>(SampleBean.class).save(list, writer);
		} finally {
			writer.close();
		}
		assertThat(sw.getBuffer().toString(), is("symbol,name,price,volume\r\nAAAA,aaa,10000,10\r\nBBBB,bbb,NULL,0\r\n"));
	}

	@Test
	public void testSave2() throws IOException {
		final List<SampleBean> list = new ArrayList<SampleBean>();
		list.add(new SampleBean("AAAA", "aaa", 10000, 10));
		list.add(new SampleBean("BBBB", "bbb", null, 0));

		final StringWriter sw = new StringWriter();
		final CsvWriter writer = new CsvWriter(sw, cfg);
		try {
			new ColumnNameMappingBeanListHandler<SampleBean>(SampleBean.class)
				.addColumn("シンボル", "symbol")
				.addColumn("名称",     "name")
				.addColumn("価格",     "price")
				.addColumn("出来高",   "volume")
				.save(list, writer);
		} finally {
			writer.close();
		}
		assertThat(sw.getBuffer().toString(), is("シンボル,名称,価格,出来高\r\nAAAA,aaa,10000,10\r\nBBBB,bbb,NULL,0\r\n"));
	}

	@Test
	public void testSave3() throws IOException {
		final List<SampleBean> list = new ArrayList<SampleBean>();
		list.add(new SampleBean("AAAA", "aaa", 10000, 10));
		list.add(new SampleBean("BBBB", "bbb", null, 0));

		final StringWriter sw = new StringWriter();
		final CsvWriter writer = new CsvWriter(sw, cfg);
		try {
			new ColumnNameMappingBeanListHandler<SampleBean>(SampleBean.class)
				.addColumn("シンボル", "symbol")
				.addColumn("価格",     "price")
				.addColumn("出来高",   "volume")
				.save(list, writer);
		} finally {
			writer.close();
		}
		assertThat(sw.getBuffer().toString(), is("シンボル,価格,出来高\r\nAAAA,10000,10\r\nBBBB,NULL,0\r\n"));
	}

	@Test
	public void testSave4() throws IOException {
		final List<SampleBean> list = new ArrayList<SampleBean>();
		list.add(new SampleBean("AAAA", "aaa", 10000, 10));
		list.add(new SampleBean("BBBB", "bbb", null, 0));

		final StringWriter sw = new StringWriter();
		final CsvWriter writer = new CsvWriter(sw, cfg);
		try {
			final Map<String, String> columnMapping = new LinkedHashMap<String, String>();
			columnMapping.put("シンボル", "symbol");
			columnMapping.put("名称",     "name");
			columnMapping.put("価格",     "price");
			columnMapping.put("出来高",   "volume");
			new ColumnNameMappingBeanListHandler<SampleBean>(SampleBean.class).columnMapping(columnMapping).save(list, writer);
		} finally {
			writer.close();
		}
		assertThat(sw.getBuffer().toString(), is("シンボル,名称,価格,出来高\r\nAAAA,aaa,10000,10\r\nBBBB,bbb,NULL,0\r\n"));
	}

	@Test
	public void testSave5() throws IOException {
		final List<SampleBean> list = new ArrayList<SampleBean>();
		list.add(new SampleBean("AAAA", "aaa", 10000, 10));
		list.add(new SampleBean("BBBB", "bbb", null, 0));

		final StringWriter sw = new StringWriter();
		final CsvWriter writer = new CsvWriter(sw, cfg);
		try {
			final Map<String, String> columnMapping = new LinkedHashMap<String, String>();
			columnMapping.put("シンボル", "symbol");
			columnMapping.put("価格",     "price");
			columnMapping.put("出来高",   "volume");
			new ColumnNameMappingBeanListHandler<SampleBean>(SampleBean.class).columnMapping(columnMapping).save(list, writer);
		} finally {
			writer.close();
		}
		assertThat(sw.getBuffer().toString(), is("シンボル,価格,出来高\r\nAAAA,10000,10\r\nBBBB,NULL,0\r\n"));
	}

}
