/*
 * Copyright (c) 2009 OrangeSignal.com All rights reserved.
 *
 * これは Apache ライセンス Version 2.0 (以下、このライセンスと記述) に
 * 従っています。このライセンスに準拠する場合以外、このファイルを使用
 * してはなりません。このライセンスのコピーは以下から入手できます。
 *
 * http://www.apache.org/licenses/LICENSE-2.0.txt
 *
 * 適用可能な法律がある、あるいは文書によって明記されている場合を除き、
 * このライセンスの下で配布されているソフトウェアは、明示的であるか暗黙の
 * うちであるかを問わず、「保証やあらゆる種類の条件を含んでおらず」、
 * 「あるがまま」の状態で提供されるものとします。
 * このライセンスが適用される特定の許諾と制限については、このライセンス
 * を参照してください。
 */

package jp.sf.orangesignal.csv.manager;

import static org.hamcrest.core.Is.is;
import static org.junit.Assert.assertThat;

import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.text.DecimalFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.List;

import jp.sf.orangesignal.csv.CsvConfig;
import jp.sf.orangesignal.csv.handlers.SampleBean;

import org.junit.Test;

/**
 * {@link CsvColumnNameMappingBeanManager} クラスの単体テストです。
 *
 * @author 杉澤 浩二
 */
public class CsvColumnNameMappingBeanManagerTest {

	@Test
	public void testCsvColumnNameMappingBeanManager() {
		new CsvColumnNameMappingBeanManager();
	}

	@Test
	public void testCsvColumnNameMappingBeanManagerCsvConfig() {
		new CsvColumnNameMappingBeanManager(new CsvConfig());
	}

	@Test
	public void testConfig() {
		new CsvColumnNameMappingBeanManager().config(new CsvConfig());
	}

	@Test
	public void testLoad() throws IOException {
		final CsvConfig cfg = new CsvConfig();
		cfg.setEscapeDisabled(false);

		final List<SamplePrice> list
			= new CsvColumnNameMappingBeanManager()
				.config(cfg)
				.load(SamplePrice.class)
				.column("日付", "date", new SimpleDateFormat("yyyy/MM/dd"))
				.column("時刻", "date", new SimpleDateFormat("HH:mm:ss"))
				.column("名称", "name")
				.column("価格", "price", new DecimalFormat("#,##0.00"))
				.from(new StringReader("日付,時刻,名称,価格\r\n2009/01/01,12:34:56,商品A,10\\,005.89\r\n2009/01/02,10:20:30,商品B,2\\,000.05"));

		assertThat(list.size(), is(2));
		SamplePrice o1 = list.get(0);
		assertThat(new SimpleDateFormat("yyyy/MM/dd HH:mm:ss").format(o1.getDate()), is("2009/01/01 12:34:56"));
		assertThat(o1.getName(), is("商品A"));
		assertThat(o1.getPrice().toString(), is("10005.89"));
		SamplePrice o2 = list.get(1);
		assertThat(new SimpleDateFormat("yyyy/MM/dd HH:mm:ss").format(o2.getDate()), is("2009/01/02 10:20:30"));
		assertThat(o2.getName(), is("商品B"));
		assertThat(o2.getPrice().toString(), is("2000.05"));
	}

	@Test
	public void testSave() throws Exception {
		final List<SampleBean> list = new ArrayList<SampleBean>();
		list.add(new SampleBean("AAAA", "aaa", 10000, 10, new SimpleDateFormat("yyyy/MM/dd HH:mm:ss").parse("2008/10/28 10:24:00")));
		list.add(new SampleBean("BBBB", "bbb", null, 0, null));
		list.add(new SampleBean("CCCC", "ccc", 20000, 100, new SimpleDateFormat("yyyy/MM/dd HH:mm:ss").parse("2008/10/26 14:20:10")));

		final CsvConfig cfg = new CsvConfig();
		cfg.setEscapeDisabled(false);
		cfg.setNullString("NULL");

		final StringWriter sw = new StringWriter();
		new CsvColumnNameMappingBeanManager()
			.config(cfg)
			.save(list, SampleBean.class)
			.column("シンボル", "symbol")
			.column("名称",     "name")
			.column("価格",     "price", new DecimalFormat("#,##0"))
			.column("出来高",   "volume")
			.column("日付",     "date", new SimpleDateFormat("yyyy/MM/dd"))
			.column("時刻",     "date", new SimpleDateFormat("HH:mm:ss"))
			.to(sw);
		assertThat(sw.getBuffer().toString(), is("シンボル,名称,価格,出来高,日付,時刻\r\nAAAA,aaa,10\\,000,10,2008/10/28,10:24:00\r\nBBBB,bbb,NULL,0,NULL,NULL\r\nCCCC,ccc,20\\,000,100,2008/10/26,14:20:10\r\n"));
	}

}
