/*
 * Copyright (c) 2008-2009 OrangeSignal.com All rights reserved.
 * 
 * これは Apache ライセンス Version 2.0 (以下、このライセンスと記述) に
 * 従っています。このライセンスに準拠する場合以外、このファイルを使用
 * してはなりません。このライセンスのコピーは以下から入手できます。
 * 
 * http://www.apache.org/licenses/LICENSE-2.0.txt
 * 
 * 適用可能な法律がある、あるいは文書によって明記されている場合を除き、
 * このライセンスの下で配布されているソフトウェアは、明示的であるか暗黙の
 * うちであるかを問わず、「保証やあらゆる種類の条件を含んでおらず」、
 * 「あるがまま」の状態で提供されるものとします。
 * このライセンスが適用される特定の許諾と制限については、このライセンス
 * を参照してください。
 */

package jp.sf.orangesignal.ta.candle;

/**
 * <p>ローソク足パターン分析用ユーティリティクラスを提供します。</p>
 * 
 * @author 杉澤 浩二
 */
public class CandlestickPatterns {

	/**
	 * デフォルトコンストラクタです。
	 */
	protected CandlestickPatterns() {}

	/* ---------------------------------------------------------------------- */
	/* 一本足【指標】反転・継続【信頼性】低 */

	/**
	 * <p>指定されたローソク足データに大陽線が存在するかどうかを列挙して返します。</p>
	 * このパターンは一本足のパターンで、現トレンドの反転や継続の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に低いと言われています。<br>
	 * 以下を基準として判断します。
	 * <p>
	 * <ol>
	 * <li>長い陽線である。※但し影(ヒゲ)が長いものは除く。</li>
	 * </ol>
	 * </p>
	 * 尚、このパターンはトレンドを確認しません。
	 * 
	 * @param c ローソク足データ
	 * @return 大陽線が存在するかどうかを列挙
	 */
	public static boolean[] bullishLongWhiteDay(final Candlestick[] c) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 0; i < len; i++) {
			if (c[i] != null)
				results[i] = c[i].isLongWhite();
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに大陰線が存在するかどうかを列挙して返します。</p>
	 * このパターンは一本足のパターンで、現トレンドの反転や継続の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に低いと言われています。<br>
	 * 以下を基準として判断します。
	 * <p>
	 * <ol>
	 * <li>長い陰線である。※但し影(ヒゲ)が長いものは除く。</li>
	 * </ol>
	 * </p>
	 * 尚、このパターンはトレンドを確認しません。
	 * 
	 * @param c ローソク足データ
	 * @return 大陰線が存在するかどうかを列挙
	 */
	public static boolean[] bearishLongBlackDay(final Candlestick[] c) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 0; i < len; i++) {
			if (c[i] != null)
				results[i] = c[i].isLongBlack();
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに小陽線が存在するかどうかを列挙して返します。</p>
	 * 尚、このパターンはトレンドを確認しません。
	 * 
	 * @param c ローソク足データ
	 * @return 小陽線が存在するかどうかを列挙
	 */
	public static boolean[] bullishShortWhiteDay(final Candlestick[] c) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 0; i < len; i++) {
			if (c[i] != null)
				results[i] = c[i].isShortWhite();
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに小陰線が存在するかどうかを列挙して返します。</p>
	 * 尚、このパターンはトレンドを確認しません。
	 * 
	 * @param c ローソク足データ
	 * @return 小陰線が存在するかどうかを列挙
	 */
	public static boolean[] bearishShortBlackDay(final Candlestick[] c) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 0; i < len; i++) {
			if (c[i] != null)
				results[i] = c[i].isShortBlack();
		}
		return results;
	}

	/* ---------------------------------------------------------------------- */
	/* 一本足【指標】反転【信頼性】低 */

	/**
	 * <p>指定されたローソク足データに陽の寄付坊主(寄り切り線)が存在するかどうかを列挙して返します。</p>
	 * このパターンは一本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に低いと言われています。<br>
	 * 以下を基準として判断します。
	 * <p>
	 * <ol>
	 * <li>下降トレンドである。</li>
	 * <li>下にギャップ(窓)を作り、陽の寄付坊主を形成している。</li>
	 * </ol>
	 * </p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bullishBeltHold(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 陽の寄付坊主が存在するかどうかを列挙
	 * @see #bullishBeltHold(Candlestick[], boolean)
	 */
	public static boolean[] bullishBeltHold(final Candlestick[] c) { return bullishBeltHold(c, true); }

	/**
	 * <p>指定されたローソク足データに陽の寄付坊主(寄り切り線)が存在するかどうかを列挙して返します。</p>
	 * このパターンは一本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に低いと言われています。<br>
	 * 以下を基準として判断します。
	 * <p>
	 * <ol>
	 * <li>下降トレンドである。</li>
	 * <li>下にギャップ(窓)を作り、陽の寄付坊主を形成している。</li>
	 * </ol>
	 * </p>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 陽の寄付坊主が存在するかどうかを列挙
	 */
	public static boolean[] bullishBeltHold(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 1; i < len; i++) {
			final int _0 = i - 1, _1 = i;
			if (c[_0] == null || c[_1] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.DOWN) &&
				c[_0].gapDownBodyByBody(c[_1]) &&
				c[_1].getColor() == CandlestickColor.WHITE && c[_1].isOpeningMarubozu();// && c[_1].upperShadow() < c[_1].body();
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに陰の大引坊主(寄り切り線)が存在するかどうかを列挙して返します。</p>
	 * このパターンは一本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に低いと言われています。<br>
	 * 以下を基準として判断します。
	 * <p>
	 * <ol>
	 * <li>上昇トレンドである。</li>
	 * <li>上にギャップ(窓)を作り、陰の大引坊主を形成している。</li>
	 * </ol>
	 * </p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bearishBeltHold(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 陰の大引坊主が存在するかどうかを列挙
	 * @see #bearishBeltHold(Candlestick[], boolean)
	 */
	public static boolean[] bearishBeltHold(final Candlestick[] c) { return bearishBeltHold(c, true); }

	/**
	 * <p>指定されたローソク足データに陰の大引坊主(寄り切り線)が存在するかどうかを列挙して返します。</p>
	 * このパターンは一本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に低いと言われています。<br>
	 * 以下を基準として判断します。
	 * <p>
	 * <ol>
	 * <li>上昇トレンドである。</li>
	 * <li>上にギャップ(窓)を作り、陰の大引坊主を形成している。</li>
	 * </ol>
	 * </p>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 陰の大引坊主が存在するかどうかを列挙
	 */
	public static boolean[] bearishBeltHold(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 1; i < len; i++) {
			final int _0 = i - 1, _1 = i;
			if (c[_0] == null || c[_1] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.UP) &&
				c[_0].gapUpBodyByBody(c[_1]) &&
				c[_1].getColor() == CandlestickColor.BLACK && c[_1].isClosingMarubozu();// && c[_1].getLow()erShadow() < c[_1].body();
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに陰のカラカサ線(たくり線)が存在するかどうかを列挙して返します。</p>
	 * このパターンは一本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * カラカサ線(たくり線)は、陰線より陽線である方が信頼性が高く、下影(下ヒゲ)が長い程反転の可能性が大きい事を示唆します。<br>
	 * このパターンの信頼性は一般的に低いと言われています。<br>
	 * 以下を基準として判断します。
	 * <p>
	 * <ol>
	 * <li>下降トレンドである。</li>
	 * <li>陽線(白)の唐傘(カラカサ)/首吊りを形成している。</li>
	 * </ol>
	 * </p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bullishHammer(Candlestick[], boolean, boolean)} をトレンドを確認する、陰陽線の種類を確認しないモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return カラカサ線(たくり線)が存在するかどうかを列挙
	 * @see #bullishHammer(Candlestick[], boolean, boolean)
	 */
	public static boolean[] bullishHammer(final Candlestick[] c) { return bullishHammer(c, true, false); }

	/**
	 * <p>指定されたローソク足データに陰のカラカサ線(たくり線)が存在するかどうかを列挙して返します。</p>
	 * このパターンは一本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * カラカサ線(たくり線)は、陰線より陽線である方が信頼性が高く、下影(下ヒゲ)が長い程反転の可能性が大きい事を示唆します。<br>
	 * このパターンの信頼性は一般的に低いと言われています。<br>
	 * 以下を基準として判断します。
	 * <p>
	 * <ol>
	 * <li>下降トレンドである。</li>
	 * <li>陽線(白)の唐傘(カラカサ)/首吊りを形成している。</li>
	 * </ol>
	 * </p>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @param color 陰陽線の種類を確認するかどうか
	 * @return カラカサ線(たくり線)が存在するかどうかを列挙
	 */
	public static boolean[] bullishHammer(final Candlestick[] c, final boolean trend, final boolean color) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 1; i < len; i++) {
			final int _0 = i - 1, _1 = i;
			if (c[_0] == null || c[_1] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.DOWN) &&
				(!color | c[_1].getColor() == CandlestickColor.WHITE) && c[_1].isHangingMan();
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに陰の首吊り線が存在するかどうかを列挙して返します。</p>
	 * このパターンは一本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * 首吊り線は、陽線より陰線である方が信頼性が高く、下影(下ヒゲ)が長い程反転の可能性が大きい事を示唆します。<br>
	 * このパターンの信頼性は一般的に低いと言われています。<br>
	 * 以下を基準として判断します。
	 * <p>
	 * <ol>
	 * <li>上昇トレンドである。</li>
	 * <li>陰線(黒)の唐傘(カラカサ)/首吊りを形成している。</li>
	 * </ol>
	 * </p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bearishHangingMan(Candlestick[], boolean, boolean)} をトレンドを確認する、陰陽線の種類を確認しないモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 首吊り線が存在するかどうかを列挙
	 * @see #bearishHangingMan(Candlestick[], boolean, boolean)
	 */
	public static boolean[] bearishHangingMan(final Candlestick[] c) { return bearishHangingMan(c, true, false); }

	/**
	 * <p>指定されたローソク足データに陰の首吊り線が存在するかどうかを列挙して返します。</p>
	 * このパターンは一本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * 首吊り線は、陽線より陰線である方が信頼性が高く、下影(下ヒゲ)が長い程反転の可能性が大きい事を示唆します。<br>
	 * このパターンの信頼性は一般的に低いと言われています。<br>
	 * 以下を基準として判断します。
	 * <p>
	 * <ol>
	 * <li>上昇トレンドである。</li>
	 * <li>陰線(黒)の唐傘(カラカサ)/首吊りを形成している。</li>
	 * </ol>
	 * </p>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @param color 陰陽線の種類を確認するかどうか
	 * @return 首吊り線が存在するかどうかを列挙
	 */
	public static boolean[] bearishHangingMan(final Candlestick[] c, final boolean trend, final boolean color) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 1; i < len; i++) {
			final int _0 = i - 1, _1 = i;
			if (c[_0] == null || c[_1] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.UP) &&
				(!color | c[_1].getColor() == CandlestickColor.BLACK) && c[_1].isHangingMan();
		}
		return results;
	}

	/* ---------------------------------------------------------------------- */
	/* 一本足【指標】反転【信頼性】中 */

	/**
	 * <p>指定されたローソク足データに強気の蜻蛉(トンボ)が存在するかどうかを列挙して返します。</p>
	 * このパターンは一本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * 以下を基準として判断します。
	 * <p>
	 * <ol>
	 * <li>下降トレンドである。</li>
	 * <li>下にギャップ(窓)を作り、蜻蛉(トンボ)を形成している。</li>
	 * </ol>
	 * </p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bullishDragonflyDoji(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 強気の蜻蛉(トンボ)が存在するかどうかを列挙
	 * @see #bullishDragonflyDoji(Candlestick[], boolean)
	 */
	public static boolean[] bullishDragonflyDoji(final Candlestick[] c) { return bullishDragonflyDoji(c, true); }

	/**
	 * <p>指定されたローソク足データに強気の蜻蛉(トンボ)が存在するかどうかを列挙して返します。</p>
	 * このパターンは一本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * 以下を基準として判断します。
	 * <p>
	 * <ol>
	 * <li>下降トレンドである。</li>
	 * <li>下にギャップ(窓)を作り、蜻蛉(トンボ)を形成している。</li>
	 * </ol>
	 * </p>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 強気の蜻蛉(トンボ)が存在するかどうかを列挙
	 */
	public static boolean[] bullishDragonflyDoji(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 1; i < len; i++) {
			final int _0 = i - 1, _1 = i;
			if (c[_0] == null || c[_1] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.DOWN) &&
				c[_0].gapDownBodyByBody(c[_1]) && c[_1].isDragonflyDoji();
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに弱気の蜻蛉(トンボ)が存在するかどうかを列挙して返します。</p>
	 * このパターンは一本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * 以下を基準として判断します。
	 * <p>
	 * <ol>
	 * <li>上昇トレンドである。</li>
	 * <li>上にギャップ(窓)を作り、蜻蛉(トンボ)を形成している。</li>
	 * </ol>
	 * </p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bearishDragonflyDoji(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 弱気の蜻蛉(トンボ)が存在するかどうかを列挙
	 * @see #bearishDragonflyDoji(Candlestick[], boolean)
	 */
	public static boolean[] bearishDragonflyDoji(final Candlestick[] c) { return bearishDragonflyDoji(c, true); }

	/**
	 * <p>指定されたローソク足データに弱気の蜻蛉(トンボ)が存在するかどうかを列挙して返します。</p>
	 * このパターンは一本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * 以下を基準として判断します。
	 * <p>
	 * <ol>
	 * <li>上昇トレンドである。</li>
	 * <li>上にギャップ(窓)を作り、蜻蛉(トンボ)を形成している。</li>
	 * </ol>
	 * </p>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 弱気の蜻蛉(トンボ)が存在するかどうかを列挙
	 */
	public static boolean[] bearishDragonflyDoji(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 1; i < len; i++) {
			final int _0 = i - 1, _1 = i;
			if (c[_0] == null || c[_1] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.UP) &&
				c[_0].gapDownBodyByBody(c[_1]) && c[_1].isDragonflyDoji();
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに強気の足長同時が存在するかどうかを列挙して返します。</p>
	 * このパターンは一本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * 以下を基準として判断します。
	 * <p>
	 * <ol>
	 * <li>下降トレンドである。</li>
	 * <li>下にギャップ(窓)を作り、足長同時を形成している。</li>
	 * </ol>
	 * </p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bullishLongLeggedDoji(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 足長同時が存在するかどうかを列挙
	 * @see #bullishLongLeggedDoji(Candlestick[], boolean)
	 */
	public static boolean[] bullishLongLeggedDoji(final Candlestick[] c) { return bullishLongLeggedDoji(c, true); }

	/**
	 * <p>指定されたローソク足データに強気の足長同時が存在するかどうかを列挙して返します。</p>
	 * このパターンは一本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * 以下を基準として判断します。
	 * <p>
	 * <ol>
	 * <li>下降トレンドである。</li>
	 * <li>下にギャップ(窓)を作り、足長同時を形成している。</li>
	 * </ol>
	 * </p>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 足長同時が存在するかどうかを列挙
	 */
	public static boolean[] bullishLongLeggedDoji(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 1; i < len; i++) {
			final int _0 = i - 1, _1 = i;
			if (c[_0] == null || c[_1] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.DOWN) &&
				c[_0].gapDownBodyByBody(c[_1]) && c[_1].isLongLeggedDoji();
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに弱気の足長同時が存在するかどうかを列挙して返します。</p>
	 * このパターンは一本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * 以下を基準として判断します。
	 * <p>
	 * <ol>
	 * <li>上昇トレンドである。</li>
	 * <li>上にギャップ(窓)を作り、足長同時を形成している。</li>
	 * </ol>
	 * </p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bearishLongLeggedDoji(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 足長同時が存在するかどうかを列挙
	 * @see #bearishLongLeggedDoji(Candlestick[], boolean)
	 */
	public static boolean[] bearishLongLeggedDoji(final Candlestick[] c) { return bearishLongLeggedDoji(c, true); }

	/**
	 * <p>指定されたローソク足データに弱気の足長同時が存在するかどうかを列挙して返します。</p>
	 * このパターンは一本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * 以下を基準として判断します。
	 * <p>
	 * <ol>
	 * <li>上昇トレンドである。</li>
	 * <li>上にギャップ(窓)を作り、足長同時を形成している。</li>
	 * </ol>
	 * </p>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 足長同時が存在するかどうかを列挙
	 */
	public static boolean[] bearishLongLeggedDoji(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 1; i < len; i++) {
			final int _0 = i - 1, _1 = i;
			if (c[_0] == null || c[_1] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.UP) &&
				c[_0].gapUpBodyByBody(c[_1]) && c[_1].isLongLeggedDoji();
		}
		return results;
	}

	/* ---------------------------------------------------------------------- */
	/* 二本足【指標】反転【信頼性】低 */

	/**
	 * <p>指定されたローソク足データに金槌(カナヅチ/トンカチ)が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * 金槌(カナヅチ/トンカチ)は陰線より陽線である方が信頼性が高く、上影(上ヒゲ)が長い程反転の可能性が大きい事を示唆します。
	 * (形容から底堅いと言います)<br>
	 * このパターンの信頼性は一般的に低いと言われています。<br>
	 * 以下を基準として判断します。
	 * <p>
	 * <ol>
	 * <li>下降トレンドである。</li>
	 * <li>長い陰線(黒)の下にギャップ(窓)を作り、陽線(白)の金槌(カナヅチ/トンカチ)又は塔婆(トウバ)を形成している。</li>
	 * </ol>
	 * </p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bullishInvertedHammer(Candlestick[], boolean, boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 金槌(カナヅチ/トンカチ)が存在するかどうかを列挙
	 * @see #bullishInvertedHammer(Candlestick[], boolean, boolean)
	 */
	public static boolean[] bullishInvertedHammer(final Candlestick[] c) { return bullishInvertedHammer(c, true, false); }

	/**
	 * <p>指定されたローソク足データに金槌(カナヅチ/トンカチ)が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * 金槌(カナヅチ/トンカチ)は陰線より陽線である方が信頼性が高く、上影(上ヒゲ)が長い程反転の可能性が大きい事を示唆します。
	 * (形容から底堅いと言います)<br>
	 * このパターンの信頼性は一般的に低いと言われています。<br>
	 * 以下を基準として判断します。
	 * <p>
	 * <ol>
	 * <li>下降トレンドである。</li>
	 * <li>長い陰線(黒)の下にギャップ(窓)を作り、陽線(白)の金槌(カナヅチ/トンカチ)又は塔婆(トウバ)を形成している。</li>
	 * </ol>
	 * </p>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @param color 陰陽線の種類を確認するかどうか
	 * @return 金槌(カナヅチ/トンカチ)が存在するかどうかを列挙
	 */
	public static boolean[] bullishInvertedHammer(final Candlestick[] c, final boolean trend, final boolean color) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 1; i < len; i++) {
			final int _0 = i - 1, _1 = i;
			if (c[_0] == null || c[_1] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.DOWN) &&
				c[_0].getColor() == CandlestickColor.BLACK && c[_0].getBodyType() == RealBodyType.LONG &&
				(c[_1].isShootingStar() || c[_1].isGravestoneDoji()) &&
				c[_0].gapDownBodyByBody(c[_1]) &&
				(!color | c[_1].getColor() == CandlestickColor.WHITE);
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに流れ星が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に低いと言われています。<br>
	 * 以下を基準として判断します。
	 * <p>
	 * <ol>
	 * <li>上昇トレンドである。</li>
	 * <li>長い陽線(白)の上にギャップ(窓)を作り、陰線(黒)の金槌(カナヅチ/トンカチ)又は塔婆(トウバ)を形成している。</li>
	 * </ol>
	 * </p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bearishShootingStar(Candlestick[], boolean, boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 流れ星が存在するかどうかを列挙
	 * @see #bearishShootingStar(Candlestick[], boolean, boolean)
	 */
	public static boolean[] bearishShootingStar(final Candlestick[] c) { return bearishShootingStar(c, true, false); }

	/**
	 * <p>指定されたローソク足データに流れ星が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に低いと言われています。<br>
	 * 以下を基準として判断します。
	 * <p>
	 * <ol>
	 * <li>上昇トレンドである。</li>
	 * <li>長い陽線(白)の上にギャップ(窓)を作り、陰線(黒)の金槌(カナヅチ/トンカチ)又は塔婆(トウバ)を形成している。</li>
	 * </ol>
	 * </p>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @param color 陰陽線の種類を確認するかどうか
	 * @return 流れ星が存在するかどうかを列挙
	 */
	public static boolean[] bearishShootingStar(final Candlestick[] c, final boolean trend, final boolean color) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 1; i < len; i++) {
			final int _0 = i - 1, _1 = i;
			if (c[_0] == null || c[_1] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.UP) &&
				c[_0].getColor() == CandlestickColor.WHITE && c[_0].getBodyType() == RealBodyType.LONG &&
				(c[_1].isShootingStar() || c[_1].isGravestoneDoji()) &&
				c[_0].gapUpBodyByBody(c[_1]) &&
				(!color | c[_1].getColor() == CandlestickColor.BLACK);
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに強気のはらみ線が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に低いと言われています。<br>
	 * 以下を基準として判断します。
	 * <p>
	 * <ol>
	 * <li>下降トレンドである。</li>
	 * <li>1日目は長い陰線(黒)である。</li>
	 * <li>2日目は小さい陽線(白)で、1日目の胴体範囲が2日目の胴体範囲を内包する。</li>
	 * </ol>
	 * </p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bullishHarami(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 強気のはらみ線が存在するかどうかを列挙
	 * @see #bullishHarami(Candlestick[], boolean)
	 */
	public static boolean[] bullishHarami(final Candlestick[] c) { return bullishHarami(c, true); }

	/**
	 * <p>指定されたローソク足データに強気のはらみ線が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に低いと言われています。<br>
	 * 以下を基準として判断します。
	 * <p>
	 * <ol>
	 * <li>下降トレンドである。</li>
	 * <li>1日目は長い陰線(黒)である。</li>
	 * <li>2日目は小さい陽線(白)で、1日目の胴体範囲が2日目の胴体範囲を内包する。</li>
	 * </ol>
	 * </p>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 強気のはらみ線が存在するかどうかを列挙
	 */
	public static boolean[] bullishHarami(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 1; i < len; i++) {
			final int _1 = i - 1, _2 = i;
			if (c[_1] == null || c[_2] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.DOWN) &&
				c[_1].getColor() == CandlestickColor.BLACK && c[_1].getBodyType() == RealBodyType.LONG &&
				c[_2].getColor() == CandlestickColor.WHITE && c[_2].getBodyType() == RealBodyType.SMALL &&
				c[_1].containsBodyByBody(c[_2]);
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに弱気のはらみ線が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に低いと言われています。<br>
	 * 以下を基準として判断します。
	 * <p>
	 * <ol>
	 * <li>上昇トレンドである。</li>
	 * <li>1日目は長い陽線(白)である。</li>
	 * <li>2日目は小さい陰線(黒)で、1日目の胴体範囲が2日目の胴体範囲を内包する。</li>
	 * </ol>
	 * </p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bearishHarami(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 弱気のはらみ線が存在するかどうかを列挙
	 * @see #bearishHarami(Candlestick[], boolean)
	 */
	public static boolean[] bearishHarami(final Candlestick[] c) { return bearishHarami(c, true); }

	/**
	 * <p>指定されたローソク足データに弱気のはらみ線が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に低いと言われています。<br>
	 * 以下を基準として判断します。
	 * <p>
	 * <ol>
	 * <li>上昇トレンドである。</li>
	 * <li>1日目は長い陽線(白)である。</li>
	 * <li>2日目は小さい陰線(黒)で、1日目の胴体範囲が2日目の胴体範囲を内包する。</li>
	 * </ol>
	 * </p>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 弱気のはらみ線が存在するかどうかを列挙
	 */
	public static boolean[] bearishHarami(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 1; i < len; i++) {
			final int _1 = i - 1, _2 = i;
			if (c[_1] == null || c[_2] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.UP) &&
				c[_1].getColor() == CandlestickColor.WHITE && c[_1].getBodyType() == RealBodyType.LONG &&
				c[_2].getColor() == CandlestickColor.BLACK && c[_2].getBodyType() == RealBodyType.SMALL &&
				c[_1].containsBodyByBody(c[_2]);
		}
		return results;
	}

	/* ---------------------------------------------------------------------- */
	/* 二本足【指標】反転【信頼性】中 */

	/**
	 * <p>指定されたローソク足データに強気のはらみ線寄せ線が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * 以下を基準として判断します。
	 * <p>
	 * <ol>
	 * <li>下降トレンドである。</li>
	 * <li>1日目は長い陰線(黒)である。</li>
	 * <li>2日目は胴体が同時の陽線(白)で、1日目の胴体範囲が2日目の胴体範囲を内包する。</li>
	 * </ol>
	 * </p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bullishHaramiCross(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 強気のはらみ寄せ線が存在するかどうかを列挙
	 * @see #bullishHaramiCross(Candlestick[], boolean)
	 */
	public static boolean[] bullishHaramiCross(final Candlestick[] c) { return bullishHaramiCross(c, true); }

	/**
	 * <p>指定されたローソク足データに強気のはらみ線寄せ線が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * 以下を基準として判断します。
	 * <p>
	 * <ol>
	 * <li>下降トレンドである。</li>
	 * <li>1日目は長い陰線(黒)である。</li>
	 * <li>2日目は胴体が同時の陽線(白)で、1日目の胴体範囲が2日目の胴体範囲を内包する。</li>
	 * </ol>
	 * </p>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 強気のはらみ寄せ線が存在するかどうかを列挙
	 */
	public static boolean[] bullishHaramiCross(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 1; i < len; i++) {
			final int _1 = i - 1, _2 = i;
			if (c[_1] == null || c[_2] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.DOWN) &&
				c[_1].getColor() == CandlestickColor.BLACK && c[_1].getBodyType() == RealBodyType.LONG &&
				c[_2].getColor() == CandlestickColor.WHITE && c[_2].getBodyType() == RealBodyType.DOJI &&
				c[_1].containsBodyByBody(c[_2]);
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに弱気のはらみ線寄せ線が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * 以下を基準として判断します。
	 * <p>
	 * <ol>
	 * <li>上昇トレンドである。</li>
	 * <li>1日目は長い陽線(白)である。</li>
	 * <li>2日目は胴体が同時の陰線(黒)で、1日目の胴体範囲が2日目の胴体範囲を内包する。</li>
	 * </ol>
	 * </p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bearishHaramiCross(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 弱気のはらみ寄せ線が存在するかどうかを列挙
	 * @see #bearishHaramiCross(Candlestick[], boolean)
	 */
	public static boolean[] bearishHaramiCross(final Candlestick[] c) { return bearishHaramiCross(c, true); }

	/**
	 * <p>指定されたローソク足データに弱気のはらみ線寄せ線が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * 以下を基準として判断します。
	 * <p>
	 * <ol>
	 * <li>上昇トレンドである。</li>
	 * <li>1日目は長い陽線(白)である。</li>
	 * <li>2日目は胴体が同時の陰線(黒)で、1日目の胴体範囲が2日目の胴体範囲を内包する。</li>
	 * </ol>
	 * </p>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 弱気のはらみ寄せ線が存在するかどうかを列挙
	 */
	public static boolean[] bearishHaramiCross(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 1; i < len; i++) {
			final int _1 = i - 1, _2 = i;
			if (c[_1] == null || c[_2] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.UP) &&
				c[_1].getColor() == CandlestickColor.WHITE && c[_1].getBodyType() == RealBodyType.LONG &&
				c[_2].getColor() == CandlestickColor.BLACK && c[_2].getBodyType() == RealBodyType.DOJI &&
				c[_1].containsBodyByBody(c[_2]);
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに強気の抱き線(包み線)が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * 以下を基準として判断します。
	 * <p>
	 * <ol>
	 * <li>下降トレンドである。</li>
	 * <li>1日目は小さい又は同時の陰線(黒)である。</li>
	 * <li>2日目は長い陽線(白)で、2日目の胴体範囲が1日目の胴体範囲を内包する。</li>
	 * </ol>
	 * </p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bullishEngulfing(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 強気の抱き線(包み線)が存在するかどうかを列挙
	 * @see #bullishEngulfing(Candlestick[], boolean)
	 */
	public static boolean[] bullishEngulfing(final Candlestick[] c) { return bullishEngulfing(c, true); }

	/**
	 * <p>指定されたローソク足データに強気の抱き線(包み線)が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * 以下を基準として判断します。
	 * <p>
	 * <ol>
	 * <li>下降トレンドである。</li>
	 * <li>1日目は小さい又は同時の陰線(黒)である。</li>
	 * <li>2日目は長い陽線(白)で、2日目の胴体範囲が1日目の胴体範囲を内包する。</li>
	 * </ol>
	 * </p>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 強気の抱き線(包み線)が存在するかどうかを列挙
	 */
	public static boolean[] bullishEngulfing(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 1; i < len; i++) {
			final int _1 = i - 1, _2 = i;
			if (c[_1] == null || c[_2] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.DOWN) &&
				c[_1].getColor() == CandlestickColor.BLACK && (c[_1].getBodyType() == RealBodyType.SMALL || c[_1].getBodyType() == RealBodyType.DOJI) &&
				c[_2].getColor() == CandlestickColor.WHITE && c[_2].getBodyType() == RealBodyType.LONG &&
				c[_2].containsBodyByBody(c[_1]);
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに弱気の抱き線(包み線)が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * 以下を基準として判断します。
	 * <p>
	 * <ol>
	 * <li>上昇トレンドである。</li>
	 * <li>1日目は小さい又は同時の陽線(白)である。</li>
	 * <li>2日目は長い陰線(黒)で、2日目の胴体範囲が1日目の胴体範囲を内包する。</li>
	 * </ol>
	 * </p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bearishEngulfing(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 弱気の抱き線(包み線)が存在するかどうかを列挙
	 * @see #bearishEngulfing(Candlestick[], boolean)
	 */
	public static boolean[] bearishEngulfing(final Candlestick[] c) { return bearishEngulfing(c, true); }

	/**
	 * <p>指定されたローソク足データに弱気の抱き線(包み線)が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * 以下を基準として判断します。
	 * <p>
	 * <ol>
	 * <li>上昇トレンドである。</li>
	 * <li>1日目は小さい又は同時の陽線(白)である。</li>
	 * <li>2日目は長い陰線(黒)で、2日目の胴体範囲が1日目の胴体範囲を内包する。</li>
	 * </ol>
	 * </p>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 弱気の抱き線(包み線)が存在するかどうかを列挙
	 */
	public static boolean[] bearishEngulfing(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 1; i < len; i++) {
			final int _1 = i - 1, _2 = i;
			if (c[_1] == null || c[_2] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.UP) &&
				c[_1].getColor() == CandlestickColor.WHITE && (c[_1].getBodyType() == RealBodyType.SMALL || c[_1].getBodyType() == RealBodyType.DOJI) &&
				c[_2].getColor() == CandlestickColor.BLACK && c[_2].getBodyType() == RealBodyType.LONG &&
				c[_2].containsBodyByBody(c[_1]);
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに強気の塔婆(トウバ)が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、下降トレンドからの反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * 以下を基準として判断します。
	 * <p>
	 * <ol>
	 * <li>下降トレンドである。</li>
	 * <li>下にギャップ(窓)を作り、陰線(黒)の塔婆(トウバ)を形成している。</li>
	 * </ol>
	 * </p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bullishGravestoneDoji(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 強気の塔婆(トウバ)が存在するかどうかを列挙
	 * @see #bullishGravestoneDoji(Candlestick[], boolean)
	 */
	public static boolean[] bullishGravestoneDoji(final Candlestick[] c) { return bullishGravestoneDoji(c, true); }

	/**
	 * <p>指定されたローソク足データに強気の塔婆(トウバ)が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、下降トレンドからの反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * 以下を基準として判断します。
	 * <p>
	 * <ol>
	 * <li>下降トレンドである。</li>
	 * <li>下にギャップ(窓)を作り、陰線(黒)の塔婆(トウバ)を形成している。</li>
	 * </ol>
	 * </p>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 強気の塔婆(トウバ)が存在するかどうかを列挙
	 */
	public static boolean[] bullishGravestoneDoji(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 1; i < len; i++) {
			final int _0 = i - 1, _1 = i;
			if (c[_0] == null || c[_1] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.DOWN) &&
				c[_0].getColor() == CandlestickColor.BLACK && c[_0].gapDownBodyByBody(c[_1]) && c[_1].isGravestoneDoji();
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに弱気の塔婆(トウバ)が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、上昇トレンドからの反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * 以下を基準として判断します。
	 * <p>
	 * <ol>
	 * <li>上昇トレンドである。</li>
	 * <li>上にギャップ(窓)を作り、陽線(白)の塔婆(トウバ)を形成している。</li>
	 * </ol>
	 * </p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bearishGravestoneDoji(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 弱気の塔婆(トウバ)が存在するかどうかを列挙
	 * @see #bearishGravestoneDoji(Candlestick[], boolean)
	 */
	public static boolean[] bearishGravestoneDoji(final Candlestick[] c) { return bearishGravestoneDoji(c, true); }

	/**
	 * <p>指定されたローソク足データに弱気の塔婆(トウバ)が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、上昇トレンドからの反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * 以下を基準として判断します。
	 * <p>
	 * <ol>
	 * <li>上昇トレンドである。</li>
	 * <li>上にギャップ(窓)を作り、陽線(白)の塔婆(トウバ)を形成している。</li>
	 * </ol>
	 * </p>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 弱気の塔婆(トウバ)が存在するかどうかを列挙
	 */
	public static boolean[] bearishGravestoneDoji(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 1; i < len; i++) {
			final int _0 = i - 1, _1 = i;
			if (c[_0] == null || c[_1] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.UP) &&
				c[_0].getColor() == CandlestickColor.WHITE && c[_0].gapUpBodyByBody(c[_1]) && c[_1].isGravestoneDoji();
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに強気の出会い線が存在するかどうかを列挙して返します。</p>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bullishMeetingLines(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 強気の出会い線が存在するかどうかを列挙
	 * @see #bullishMeetingLines(Candlestick[], boolean)
	 */
	public static boolean[] bullishCounterAttack(final Candlestick[] c) { return bullishMeetingLines(c, true); }

	/**
	 * <p>指定されたローソク足データに強気の出会い線が存在するかどうかを列挙して返します。</p>
	 * このメソッドは {@link #bullishMeetingLines(Candlestick[], boolean)} の別称です。<br>
	 * 詳細は {@link #bullishMeetingLines(Candlestick[], boolean)} を参照して下さい。
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 強気の出会い線が存在するかどうかを列挙
	 * @see #bullishMeetingLines(Candlestick[], boolean)
	 */
	public static boolean[] bullishCounterAttack(final Candlestick[] c, final boolean trend) { return bullishMeetingLines(c, trend); }

	/**
	 * <p>指定されたローソク足データに弱気の出会い線が存在するかどうかを列挙して返します。</p>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bearishMeetingLines(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 弱気の出会い線が存在するかどうかを列挙
	 * @see #bearishMeetingLines(Candlestick[], boolean)
	 */
	public static boolean[] bearishCounterAttack(final Candlestick[] c) { return bearishMeetingLines(c, true); }

	/**
	 * <p>指定されたローソク足データに弱気の出会い線が存在するかどうかを列挙して返します。</p>
	 * このメソッドは {@link #bearishMeetingLines(Candlestick[], boolean)} の別称です。<br>
	 * 詳細は {@link #bearishMeetingLines(Candlestick[], boolean)} を参照して下さい。
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 弱気の出会い線が存在するかどうかを列挙
	 * @see #bearishMeetingLines(Candlestick[], boolean)
	 */
	public static boolean[] bearishCounterAttack(final Candlestick[] c, final boolean trend) { return bearishMeetingLines(c, trend); }

	/**
	 * <p>指定されたローソク足データに強気の出会い線が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bullishMeetingLines(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 強気の出会い線が存在するかどうかを列挙
	 * @see #bullishMeetingLines(Candlestick[], boolean)
	 */
	public static boolean[] bullishMeetingLines(final Candlestick[] c) { return bullishMeetingLines(c, true); }

	/**
	 * <p>指定されたローソク足データに強気の出会い線が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 強気の出会い線が存在するかどうかを列挙
	 */
	public static boolean[] bullishMeetingLines(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 1; i < len; i++) {
			final int _1 = i - 1, _2 = i;
			if (c[_1] == null || c[_2] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.DOWN) &&
				c[_1].getColor() == CandlestickColor.BLACK && c[_1].getBodyType() == RealBodyType.LONG &&
				c[_2].getColor() == CandlestickColor.WHITE && c[_2].getBodyType() == RealBodyType.LONG &&
				c[_1].body() > c[_2].body() && c[_1].gapDownBodyByBody(c[_2]) &&
				c[_2].isSameClose(c[_1].getClose());
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに弱気の出会い線が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bearishMeetingLines(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 弱気の出会い線が存在するかどうかを列挙
	 * @see #bearishMeetingLines(Candlestick[], boolean)
	 */
	public static boolean[] bearishMeetingLines(final Candlestick[] c) { return bearishMeetingLines(c, true); }

	/**
	 * <p>指定されたローソク足データに弱気の出会い線が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 弱気の出会い線が存在するかどうかを列挙
	 */
	public static boolean[] bearishMeetingLines(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 1; i < len; i++) {
			final int _1 = i - 1, _2 = i;
			if (c[_1] == null || c[_2] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.UP) &&
				c[_1].getColor() == CandlestickColor.WHITE && c[_1].getBodyType() == RealBodyType.LONG &&
				c[_2].getColor() == CandlestickColor.BLACK && c[_2].getBodyType() == RealBodyType.LONG &&
				c[_1].body() > c[_2].body() && c[_1].gapUpBodyByBody(c[_2]) &&
				c[_2].isSameClose(c[_1].getClose());
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに強気の寄り引き同事線が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bullishDojiStar(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 強気の寄り引き同事線が存在するかどうかを列挙
	 * @see #bullishDojiStar(Candlestick[], boolean)
	 */
	public static boolean[] bullishDojiStar(final Candlestick[] c) { return bullishDojiStar(c, true); }

	/**
	 * <p>指定されたローソク足データに強気の寄り引き同事線が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 強気の寄り引き同事線が存在するかどうかを列挙
	 */
	public static boolean[] bullishDojiStar(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 1; i < len; i++) {
			final int _1 = i - 1, _2 = i;
			if (c[_1] == null || c[_2] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.DOWN) &&
				c[_1].getColor() == CandlestickColor.BLACK && c[_1].getBodyType() == RealBodyType.LONG &&
				c[_1].gapDownBodyByBody(c[_2]) &&
				c[_2].getBodyType() == RealBodyType.DOJI && c[_2].getUpperShadowType() != ShadowType.LONG && c[_2].getLowerShadowType() != ShadowType.LONG;
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに弱気の寄り引き同事線が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bearishDojiStar(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足
	 * @return 弱気の寄り引き同事線が存在するかどうかを列挙
	 * @see #bearishDojiStar(Candlestick[], boolean)
	 */
	public static boolean[] bearishDojiStar(final Candlestick[] c) { return bearishDojiStar(c, true); }

	/**
	 * <p>指定されたローソク足データに弱気の寄り引き同事線が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * 
	 * @param c ローソク足
	 * @param trend トレンドを確認するかどうか
	 * @return 弱気の寄り引き同事線が存在するかどうかを列挙
	 */
	public static boolean[] bearishDojiStar(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 1; i < len; i++) {
			final int _1 = i - 1, _2 = i;
			if (c[_1] == null || c[_2] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.UP) &&
				c[_1].getColor() == CandlestickColor.WHITE && c[_1].getBodyType() == RealBodyType.LONG &&
				c[_1].gapUpBodyByBody(c[_2]) &&
				c[_2].getBodyType() == RealBodyType.DOJI && c[_2].getUpperShadowType() != ShadowType.LONG && c[_2].getLowerShadowType() != ShadowType.LONG;
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに小鳩返しが存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bullishHomingPigeon(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 小鳩返しが存在するかどうかを列挙
	 * @see #bullishHomingPigeon(Candlestick[], boolean)
	 */
	public static boolean[] bullishHomingPigeon(final Candlestick[] c) { return bullishHomingPigeon(c, true); }

	/**
	 * <p>指定されたローソク足データに小鳩返しが存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 小鳩返しが存在するかどうかを列挙
	 */
	public static boolean[] bullishHomingPigeon(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 1; i < len; i++) {
			final int _1 = i - 1, _2 = i;
			if (c[_1] == null || c[_2] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.DOWN) &&
				c[_1].getColor() == CandlestickColor.BLACK && c[_1].getBodyType() == RealBodyType.LONG &&
				c[_2].getColor() == CandlestickColor.BLACK && c[_2].getBodyType() == RealBodyType.SMALL &&
				c[_1].containsBodyByBody(c[_2]);
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに下降中の鷹が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bearishDescendingHawk(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 下降中の鷹が存在するかどうかを列挙
	 * @see #bearishDescendingHawk(Candlestick[], boolean)
	 */
	public static boolean[] bearishDescendingHawk(final Candlestick[] c) { return bearishDescendingHawk(c, true); }

	/**
	 * <p>指定されたローソク足データに下降中の鷹が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 下降中の鷹が存在するかどうかを列挙
	 */
	// bullishHomingPigeon 保管用に作られた
	public static boolean[] bearishDescendingHawk(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 1; i < len; i++) {
			final int _1 = i - 1, _2 = i;
			if (c[_1] == null || c[_2] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.UP) &&
				c[_1].getColor() == CandlestickColor.WHITE && c[_1].getBodyType() == RealBodyType.LONG &&
				c[_2].getColor() == CandlestickColor.WHITE && c[_2].getBodyType() == RealBodyType.SMALL &&
				c[_1].containsBodyByBody(c[_2]);
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに二点底が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bullishMatchingLow(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 二点底が存在するかどうかを列挙
	 * @see #bullishMatchingLow(Candlestick[], boolean)
	 */
	public static boolean[] bullishMatchingLow(final Candlestick[] c) { return bullishMatchingLow(c, true); }

	/**
	 * <p>指定されたローソク足データに二点底が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 二点底が存在するかどうかを列挙
	 */
	public static boolean[] bullishMatchingLow(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		final boolean[] bullishHomingPigeon = bullishHomingPigeon(c, trend);
		for (int i = 1; i < len; i++) {
			final int _1 = i - 1, _2 = i;
			if (c[_1] == null || c[_2] == null)
				continue;
			results[i] = bullishHomingPigeon[_2] && c[_2].isSameClose(c[_1].getClose());
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに二点天井が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bearishMatchingHigh(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 二点天井が存在するかどうかを列挙
	 * @see #bearishMatchingHigh(Candlestick[], boolean)
	 */
	public static boolean[] bearishMatchingHigh(final Candlestick[] c) {
		return bearishMatchingHigh(c, true);
	}

	/**
	 * <p>指定されたローソク足データに二点天井が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 二点天井が存在するかどうかを列挙
	 */
	public static boolean[] bearishMatchingHigh(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		final boolean[] bearishDescendingHawk = bearishDescendingHawk(c, trend);
		for (int i = 1; i < len; i++) {
			final int _1 = i - 1, _2 = i;
			if (c[_1] == null || c[_2] == null)
				continue;
			results[i] = bearishDescendingHawk[_2] && c[_2].isSameClose(c[_1].getClose());
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに毛抜き底が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bullishTweezerBottom(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 毛抜き底が存在するかどうかを列挙
	 * @see #bullishTweezerBottom(Candlestick[], boolean)
	 */
	public static boolean[] bullishTweezerBottom(final Candlestick[] c) { return bullishTweezerBottom(c, true); }

	/**
	 * <p>指定されたローソク足データに毛抜き底が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 毛抜き底が存在するかどうかを列挙
	 */
	public static boolean[] bullishTweezerBottom(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 1; i < len; i++) {
			final int _1 = i - 1, _2 = i;
			if (c[_1] == null || c[_2] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.DOWN) &&
				c[_1].getColor() == CandlestickColor.BLACK && c[_1].getBodyType() == RealBodyType.LONG &&
				c[_2].getColor() == CandlestickColor.WHITE && (c[_2].getBodyType() == RealBodyType.SMALL || c[_2].getBodyType() == RealBodyType.DOJI) &&
				c[_1].contains(c[_2]) && c[_2].isSameLow(c[_1].getLow());
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに毛抜き天井が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bearishTweezerTop(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 毛抜き天井が存在するかどうかを列挙
	 * @see #bearishTweezerTop(Candlestick[], boolean)
	 */
	public static boolean[] bearishTweezerTop(final Candlestick[] c) { return bearishTweezerTop(c, true); }

	/**
	 * <p>指定されたローソク足データに毛抜き天井が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 毛抜き天井が存在するかどうかを列挙
	 */
	public static boolean[] bearishTweezerTop(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 1; i < len; i++) {
			final int _1 = i - 1, _2 = i;
			if (c[_1] == null || c[_2] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.UP) &&
				c[_1].getColor() == CandlestickColor.WHITE && c[_1].getBodyType() == RealBodyType.LONG &&
				c[_2].getColor() == CandlestickColor.BLACK && (c[_2].getBodyType() == RealBodyType.SMALL || c[_2].getBodyType() == RealBodyType.DOJI) &&
				c[_1].contains(c[_2]) && c[_2].isSameHigh(c[_1].getHigh());
		}
		return results;
	}

	/* ---------------------------------------------------------------------- */
	/* 二本足【指標】反転【信頼性】高 */

	/**
	 * <p>指定されたローソク足データに強気の行き違い線が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に高いと言われています。<br>
	 * 
	 * @param c ローソク足データ
	 * @return 強気の行き違い線が存在するかどうかを列挙
	 */
	public static boolean[] bullishKicking(final Candlestick[] c) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 1; i < len; i++) {
			final int _1 = i - 1, _2 = i;
			if (c[_1] == null || c[_2] == null)
				continue;
			// トレンドは関係ない
			results[i] =
				c[_1].getColor() == CandlestickColor.BLACK && c[_1].isMarubozu() &&
				c[_2].getColor() == CandlestickColor.WHITE && c[_2].isMarubozu() &&
				c[_1].gapUpBodyByBody(c[_2]);
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに弱気の行き違い線が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に高いと言われています。<br>
	 * 
	 * @param c ローソク足データ
	 * @return 弱気の行き違い線が存在するかどうかを列挙
	 */
	public static boolean[] bearishKicking(final Candlestick[] c) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 1; i < len; i++) {
			final int _1 = i - 1, _2 = i;
			if (c[_1] == null || c[_2] == null)
				continue;
			// トレンドは関係ない
			results[i] =
				c[_1].getColor() == CandlestickColor.WHITE && c[_1].isMarubozu() &&
				c[_2].getColor() == CandlestickColor.BLACK && c[_2].isMarubozu() &&
				c[_1].gapDownBodyByBody(c[_2]);
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに切り込み線(切り返し線)が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に高いと言われています。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bullishPiercingLine(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 切り込み線(切り返し線)が存在するかどうかを列挙
	 * @see #bullishPiercingLine(Candlestick[], boolean)
	 */
	public static boolean[] bullishPiercingLine(final Candlestick[] c) { return bullishPiercingLine(c, true); }

	/**
	 * <p>指定されたローソク足データに切り込み線(切り返し線)が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に高いと言われています。<br>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 切り込み線(切り返し線)が存在するかどうかを列挙
	 */
	public static boolean[] bullishPiercingLine(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 1; i < len; i++) {
			final int _1 = i - 1, _2 = i;
			if (c[_1] == null || c[_2] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.DOWN) &&
				c[_1].getColor() == CandlestickColor.BLACK && c[_1].getBodyType() == RealBodyType.LONG &&
				c[_2].getColor() == CandlestickColor.WHITE && c[_2].getBodyType() == RealBodyType.LONG &&
				c[_1].getLow() > c[_2].getOpen() &&
				c[_1].containsBody(c[_2].getClose()) && c[_1].midpoint() < c[_2].getClose() &&
				!c[_2].isSameClose(c[_1].getOpen());
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに被せ線が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に高いと言われています。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bearishDarkCloudCover(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 被せ線が存在するかどうかを列挙
	 * @see #bearishDarkCloudCover(Candlestick[], boolean)
	 */
	public static boolean[] bearishDarkCloudCover(final Candlestick[] c) { return bearishDarkCloudCover(c, true); }

	/**
	 * <p>指定されたローソク足データに被せ線が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に高いと言われています。<br>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 被せ線が存在するかどうかを列挙
	 */
	public static boolean[] bearishDarkCloudCover(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 1; i < len; i++) {
			final int _1 = i - 1, _2 = i;
			if (c[_1] == null || c[_2] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.UP) &&
				c[_1].getColor() == CandlestickColor.WHITE && c[_1].getBodyType() == RealBodyType.LONG &&
				c[_2].getColor() == CandlestickColor.BLACK && c[_2].getBodyType() == RealBodyType.LONG &&
				c[_1].getHigh() < c[_2].getOpen() &&
				c[_1].containsBody(c[_2].getClose()) && c[_1].midpoint() > c[_2].getClose() &&
				!c[_2].isSameClose(c[_1].getOpen());
		}
		return results;
	}

	/* ---------------------------------------------------------------------- */
	/* 二本足【指標】継続【信頼性】低 */

	/**
	 * <p>指定されたローソク足データに強気の振分け線が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、上昇トレンド継続の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に低いと言われています。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bullishSeparatingLines(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 強気の振分け線が存在するかどうかを列挙
	 * @see #bullishSeparatingLines(Candlestick[], boolean)
	 */
	public static boolean[] bullishSeparatingLines(final Candlestick[] c) { return bullishSeparatingLines(c, true); }

	/**
	 * <p>指定されたローソク足データに強気の振分け線が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、上昇トレンド継続の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に低いと言われています。<br>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 強気の振分け線が存在するかどうかを列挙
	 */
	public static boolean[] bullishSeparatingLines(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 1; i < len; i++) {
			final int _1 = i - 1, _2 = i;
			if (c[_1] == null || c[_2] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.UP) &&
				c[_1].getColor() == CandlestickColor.BLACK && c[_1].getBodyType() == RealBodyType.LONG &&
				c[_2].getColor() == CandlestickColor.WHITE && c[_2].getBodyType() != RealBodyType.DOJI && c[_2].getLowerShadowType() == ShadowType.NO &&
				c[_2].isSameOpen(c[_1].getOpen());
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに弱気の振分け線が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、下降トレンド継続の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に低いと言われています。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bearishSeparatingLines(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 弱気の振分け線が存在するかどうかを列挙
	 * @see #bearishSeparatingLines(Candlestick[], boolean)
	 */
	public static boolean[] bearishSeparatingLines(final Candlestick[] c) { return bearishSeparatingLines(c, true); }

	/**
	 * <p>指定されたローソク足データに弱気の振分け線が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、下降トレンド継続の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に低いと言われています。<br>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 弱気の振分け線が存在するかどうかを列挙
	 */
	public static boolean[] bearishSeparatingLines(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 1; i < len; i++) {
			final int _1 = i - 1, _2 = i;
			if (c[_1] == null || c[_2] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.DOWN) &&
				c[_1].getColor() == CandlestickColor.WHITE && c[_1].getBodyType() == RealBodyType.LONG &&
				c[_2].getColor() == CandlestickColor.BLACK && c[_2].getBodyType() != RealBodyType.DOJI && c[_2].getUpperShadowType() == ShadowType.NO &&
				c[_2].isSameOpen(c[_1].getOpen());
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに差込み線が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、下降トレンド継続の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に低いと言われています。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bearishThrusting(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 差込み線が存在するかどうかを列挙
	 * @see #bearishThrusting(Candlestick[], boolean)
	 */
	public static boolean[] bearishThrusting(final Candlestick[] c) { return bearishThrusting(c, true); }

	/**
	 * <p>指定されたローソク足データに差込み線が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、下降トレンド継続の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に低いと言われています。<br>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 差込み線が存在するかどうかを列挙
	 */
	public static boolean[] bearishThrusting(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 1; i < len; i++) {
			final int _1 = i - 1, _2 = i;
			if (c[_1] == null || c[_2] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.DOWN) &&
				c[_1].getColor() == CandlestickColor.BLACK && c[_1].getBodyType() == RealBodyType.LONG &&
				c[_2].getColor() == CandlestickColor.WHITE &&
				c[_1].getLow() > c[_2].getOpen() && c[_1].getClose() < c[_2].getClose() &&
				c[_1].midpoint() > c[_2].getClose() &&
				!c[_2].isNearClose(c[_1].getClose());
		}
		return results;
	}

	/* ---------------------------------------------------------------------- */
	/* 二本足【指標】継続【信頼性】中 */

	/**
	 * <p>指定されたローソク足データにあて首線が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、下降トレンド継続の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bearishOnNeck(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return あて首線が存在するかどうかを列挙
	 * @see #bearishOnNeck(Candlestick[], boolean)
	 */
	public static boolean[] bearishOnNeck(final Candlestick[] c) { return bearishOnNeck(c, true); }

	/**
	 * <p>指定されたローソク足データにあて首線が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、下降トレンド継続の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return あて首線が存在するかどうかを列挙
	 */
	public static boolean[] bearishOnNeck(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 1; i < len; i++) {
			final int _1 = i - 1, _2 = i;
			if (c[_1] == null || c[_2] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.DOWN) &&
				c[_1].getColor() == CandlestickColor.BLACK && c[_1].getBodyType() == RealBodyType.LONG &&
				c[_2].getColor() == CandlestickColor.WHITE &&
				c[_1].getLow() > c[_2].getOpen() && c[_1].getClose() > c[_2].getClose();
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに入り首線が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、下降トレンド継続の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bearishInNeck(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 入り首線が存在するかどうかを列挙
	 * @see #bearishInNeck(Candlestick[], boolean)
	 */
	public static boolean[] bearishInNeck(final Candlestick[] c) { return bearishInNeck(c, true); }

	/**
	 * <p>指定されたローソク足データに入り首線が存在するかどうかを列挙して返します。</p>
	 * このパターンは二本足のパターンで、下降トレンド継続の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 入り首線が存在するかどうかを列挙
	 */
	public static boolean[] bearishInNeck(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 1; i < len; i++) {
			final int _1 = i - 1, _2 = i;
			if (c[_1] == null || c[_2] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.DOWN) &&
				c[_1].getColor() == CandlestickColor.BLACK && c[_1].getBodyType() == RealBodyType.LONG &&
				c[_2].getColor() == CandlestickColor.WHITE &&
				c[_1].getLow() > c[_2].getOpen() && c[_1].getClose() < c[_2].getClose() &&
				c[_1].midpoint() > c[_2].getClose() &&
				c[_2].isNearClose(c[_1].getClose());
		}
		return results;
	}

	/* ---------------------------------------------------------------------- */
	/* 三本足【指標】反転【信頼性】中 */

	/**
	 * <p>指定されたローソク足データに強気の三ツ星が存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bullishTriStar(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 強気の三ツ星が存在するかどうかを列挙
	 * @see #bullishTriStar(Candlestick[], boolean)
	 */
	public static boolean[] bullishTriStar(final Candlestick[] c) { return bullishTriStar(c, true); }

	/**
	 * <p>指定されたローソク足データに強気の三ツ星が存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 強気の三ツ星が存在するかどうかを列挙
	 */
	public static boolean[] bullishTriStar(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 2; i < len; i++) {
			final int _1 = i - 2, _2 = i - 1, _3 = i;
			if (c[_1] == null || c[_2] == null || c[_3] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.DOWN) &&
				c[_1].getBodyType() == RealBodyType.DOJI && c[_2].getBodyType() == RealBodyType.DOJI && c[_3].getBodyType() == RealBodyType.DOJI &&
				c[_1].gapDownBodyByBody(c[_2]) && c[_2].gapUpBodyByBody(c[_3]);
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに弱気の三ツ星が存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bearishTriStar(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 弱気の三ツ星が存在するかどうかを列挙
	 * @see #bearishTriStar(Candlestick[], boolean)
	 */
	public static boolean[] bearishTriStar(final Candlestick[] c) { return bearishTriStar(c, true); }

	/**
	 * <p>指定されたローソク足データに弱気の三ツ星が存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 弱気の三ツ星が存在するかどうかを列挙
	 */
	public static boolean[] bearishTriStar(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 2; i < len; i++) {
			final int _1 = i - 2, _2 = i - 1, _3 = i;
			if (c[_1] == null || c[_2] == null || c[_3] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.UP) &&
				c[_1].getBodyType() == RealBodyType.DOJI && c[_2].getBodyType() == RealBodyType.DOJI && c[_3].getBodyType() == RealBodyType.DOJI &&
				c[_1].gapUpBodyByBody(c[_2]) && c[_2].gapDownBodyByBody(c[_3]);
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに二羽烏が存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bearishTwoCrows(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足
	 * @return 二羽烏が存在するかどうかを列挙
	 * @see #bearishTwoCrows(Candlestick[], boolean)
	 */
	public static boolean[] bearishTwoCrows(final Candlestick[] c) { return bearishTwoCrows(c, true); }

	/**
	 * <p>指定されたローソク足データに二羽烏が存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * 
	 * @param c ローソク足
	 * @param trend トレンドを確認するかどうか
	 * @return 二羽烏が存在するかどうかを列挙
	 */
	public static boolean[] bearishTwoCrows(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 2; i < len; i++) {
			final int _1 = i - 2, _2 = i - 1, _3 = i;
			if (c[_1] == null || c[_2] == null || c[_3] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.UP) &&
				c[_1].getColor() == CandlestickColor.WHITE && c[_1].getBodyType() == RealBodyType.LONG &&
				c[_1].gapUpBodyByBody(c[_2]) &&
				c[_2].getColor() == CandlestickColor.BLACK &&
				c[_3].getColor() == CandlestickColor.BLACK &&
				c[_2].containsBody(c[_3].getOpen()) && c[_1].containsBody(c[_3].getClose());
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに南の三ツ星が存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bullishThreeStarsInTheSouth(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 南の三ツ星が存在するかどうかを列挙
	 * @see #bullishThreeStarsInTheSouth(Candlestick[], boolean)
	 */
	public static boolean[] bullishThreeStarsInTheSouth(final Candlestick[] c) { return bullishThreeStarsInTheSouth(c, true); }

	/**
	 * <p>指定されたローソク足データに南の三ツ星が存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 南の三ツ星が存在するかどうかを列挙
	 */
	public static boolean[] bullishThreeStarsInTheSouth(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 2; i < len; i++) {
			final int _1 = i - 2, _2 = i - 1, _3 = i;
			if (c[_1] == null || c[_2] == null || c[_3] == null)
				continue;

			// FIXME - 判定が辛すぎるのでかえること
			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.DOWN) &&
				c[_1].getColor() == CandlestickColor.BLACK && c[_1].isOpeningMarubozu() &&
				c[_2].getColor() == CandlestickColor.BLACK && c[_2].isSmallOpeningMarubozu() &&
				c[_3].getColor() == CandlestickColor.BLACK && c[_3].isSmallMarubozu() &&
				c[_1].body() > c[_2].body() && c[_2].body() > c[_3].body() &&
				c[_1].lowerShadow() > c[_2].lowerShadow() &&
				c[_1].contains(c[_2]) && c[_2].contains(c[_3]) &&
				c[_1].getClose() > c[_2].getClose() && c[_2].getClose() > c[_3].getClose();
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに変形三川底が存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bullishUniqueThreeRiverBottom(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 変形三川底が存在するかどうかを列挙
	 * @see #bullishUniqueThreeRiverBottom(Candlestick[], boolean)
	 */
	public static boolean[] bullishUniqueThreeRiverBottom(final Candlestick[] c) { return bullishUniqueThreeRiverBottom(c, true); }

	/**
	 * <p>指定されたローソク足データに変形三川底が存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 変形三川底が存在するかどうかを列挙
	 */
	public static boolean[] bullishUniqueThreeRiverBottom(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 2; i < len; i++) {
			final int _1 = i - 2, _2 = i - 1, _3 = i;
			if (c[_1] == null || c[_2] == null || c[_3] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.DOWN) &&
				c[_1].getColor() == CandlestickColor.BLACK &&
				c[_2].getColor() == CandlestickColor.BLACK &&
				c[_3].getColor() == CandlestickColor.WHITE && c[_3].getBodyType() == RealBodyType.SMALL &&
				c[_1].containsBodyByBody(c[_2]) && c[_1].getLow() > c[_2].getLow() &&
				c[_2].gapDownBodyByBody(c[_3]);
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに逆差し二点底が存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bullishStickSandwich(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 逆差し二点底が存在するかどうかを列挙
	 * @see #bullishStickSandwich(Candlestick[], boolean)
	 */
	public static boolean[] bullishStickSandwich(final Candlestick[] c) { return bullishStickSandwich(c, true); }

	/**
	 * <p>指定されたローソク足データに逆差し二点底が存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 逆差し二点底が存在するかどうかを列挙
	 */
	public static boolean[] bullishStickSandwich(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 2; i < len; i++) {
			final int _1 = i - 2, _2 = i - 1, _3 = i;
			if (c[_1] == null || c[_2] == null || c[_3] == null)
				continue;

			// TODO: 判定が辛いかも
			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.DOWN) &&
				c[_1].getColor() == CandlestickColor.BLACK && /*c[_1].isVeryShortLowerShadow() &&*/
				c[_2].getColor() == CandlestickColor.WHITE &&
				c[_3].getColor() == CandlestickColor.BLACK && /*c[_3].isVeryShortLowerShadow() &&*/
				c[_2].containsBody(c[_1].getOpen()) &&
				c[_3].containsBodyByBody(c[_2]) &&
				c[_3].isSameClose(c[_1].getClose());
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに赤三兵先詰まりが存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bearishAdvanceBlock(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 赤三兵先詰まりが存在するかどうかを列挙
	 * @see #bearishAdvanceBlock(Candlestick[], boolean)
	 */
	public static boolean[] bearishAdvanceBlock(final Candlestick[] c) { return bearishAdvanceBlock(c, true); }

	/**
	 * <p>指定されたローソク足データに赤三兵先詰まりが存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 赤三兵先詰まりが存在するかどうかを列挙
	 */
	public static boolean[] bearishAdvanceBlock(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 2; i < len; i++) {
			final int _1 = i - 2, _2 = i - 1, _3 = i;
			if (c[_1] == null || c[_2] == null || c[_3] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.UP) &&
				c[_1].getColor() == CandlestickColor.WHITE && //c[_1].getBodyType() == Body.LONG && c[_1].isShortUpperShadow() &&
				c[_2].getColor() == CandlestickColor.WHITE && //c[_2].isLongUpperShadow() &&
				c[_3].getColor() == CandlestickColor.WHITE && //c[_3].isLongUpperShadow() &&
				c[_1].getClose() < c[_2].getClose() && c[_2].getClose() < c[_3].getClose() &&
				c[_1].containsBody(c[_2].getOpen()) && c[_2].containsBody(c[_3].getOpen()) &&
				c[_1].body() > c[_2].body() && c[_2].body() > c[_3].body();
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに赤三兵思案星が存在するかどうかを列挙して返します。</p>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bearishDeliberation(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 赤三兵思案星が存在するかどうかを列挙
	 * @see #bearishDeliberation(Candlestick[], boolean)
	 */
	public static boolean[] bearishStalledPattern(final Candlestick[] c) { return bearishDeliberation(c, true); }

	/**
	 * <p>指定されたローソク足データに赤三兵思案星が存在するかどうかを列挙して返します。</p>
	 * このメソッドは {@link #bearishDeliberation(Candlestick[], boolean)} の別称です。<br>
	 * 詳細は {@link #bearishDeliberation(Candlestick[], boolean)} を参照して下さい。
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 赤三兵思案星が存在するかどうかを列挙
	 * @see #bearishDeliberation(Candlestick[], boolean)
	 */
	public static boolean[] bearishStalledPattern(final Candlestick[] c, final boolean trend) { return bearishDeliberation(c, trend); }

	/**
	 * <p>指定されたローソク足データに赤三兵思案星が存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bearishDeliberation(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 赤三兵思案星が存在するかどうかを列挙
	 * @see #bearishDeliberation(Candlestick[], boolean)
	 */
	public static boolean[] bearishDeliberation(final Candlestick[] c) { return bearishDeliberation(c, true); }

	/**
	 * <p>指定されたローソク足データに赤三兵思案星が存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 赤三兵思案星が存在するかどうかを列挙
	 */
	public static boolean[] bearishDeliberation(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 2; i < len; i++) {
			final int _1 = i - 2, _2 = i - 1, _3 = i;
			if (c[_1] == null || c[_2] == null || c[_3] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.UP) &&
				c[_1].getColor() == CandlestickColor.WHITE && c[_1].getBodyType() == RealBodyType.LONG &&
				c[_2].getColor() == CandlestickColor.WHITE && c[_2].getBodyType() == RealBodyType.LONG &&
				c[_1].containsBody(c[_2].getOpen()) && c[_1].getClose() < c[_2].getClose() &&
				c[_2].gapUpBodyByBody(c[_3]) &&
				(c[_3].getBodyType() == RealBodyType.SMALL || c[_3].getBodyType() == RealBodyType.DOJI);
		}
		return results;
	}

	/* ---------------------------------------------------------------------- */
	/* 三本足【指標】反転【信頼性】高 */

	/**
	 * <p>指定されたローソク足データに強気の捨て子線が存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に高いと言われています。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bullishAbandonedBaby(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 強気の捨て子線が存在するかどうかを列挙
	 * @see #bullishAbandonedBaby(Candlestick[], boolean)
	 */
	public static boolean[] bullishAbandonedBaby(final Candlestick[] c) { return bullishAbandonedBaby(c, true); }

	/**
	 * <p>指定されたローソク足データに強気の捨て子線が存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に高いと言われています。<br>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 強気の捨て子線が存在するかどうかを列挙
	 */
	public static boolean[] bullishAbandonedBaby(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 2; i < len; i++) {
			final int _1 = i - 2, _2 = i - 1, _3 = i;
			if (c[_1] == null || c[_2] == null || c[_3] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.DOWN) &&
				c[_1].getColor() == CandlestickColor.BLACK && c[_1].getBodyType() == RealBodyType.LONG &&
				c[_1].gapDown(c[_2]) &&
				c[_2].getBodyType() == RealBodyType.DOJI &&
				c[_2].gapUp(c[_3]) &&
				c[_3].getColor() == CandlestickColor.WHITE && c[_3].getBodyType() == RealBodyType.SMALL &&
				c[_1].midpoint() < c[_3].getClose();
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに弱気の捨て子線が存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に高いと言われています。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bearishAbandonedBaby(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 弱気の捨て子線が存在するかどうかを列挙
	 * @see #bearishAbandonedBaby(Candlestick[], boolean)
	 */
	public static boolean[] bearishAbandonedBaby(final Candlestick[] c) { return bearishAbandonedBaby(c, true); }

	/**
	 * <p>指定されたローソク足データに弱気の捨て子線が存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に高いと言われています。<br>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 弱気の捨て子線が存在するかどうかを列挙
	 */
	public static boolean[] bearishAbandonedBaby(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 2; i < len; i++) {
			final int _1 = i - 2, _2 = i - 1, _3 = i;
			if (c[_1] == null || c[_2] == null || c[_3] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.UP) &&
				c[_1].getColor() == CandlestickColor.WHITE && c[_1].getBodyType() == RealBodyType.LONG &&
				c[_1].gapUp(c[_2]) &&
				c[_2].getBodyType() == RealBodyType.DOJI &&
				c[_2].gapDown(c[_3]) &&
				c[_3].getColor() == CandlestickColor.BLACK && c[_3].getBodyType() == RealBodyType.SMALL &&
				c[_1].midpoint() > c[_3].getClose();
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに三川明けの十字星が存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に高いと言われています。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bullishMorningDojiStar(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 三川明けの十字星が存在するかどうかを列挙
	 * @see #bullishMorningDojiStar(Candlestick[], boolean)
	 */
	public static boolean[] bullishMorningDojiStar(final Candlestick[] c) { return bullishMorningDojiStar(c, true); }

	/**
	 * <p>指定されたローソク足データに三川明けの十字星が存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に高いと言われています。<br>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 三川明けの十字星が存在するかどうかを列挙
	 */
	public static boolean[] bullishMorningDojiStar(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 2; i < len; i++) {
			final int _1 = i - 2, _2 = i - 1, _3 = i;
			if (c[_1] == null || c[_2] == null || c[_3] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.DOWN) &&
				c[_1].getColor() == CandlestickColor.BLACK && c[_1].getBodyType() == RealBodyType.LONG &&
				c[_1].gapDownBodyByBody(c[_2]) &&
				c[_2].getBodyType() == RealBodyType.DOJI &&
				c[_2].gapUpBodyByBody(c[_3]) &&
				c[_3].getColor() == CandlestickColor.WHITE && c[_3].getBodyType() != RealBodyType.DOJI &&
				c[_1].midpoint() > c[_3].getClose();
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに三川宵の十字星が存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に高いと言われています。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bearishEveningDojiStar(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 三川宵の十字星が存在するかどうかを列挙
	 * @see #bearishEveningDojiStar(Candlestick[], boolean)
	 */
	public static boolean[] bearishEveningDojiStar(final Candlestick[] c) { return bearishEveningDojiStar(c, true); }

	/**
	 * <p>指定されたローソク足データに三川宵の十字星が存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に高いと言われています。<br>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 三川宵の十字星が存在するかどうかを列挙
	 */
	public static boolean[] bearishEveningDojiStar(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 2; i < len; i++) {
			final int _1 = i - 2, _2 = i - 1, _3 = i;
			if (c[_1] == null || c[_2] == null || c[_3] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.UP) &&
				c[_1].getColor() == CandlestickColor.WHITE && c[_1].getBodyType() == RealBodyType.LONG &&
				c[_1].gapUpBodyByBody(c[_2]) &&
				c[_2].getBodyType() == RealBodyType.DOJI &&
				c[_2].gapDownBodyByBody(c[_3]) &&
				c[_3].getColor() == CandlestickColor.BLACK && c[_3].getBodyType() != RealBodyType.DOJI &&
				c[_1].midpoint() < c[_3].getClose();
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに三川明けの明星が存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に高いと言われています。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bullishMorningStar(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 三川明けの明星が存在するかどうかを列挙
	 * @see #bullishMorningStar(Candlestick[], boolean)
	 */
	public static boolean[] bullishMorningStar(final Candlestick[] c) { return bullishMorningStar(c, true); }

	/**
	 * <p>指定されたローソク足データに三川明けの明星が存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に高いと言われています。<br>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 三川明けの明星が存在するかどうかを列挙
	 */
	public static boolean[] bullishMorningStar(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 2; i < len; i++) {
			final int _1 = i - 2, _2 = i - 1, _3 = i;
			if (c[_1] == null || c[_2] == null || c[_3] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.DOWN) &&
				c[_1].getColor() == CandlestickColor.BLACK && c[_1].getBodyType() == RealBodyType.LONG &&
				c[_1].gapDownBodyByBody(c[_2]) &&
				c[_2].getBodyType() == RealBodyType.SMALL &&
				c[_2].gapUpBodyByBody(c[_3]) &&
				c[_3].getColor() == CandlestickColor.WHITE && c[_3].getBodyType() != RealBodyType.DOJI &&
				c[_1].midpoint() > c[_3].getClose();
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに三川宵の明星が存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に高いと言われています。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bearishEveningStar(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 三川宵の明星が存在するかどうかを列挙
	 * @see #bearishEveningStar(Candlestick[], boolean)
	 */
	public static boolean[] bearishEveningStar(final Candlestick[] c) { return bearishEveningStar(c, true); }

	/**
	 * <p>指定されたローソク足データに三川宵の明星が存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に高いと言われています。<br>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 三川宵の明星が存在するかどうかを列挙
	 */
	public static boolean[] bearishEveningStar(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 2; i < len; i++) {
			final int _1 = i - 2, _2 = i - 1, _3 = i;
			if (c[_1] == null || c[_2] == null || c[_3] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.UP) &&
				c[_1].getColor() == CandlestickColor.WHITE && c[_1].getBodyType() == RealBodyType.LONG &&
				c[_1].gapUpBodyByBody(c[_2]) &&
				c[_2].getBodyType() == RealBodyType.SMALL &&
				c[_2].gapDownBodyByBody(c[_3]) &&
				c[_3].getColor() == CandlestickColor.BLACK && c[_3].getBodyType() != RealBodyType.DOJI &&
				c[_1].midpoint() < c[_3].getClose();
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに包み上げが存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に高いと言われています。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bullishThreeOutsideUp(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 包み上げが存在するかどうかを列挙
	 * @see #bullishThreeOutsideUp(Candlestick[], boolean)
	 */
	public static boolean[] bullishThreeOutsideUp(final Candlestick[] c) { return bullishThreeOutsideUp(c, true); }

	/**
	 * <p>指定されたローソク足データに包み上げが存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に高いと言われています。<br>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 包み上げが存在するかどうかを列挙
	 */
	public static boolean[] bullishThreeOutsideUp(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		final boolean[] bullishEngulfing = bullishEngulfing(c, trend);
		for (int i = 2; i < len; i++) {
			final int _1 = i - 2, _2 = i - 1, _3 = i;
			if (c[_1] == null || c[_2] == null || c[_3] == null)
				continue;
			results[i] = bullishEngulfing[_2] &&
				c[_3].getColor() == CandlestickColor.WHITE &&
				c[_2].midpoint() < c[_3].getOpen() &&
				c[_2].getClose() < c[_3].getClose();
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに包み下げが存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に高いと言われています。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bearishThreeOutsideDown(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 包み下げが存在するかどうかを列挙
	 * @see #bearishThreeOutsideDown(Candlestick[], boolean)
	 */
	public static boolean[] bearishThreeOutsideDown(final Candlestick[] c) { return bearishThreeOutsideDown(c, true); }

	/**
	 * <p>指定されたローソク足データに包み下げが存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に高いと言われています。<br>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 包み下げが存在するかどうかを列挙
	 */
	public static boolean[] bearishThreeOutsideDown(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		final boolean[] bearishEngulfing = bearishEngulfing(c, trend);
		for (int i = 2; i < len; i++) {
			final int _1 = i - 2, _2 = i - 1, _3 = i;
			if (c[_1] == null || c[_2] == null || c[_3] == null)
				continue;
			results[i] = bearishEngulfing[_2] &&
				c[_3].getColor() == CandlestickColor.BLACK &&
				c[_2].midpoint() > c[_3].getOpen() &&
				c[_2].getClose() > c[_3].getClose();
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データにはらみ上げが存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に高いと言われています。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bullishThreeInsideUp(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return はらみ上げが存在するかどうかを列挙
	 * @see #bullishThreeInsideUp(Candlestick[], boolean)
	 */
	public static boolean[] bullishThreeInsideUp(final Candlestick[] c) { return bullishThreeInsideUp(c, true); }

	/**
	 * <p>指定されたローソク足データにはらみ上げが存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に高いと言われています。<br>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return はらみ上げが存在するかどうかを列挙
	 */
	public static boolean[] bullishThreeInsideUp(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		final boolean[] bullishHarami = bullishHarami(c, trend);
		for (int i = 2; i < len; i++) {
			final int _1 = i - 2, _2 = i - 1, _3 = i;
			if (c[_1] == null || c[_2] == null || c[_3] == null)
				continue;
			results[i] = bullishHarami[_2] &&
				c[_3].getColor() == CandlestickColor.WHITE &&
				c[_2].midpoint() < c[_3].getOpen() &&
				c[_2].getClose() < c[_3].getClose();			
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データにはらみ下げが存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に高いと言われています。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bearishThreeInsideDown(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return はらみ下げが存在するかどうかを列挙
	 * @see #bearishThreeInsideDown(Candlestick[], boolean)
	 */
	public static boolean[] bearishThreeInsideDown(final Candlestick[] c) { return bearishThreeInsideDown(c, true); }

	/**
	 * <p>指定されたローソク足データにはらみ下げが存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に高いと言われています。<br>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return はらみ下げが存在するかどうかを列挙
	 */
	public static boolean[] bearishThreeInsideDown(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		final boolean[] bearishHarami = bearishHarami(c, trend);
		for (int i = 2; i < len; i++) {
			final int _1 = i - 2, _2 = i - 1, _3 = i;
			if (c[_1] == null || c[_2] == null || c[_3] == null)
				continue;
			results[i] = bearishHarami[_2] &&
				c[_3].getColor() == CandlestickColor.BLACK &&
				c[_2].midpoint() > c[_3].getOpen() &&
				c[_2].getClose() > c[_3].getClose();
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに三川上放れ二羽烏が存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に高いと言われています。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bearishUpsideGapTwoCrows(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 三川上放れ二羽烏が存在するかどうかを列挙
	 * @see #bearishUpsideGapTwoCrows(Candlestick[], boolean)
	 */
	public static boolean[] bearishUpsideGapTwoCrows(final Candlestick[] c) { return bearishUpsideGapTwoCrows(c, true); }

	/**
	 * <p>指定されたローソク足データに三川上放れ二羽烏が存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に高いと言われています。<br>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 三川上放れ二羽烏が存在するかどうかを列挙
	 */
	public static boolean[] bearishUpsideGapTwoCrows(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 2; i < len; i++) {
			final int _1 = i - 2, _2 = i - 1, _3 = i;
			if (c[_1] == null || c[_2] == null || c[_3] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.UP) &&
				c[_1].getColor() == CandlestickColor.WHITE && c[_1].getBodyType() == RealBodyType.LONG &&
				c[_1].gapUpBodyByBody(c[_2]) &&
				c[_2].getColor() == CandlestickColor.BLACK &&
				c[_3].getColor() == CandlestickColor.BLACK &&
				c[_3].containsBodyByBody(c[_2]) && c[_3].contains(c[_2]) &&
				c[_1].getClose() < c[_3].getClose();
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに赤三兵が存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に高いと言われています。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bullishThreeWhiteSoldiers(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 赤三兵が存在するかどうかを列挙
	 * @see #bullishThreeWhiteSoldiers(Candlestick[], boolean)
	 */
	public static boolean[] bullishThreeWhiteSoldiers(final Candlestick[] c) { return bullishThreeWhiteSoldiers(c, true); }

	/**
	 * <p>指定されたローソク足データに赤三兵が存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に高いと言われています。<br>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 赤三兵が存在するかどうかを列挙
	 */
	public static boolean[] bullishThreeWhiteSoldiers(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 3; i < len; i++) {
			final int _0 = i - 3, _1 = i - 2, _2 = i - 1, _3 = i;
			if (c[_0] == null || c[_1] == null || c[_2] == null || c[_3] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.DOWN) &&
//				c[_0].getColor() == CandlestickColor.BLACK &&// 3-line strike でも使われるので、ここで陰線を見るのではなくトレンドの中でみる必要がある
				c[_1].getColor() == CandlestickColor.WHITE && c[_1].getBodyType() == RealBodyType.LONG &&
				c[_2].getColor() == CandlestickColor.WHITE && c[_2].getBodyType() == RealBodyType.LONG &&
				c[_3].getColor() == CandlestickColor.WHITE && c[_3].getBodyType() == RealBodyType.LONG &&
				c[_1].getClose() < c[_2].getClose() && c[_2].getClose() < c[_3].getClose() &&
				c[_1].containsBody(c[_2].getOpen()) && c[_2].containsBody(c[_3].getOpen());
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに黒三兵(三羽鳥)が存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に高いと言われています。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bearishThreeBlackCrows(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 黒三兵(三羽鳥)が存在するかどうかを列挙
	 * @see #bearishThreeBlackCrows(Candlestick[], boolean)
	 */
	public static boolean[] bearishThreeBlackCrows(final Candlestick[] c) { return bearishThreeBlackCrows(c, true); }

	/**
	 * <p>指定されたローソク足データに黒三兵(三羽鳥)が存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に高いと言われています。<br>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 黒三兵(三羽鳥)が存在するかどうかを列挙
	 */
	public static boolean[] bearishThreeBlackCrows(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 3; i < len; i++) {
			final int _0 = i - 3, _1 = i - 2, _2 = i - 1, _3 = i;
			if (c[_0] == null || c[_1] == null || c[_2] == null || c[_3] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.UP) &&
				c[_1].getColor() == CandlestickColor.BLACK && c[_1].getBodyType() == RealBodyType.LONG &&
				c[_2].getColor() == CandlestickColor.BLACK && c[_2].getBodyType() == RealBodyType.LONG &&
				c[_3].getColor() == CandlestickColor.BLACK && c[_3].getBodyType() == RealBodyType.LONG &&
				c[_1].getClose() > c[_2].getClose() && c[_2].getClose() > c[_3].getClose() &&
				c[_1].containsBody(c[_2].getOpen()) && c[_2].containsBody(c[_3].getOpen());
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに雪崩三羽烏(同時三羽)が存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に高いと言われています。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bearishIdenticalThreeCrows(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 雪崩三羽烏(同時三羽)が存在するかどうかを列挙
	 * @see #bearishIdenticalThreeCrows(Candlestick[], boolean)
	 */
	public static boolean[] bearishIdenticalThreeCrows(final Candlestick[] c) { return bearishIdenticalThreeCrows(c, true); }

	/**
	 * <p>指定されたローソク足データに雪崩三羽烏(同時三羽)が存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に高いと言われています。<br>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 雪崩三羽烏(同時三羽)が存在するかどうかを列挙
	 */
	public static boolean[] bearishIdenticalThreeCrows(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 2; i < len; i++) {
			final int _1 = i - 2, _2 = i - 1, _3 = i;
			if (c[_1] == null || c[_2] == null || c[_3] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.UP) &&
				c[_1].getColor() == CandlestickColor.BLACK && c[_1].getBodyType() == RealBodyType.LONG &&
				c[_2].getColor() == CandlestickColor.BLACK && c[_2].getBodyType() == RealBodyType.LONG &&
				c[_3].getColor() == CandlestickColor.BLACK && c[_3].getBodyType() == RealBodyType.LONG &&
				c[_1].getClose() > c[_2].getClose() && c[_2].getClose() > c[_3].getClose() &&
				c[_2].isSameOpen(c[_1].getClose()) && c[_3].isSameOpen(c[_2].getClose());
		}
		return results;
	}

	/* ---------------------------------------------------------------------- */
	/* 三本足【指標】継続【信頼性】中 */

	/**
	 * <p>指定されたローソク足データに上放れたすき線が存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、上昇トレンド継続の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bullishUpsideTasukiGap(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 上放れたすき線が存在するかどうかを列挙
	 * @see #bullishUpsideTasukiGap(Candlestick[], boolean)
	 */
	public static boolean[] bullishUpsideTasukiGap(final Candlestick[] c) { return bullishUpsideTasukiGap(c, true); }

	/**
	 * <p>指定されたローソク足データに上放れたすき線が存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、上昇トレンド継続の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 上放れたすき線が存在するかどうかを列挙
	 */
	public static boolean[] bullishUpsideTasukiGap(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 2; i < len; i++) {
			final int _1 = i - 2, _2 = i - 1, _3 = i;
			if (c[_1] == null || c[_2] == null || c[_3] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.UP) &&
				c[_1].getColor() == CandlestickColor.WHITE && c[_1].getBodyType() == RealBodyType.LONG &&
				c[_2].getColor() == CandlestickColor.WHITE && c[_2].getBodyType() == RealBodyType.LONG &&
				c[_3].getColor() == CandlestickColor.BLACK &&
				c[_1].gapUpBodyByBody(c[_2]) &&
				c[_2].containsBody(c[_3].getOpen()) && c[_1].getClose() < c[_3].getClose() && c[_2].getOpen() > c[_3].getClose() &&
				c[_3].isFarClose(c[_1].getClose());
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに下放れたすき線が存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、下降トレンド継続の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bearishDownsideTasukiGap(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 下放れたすき線が存在するかどうかを列挙
	 * @see #bearishDownsideTasukiGap(Candlestick[], boolean)
	 */
	public static boolean[] bearishDownsideTasukiGap(final Candlestick[] c) { return bearishDownsideTasukiGap(c, true); }

	/**
	 * <p>指定されたローソク足データに下放れたすき線が存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、下降トレンド継続の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 下放れたすき線が存在するかどうかを列挙
	 */
	public static boolean[] bearishDownsideTasukiGap(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 2; i < len; i++) {
			final int _1 = i - 2, _2 = i - 1, _3 = i;
			if (c[_1] == null || c[_2] == null || c[_3] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.DOWN) &&
				c[_1].getColor() == CandlestickColor.BLACK && c[_1].getBodyType() == RealBodyType.LONG &&
				c[_2].getColor() == CandlestickColor.BLACK && c[_2].getBodyType() == RealBodyType.LONG &&
				c[_3].getColor() == CandlestickColor.WHITE &&
				c[_1].gapDownBodyByBody(c[_2]) &&
				c[_2].containsBody(c[_3].getOpen()) && c[_1].getClose() > c[_3].getClose() && c[_2].getOpen() < c[_3].getClose() &&
				c[_3].isFarClose(c[_1].getClose());
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに上放れ三法が存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、上昇トレンド継続の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bullishUpsideGapThreeMethods(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 上げ放れ三法が存在するかどうかを列挙
	 * @see #bullishUpsideGapThreeMethods(Candlestick[], boolean)
	 */
	public static boolean[] bullishUpsideGapThreeMethods(final Candlestick[] c) { return bullishUpsideGapThreeMethods(c, true); }

	/**
	 * <p>指定されたローソク足データに上放れ三法が存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、上昇トレンド継続の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 上げ放れ三法が存在するかどうかを列挙
	 */
	public static boolean[] bullishUpsideGapThreeMethods(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 2; i < len; i++) {
			final int _1 = i - 2, _2 = i - 1, _3 = i;
			if (c[_1] == null || c[_2] == null || c[_3] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.UP) &&
				c[_1].getColor() == CandlestickColor.WHITE && c[_1].getBodyType() == RealBodyType.LONG &&
				c[_2].getColor() == CandlestickColor.WHITE && c[_2].getBodyType() == RealBodyType.LONG &&
				c[_3].getColor() == CandlestickColor.BLACK &&
				c[_1].gapUpBodyByBody(c[_2]) &&
				c[_2].containsBody(c[_3].getOpen()) && c[_1].getClose() > c[_3].getClose();
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに下放れ三法が存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、上昇トレンド継続の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bearishDownsideGapThreeMethods(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 下げ放れ三法が存在するかどうかを列挙
	 * @see #bearishDownsideGapThreeMethods(Candlestick[], boolean)
	 */
	public static boolean[] bearishDownsideGapThreeMethods(final Candlestick[] c) { return bearishDownsideGapThreeMethods(c, true); }

	/**
	 * <p>指定されたローソク足データに下放れ三法が存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、上昇トレンド継続の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 下げ放れ三法が存在するかどうかを列挙
	 */
	public static boolean[] bearishDownsideGapThreeMethods(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 2; i < len; i++) {
			final int _1 = i - 2, _2 = i - 1, _3 = i;
			if (c[_1] == null || c[_2] == null || c[_3] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.DOWN) &&
				c[_1].getColor() == CandlestickColor.BLACK && c[_1].getBodyType() == RealBodyType.LONG &&
				c[_2].getColor() == CandlestickColor.BLACK && c[_2].getBodyType() == RealBodyType.LONG &&
				c[_3].getColor() == CandlestickColor.WHITE &&
				c[_1].gapDownBodyByBody(c[_2]) &&
				c[_2].containsBody(c[_3].getOpen()) && c[_1].getClose() < c[_3].getClose();
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに上放れ並び赤が存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、下降トレンド継続の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bearishSideBySideWhiteLines(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 上放れ並び赤が存在するかどうかを列挙
	 * @see #bearishSideBySideWhiteLines(Candlestick[], boolean)
	 */
	public static boolean[] bearishSideBySideWhiteLines(final Candlestick[] c) { return bearishSideBySideWhiteLines(c, true); }

	/**
	 * <p>指定されたローソク足データに上放れ並び赤が存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、下降トレンド継続の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 上放れ並び赤が存在するかどうかを列挙
	 */
	public static boolean[] bearishSideBySideWhiteLines(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 2; i < len; i++) {
			final int _1 = i - 2, _2 = i - 1, _3 = i;
			if (c[_1] == null || c[_2] == null || c[_3] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.DOWN) &&
				c[_1].getColor() == CandlestickColor.BLACK &&
				c[_2].getColor() == CandlestickColor.WHITE &&
				c[_3].getColor() == CandlestickColor.WHITE &&
				c[_1].gapDownBodyByBody(c[_2]) &&
				c[_3].isSameOpen(c[_2].getOpen()) && c[_3].isSameClose(c[_2].getClose()) &&
				c[_2].getLow() > c[_3].getLow();
		}
		return results;
	}

	/* ---------------------------------------------------------------------- */
	/* 三本足【指標】継続【信頼性】高 */

	/**
	 * <p>指定されたローソク足データに下放れ並び赤が存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、上昇トレンド継続の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に高いと言われています。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bullishSideBySideWhiteLines(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 下放れ並び赤が存在するかどうかを列挙
	 * @see #bullishSideBySideWhiteLines(Candlestick[], boolean)
	 */
	public static boolean[] bullishSideBySideWhiteLines(final Candlestick[] c) { return bullishSideBySideWhiteLines(c, true); }

	/**
	 * <p>指定されたローソク足データに下放れ並び赤が存在するかどうかを列挙して返します。</p>
	 * このパターンは三本足のパターンで、上昇トレンド継続の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に高いと言われています。<br>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 下放れ並び赤が存在するかどうかを列挙
	 */
	public static boolean[] bullishSideBySideWhiteLines(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 2; i < len; i++) {
			final int _1 = i - 2, _2 = i - 1, _3 = i;
			if (c[_1] == null || c[_2] == null || c[_3] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.UP) &&
				c[_1].getColor() == CandlestickColor.WHITE &&
				c[_2].getColor() == CandlestickColor.WHITE &&
				c[_3].getColor() == CandlestickColor.WHITE &&
				c[_1].gapUpBodyByBody(c[_2]) &&
				c[_3].isSameOpen(c[_2].getOpen()) && c[_3].isSameClose(c[_2].getClose()) &&
				c[_2].getHigh() < c[_3].getHigh();
		}
		return results;
	}

	/* ---------------------------------------------------------------------- */
	/* 四本足【指標】反転【信頼性】高 */

	/**
	 * <p>指定されたローソク足データに小燕包みが存在するかどうかを列挙して返します。</p>
	 * このパターンは四本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に高いと言われています。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bullishConcealingBabySwallow(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 小燕包みが存在するかどうかを列挙
	 * @see #bullishConcealingBabySwallow(Candlestick[], boolean)
	 */
	public static boolean[] bullishConcealingBabySwallow(final Candlestick[] c) { return bullishConcealingBabySwallow(c, true); }

	/**
	 * <p>指定されたローソク足データに小燕包みが存在するかどうかを列挙して返します。</p>
	 * このパターンは四本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に高いと言われています。<br>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 小燕包みが存在するかどうかを列挙
	 */
	public static boolean[] bullishConcealingBabySwallow(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 3; i < len; i++) {
			final int _1 = i - 3, _2 = i - 2, _3 = i - 1, _4 = i;
			if (c[_1] == null || c[_2] == null || c[_3] == null || c[_4] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.DOWN) &&
				c[_1].getColor() == CandlestickColor.BLACK && c[_1].isMarubozu() &&
				c[_2].getColor() == CandlestickColor.BLACK && c[_2].isMarubozu() &&
				c[_3].getColor() == CandlestickColor.BLACK && c[_3].getUpperShadowType() == ShadowType.LONG &&
				c[_4].getColor() == CandlestickColor.BLACK && c[_4].isMarubozu() &&
				c[_2].containsBody(c[_3].getHigh()) && c[_2].gapDownBodyByBody(c[_3]) &&
				c[_4].containsBody(c[_3]);
		}
		return results;
	}

	/* ---------------------------------------------------------------------- */
	/* 四本足【指標】継続【信頼性】低 */

	/**
	 * <p>指定されたローソク足データに三手打ちが存在するかどうかを列挙して返します。</p>
	 * このパターンは四本足のパターンで、上昇トレンド継続の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に低いと言われています。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bullishThreeLineStrike(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 三手打ちが存在するかどうかを列挙
	 * @see #bullishThreeLineStrike(Candlestick[], boolean)
	 */
	public static boolean[] bullishThreeLineStrike(final Candlestick[] c) { return bullishThreeLineStrike(c, true); }

	/**
	 * <p>指定されたローソク足データに三手打ちが存在するかどうかを列挙して返します。</p>
	 * このパターンは四本足のパターンで、上昇トレンド継続の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に低いと言われています。<br>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 三手打ちが存在するかどうかを列挙
	 */
	public static boolean[] bullishThreeLineStrike(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		final boolean[] threeWhiteSoldiers = bullishThreeWhiteSoldiers(c, false); 
		for (int i = 3; i < len; i++) {
			final int _1 = i - 3, _2 = i - 2, _3 = i - 1, _4 = i;
			if (c[_1] == null || c[_2] == null || c[_3] == null || c[_4] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.UP) &&
				threeWhiteSoldiers[_3] && c[_4].getColor() == CandlestickColor.BLACK
				&& c[_4].getOpen() > c[_3].getClose() && c[_4].getClose() < c[_1].getOpen();
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに三手打ちが存在するかどうかを列挙して返します。</p>
	 * このパターンは四本足のパターンで、下降トレンド継続の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に低いと言われています。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bearishThreeLineStrike(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 三手打ちが存在するかどうかを列挙
	 * @see #bearishThreeLineStrike(Candlestick[], boolean)
	 */
	public static boolean[] bearishThreeLineStrike(final Candlestick[] c) { return bearishThreeLineStrike(c, true); }

	/**
	 * <p>指定されたローソク足データに三手打ちが存在するかどうかを列挙して返します。</p>
	 * このパターンは四本足のパターンで、下降トレンド継続の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に低いと言われています。<br>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 三手打ちが存在するかどうかを列挙
	 */
	public static boolean[] bearishThreeLineStrike(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		final boolean[] threeBlackCrows = bearishThreeBlackCrows(c, false); 
		for (int i = 3; i < len; i++) {
			final int _1 = i - 3, _2 = i - 2, _3 = i - 1, _4 = i;
			if (c[_1] == null || c[_2] == null || c[_3] == null || c[_4] == null)
				continue;
			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.DOWN) &&
				threeBlackCrows[_3] && c[_4].getColor() == CandlestickColor.WHITE
				&& c[_4].getOpen() < c[_3].getClose() && c[_4].getClose() > c[_1].getOpen();
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに三空踏み上げが存在するかどうかを列挙して返します。</p>
	 * このパターンは四本足のパターンで、上昇トレンド継続の可能性を示唆します。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bullishThreeGaps(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 三空踏み上げが存在するかどうかを列挙
	 * @see #bullishThreeGaps(Candlestick[], boolean)
	 */
	public static boolean[] bullishThreeGaps(final Candlestick[] c) { return bullishThreeGaps(c, true); }

	/**
	 * <p>指定されたローソク足データに三空踏み上げが存在するかどうかを列挙して返します。</p>
	 * このパターンは四本足のパターンで、上昇トレンド継続の可能性を示唆します。<br>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 三空踏み上げが存在するかどうかを列挙
	 */
	public static boolean[] bullishThreeGaps(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 3; i < len; i++) {
			final int _1 = i - 3, _2 = i - 2, _3 = i - 1, _4 = i;
			if (c[_1] == null || c[_2] == null || c[_3] == null || c[_4] == null)
				continue;
			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.UP) &&
				c[_1].getColor() == CandlestickColor.WHITE &&
				c[_1].gapUpBodyByBody(c[_2]) &&
				c[_2].getColor() == CandlestickColor.WHITE &&
				c[_2].gapUpBodyByBody(c[_3]) &&
				c[_3].getColor() == CandlestickColor.WHITE &&
				c[_3].gapUpBodyByBody(c[_4]) &&
				c[_4].getColor() == CandlestickColor.WHITE;
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに三空叩き込みが存在するかどうかを列挙して返します。</p>
	 * このパターンは四本足のパターンで、上昇トレンド継続の可能性を示唆します。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bearishThreeGaps(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 三空叩き込みが存在するかどうかを列挙
	 * @see #bearishThreeGaps(Candlestick[], boolean)
	 */
	public static boolean[] bearishThreeGaps(final Candlestick[] c) {
		return bearishThreeGaps(c, true);
	}

	/**
	 * <p>指定されたローソク足データに三空叩き込みが存在するかどうかを列挙して返します。</p>
	 * このパターンは四本足のパターンで、上昇トレンド継続の可能性を示唆します。<br>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 三空叩き込みが存在するかどうかを列挙
	 */
	public static boolean[] bearishThreeGaps(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 3; i < len; i++) {
			final int _1 = i - 3, _2 = i - 2, _3 = i - 1, _4 = i;
			if (c[_1] == null || c[_2] == null || c[_3] == null || c[_4] == null)
				continue;
			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.DOWN) &&
				c[_1].getColor() == CandlestickColor.BLACK &&
				c[_1].gapDownBodyByBody(c[_2]) &&
				c[_2].getColor() == CandlestickColor.BLACK &&
				c[_2].gapDownBodyByBody(c[_3]) &&
				c[_3].getColor() == CandlestickColor.BLACK &&
				c[_3].gapDownBodyByBody(c[_4]) &&
				c[_4].getColor() == CandlestickColor.BLACK;
		}
		return results;
	}

	/* ---------------------------------------------------------------------- */
	/* 五本足【指標】反転【信頼性】中 */

	/**
	 * <p>指定されたローソク足データに放れ三手の新値が存在するかどうかを列挙して返します。</p>
	 * このパターンは五本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bullishBreakaway(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 放れ三手の新値が存在するかどうかを列挙
	 * @see #bullishBreakaway(Candlestick[], boolean)
	 */
	public static boolean[] bullishBreakaway(final Candlestick[] c) { return bullishBreakaway(c, true); }

	/**
	 * <p>指定されたローソク足データに放れ三手の新値が存在するかどうかを列挙して返します。</p>
	 * このパターンは五本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 放れ三手の新値が存在するかどうかを列挙
	 */
	public static boolean[] bullishBreakaway(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 4; i < len; i++) {
			final int _1 = i - 4, _2 = i - 3, _3 = i - 2, _4 = i - 1, _5 = i;
			if (c[_1] == null || c[_2] == null || c[_3] == null || c[_4] == null || c[_5] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.DOWN) &&
				c[_1].getColor() == CandlestickColor.BLACK &&
				c[_2].getColor() == CandlestickColor.BLACK &&
				c[_4].getColor() == CandlestickColor.BLACK &&
				c[_5].getColor() == CandlestickColor.WHITE &&
				c[_1].getBodyType() == RealBodyType.LONG && c[_5].getBodyType() == RealBodyType.LONG &&
				c[_1].gapDownBodyByBody(c[_2]) &&
				c[_2].getHigh() > c[_3].getHigh() && c[_3].getHigh() > c[_4].getHigh() &&
				c[_2].getLow() > c[_3].getLow() && c[_3].getLow() > c[_4].getLow() &&
				c[_5].getClose() > c[_2].upperBody() && c[_5].getClose() < c[_1].lowerBody();
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに放れ三手の新値が存在するかどうかを列挙して返します。</p>
	 * このパターンは五本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bearishBreakaway(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 放れ三手の新値が存在するかどうかを列挙
	 * @see #bearishBreakaway(Candlestick[], boolean)
	 */
	public static boolean[] bearishBreakaway(final Candlestick[] c) { return bearishBreakaway(c, true); }

	/**
	 * <p>指定されたローソク足データに放れ三手の新値が存在するかどうかを列挙して返します。</p>
	 * このパターンは五本足のパターンで、上昇トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 放れ三手の新値が存在するかどうかを列挙
	 */
	public static boolean[] bearishBreakaway(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 4; i < len; i++) {
			final int _1 = i - 4, _2 = i - 3, _3 = i - 2, _4 = i - 1, _5 = i;
			if (c[_1] == null || c[_2] == null || c[_3] == null || c[_4] == null || c[_5] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.UP) &&
				c[_1].getColor() == CandlestickColor.WHITE &&
				c[_2].getColor() == CandlestickColor.WHITE &&
				c[_4].getColor() == CandlestickColor.WHITE &&
				c[_5].getColor() == CandlestickColor.BLACK &&
				c[_1].getBodyType() == RealBodyType.LONG && c[_5].getBodyType() == RealBodyType.LONG &&
				c[_1].gapUpBodyByBody(c[_2]) &&
				c[_2].getHigh() < c[_3].getHigh() && c[_3].getHigh() < c[_4].getHigh() &&
				c[_2].getLow() < c[_3].getLow() && c[_3].getLow() < c[_4].getLow() &&
				c[_5].getClose() < c[_2].lowerBody() && c[_5].getClose() > c[_1].upperBody();
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データにはしご底が存在するかどうかを列挙して返します。</p>
	 * このパターンは五本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bullishLadderBottom(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return はしご底が存在するかどうかを列挙
	 * @see #bullishLadderBottom(Candlestick[], boolean)
	 */
	public static boolean[] bullishLadderBottom(final Candlestick[] c) { return bullishLadderBottom(c, true); }

	/**
	 * <p>指定されたローソク足データにはしご底が存在するかどうかを列挙して返します。</p>
	 * このパターンは五本足のパターンで、下降トレンド反転の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に普通と言われています。<br>
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return はしご底が存在するかどうかを列挙
	 */
	public static boolean[] bullishLadderBottom(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		final boolean[] threeBlackCrows = bearishThreeBlackCrows(c, false);
		final boolean[] invertedHammer = bullishInvertedHammer(c, trend, false);
		for (int i = 4; i < len; i++) {
			final int _1 = i - 4, _2 = i - 3, _3 = i - 2, _4 = i - 1, _5 = i;
			if (c[_1] == null || c[_2] == null || c[_3] == null || c[_4] == null || c[_5] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.DOWN) &&
				threeBlackCrows[_3] && invertedHammer[_4] &&
				c[_4].getColor() == CandlestickColor.BLACK &&
				c[_5].getColor() == CandlestickColor.WHITE &&
				c[_4].upperBody() < c[_5].getOpen();
		}
		return results;
	}

	/* ---------------------------------------------------------------------- */
	/* 五本足【指標】継続【信頼性】高 */

	/**
	 * <p>指定されたローソク足データに上げ三法が存在するかどうかを列挙して返します。</p>
	 * このパターンは五本足のパターンで、上昇トレンド継続の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に高いと言われています。
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bullishRisingThreeMethods(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 上げ三法が存在するかどうかを列挙
	 * @see #bullishRisingThreeMethods(Candlestick[], boolean)
	 */
	public static boolean[] bullishRisingThreeMethods(final Candlestick[] c) { return bullishRisingThreeMethods(c, true); }

	/**
	 * <p>指定されたローソク足データに上げ三法が存在するかどうかを列挙して返します。</p>
	 * このパターンは五本足のパターンで、上昇トレンド継続の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に高いと言われています。
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 上げ三法が存在するかどうかを列挙
	 */
	public static boolean[] bullishRisingThreeMethods(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 4; i < len; i++) {
			final int _1 = i - 4, _2 = i - 3, _3 = i - 2, _4 = i - 1, _5 = i;
			if (c[_1] == null || c[_2] == null || c[_3] == null || c[_4] == null || c[_5] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.UP) &&
				c[_1].getBodyType() == RealBodyType.LONG && c[_1].getColor() == CandlestickColor.WHITE &&
				c[_2].getBodyType() == RealBodyType.SMALL && c[_2].getColor() == CandlestickColor.BLACK &&
				c[_3].getBodyType() == RealBodyType.SMALL &&
				c[_4].getBodyType() == RealBodyType.SMALL && c[_4].getColor() == CandlestickColor.BLACK &&
				c[_5].getBodyType() == RealBodyType.LONG && c[_5].getColor() == CandlestickColor.WHITE &&
				c[_2].upperBody() > c[_3].upperBody() && c[_3].upperBody() > c[_4].upperBody() &&
				c[_2].lowerBody() > c[_3].lowerBody() && c[_3].lowerBody() > c[_4].lowerBody() &&
				c[_1].containsBody(c[_2].getClose()) && c[_1].containsBody(c[_3].getClose()) && c[_1].containsBody(c[_4].getClose()) &&
				c[_1].getClose() < c[_5].getClose();
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに下げ三法が存在するかどうかを列挙して返します。</p>
	 * このパターンは五本足のパターンで、下降トレンド継続の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に高いと言われています。
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bearishFallingThreeMethods(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 下げ三法が存在するかどうかを列挙
	 * @see #bearishFallingThreeMethods(Candlestick[], boolean)
	 */
	public static boolean[] bearishFallingThreeMethods(final Candlestick[] c) { return bearishFallingThreeMethods(c, true); }

	/**
	 * <p>指定されたローソク足データに下げ三法が存在するかどうかを列挙して返します。</p>
	 * このパターンは五本足のパターンで、下降トレンド継続の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に高いと言われています。
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 下げ三法が存在するかどうかを列挙
	 */
	public static boolean[] bearishFallingThreeMethods(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 4; i < len; i++) {
			final int _1 = i - 4, _2 = i - 3, _3 = i - 2, _4 = i - 1, _5 = i;
			if (c[_1] == null || c[_2] == null || c[_3] == null || c[_4] == null || c[_5] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.DOWN) &&
				c[_1].getBodyType() == RealBodyType.LONG && c[_1].getColor() == CandlestickColor.BLACK &&
				c[_2].getBodyType() == RealBodyType.SMALL && c[_2].getColor() == CandlestickColor.WHITE &&
				c[_3].getBodyType() == RealBodyType.SMALL &&
				c[_4].getBodyType() == RealBodyType.SMALL && c[_4].getColor() == CandlestickColor.WHITE &&
				c[_5].getBodyType() == RealBodyType.LONG && c[_5].getColor() == CandlestickColor.BLACK &&
				c[_2].upperBody() < c[_3].upperBody() && c[_3].upperBody() < c[_4].upperBody() &&
				c[_2].lowerBody() < c[_3].lowerBody() && c[_3].lowerBody() < c[_4].lowerBody() &&
				c[_1].containsBody(c[_2].getClose()) && c[_1].containsBody(c[_3].getClose()) && c[_1].containsBody(c[_4].getClose()) &&
				c[_1].getClose() > c[_5].getClose();
		}
		return results;
	}

	/**
	 * <p>指定されたローソク足データに押え込み線が存在するかどうかを列挙して返します。</p>
	 * このパターンは五本足のパターンで、上昇トレンド継続の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に高いと言われています。
	 * <p>
	 * このメソッドは利便性の為に提供しています。<br>
	 * 実装は {@link #bullishMatHold(Candlestick[], boolean)} をトレンドを確認するモードで呼出すだけです。
	 * 
	 * @param c ローソク足データ
	 * @return 押え込み線が存在するかどうかを列挙
	 * @see #bullishMatHold(Candlestick[], boolean)
	 */
	public static boolean[] bullishMatHold(final Candlestick[] c) { return bullishMatHold(c, true); }

	/**
	 * <p>指定されたローソク足データに押え込み線が存在するかどうかを列挙して返します。</p>
	 * このパターンは五本足のパターンで、上昇トレンド継続の可能性を示唆します。<br>
	 * このパターンの信頼性は一般的に高いと言われています。
	 * 
	 * @param c ローソク足データ
	 * @param trend トレンドを確認するかどうか
	 * @return 押え込み線が存在するかどうかを列挙
	 */
	public static boolean[] bullishMatHold(final Candlestick[] c, final boolean trend) {
		final int len = c.length;
		final boolean[] results = new boolean[len];
		for (int i = 4; i < len; i++) {
			final int _1 = i - 4, _2 = i - 3, _3 = i - 2, _4 = i - 1, _5 = i;
			if (c[_1] == null || c[_2] == null || c[_3] == null || c[_4] == null || c[_5] == null)
				continue;

			results[i] = (!trend | c[_1].getTrend() == null | c[_1].getTrend() == TrendType.UP) &&
				c[_1].getBodyType() == RealBodyType.LONG && c[_1].getColor() == CandlestickColor.WHITE &&
				c[_1].gapUpBodyByBody(c[_2]) &&
				c[_2].getBodyType() == RealBodyType.SMALL && c[_2].getColor() == CandlestickColor.BLACK &&
				c[_3].getBodyType() == RealBodyType.SMALL &&
				c[_4].getBodyType() == RealBodyType.SMALL &&
				c[_5].getColor() == CandlestickColor.WHITE &&
				c[_1].getClose() > c[_3].lowerBody() &&
				c[_1].getClose() > c[_4].lowerBody() &&
				c[_2].getOpen() > c[_3].upperBody() &&
				c[_3].upperBody() > c[_4].upperBody() &&
				c[_4].getClose() < c[_5].getOpen() &&
				c[_2].getHigh() < c[_5].getClose() && c[_3].getHigh() < c[_5].getClose() && c[_4].getHigh() < c[_5].getClose();
		}
		return results;
	}

}
