/*
 * Copyright (c) 2006-2009 OrangeSignal.com All rights reserved.
 * 
 * これは Apache ライセンス Version 2.0 (以下、このライセンスと記述) に
 * 従っています。このライセンスに準拠する場合以外、このファイルを使用
 * してはなりません。このライセンスのコピーは以下から入手できます。
 * 
 * http://www.apache.org/licenses/LICENSE-2.0.txt
 * 
 * 適用可能な法律がある、あるいは文書によって明記されている場合を除き、
 * このライセンスの下で配布されているソフトウェアは、明示的であるか暗黙の
 * うちであるかを問わず、「保証やあらゆる種類の条件を含んでおらず」、
 * 「あるがまま」の状態で提供されるものとします。
 * このライセンスが適用される特定の許諾と制限については、このライセンス
 * を参照してください。
 */

package jp.sf.orangesignal.ta.data.model;

import java.io.Serializable;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.text.DateFormat;
import java.util.Date;

import jp.sf.orangesignal.ta.data.CompressType;
import jp.sf.orangesignal.ta.data.annotation.DateItem;
import jp.sf.orangesignal.ta.data.annotation.NumberCompressorType;
import jp.sf.orangesignal.ta.data.annotation.NumberItem;

/**
 * 株式分割データのモデルクラスを提供します。
 * 
 * @author 杉澤 浩二
 * @since 2.2
 */
@NumberCompressorType(CompressType.SUM)
public class StockSplit implements Serializable, Comparable<StockSplit> {

	private static final long serialVersionUID = 3270022426162558867L;

	/**
	 * 日時を保持します。
	 */
	@DateItem
	protected Date date;

	/**
	 * 株式分割数を保持します。
	 */
	@NumberItem
	@NumberCompressorType(CompressType.SUM)
	protected Number split;

	/**
	 * デフォルトコンストラクタです。
	 */
	public StockSplit() {}

	/**
	 * 日時を返します。
	 * 
	 * @return 日時
	 */
	public Date getDate() { return date; }

	/**
	 * 日時を設定します。
	 * 
	 * @param date 日時
	 */
	public void setDate(final Date date) { this.date = date; }

	/**
	 * 株式分割数を返します。
	 * 
	 * @return 株式分割数
	 */
	public Number getSplit() { return split; }

	/**
	 * 株式分割数を設定します。
	 * 
	 * @param split 株式分割数
	 */
	public void setSplit(final Number split) { this.split = split; }

	/**
	 * 実装は日時で比較します。
	 */
	@Override public int compareTo(final StockSplit o) { return date.compareTo(o.date); }

	@Override
	public String toString() {
		final StringBuilder sb = new StringBuilder();
		final Field[] fields = this.getClass().getDeclaredFields();
		for (final Field field : fields) {
			if (Modifier.isStatic(field.getModifiers())) {
				continue;
			}
			if (sb.length() > 0) {
				sb.append(", ");
			}
			sb.append(field.getName()).append('=');
			try {
				final Object value = field.get(this);
				if (value == null) {
					sb.append("null");
				} else if (value instanceof Date) {
					sb.append(DateFormat.getDateTimeInstance().format((Date) value));
				} else {
					sb.append(value.toString());
				}
			} catch (IllegalAccessException  e) {
				throw new IllegalStateException(e.getMessage(), e);
			}
		}
		return new StringBuilder().append(getClass().getSimpleName()).append(' ').append(sb).toString();
	}

}
