/*
 * Copyright (c) 2006-2009 OrangeSignal.com All rights reserved.
 * 
 * これは Apache ライセンス Version 2.0 (以下、このライセンスと記述) に
 * 従っています。このライセンスに準拠する場合以外、このファイルを使用
 * してはなりません。このライセンスのコピーは以下から入手できます。
 * 
 * http://www.apache.org/licenses/LICENSE-2.0.txt
 * 
 * 適用可能な法律がある、あるいは文書によって明記されている場合を除き、
 * このライセンスの下で配布されているソフトウェアは、明示的であるか暗黙の
 * うちであるかを問わず、「保証やあらゆる種類の条件を含んでおらず」、
 * 「あるがまま」の状態で提供されるものとします。
 * このライセンスが適用される特定の許諾と制限については、このライセンス
 * を参照してください。
 */

package jp.sf.orangesignal.ta.util;

import java.text.MessageFormat;
import java.util.Hashtable;
import java.util.Map;
import java.util.ResourceBundle;

/**
 * パッケージ単位の文字列リソース管理クラスを提供します。
 * 
 * @author <a href="mailto:sugisawa@orangesignal.com">杉澤 浩二</a>
 */
public final class StringManager {

	/**
	 * 文字列リソースを保持します。
	 */
	private ResourceBundle bundle;

	/**
	 * インスタンス化できない事を強制します。
	 * 
	 * @param packageName パッケージ名
	 */
	private StringManager(final String packageName) {
		this.bundle = ResourceBundle.getBundle(packageName + ".LocalStrings");
	}

	/**
	 * アプリケーションリソースからキーに対するメッセージを取得して返します。
	 * 
	 * @param key キー
	 * @param obj 書式
	 * @return メッセージ
	 */
	public String getString(final String key, final Object... obj) {
		final String s = this.bundle.getString(key);
		if (obj != null && obj.length > 0)
			return new MessageFormat(s).format(obj);
		return s;
	}

	/**
	 * <code>StringManager</code> のマッピングリストを保持します。
	 */
	private static final Map<String, StringManager> managers = new Hashtable<String, StringManager>();

	/**
	 * 指定されたパッケージ単位の <code>StringManager</code> オブジェクトを返します。
	 * 
	 * @param cls パッケージ名
	 * @return <code>StringManager</code> オブジェクト
	 */
	public static StringManager getManager(final Class<?> cls) {
		return getManager(ClassUtils.getPackageName(cls));
	}

	/**
	 * 指定されたパッケージ単位の <code>StringManager</code> オブジェクトを返します。
	 * 
	 * @param packageName パッケージ名
	 * @return <code>StringManager</code> オブジェクト
	 */
	public synchronized static StringManager getManager(final String packageName) {
		StringManager manager = managers.get(packageName);
		if (manager == null) {
			manager = new StringManager(packageName);
			managers.put(packageName, manager);
		}
		return manager;
	}

}
