/*
 * Copyright (c) 2009 OrangeSignal.com All rights reserved.
 * 
 * これは Apache ライセンス Version 2.0 (以下、このライセンスと記述) に
 * 従っています。このライセンスに準拠する場合以外、このファイルを使用
 * してはなりません。このライセンスのコピーは以下から入手できます。
 * 
 * http://www.apache.org/licenses/LICENSE-2.0.txt
 * 
 * 適用可能な法律がある、あるいは文書によって明記されている場合を除き、
 * このライセンスの下で配布されているソフトウェアは、明示的であるか暗黙の
 * うちであるかを問わず、「保証やあらゆる種類の条件を含んでおらず」、
 * 「あるがまま」の状態で提供されるものとします。
 * このライセンスが適用される特定の許諾と制限については、このライセンス
 * を参照してください。
 */

package jp.sf.orangesignal.ta.data;

import static org.hamcrest.core.Is.is;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.fail;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.Locale;
import java.util.Set;
import java.util.SortedMap;
import java.util.TreeMap;

import org.junit.Ignore;
import org.junit.Test;

/**
 * {@link DataConvertUtils} クラスの単体テストです。
 * 
 * @author 杉澤 浩二
 */
public class DataConvertUtilsTest {

	// ------------------------------------------------------------------------
	// toDate

	@Ignore("テスト対象外とする")
	@Test
	public void testToDateObject() {
		fail("まだ実装されていません");
	}

	@Test
	public void testToDateObjectDateFormatConfig() {
		DateFormatConfig cfg = null;

		final Date date = new Date();
		assertEquals(DataConvertUtils.toDate(date, cfg), date);

		final Calendar calendar = Calendar.getInstance();
		assertEquals(DataConvertUtils.toDate(calendar, cfg), calendar.getTime());

		final Number number = Long.valueOf(date.getTime());
		assertEquals(DataConvertUtils.toDate(number, cfg), date);

		final String str = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss.SSS").format(date);
		assertEquals(DataConvertUtils.toDate(str, new DateFormatConfig("yyyy/MM/dd HH:mm:ss.SSS", null, null)), date);

		Object obj = null;
		cfg = null;
		assertNull(DataConvertUtils.toDate(obj, cfg));
	}

	@Test
	public void testToDateNumber() {
		final Date date = new Date();
		final Number number = Long.valueOf(date.getTime());
		assertEquals(DataConvertUtils.toDate(number), date);
	}

	@Test
	public void testToDateNumberClassOfT() {
		final Date date = new Date();
		final Number number = Long.valueOf(date.getTime());
		assertEquals(DataConvertUtils.toDate(number, java.util.Date.class), date);
		assertEquals(DataConvertUtils.toDate(number, java.sql.Date.class), date);
	}

	// ------------------------------------------------------------------------
	// toNumber

	@Ignore("テスト対象外とする")
	@Test
	public void testToNumberObject() {
		fail("まだ実装されていません");
	}

	@Test
	public void testToNumberObjectNumberFormatConfig() {
		NumberFormatConfig cfg = null;

		final Date date = new Date();
		assertEquals(DataConvertUtils.toNumber(date, cfg), date.getTime());

		final Calendar calendar = Calendar.getInstance();
		assertEquals(DataConvertUtils.toNumber(calendar, cfg), calendar.getTimeInMillis());

		assertEquals(DataConvertUtils.toNumber("123,456.78", new NumberFormatConfig("#,##0.00", null, null)), 123456.78);
		assertEquals(DataConvertUtils.toNumber("JPY 123,456.78", new NumberFormatConfig("\u00A4\u00A4 #,##0.00", Locale.JAPAN, "JPY")), 123456.78);
		assertEquals(DataConvertUtils.toNumber("USD 123,456.78", new NumberFormatConfig("\u00A4\u00A4 #,##0.00", Locale.JAPAN, "USD")), 123456.78);

		Object obj = null;
		cfg = null;
		assertNull(DataConvertUtils.toNumber(obj, cfg));
	}

	// ------------------------------------------------------------------------
	// toDateArray

/*
	@Test
	public void testToDateArrayObjectDateFormatConfig() {
		fail("まだ実装されていません");
	}
*/

	@Test
	public void testToDateArrayCollectionOfQ() throws ParseException {
		final SimpleDateFormat ymd = new SimpleDateFormat("yyyy/MM/dd");

		// Arrange
		final Collection<Date> c1 = new ArrayList<Date>(3);
		c1.add(ymd.parse("2008/01/01"));
		c1.add(ymd.parse("2008/01/02"));
		c1.add(ymd.parse("2008/01/03"));
		// Act
		final Date[] r1 = DataConvertUtils.toDateArray(c1);
		// Assert
		assertThat(r1.length, is(3));
		assertThat(r1[0], is(ymd.parse("2008/01/01")));
		assertThat(r1[1], is(ymd.parse("2008/01/02")));
		assertThat(r1[2], is(ymd.parse("2008/01/03")));

		// Arrange
		final Collection<Number> c2 = new ArrayList<Number>(3);
		c2.add(ymd.parse("2008/01/01").getTime());
		c2.add(ymd.parse("2008/01/02").getTime());
		c2.add(ymd.parse("2008/01/03").getTime());
		// Act
		final Date[] r2 = DataConvertUtils.toDateArray(c2);
		// Assert
		assertThat(r2.length, is(3));
		assertThat(r2[0], is(ymd.parse("2008/01/01")));
		assertThat(r2[1], is(ymd.parse("2008/01/02")));
		assertThat(r2[2], is(ymd.parse("2008/01/03")));

		// Arrange
		final Collection<?> c4 = null;
		// Act + Assert
		assertNull(DataConvertUtils.toDateArray(c4));
	}

	@Test(expected = IllegalArgumentException.class)
	public void testToDateArrayCollectionOfQIllegalArgumentException() {
		// Arrange
		final Collection<String> c3 = new ArrayList<String>(3);
		c3.add("2008/01/01");
		c3.add("2008/01/02");
		c3.add("2008/01/03");
		// Act
		DataConvertUtils.toDateArray(c3);
	}

	@Test
	public void testToDateArrayCollectionOfQDateFormatConfig() throws ParseException {
		final SimpleDateFormat ymd = new SimpleDateFormat("yyyy/MM/dd");

		// Arrange
		final Collection<Date> c1 = new ArrayList<Date>(3);
		c1.add(ymd.parse("2008/01/01"));
		c1.add(ymd.parse("2008/01/02"));
		c1.add(ymd.parse("2008/01/03"));
		// Act
		final Date[] r1 = DataConvertUtils.toDateArray(c1, null);
		// Assert
		assertThat(r1.length, is(3));
		assertThat(r1[0], is(ymd.parse("2008/01/01")));
		assertThat(r1[1], is(ymd.parse("2008/01/02")));
		assertThat(r1[2], is(ymd.parse("2008/01/03")));

		// Arrange
		final Collection<Number> c2 = new ArrayList<Number>(3);
		c2.add(ymd.parse("2008/01/01").getTime());
		c2.add(ymd.parse("2008/01/02").getTime());
		c2.add(ymd.parse("2008/01/03").getTime());
		// Act
		final Date[] r2 = DataConvertUtils.toDateArray(c2, null);
		// Assert
		assertThat(r2.length, is(3));
		assertThat(r2[0], is(ymd.parse("2008/01/01")));
		assertThat(r2[1], is(ymd.parse("2008/01/02")));
		assertThat(r2[2], is(ymd.parse("2008/01/03")));

		// Arrange
		final Collection<String> c3 = new ArrayList<String>(3);
		c3.add("2008/01/01");
		c3.add("2008/01/02");
		c3.add("2008/01/03");
		// Act
		final Date[] r3 = DataConvertUtils.toDateArray(c3, new DateFormatConfig("yyyy/MM/dd", null, null));
		// Assert
		assertThat(r3.length, is(3));
		assertThat(r3[0], is(ymd.parse("2008/01/01")));
		assertThat(r3[1], is(ymd.parse("2008/01/02")));
		assertThat(r3[2], is(ymd.parse("2008/01/03")));

		// Arrange
		final Collection<?> c4 = null;
		// Act + Assert
		assertNull(DataConvertUtils.toDateArray(c4, null));
	}

	@Test
	public void testToDateArrayObjectArray() throws ParseException {
		final SimpleDateFormat ymd = new SimpleDateFormat("yyyy/MM/dd");

		// Arrange + Act
		final Date[] r1 = DataConvertUtils.toDateArray(
			new Date[]{
				ymd.parse("2008/01/01"),
				ymd.parse("2008/01/02"),
				ymd.parse("2008/01/03")
			});
		// Assert
		assertThat(r1.length, is(3));
		assertThat(r1[0], is(ymd.parse("2008/01/01")));
		assertThat(r1[1], is(ymd.parse("2008/01/02")));
		assertThat(r1[2], is(ymd.parse("2008/01/03")));

		// Arrange + Act
		final Date[] r2 = DataConvertUtils.toDateArray(
			new Number[]{
				ymd.parse("2008/01/01").getTime(),
				ymd.parse("2008/01/02").getTime(),
				ymd.parse("2008/01/03").getTime()
			});
		// Assert
		assertThat(r2.length, is(3));
		assertThat(r2[0], is(ymd.parse("2008/01/01")));
		assertThat(r2[1], is(ymd.parse("2008/01/02")));
		assertThat(r2[2], is(ymd.parse("2008/01/03")));

		// Arrange
		final Object[] obj = null;
		// Act + Assert
		assertNull(DataConvertUtils.toDateArray(obj));
	}

	@Test(expected = IllegalArgumentException.class)
	public void testToDateArrayObjectArrayIllegalArgumentException() {
		// Act
		DataConvertUtils.toDateArray(new String[]{ "2008/01/01", "2008/01/02", "2008/01/03" });
	}

	@Test
	public void testToDateArrayObjectArrayDateFormatConfig() throws ParseException {
		final SimpleDateFormat ymd = new SimpleDateFormat("yyyy/MM/dd");

		// Arrange + Act
		final Date[] r1 = DataConvertUtils.toDateArray(
			new Date[]{
				ymd.parse("2008/01/01"),
				ymd.parse("2008/01/02"),
				ymd.parse("2008/01/03")
			}, null);
		// Assert
		assertThat(r1.length, is(3));
		assertThat(r1[0], is(ymd.parse("2008/01/01")));
		assertThat(r1[1], is(ymd.parse("2008/01/02")));
		assertThat(r1[2], is(ymd.parse("2008/01/03")));

		// Arrange + Act
		final Date[] r2 = DataConvertUtils.toDateArray(
			new Number[]{
				ymd.parse("2008/01/01").getTime(),
				ymd.parse("2008/01/02").getTime(),
				ymd.parse("2008/01/03").getTime()
			}, null);
		// Assert
		assertThat(r2.length, is(3));
		assertThat(r2[0], is(ymd.parse("2008/01/01")));
		assertThat(r2[1], is(ymd.parse("2008/01/02")));
		assertThat(r2[2], is(ymd.parse("2008/01/03")));

		// Arrange + Act
		final Date[] r3 = DataConvertUtils.toDateArray(
			new String[]{
				"2008/01/01",
				"2008/01/02",
				"2008/01/03"
			}, new DateFormatConfig("yyyy/MM/dd", null, null));
		// Assert
		assertThat(r3.length, is(3));
		assertThat(r3[0], is(ymd.parse("2008/01/01")));
		assertThat(r3[1], is(ymd.parse("2008/01/02")));
		assertThat(r3[2], is(ymd.parse("2008/01/03")));

		// Arrange
		final Object[] obj = null;
		// Act + Assert
		assertNull(DataConvertUtils.toDateArray(obj, null));
	}

	// ------------------------------------------------------------------------
	// toNumberArray

	@Test
	public void testToNumberArrayIntArray() {
		int[] data = new int[]{ 1,2,3,0,4,5 };
		final Number[] results = DataConvertUtils.toNumberArray(data);
		assertThat(results.length, is(data.length));
		assertEquals(results[0], 1);
		assertEquals(results[1], 2);
		assertEquals(results[2], 3);
		assertEquals(results[3], 0);
		assertEquals(results[4], 4);
		assertEquals(results[5], 5);

		data = null;
		assertNull(DataConvertUtils.toNumberArray(data));
	}

	@Test
	public void testToNumberArrayLongArray() {
		long[] data = new long[]{ 1,2,3,0,4,5 };
		final Number[] results = DataConvertUtils.toNumberArray(data);
		assertThat(results.length, is(data.length));
		assertEquals(results[0], 1L);
		assertEquals(results[1], 2L);
		assertEquals(results[2], 3L);
		assertEquals(results[3], 0L);
		assertEquals(results[4], 4L);
		assertEquals(results[5], 5L);

		data = null;
		assertNull(DataConvertUtils.toNumberArray(data));
	}

	@Test
	public void testToNumberArrayFloatArray() {
		float[] data = new float[]{ 1,2,3,0,4,5 };
		final Number[] results = DataConvertUtils.toNumberArray(data);
		assertThat(results.length, is(data.length));
		assertEquals(results[0], 1F);
		assertEquals(results[1], 2F);
		assertEquals(results[2], 3F);
		assertEquals(results[3], 0F);
		assertEquals(results[4], 4F);
		assertEquals(results[5], 5F);

		data = null;
		assertNull(DataConvertUtils.toNumberArray(data));
	}

	@Test
	public void testToNumberArrayDoubleArray() {
		double[] data = new double[]{ 1,2,3,0,4,5 };
		final Number[] results = DataConvertUtils.toNumberArray(data);
		assertThat(results.length, is(data.length));
		assertEquals(results[0], 1D);
		assertEquals(results[1], 2D);
		assertEquals(results[2], 3D);
		assertEquals(results[3], 0D);
		assertEquals(results[4], 4D);
		assertEquals(results[5], 5D);

		data = null;
		assertNull(DataConvertUtils.toNumberArray(data));
	}

	// ------------------------------------------------------------------------

	@Test
	public void testToNumberArrayCollectionOfQ() {
		// Arrange
		final Collection<Number> c2 = new ArrayList<Number>(3);
		c2.add(1000.01D);
		c2.add(2000.02D);
		c2.add(3000.03D);
		// Act
		final Number[] r2 = DataConvertUtils.toNumberArray(c2);
		// Assert
		assertThat(r2.length, is(3));
		assertThat(r2[0].doubleValue(), is(1000.01));
		assertThat(r2[1].doubleValue(), is(2000.02));
		assertThat(r2[2].doubleValue(), is(3000.03));

		// Arrange
		final Collection<Number> c4 = null;
		// Act + Assert
		assertNull(DataConvertUtils.toNumberArray(c4));
	}

	@Test(expected = IllegalArgumentException.class)
	public void testToNumberArrayCollectionOfQIllegalArgumentException() {
		// Arrange
		final Collection<String> c3 = new ArrayList<String>(3);
		c3.add("USD 1,000.01");
		c3.add("USD 2,000.02");
		c3.add("USD 3,000.03");
		// Act
		DataConvertUtils.toNumberArray(c3);
	}

	@Test
	public void testToNumberArrayCollectionOfQNumberFormatConfig() {
		// Arrange
		final Collection<Number> c2 = new ArrayList<Number>(3);
		c2.add(1000.01D);
		c2.add(2000.02D);
		c2.add(3000.03D);
		// Act
		final Number[] r2 = DataConvertUtils.toNumberArray(c2, null);
		// Assert
		assertThat(r2.length, is(3));
		assertThat(r2[0].doubleValue(), is(1000.01));
		assertThat(r2[1].doubleValue(), is(2000.02));
		assertThat(r2[2].doubleValue(), is(3000.03));

		// Arrange
		final Collection<String> c3 = new ArrayList<String>(3);
		c3.add("USD 1,000.01");
		c3.add("USD 2,000.02");
		c3.add("USD 3,000.03");
		// Act
		final Number[] r3 = DataConvertUtils.toNumberArray(c3, new NumberFormatConfig("\u00A4\u00A4 #,##0.00", Locale.JAPAN, "USD"));
		// Assert
		assertThat(r3.length, is(3));
		assertThat(r3[0].doubleValue(), is(1000.01));
		assertThat(r3[1].doubleValue(), is(2000.02));
		assertThat(r3[2].doubleValue(), is(3000.03));

		// Arrange
		final Collection<Number> c4 = null;
		// Act + Assert
		assertNull(DataConvertUtils.toNumberArray(c4, null));
	}

	@Test
	public void testToNumberArrayObjectArray() {
		// Arrange + Act
		final Number[] r2 = DataConvertUtils.toNumberArray(new Number[]{ 1000.01D, 2000.02D, 3000.03D });
		// Assert
		assertThat(r2.length, is(3));
		assertThat(r2[0].doubleValue(), is(1000.01));
		assertThat(r2[1].doubleValue(), is(2000.02));
		assertThat(r2[2].doubleValue(), is(3000.03));

		// Arrange
		final Object[] obj = null;
		// Act + Assert
		assertNull(DataConvertUtils.toNumberArray(obj, null));
	}

	@Test(expected = IllegalArgumentException.class)
	public void testToNumberArrayObjectArrayIllegalArgumentException() {
		// Act
		DataConvertUtils.toNumberArray(new String[]{ "USD 1,000.01", "USD 2,000.02", "USD 3,000.03" });
	}

	@Test
	public void testToNumberArrayObjectArrayNumberFormatConfig() {
		// Arrange + Act
		final Number[] r2 = DataConvertUtils.toNumberArray(new Number[]{ 1000.01D, 2000.02D, 3000.03D }, null);
		// Assert
		assertThat(r2.length, is(3));
		assertThat(r2[0].doubleValue(), is(1000.01));
		assertThat(r2[1].doubleValue(), is(2000.02));
		assertThat(r2[2].doubleValue(), is(3000.03));

		// Arrange + Act
		final Number[] r3 = DataConvertUtils.toNumberArray(
				new String[]{ "USD 1,000.01", "USD 2,000.02", "USD 3,000.03" },
				new NumberFormatConfig("\u00A4\u00A4 #,##0.00", Locale.JAPAN, "USD")
			);
		// Assert
		assertThat(r3.length, is(3));
		assertThat(r3[0].doubleValue(), is(1000.01));
		assertThat(r3[1].doubleValue(), is(2000.02));
		assertThat(r3[2].doubleValue(), is(3000.03));

		// Arrange
		final Object[] obj = null;
		// Act + Assert
		assertNull(DataConvertUtils.toNumberArray(obj, null));
	}

	// ------------------------------------------------------------------------
	// toNumberArrayWithNullIfZero

	@Test
	public void testToNumberArrayWithNullIfZeroIntArray() {
		int[] data = new int[]{ 1,2,3,0,4,5 };
		final Number[] results = DataConvertUtils.toNumberArrayWithNullIfZero(data);
		assertThat(results.length, is(data.length));
		assertEquals(results[0], 1);
		assertEquals(results[1], 2);
		assertEquals(results[2], 3);
		assertNull(results[3]);
		assertEquals(results[4], 4);
		assertEquals(results[5], 5);

		data = null;
		assertNull(DataConvertUtils.toNumberArrayWithNullIfZero(data));
	}

	@Test
	public void testToNumberArrayWithNullIfZeroLongArray() {
		long[] data = new long[]{ 1,2,3,0,4,5 };
		final Number[] results = DataConvertUtils.toNumberArrayWithNullIfZero(data);
		assertThat(results.length, is(data.length));
		assertEquals(results[0], 1L);
		assertEquals(results[1], 2L);
		assertEquals(results[2], 3L);
		assertNull(results[3]);
		assertEquals(results[4], 4L);
		assertEquals(results[5], 5L);

		data = null;
		assertNull(DataConvertUtils.toNumberArrayWithNullIfZero(data));
	}

	@Test
	public void testToNumberArrayWithNullIfZeroFloatArray() {
		// Arrange
		float[] data = new float[]{ 1,2,3,0,4,5 };
		// Act
		final Number[] results = DataConvertUtils.toNumberArrayWithNullIfZero(data);
		// Assert
		assertThat(results.length, is(data.length));
		assertEquals(results[0], 1F);
		assertEquals(results[1], 2F);
		assertEquals(results[2], 3F);
		assertNull(results[3]);
		assertEquals(results[4], 4F);
		assertEquals(results[5], 5F);

		// Arrange
		data = null;
		// Act + Assert
		assertNull(DataConvertUtils.toNumberArrayWithNullIfZero(data));
	}

	@Test
	public void testToNumberArrayWithNullIfZeroDoubleArray() {
		// Arrange
		double[] data = new double[]{ 1,2,3,0,4,5 };
		// Act
		final Number[] results = DataConvertUtils.toNumberArrayWithNullIfZero(data);
		// Assert
		assertThat(results.length, is(data.length));
		assertEquals(results[0], 1D);
		assertEquals(results[1], 2D);
		assertEquals(results[2], 3D);
		assertNull(results[3]);
		assertEquals(results[4], 4D);
		assertEquals(results[5], 5D);

		// Arrange
		data = null;
		// Act + Assert
		assertNull(DataConvertUtils.toNumberArrayWithNullIfZero(data));
	}

	// ------------------------------------------------------------------------
	// toNumberArrayWithPreviousIfZero

	@Test
	public void testToNumberArrayWithPreviousIfZeroIntArray() {
		// Arrange
		int[] data = new int[]{ 1,2,3,0,4,5 };
		// Act
		final Number[] results = DataConvertUtils.toNumberArrayWithPreviousIfZero(data);
		// Assert
		assertThat(results.length, is(data.length));
		assertEquals(results[0], 1);
		assertEquals(results[1], 2);
		assertEquals(results[2], 3);
		assertEquals(results[3], 3);
		assertEquals(results[4], 4);
		assertEquals(results[5], 5);

		data = null;
		assertNull(DataConvertUtils.toNumberArrayWithPreviousIfZero(data));
	}

	@Test
	public void testToNumberArrayWithPreviousIfZeroLongArray() {
		// Arrange
		long[] data = new long[]{ 1,2,3,0,4,5 };
		// Act
		final Number[] results = DataConvertUtils.toNumberArrayWithPreviousIfZero(data);
		// Assert
		assertThat(results.length, is(data.length));
		assertEquals(results[0], 1L);
		assertEquals(results[1], 2L);
		assertEquals(results[2], 3L);
		assertEquals(results[3], 3L);
		assertEquals(results[4], 4L);
		assertEquals(results[5], 5L);

		data = null;
		assertNull(DataConvertUtils.toNumberArrayWithPreviousIfZero(data));
	}

	@Test
	public void testToNumberArrayWithPreviousIfZeroFloatArray() {
		// Arrange
		float[] data = new float[]{ 1,2,3,0,4,5 };
		// Act
		final Number[] results = DataConvertUtils.toNumberArrayWithPreviousIfZero(data);
		// Assert
		assertThat(results.length, is(data.length));
		assertEquals(results[0], 1F);
		assertEquals(results[1], 2F);
		assertEquals(results[2], 3F);
		assertEquals(results[3], 3F);
		assertEquals(results[4], 4F);
		assertEquals(results[5], 5F);

		data = null;
		assertNull(DataConvertUtils.toNumberArrayWithPreviousIfZero(data));
	}

	@Test
	public void testToNumberArrayWithPreviousIfZeroDoubleArray() {
		// Arrange
		double[] data = new double[]{ 1,2,3,0,4,5 };
		// Act
		final Number[] results = DataConvertUtils.toNumberArrayWithPreviousIfZero(data);
		// Assert
		assertThat(results.length, is(data.length));
		assertEquals(results[0], 1D);
		assertEquals(results[1], 2D);
		assertEquals(results[2], 3D);
		assertEquals(results[3], 3D);
		assertEquals(results[4], 4D);
		assertEquals(results[5], 5D);

		data = null;
		assertNull(DataConvertUtils.toNumberArrayWithPreviousIfZero(data));
	}

	// ------------------------------------------------------------------------
	// zeroIfNull

	@Test
	public void testZeroIfNull() {
		// Arrange
		Number[] data = new Number[]{ 1D,2D,3D,null,4D,5D };
		// Act
		final Number[] results = DataConvertUtils.zeroIfNull(data);
		// Assert
		assertThat(results.length, is(data.length));
		assertEquals(results[0], 1D);
		assertEquals(results[1], 2D);
		assertEquals(results[2], 3D);
		assertEquals(results[3], 0);
		assertEquals(results[4], 4D);
		assertEquals(results[5], 5D);

		data = null;
		assertNull(DataConvertUtils.zeroIfNull(data));
	}

	@Test
	public void testPreviousIfNull() {
		// Arrange
		Number[] data = new Number[]{ 1D,2D,3D,null,4D,5D };
		// Act
		final Number[] results = DataConvertUtils.previousIfNull(data);
		// Assert
		assertEquals(results.length, data.length);
		assertEquals(results[0], 1D);
		assertEquals(results[1], 2D);
		assertEquals(results[2], 3D);
		assertEquals(results[3], 3D);
		assertEquals(results[4], 4D);
		assertEquals(results[5], 5D);

		data = null;
		assertNull(DataConvertUtils.previousIfNull(data));
	}

	// ------------------------------------------------------------------------
	// toSortedMap

	@Test
	public void testToSortedMap() throws ParseException {
		///////////////////////////////////////////////////////////////////////
		// Arrange

		final SimpleDateFormat df = new SimpleDateFormat("yyyy/MM/dd");

		final Date[] date = new Date[]{
			df.parse("2009/01/03"),
			df.parse("2009/01/02"),
			df.parse("2009/01/04"),
			df.parse("2009/01/05"),
			df.parse("2009/01/07"),
			df.parse("2009/01/08")
		};

		final Number[] data = new Number[]{ 300, 200, 400, 500, 700, null };

		///////////////////////////////////////////////////////////////////////

		// Act
		final SortedMap<Date, Number> r1 = DataConvertUtils.toSortedMap(date, data);
		// Assert
		assertThat(r1.size(), is(6));
		assertThat(r1.get(df.parse("2009/01/02")).intValue(), is(200));
		assertThat(r1.get(df.parse("2009/01/03")).intValue(), is(300));
		assertThat(r1.get(df.parse("2009/01/04")).intValue(), is(400));
		assertThat(r1.get(df.parse("2009/01/05")).intValue(), is(500));
		assertThat(r1.get(df.parse("2009/01/07")).intValue(), is(700));
		assertNull(r1.get(df.parse("2009/01/08")));

		// Act
		final SortedMap<Date, Number> r2 = DataConvertUtils.toSortedMap(date, null);
		// Assert
		assertThat(r2.size(), is(6));
		assertNull(r2.get(df.parse("2009/01/02")));
		assertNull(r2.get(df.parse("2009/01/03")));
		assertNull(r2.get(df.parse("2009/01/04")));
		assertNull(r2.get(df.parse("2009/01/05")));
		assertNull(r2.get(df.parse("2009/01/07")));
		assertNull(r2.get(df.parse("2009/01/08")));
	}

	@Test(expected = IllegalArgumentException.class)
	public void testToSortedMapIllegalArgumentException() {
		// Act
		DataConvertUtils.toSortedMap(null, new Number[]{ 300, 200, 400, 500, 700, null });
	}

	// ------------------------------------------------------------------------
	// merge

	@Test
	public void testMergeDateArrayDateArrayTArray() throws ParseException {
		///////////////////////////////////////////////////////////////////////
		// Arrange

		final SimpleDateFormat df = new SimpleDateFormat("yyyy/MM/dd");

		final Date[] base = new Date[]{
			df.parse("2009/01/03"),
			df.parse("2009/01/06"),
			df.parse("2009/01/05"),
			df.parse("2009/01/10")
		};

		final Date[] date = new Date[]{
			df.parse("2009/01/03"),
			df.parse("2009/01/02"),
			df.parse("2009/01/04"),
			df.parse("2009/01/05"),
			df.parse("2009/01/07"),
			df.parse("2009/01/08")
		};

		final Number[] data = new Number[]{ 300, 200, 400, 500, 700, null };

		///////////////////////////////////////////////////////////////////////

		// Act
		final SortedMap<Date, Number> r1 = DataConvertUtils.merge(base, date, data);
		// Assert
		assertThat(r1.size(), is(4));
		assertThat(r1.get(df.parse("2009/01/03")).intValue(), is(400));
		assertThat(r1.get(df.parse("2009/01/05")).intValue(), is(500));
		assertNull(r1.get(df.parse("2009/01/06")));
		assertNull(r1.get(df.parse("2009/01/10")));
	}

	@Test
	public void testMergeDateArrayDateArrayTArrayMergeMatchTypeMergeGapFillTypeT() throws ParseException {
		///////////////////////////////////////////////////////////////////////
		// Arrange

		final SimpleDateFormat df = new SimpleDateFormat("yyyy/MM/dd");

		final Date[] base = new Date[]{
			df.parse("2009/01/03"),
			df.parse("2009/01/06"),
			df.parse("2009/01/05"),
			df.parse("2009/01/10")
		};

		final Date[] date = new Date[]{
			df.parse("2009/01/03"),
			df.parse("2009/01/02"),
			df.parse("2009/01/04"),
			df.parse("2009/01/05"),
			df.parse("2009/01/07"),
			df.parse("2009/01/08")
		};

		final Number[] data = new Number[]{ 300, 200, 400, 500, 700, null };

		///////////////////////////////////////////////////////////////////////

		// Act
		final SortedMap<Date, Number> r1 = DataConvertUtils.merge(base, date, data, MergeMatchType.STRICT, null, null);
		// Assert
		assertThat(r1.size(), is(4));
		assertThat(r1.get(df.parse("2009/01/03")).intValue(), is(300));
		assertThat(r1.get(df.parse("2009/01/05")).intValue(), is(500));
		assertNull(r1.get(df.parse("2009/01/06")));
		assertNull(r1.get(df.parse("2009/01/10")));

		// Act
		final SortedMap<Date, Number> r2 = DataConvertUtils.merge(base, date, data, MergeMatchType.STRICT, MergeGapFillType.PARAMETER, 0);
		// Assert
		assertThat(r2.size(), is(4));
		assertThat(r2.get(df.parse("2009/01/03")).intValue(), is(300));
		assertThat(r2.get(df.parse("2009/01/05")).intValue(), is(500));
		assertThat(r2.get(df.parse("2009/01/06")).intValue(), is(0));
		assertThat(r2.get(df.parse("2009/01/10")).intValue(), is(0));

		// Act
		final SortedMap<Date, Number> r3 = DataConvertUtils.merge(base, date, data, MergeMatchType.STRICT, MergeGapFillType.PREVIOUS, null);
		// Assert
		assertThat(r3.size(), is(4));
		assertThat(r3.get(df.parse("2009/01/03")).intValue(), is(300));
		assertThat(r3.get(df.parse("2009/01/05")).intValue(), is(500));
		assertThat(r3.get(df.parse("2009/01/06")).intValue(), is(500));
		assertThat(r3.get(df.parse("2009/01/10")).intValue(), is(500));

		// Act
		final SortedMap<Date, Number> r4 = DataConvertUtils.merge(base, date, data, MergeMatchType.CURRENT, null, null);
		// Assert
		assertThat(r4.size(), is(4));
		assertThat(r4.get(df.parse("2009/01/03")).intValue(), is(400));
		assertThat(r4.get(df.parse("2009/01/05")).intValue(), is(500));
		assertNull(r4.get(df.parse("2009/01/06")));
		assertNull(r4.get(df.parse("2009/01/10")));

		// Act
		final SortedMap<Date, Number> r5 = DataConvertUtils.merge(base, date, data, MergeMatchType.CURRENT, MergeGapFillType.PARAMETER, 0);
		// Assert
		assertThat(r5.size(), is(4));
		assertThat(r5.get(df.parse("2009/01/03")).intValue(), is(400));
		assertThat(r5.get(df.parse("2009/01/05")).intValue(), is(500));
		assertNull(r5.get(df.parse("2009/01/06")));
		assertThat(r5.get(df.parse("2009/01/10")).intValue(), is(0));

		// Act
		final SortedMap<Date, Number> r6 = DataConvertUtils.merge(base, date, data, MergeMatchType.CURRENT, MergeGapFillType.PREVIOUS, null);
		// Assert
		assertThat(r6.size(), is(4));
		assertThat(r6.get(df.parse("2009/01/03")).intValue(), is(400));
		assertThat(r6.get(df.parse("2009/01/05")).intValue(), is(500));
		assertNull(r6.get(df.parse("2009/01/06")));
		assertNull(r6.get(df.parse("2009/01/10")));

		// Act
		final SortedMap<Date, Number> r7 = DataConvertUtils.merge(base, date, data, MergeMatchType.INSERT, null, null);
		// Assert
		assertThat(r7.size(), is(8));
		assertThat(r7.get(df.parse("2009/01/02")).intValue(), is(200));
		assertThat(r7.get(df.parse("2009/01/03")).intValue(), is(300));
		assertThat(r7.get(df.parse("2009/01/04")).intValue(), is(400));
		assertThat(r7.get(df.parse("2009/01/05")).intValue(), is(500));
		assertNull(r7.get(df.parse("2009/01/06")));
		assertThat(r7.get(df.parse("2009/01/07")).intValue(), is(700));
		assertNull(r7.get(df.parse("2009/01/08")));
		assertNull(r7.get(df.parse("2009/01/10")));

		// Act
		final SortedMap<Date, Number> r8 = DataConvertUtils.merge(base, date, data, MergeMatchType.INSERT, MergeGapFillType.PARAMETER, 0);
		// Assert
		assertThat(r8.size(), is(8));
		assertThat(r8.get(df.parse("2009/01/02")).intValue(), is(200));
		assertThat(r8.get(df.parse("2009/01/03")).intValue(), is(300));
		assertThat(r8.get(df.parse("2009/01/04")).intValue(), is(400));
		assertThat(r8.get(df.parse("2009/01/05")).intValue(), is(500));
		assertThat(r8.get(df.parse("2009/01/06")).intValue(), is(0));
		assertThat(r8.get(df.parse("2009/01/07")).intValue(), is(700));
		assertNull(r8.get(df.parse("2009/01/08")));
		assertThat(r8.get(df.parse("2009/01/10")).intValue(), is(0));

		// Act
		final SortedMap<Date, Number> r9 = DataConvertUtils.merge(base, date, data, MergeMatchType.INSERT, MergeGapFillType.PREVIOUS, null);
		// Assert
		assertThat(r9.size(), is(8));
		assertThat(r9.get(df.parse("2009/01/02")).intValue(), is(200));
		assertThat(r9.get(df.parse("2009/01/03")).intValue(), is(300));
		assertThat(r9.get(df.parse("2009/01/04")).intValue(), is(400));
		assertThat(r9.get(df.parse("2009/01/05")).intValue(), is(500));
		assertThat(r9.get(df.parse("2009/01/06")).intValue(), is(500));
		assertThat(r9.get(df.parse("2009/01/07")).intValue(), is(700));
		assertNull(r9.get(df.parse("2009/01/08")));
		assertNull(r9.get(df.parse("2009/01/10")));
	}

	@Test
	public void testMergeCollectionOfDateSortedMapOfDateT() throws ParseException {
		///////////////////////////////////////////////////////////////////////
		// Arrange

		final SimpleDateFormat df = new SimpleDateFormat("yyyy/MM/dd");

		final Set<Date> base = new HashSet<Date>();
		base.add(df.parse("2009/01/03"));
		base.add(df.parse("2009/01/06"));
		base.add(df.parse("2009/01/05"));
		base.add(df.parse("2009/01/10"));

		final SortedMap<Date, Number> data = new TreeMap<Date, Number>();
		data.put(df.parse("2009/01/03"), 300);
		data.put(df.parse("2009/01/02"), 200);
		data.put(df.parse("2009/01/04"), 400);
		data.put(df.parse("2009/01/05"), 500);
		data.put(df.parse("2009/01/07"), 700);
		data.put(df.parse("2009/01/08"), null);

		///////////////////////////////////////////////////////////////////////

		// Act
		final SortedMap<Date, Number> r1 = DataConvertUtils.merge(base, data);
		// Assert
		assertThat(r1.size(), is(4));
		assertThat(r1.get(df.parse("2009/01/03")).intValue(), is(400));
		assertThat(r1.get(df.parse("2009/01/05")).intValue(), is(500));
		assertNull(r1.get(df.parse("2009/01/06")));
		assertNull(r1.get(df.parse("2009/01/10")));
	}

	@Test
	public void testMergeCollectionOfDateSortedMapOfDateTMergeMatchTypeMergeGapFillTypeT() throws ParseException {
		///////////////////////////////////////////////////////////////////////
		// Arrange

		final SimpleDateFormat df = new SimpleDateFormat("yyyy/MM/dd");

		final Set<Date> base = new HashSet<Date>();
		base.add(df.parse("2009/01/03"));
		base.add(df.parse("2009/01/06"));
		base.add(df.parse("2009/01/05"));
		base.add(df.parse("2009/01/10"));

		final SortedMap<Date, Number> data = new TreeMap<Date, Number>();
		data.put(df.parse("2009/01/03"), 300);
		data.put(df.parse("2009/01/02"), 200);
		data.put(df.parse("2009/01/04"), 400);
		data.put(df.parse("2009/01/05"), 500);
		data.put(df.parse("2009/01/07"), 700);
		data.put(df.parse("2009/01/08"), null);

		///////////////////////////////////////////////////////////////////////

		// Act
		final SortedMap<Date, Number> r1 = DataConvertUtils.merge(base, data, MergeMatchType.STRICT, null, null);
		// Assert
		assertThat(r1.size(), is(4));
		assertThat(r1.get(df.parse("2009/01/03")).intValue(), is(300));
		assertThat(r1.get(df.parse("2009/01/05")).intValue(), is(500));
		assertNull(r1.get(df.parse("2009/01/06")));
		assertNull(r1.get(df.parse("2009/01/10")));

		// Act
		final SortedMap<Date, Number> r2 = DataConvertUtils.merge(base, data, MergeMatchType.STRICT, MergeGapFillType.PARAMETER, 0);
		// Assert
		assertThat(r2.size(), is(4));
		assertThat(r2.get(df.parse("2009/01/03")).intValue(), is(300));
		assertThat(r2.get(df.parse("2009/01/05")).intValue(), is(500));
		assertThat(r2.get(df.parse("2009/01/06")).intValue(), is(0));
		assertThat(r2.get(df.parse("2009/01/10")).intValue(), is(0));

		// Act
		final SortedMap<Date, Number> r3 = DataConvertUtils.merge(base, data, MergeMatchType.STRICT, MergeGapFillType.PREVIOUS, null);
		// Assert
		assertThat(r3.size(), is(4));
		assertThat(r3.get(df.parse("2009/01/03")).intValue(), is(300));
		assertThat(r3.get(df.parse("2009/01/05")).intValue(), is(500));
		assertThat(r3.get(df.parse("2009/01/06")).intValue(), is(500));
		assertThat(r3.get(df.parse("2009/01/10")).intValue(), is(500));

		// Act
		final SortedMap<Date, Number> r4 = DataConvertUtils.merge(base, data, MergeMatchType.CURRENT, null, null);
		// Assert
		assertThat(r4.size(), is(4));
		assertThat(r4.get(df.parse("2009/01/03")).intValue(), is(400));
		assertThat(r4.get(df.parse("2009/01/05")).intValue(), is(500));
		assertNull(r4.get(df.parse("2009/01/06")));
		assertNull(r4.get(df.parse("2009/01/10")));

		// Act
		final SortedMap<Date, Number> r5 = DataConvertUtils.merge(base, data, MergeMatchType.CURRENT, MergeGapFillType.PARAMETER, 0);
		// Assert
		assertThat(r5.size(), is(4));
		assertThat(r5.get(df.parse("2009/01/03")).intValue(), is(400));
		assertThat(r5.get(df.parse("2009/01/05")).intValue(), is(500));
		assertNull(r5.get(df.parse("2009/01/06")));
		assertThat(r5.get(df.parse("2009/01/10")).intValue(), is(0));

		// Act
		final SortedMap<Date, Number> r6 = DataConvertUtils.merge(base, data, MergeMatchType.CURRENT, MergeGapFillType.PREVIOUS, null);
		// Assert
		assertThat(r6.size(), is(4));
		assertThat(r6.get(df.parse("2009/01/03")).intValue(), is(400));
		assertThat(r6.get(df.parse("2009/01/05")).intValue(), is(500));
		assertNull(r6.get(df.parse("2009/01/06")));
		assertNull(r6.get(df.parse("2009/01/10")));

		// Act
		final SortedMap<Date, Number> r7 = DataConvertUtils.merge(base, data, MergeMatchType.INSERT, null, null);
		// Assert
		assertThat(r7.size(), is(8));
		assertThat(r7.get(df.parse("2009/01/02")).intValue(), is(200));
		assertThat(r7.get(df.parse("2009/01/03")).intValue(), is(300));
		assertThat(r7.get(df.parse("2009/01/04")).intValue(), is(400));
		assertThat(r7.get(df.parse("2009/01/05")).intValue(), is(500));
		assertNull(r7.get(df.parse("2009/01/06")));
		assertThat(r7.get(df.parse("2009/01/07")).intValue(), is(700));
		assertNull(r7.get(df.parse("2009/01/08")));
		assertNull(r7.get(df.parse("2009/01/10")));

		// Act
		final SortedMap<Date, Number> r8 = DataConvertUtils.merge(base, data, MergeMatchType.INSERT, MergeGapFillType.PARAMETER, 0);
		// Assert
		assertThat(r8.size(), is(8));
		assertThat(r8.get(df.parse("2009/01/02")).intValue(), is(200));
		assertThat(r8.get(df.parse("2009/01/03")).intValue(), is(300));
		assertThat(r8.get(df.parse("2009/01/04")).intValue(), is(400));
		assertThat(r8.get(df.parse("2009/01/05")).intValue(), is(500));
		assertThat(r8.get(df.parse("2009/01/06")).intValue(), is(0));
		assertThat(r8.get(df.parse("2009/01/07")).intValue(), is(700));
		assertNull(r8.get(df.parse("2009/01/08")));
		assertThat(r8.get(df.parse("2009/01/10")).intValue(), is(0));

		// Act
		final SortedMap<Date, Number> r9 = DataConvertUtils.merge(base, data, MergeMatchType.INSERT, MergeGapFillType.PREVIOUS, null);
		// Assert
		assertThat(r9.size(), is(8));
		assertThat(r9.get(df.parse("2009/01/02")).intValue(), is(200));
		assertThat(r9.get(df.parse("2009/01/03")).intValue(), is(300));
		assertThat(r9.get(df.parse("2009/01/04")).intValue(), is(400));
		assertThat(r9.get(df.parse("2009/01/05")).intValue(), is(500));
		assertThat(r9.get(df.parse("2009/01/06")).intValue(), is(500));
		assertThat(r9.get(df.parse("2009/01/07")).intValue(), is(700));
		assertNull(r9.get(df.parse("2009/01/08")));
		assertNull(r9.get(df.parse("2009/01/10")));
	}

/*
	@Test
	public void testMergeMapOfDateTSortedMapOfDateTMergeMatchTypeMergeGapFillTypeT() {
		
	}
*/

	// ------------------------------------------------------------------------
	// compress

	@Test
	public void testCompressDateArrayNumberArrayCompressTypeInt() throws ParseException {
		///////////////////////////////////////////////////////////////////////
		// Arrange

		final SimpleDateFormat ymd = new SimpleDateFormat("yyyy/MM/dd");
		final SimpleDateFormat ym = new SimpleDateFormat("yyyy/MM");

		final SortedMap<Date, Number> data = new TreeMap<Date, Number>();
		data.put(ymd.parse("2009/02/03"), 230);
		data.put(ymd.parse("2009/02/04"), 300);
		data.put(ymd.parse("2009/02/05"), 200);
		data.put(ymd.parse("2009/02/06"), 260);
		data.put(ymd.parse("2009/01/02"), 120);
		data.put(ymd.parse("2009/01/03"), 130);
		data.put(ymd.parse("2009/01/04"), 140);
		data.put(ymd.parse("2009/03/04"), 340);
		data.put(ymd.parse("2009/03/05"), 350);
		data.put(ymd.parse("2009/03/06"), 360);
		data.put(ymd.parse("2009/04/05"), null);
		data.put(ymd.parse("2009/04/06"), 460);
		data.put(ymd.parse("2009/04/07"), null);
		data.put(ymd.parse("2009/05/07"), 570);

		///////////////////////////////////////////////////////////////////////

		// Act
		final SortedMap<Date, Number> r0 = DataConvertUtils.compress(data.keySet().toArray(new Date[0]), null, CompressType.FIRST, Calendar.DATE);
		// Assert
		assertNotNull(r0);

		// Act
		final SortedMap<Date, Number> r1 = DataConvertUtils.compress(data.keySet().toArray(new Date[0]), data.values().toArray(new Number[0]), CompressType.FIRST, Calendar.DATE);
		// Assert
		assertThat(r1.size(), is(5));
		assertThat(r1.get(ym.parse("2009/01")).intValue(), is(120));
		assertThat(r1.get(ym.parse("2009/02")).intValue(), is(230));
		assertThat(r1.get(ym.parse("2009/03")).intValue(), is(340));
		assertNull(r1.get(ym.parse("2009/04")));
		assertThat(r1.get(ym.parse("2009/05")).intValue(), is(570));

		// Act
		final SortedMap<Date, Number> r2 = DataConvertUtils.compress(data.keySet().toArray(new Date[0]), data.values().toArray(new Number[0]), CompressType.HIGHEST, Calendar.DATE);
		// Assert
		assertThat(r2.get(ym.parse("2009/01")).intValue(), is(140));
		assertThat(r2.get(ym.parse("2009/02")).intValue(), is(300));
		assertThat(r2.get(ym.parse("2009/03")).intValue(), is(360));
		assertThat(r2.get(ym.parse("2009/04")).intValue(), is(460));
		assertThat(r2.get(ym.parse("2009/05")).intValue(), is(570));

		// Act
		final SortedMap<Date, Number> r3 = DataConvertUtils.compress(data.keySet().toArray(new Date[0]), data.values().toArray(new Number[0]), CompressType.LOWEST, Calendar.DATE);
		// Assert
		assertThat(r3.get(ym.parse("2009/01")).intValue(), is(120));
		assertThat(r3.get(ym.parse("2009/02")).intValue(), is(200));
		assertThat(r3.get(ym.parse("2009/03")).intValue(), is(340));
		assertThat(r3.get(ym.parse("2009/04")).intValue(), is(460));
		assertThat(r3.get(ym.parse("2009/05")).intValue(), is(570));

		// Act
		final SortedMap<Date, Number> r4 = DataConvertUtils.compress(data.keySet().toArray(new Date[0]), data.values().toArray(new Number[0]), CompressType.LAST, Calendar.DATE);
		// Assert
		assertThat(r4.get(ym.parse("2009/01")).intValue(), is(140));
		assertThat(r4.get(ym.parse("2009/02")).intValue(), is(260));
		assertThat(r4.get(ym.parse("2009/03")).intValue(), is(360));
		assertNull(r4.get(ym.parse("2009/04")));
		assertThat(r4.get(ym.parse("2009/05")).intValue(), is(570));

		// Act
		final SortedMap<Date, Number> r5 = DataConvertUtils.compress(data.keySet().toArray(new Date[0]), data.values().toArray(new Number[0]), CompressType.SUM, Calendar.DATE);
		// Assert
		assertThat(r5.get(ym.parse("2009/01")).intValue(), is(390));
		assertThat(r5.get(ym.parse("2009/02")).intValue(), is(990));
		assertThat(r5.get(ym.parse("2009/03")).intValue(), is(1050));
		assertThat(r5.get(ym.parse("2009/04")).intValue(), is(460));
		assertThat(r5.get(ym.parse("2009/05")).intValue(), is(570));
	}

	@Test
	public void testCompressDateArrayNumberArrayCompressTypeIntCalendar() throws ParseException {
		///////////////////////////////////////////////////////////////////////
		// Arrange

		final SimpleDateFormat ymd = new SimpleDateFormat("yyyy/MM/dd");
		final SimpleDateFormat ym = new SimpleDateFormat("yyyy/MM");

		final SortedMap<Date, Number> data = new TreeMap<Date, Number>();
		data.put(ymd.parse("2009/02/03"), 230);
		data.put(ymd.parse("2009/02/04"), 300);
		data.put(ymd.parse("2009/02/05"), 200);
		data.put(ymd.parse("2009/02/06"), 260);
		data.put(ymd.parse("2009/01/02"), 120);
		data.put(ymd.parse("2009/01/03"), 130);
		data.put(ymd.parse("2009/01/04"), 140);
		data.put(ymd.parse("2009/03/04"), 340);
		data.put(ymd.parse("2009/03/05"), 350);
		data.put(ymd.parse("2009/03/06"), 360);
		data.put(ymd.parse("2009/04/05"), null);
		data.put(ymd.parse("2009/04/06"), 460);
		data.put(ymd.parse("2009/04/07"), null);
		data.put(ymd.parse("2009/05/07"), 570);

		///////////////////////////////////////////////////////////////////////

		// Act
		final SortedMap<Date, Number> r0 = DataConvertUtils.compress(data.keySet().toArray(new Date[0]), null, CompressType.FIRST, Calendar.DATE, Calendar.getInstance());
		// Assert
		assertNotNull(r0);

		// Act
		final SortedMap<Date, Number> r1 = DataConvertUtils.compress(data.keySet().toArray(new Date[0]), data.values().toArray(new Number[0]), CompressType.FIRST, Calendar.DATE, Calendar.getInstance());
		// Assert
		assertThat(r1.size(), is(5));
		assertThat(r1.get(ym.parse("2009/01")).intValue(), is(120));
		assertThat(r1.get(ym.parse("2009/02")).intValue(), is(230));
		assertThat(r1.get(ym.parse("2009/03")).intValue(), is(340));
		assertNull(r1.get(ym.parse("2009/04")));
		assertThat(r1.get(ym.parse("2009/05")).intValue(), is(570));

		// Act
		final SortedMap<Date, Number> r2 = DataConvertUtils.compress(data.keySet().toArray(new Date[0]), data.values().toArray(new Number[0]), CompressType.HIGHEST, Calendar.DATE, Calendar.getInstance());
		// Assert
		assertThat(r2.get(ym.parse("2009/01")).intValue(), is(140));
		assertThat(r2.get(ym.parse("2009/02")).intValue(), is(300));
		assertThat(r2.get(ym.parse("2009/03")).intValue(), is(360));
		assertThat(r2.get(ym.parse("2009/04")).intValue(), is(460));
		assertThat(r2.get(ym.parse("2009/05")).intValue(), is(570));

		// Act
		final SortedMap<Date, Number> r3 = DataConvertUtils.compress(data.keySet().toArray(new Date[0]), data.values().toArray(new Number[0]), CompressType.LOWEST, Calendar.DATE, Calendar.getInstance());
		// Assert
		assertThat(r3.get(ym.parse("2009/01")).intValue(), is(120));
		assertThat(r3.get(ym.parse("2009/02")).intValue(), is(200));
		assertThat(r3.get(ym.parse("2009/03")).intValue(), is(340));
		assertThat(r3.get(ym.parse("2009/04")).intValue(), is(460));
		assertThat(r3.get(ym.parse("2009/05")).intValue(), is(570));

		// Act
		final SortedMap<Date, Number> r4 = DataConvertUtils.compress(data.keySet().toArray(new Date[0]), data.values().toArray(new Number[0]), CompressType.LAST, Calendar.DATE, Calendar.getInstance());
		// Assert
		assertThat(r4.get(ym.parse("2009/01")).intValue(), is(140));
		assertThat(r4.get(ym.parse("2009/02")).intValue(), is(260));
		assertThat(r4.get(ym.parse("2009/03")).intValue(), is(360));
		assertNull(r4.get(ym.parse("2009/04")));
		assertThat(r4.get(ym.parse("2009/05")).intValue(), is(570));

		// Act
		final SortedMap<Date, Number> r5 = DataConvertUtils.compress(data.keySet().toArray(new Date[0]), data.values().toArray(new Number[0]), CompressType.SUM, Calendar.DATE, Calendar.getInstance());
		// Assert
		assertThat(r5.get(ym.parse("2009/01")).intValue(), is(390));
		assertThat(r5.get(ym.parse("2009/02")).intValue(), is(990));
		assertThat(r5.get(ym.parse("2009/03")).intValue(), is(1050));
		assertThat(r5.get(ym.parse("2009/04")).intValue(), is(460));
		assertThat(r5.get(ym.parse("2009/05")).intValue(), is(570));
	}

	@Test
	public void testCompressDateArrayNumberArrayCompressTypeIntCalendarDateTruncater() throws ParseException {
		///////////////////////////////////////////////////////////////////////
		// Arrange

		final SimpleDateFormat ymd = new SimpleDateFormat("yyyy/MM/dd");
		final SimpleDateFormat ym = new SimpleDateFormat("yyyy/MM");

		final SortedMap<Date, Number> data = new TreeMap<Date, Number>();
		data.put(ymd.parse("2009/02/03"), 230);
		data.put(ymd.parse("2009/02/04"), 300);
		data.put(ymd.parse("2009/02/05"), 200);
		data.put(ymd.parse("2009/02/06"), 260);
		data.put(ymd.parse("2009/01/02"), 120);
		data.put(ymd.parse("2009/01/03"), 130);
		data.put(ymd.parse("2009/01/04"), 140);
		data.put(ymd.parse("2009/03/04"), 340);
		data.put(ymd.parse("2009/03/05"), 350);
		data.put(ymd.parse("2009/03/06"), 360);
		data.put(ymd.parse("2009/04/05"), null);
		data.put(ymd.parse("2009/04/06"), 460);
		data.put(ymd.parse("2009/04/07"), null);
		data.put(ymd.parse("2009/05/07"), 570);

		///////////////////////////////////////////////////////////////////////

		// Act
		final SortedMap<Date, Number> r0 = DataConvertUtils.compress(data.keySet().toArray(new Date[0]), null, CompressType.FIRST, Calendar.DATE, Calendar.getInstance(), new DateTruncater());
		// Assert
		assertNotNull(r0);

		// Act
		final SortedMap<Date, Number> r1 = DataConvertUtils.compress(data.keySet().toArray(new Date[0]), data.values().toArray(new Number[0]), CompressType.FIRST, Calendar.DATE, Calendar.getInstance(), new DateTruncater());
		// Assert
		assertThat(r1.size(), is(5));
		assertThat(r1.get(ym.parse("2009/01")).intValue(), is(120));
		assertThat(r1.get(ym.parse("2009/02")).intValue(), is(230));
		assertThat(r1.get(ym.parse("2009/03")).intValue(), is(340));
		assertNull(r1.get(ym.parse("2009/04")));
		assertThat(r1.get(ym.parse("2009/05")).intValue(), is(570));

		// Act
		final SortedMap<Date, Number> r2 = DataConvertUtils.compress(data.keySet().toArray(new Date[0]), data.values().toArray(new Number[0]), CompressType.HIGHEST, Calendar.DATE, Calendar.getInstance(), new DateTruncater());
		// Assert
		assertThat(r2.get(ym.parse("2009/01")).intValue(), is(140));
		assertThat(r2.get(ym.parse("2009/02")).intValue(), is(300));
		assertThat(r2.get(ym.parse("2009/03")).intValue(), is(360));
		assertThat(r2.get(ym.parse("2009/04")).intValue(), is(460));
		assertThat(r2.get(ym.parse("2009/05")).intValue(), is(570));

		// Act
		final SortedMap<Date, Number> r3 = DataConvertUtils.compress(data.keySet().toArray(new Date[0]), data.values().toArray(new Number[0]), CompressType.LOWEST, Calendar.DATE, Calendar.getInstance(), new DateTruncater());
		// Assert
		assertThat(r3.get(ym.parse("2009/01")).intValue(), is(120));
		assertThat(r3.get(ym.parse("2009/02")).intValue(), is(200));
		assertThat(r3.get(ym.parse("2009/03")).intValue(), is(340));
		assertThat(r3.get(ym.parse("2009/04")).intValue(), is(460));
		assertThat(r3.get(ym.parse("2009/05")).intValue(), is(570));

		// Act
		final SortedMap<Date, Number> r4 = DataConvertUtils.compress(data.keySet().toArray(new Date[0]), data.values().toArray(new Number[0]), CompressType.LAST, Calendar.DATE, Calendar.getInstance(), new DateTruncater());
		// Assert
		assertThat(r4.get(ym.parse("2009/01")).intValue(), is(140));
		assertThat(r4.get(ym.parse("2009/02")).intValue(), is(260));
		assertThat(r4.get(ym.parse("2009/03")).intValue(), is(360));
		assertNull(r4.get(ym.parse("2009/04")));
		assertThat(r4.get(ym.parse("2009/05")).intValue(), is(570));

		// Act
		final SortedMap<Date, Number> r5 = DataConvertUtils.compress(data.keySet().toArray(new Date[0]), data.values().toArray(new Number[0]), CompressType.SUM, Calendar.DATE, Calendar.getInstance(), new DateTruncater());
		// Assert
		assertThat(r5.get(ym.parse("2009/01")).intValue(), is(390));
		assertThat(r5.get(ym.parse("2009/02")).intValue(), is(990));
		assertThat(r5.get(ym.parse("2009/03")).intValue(), is(1050));
		assertThat(r5.get(ym.parse("2009/04")).intValue(), is(460));
		assertThat(r5.get(ym.parse("2009/05")).intValue(), is(570));
	}

	@Test
	public void testCompressSortedMapOfDateNumberCompressTypeInt() throws ParseException {
		///////////////////////////////////////////////////////////////////////
		// Arrange

		final SimpleDateFormat ymd = new SimpleDateFormat("yyyy/MM/dd");
		final SimpleDateFormat ym = new SimpleDateFormat("yyyy/MM");

		final SortedMap<Date, Number> data = new TreeMap<Date, Number>();
		data.put(ymd.parse("2009/02/03"), 230);
		data.put(ymd.parse("2009/02/04"), 300);
		data.put(ymd.parse("2009/02/05"), 200);
		data.put(ymd.parse("2009/02/06"), 260);
		data.put(ymd.parse("2009/01/02"), 120);
		data.put(ymd.parse("2009/01/03"), 130);
		data.put(ymd.parse("2009/01/04"), 140);
		data.put(ymd.parse("2009/03/04"), 340);
		data.put(ymd.parse("2009/03/05"), 350);
		data.put(ymd.parse("2009/03/06"), 360);
		data.put(ymd.parse("2009/04/05"), null);
		data.put(ymd.parse("2009/04/06"), 460);
		data.put(ymd.parse("2009/04/07"), null);
		data.put(ymd.parse("2009/05/07"), 570);

		///////////////////////////////////////////////////////////////////////

		// Act
		final SortedMap<Date, Number> r0 = DataConvertUtils.compress(null, CompressType.FIRST, Calendar.DATE);
		// Assert
		assertNull(r0);

		// Act
		final SortedMap<Date, Number> r1 = DataConvertUtils.compress(data, CompressType.FIRST, Calendar.DATE);
		// Assert
		assertThat(r1.size(), is(5));
		assertThat(r1.get(ym.parse("2009/01")).intValue(), is(120));
		assertThat(r1.get(ym.parse("2009/02")).intValue(), is(230));
		assertThat(r1.get(ym.parse("2009/03")).intValue(), is(340));
		assertNull(r1.get(ym.parse("2009/04")));
		assertThat(r1.get(ym.parse("2009/05")).intValue(), is(570));

		// Act
		final SortedMap<Date, Number> r2 = DataConvertUtils.compress(data, CompressType.HIGHEST, Calendar.DATE);
		// Assert
		assertThat(r2.get(ym.parse("2009/01")).intValue(), is(140));
		assertThat(r2.get(ym.parse("2009/02")).intValue(), is(300));
		assertThat(r2.get(ym.parse("2009/03")).intValue(), is(360));
		assertThat(r2.get(ym.parse("2009/04")).intValue(), is(460));
		assertThat(r2.get(ym.parse("2009/05")).intValue(), is(570));

		// Act
		final SortedMap<Date, Number> r3 = DataConvertUtils.compress(data, CompressType.LOWEST, Calendar.DATE);
		// Assert
		assertThat(r3.get(ym.parse("2009/01")).intValue(), is(120));
		assertThat(r3.get(ym.parse("2009/02")).intValue(), is(200));
		assertThat(r3.get(ym.parse("2009/03")).intValue(), is(340));
		assertThat(r3.get(ym.parse("2009/04")).intValue(), is(460));
		assertThat(r3.get(ym.parse("2009/05")).intValue(), is(570));

		// Act
		final SortedMap<Date, Number> r4 = DataConvertUtils.compress(data, CompressType.LAST, Calendar.DATE);
		// Assert
		assertThat(r4.get(ym.parse("2009/01")).intValue(), is(140));
		assertThat(r4.get(ym.parse("2009/02")).intValue(), is(260));
		assertThat(r4.get(ym.parse("2009/03")).intValue(), is(360));
		assertNull(r4.get(ym.parse("2009/04")));
		assertThat(r4.get(ym.parse("2009/05")).intValue(), is(570));

		// Act
		final SortedMap<Date, Number> r5 = DataConvertUtils.compress(data, CompressType.SUM, Calendar.DATE);
		// Assert
		assertThat(r5.get(ym.parse("2009/01")).intValue(), is(390));
		assertThat(r5.get(ym.parse("2009/02")).intValue(), is(990));
		assertThat(r5.get(ym.parse("2009/03")).intValue(), is(1050));
		assertThat(r5.get(ym.parse("2009/04")).intValue(), is(460));
		assertThat(r5.get(ym.parse("2009/05")).intValue(), is(570));
	}

	@Test
	public void testCompressSortedMapOfDateNumberCompressTypeIntCalendar() throws ParseException {
		///////////////////////////////////////////////////////////////////////
		// Arrange

		final SimpleDateFormat ymd = new SimpleDateFormat("yyyy/MM/dd");
		final SimpleDateFormat ym = new SimpleDateFormat("yyyy/MM");

		final SortedMap<Date, Number> data = new TreeMap<Date, Number>();
		data.put(ymd.parse("2009/02/03"), 230);
		data.put(ymd.parse("2009/02/04"), 300);
		data.put(ymd.parse("2009/02/05"), 200);
		data.put(ymd.parse("2009/02/06"), 260);
		data.put(ymd.parse("2009/01/02"), 120);
		data.put(ymd.parse("2009/01/03"), 130);
		data.put(ymd.parse("2009/01/04"), 140);
		data.put(ymd.parse("2009/03/04"), 340);
		data.put(ymd.parse("2009/03/05"), 350);
		data.put(ymd.parse("2009/03/06"), 360);
		data.put(ymd.parse("2009/04/05"), null);
		data.put(ymd.parse("2009/04/06"), 460);
		data.put(ymd.parse("2009/04/07"), null);
		data.put(ymd.parse("2009/05/07"), 570);

		///////////////////////////////////////////////////////////////////////

		// Act
		final SortedMap<Date, Number> r0 = DataConvertUtils.compress(null, CompressType.FIRST, Calendar.DATE, Calendar.getInstance());
		// Assert
		assertNull(r0);

		// Act
		final SortedMap<Date, Number> r1 = DataConvertUtils.compress(data, CompressType.FIRST, Calendar.DATE, Calendar.getInstance());
		// Assert
		assertThat(r1.size(), is(5));
		assertThat(r1.get(ym.parse("2009/01")).intValue(), is(120));
		assertThat(r1.get(ym.parse("2009/02")).intValue(), is(230));
		assertThat(r1.get(ym.parse("2009/03")).intValue(), is(340));
		assertNull(r1.get(ym.parse("2009/04")));
		assertThat(r1.get(ym.parse("2009/05")).intValue(), is(570));

		// Act
		final SortedMap<Date, Number> r2 = DataConvertUtils.compress(data, CompressType.HIGHEST, Calendar.DATE, Calendar.getInstance());
		// Assert
		assertThat(r2.get(ym.parse("2009/01")).intValue(), is(140));
		assertThat(r2.get(ym.parse("2009/02")).intValue(), is(300));
		assertThat(r2.get(ym.parse("2009/03")).intValue(), is(360));
		assertThat(r2.get(ym.parse("2009/04")).intValue(), is(460));
		assertThat(r2.get(ym.parse("2009/05")).intValue(), is(570));

		// Act
		final SortedMap<Date, Number> r3 = DataConvertUtils.compress(data, CompressType.LOWEST, Calendar.DATE, Calendar.getInstance());
		// Assert
		assertThat(r3.get(ym.parse("2009/01")).intValue(), is(120));
		assertThat(r3.get(ym.parse("2009/02")).intValue(), is(200));
		assertThat(r3.get(ym.parse("2009/03")).intValue(), is(340));
		assertThat(r3.get(ym.parse("2009/04")).intValue(), is(460));
		assertThat(r3.get(ym.parse("2009/05")).intValue(), is(570));

		// Act
		final SortedMap<Date, Number> r4 = DataConvertUtils.compress(data, CompressType.LAST, Calendar.DATE, Calendar.getInstance());
		// Assert
		assertThat(r4.get(ym.parse("2009/01")).intValue(), is(140));
		assertThat(r4.get(ym.parse("2009/02")).intValue(), is(260));
		assertThat(r4.get(ym.parse("2009/03")).intValue(), is(360));
		assertNull(r4.get(ym.parse("2009/04")));
		assertThat(r4.get(ym.parse("2009/05")).intValue(), is(570));

		// Act
		final SortedMap<Date, Number> r5 = DataConvertUtils.compress(data, CompressType.SUM, Calendar.DATE, Calendar.getInstance());
		// Assert
		assertThat(r5.get(ym.parse("2009/01")).intValue(), is(390));
		assertThat(r5.get(ym.parse("2009/02")).intValue(), is(990));
		assertThat(r5.get(ym.parse("2009/03")).intValue(), is(1050));
		assertThat(r5.get(ym.parse("2009/04")).intValue(), is(460));
		assertThat(r5.get(ym.parse("2009/05")).intValue(), is(570));
	}

	@Test
	public void testCompressSortedMapOfDateNumberCompressTypeIntCalendarDateTruncater() throws ParseException {
		///////////////////////////////////////////////////////////////////////
		// Arrange

		final SimpleDateFormat ymd = new SimpleDateFormat("yyyy/MM/dd");
		final SimpleDateFormat ym = new SimpleDateFormat("yyyy/MM");

		final SortedMap<Date, Number> data = new TreeMap<Date, Number>();
		data.put(ymd.parse("2009/02/03"), 230);
		data.put(ymd.parse("2009/02/04"), 300);
		data.put(ymd.parse("2009/02/05"), 200);
		data.put(ymd.parse("2009/02/06"), 260);
		data.put(ymd.parse("2009/01/02"), 120);
		data.put(ymd.parse("2009/01/03"), 130);
		data.put(ymd.parse("2009/01/04"), 140);
		data.put(ymd.parse("2009/03/04"), 340);
		data.put(ymd.parse("2009/03/05"), 350);
		data.put(ymd.parse("2009/03/06"), 360);
		data.put(ymd.parse("2009/04/05"), null);
		data.put(ymd.parse("2009/04/06"), 460);
		data.put(ymd.parse("2009/04/07"), null);
		data.put(ymd.parse("2009/05/07"), 570);

		///////////////////////////////////////////////////////////////////////

		// Act
		final SortedMap<Date, Number> r0 = DataConvertUtils.compress(null, CompressType.FIRST, Calendar.DATE, Calendar.getInstance(), new DateTruncater());
		// Assert
		assertNull(r0);

		// Act
		final SortedMap<Date, Number> r1 = DataConvertUtils.compress(data, CompressType.FIRST, Calendar.DATE, Calendar.getInstance(), new DateTruncater());
		// Assert
		assertThat(r1.size(), is(5));
		assertThat(r1.get(ym.parse("2009/01")).intValue(), is(120));
		assertThat(r1.get(ym.parse("2009/02")).intValue(), is(230));
		assertThat(r1.get(ym.parse("2009/03")).intValue(), is(340));
		assertNull(r1.get(ym.parse("2009/04")));
		assertThat(r1.get(ym.parse("2009/05")).intValue(), is(570));

		// Act
		final SortedMap<Date, Number> r2 = DataConvertUtils.compress(data, CompressType.HIGHEST, Calendar.DATE, Calendar.getInstance(), new DateTruncater());
		// Assert
		assertThat(r2.get(ym.parse("2009/01")).intValue(), is(140));
		assertThat(r2.get(ym.parse("2009/02")).intValue(), is(300));
		assertThat(r2.get(ym.parse("2009/03")).intValue(), is(360));
		assertThat(r2.get(ym.parse("2009/04")).intValue(), is(460));
		assertThat(r2.get(ym.parse("2009/05")).intValue(), is(570));

		// Act
		final SortedMap<Date, Number> r3 = DataConvertUtils.compress(data, CompressType.LOWEST, Calendar.DATE, Calendar.getInstance(), new DateTruncater());
		// Assert
		assertThat(r3.get(ym.parse("2009/01")).intValue(), is(120));
		assertThat(r3.get(ym.parse("2009/02")).intValue(), is(200));
		assertThat(r3.get(ym.parse("2009/03")).intValue(), is(340));
		assertThat(r3.get(ym.parse("2009/04")).intValue(), is(460));
		assertThat(r3.get(ym.parse("2009/05")).intValue(), is(570));

		// Act
		final SortedMap<Date, Number> r4 = DataConvertUtils.compress(data, CompressType.LAST, Calendar.DATE, Calendar.getInstance(), new DateTruncater());
		// Assert
		assertThat(r4.get(ym.parse("2009/01")).intValue(), is(140));
		assertThat(r4.get(ym.parse("2009/02")).intValue(), is(260));
		assertThat(r4.get(ym.parse("2009/03")).intValue(), is(360));
		assertNull(r4.get(ym.parse("2009/04")));
		assertThat(r4.get(ym.parse("2009/05")).intValue(), is(570));

		// Act
		final SortedMap<Date, Number> r5 = DataConvertUtils.compress(data, CompressType.SUM, Calendar.DATE, Calendar.getInstance(), new DateTruncater());
		// Assert
		assertThat(r5.get(ym.parse("2009/01")).intValue(), is(390));
		assertThat(r5.get(ym.parse("2009/02")).intValue(), is(990));
		assertThat(r5.get(ym.parse("2009/03")).intValue(), is(1050));
		assertThat(r5.get(ym.parse("2009/04")).intValue(), is(460));
		assertThat(r5.get(ym.parse("2009/05")).intValue(), is(570));
	}

	@Test(expected = IllegalArgumentException.class)
	public void testCompressSortedMapOfDateNumberCompressTypeIntCalendarDateTruncaterIllegalArgumentException1() {
		// Arrange
		final CompressType compressType = null;
		// Act
		DataConvertUtils.compress(null, compressType, Calendar.DATE, Calendar.getInstance(), new DateTruncater());
	}

	@Test(expected = IllegalArgumentException.class)
	public void testCompressSortedMapOfDateNumberCompressTypeIntCalendarDateTruncaterIllegalArgumentException2() {
		// Arrange
		final Calendar calendar = null;
		// Act
		DataConvertUtils.compress(null, CompressType.FIRST, Calendar.DATE, calendar, new DateTruncater());
	}

	@Test(expected = IllegalArgumentException.class)
	public void testCompressSortedMapOfDateNumberCompressTypeIntCalendarDateTruncaterIllegalArgumentException3() {
		// Arrange
		final DateTruncater truncater = null;
		// Act
		DataConvertUtils.compress(null, CompressType.FIRST, Calendar.DATE, Calendar.getInstance(), truncater);
	}

}
