/*
 * Copyright (c) 2006-2009 OrangeSignal.com All rights reserved.
 * 
 * これは Apache ライセンス Version 2.0 (以下、このライセンスと記述) に
 * 従っています。このライセンスに準拠する場合以外、このファイルを使用
 * してはなりません。このライセンスのコピーは以下から入手できます。
 * 
 * http://www.apache.org/licenses/LICENSE-2.0.txt
 * 
 * 適用可能な法律がある、あるいは文書によって明記されている場合を除き、
 * このライセンスの下で配布されているソフトウェアは、明示的であるか暗黙の
 * うちであるかを問わず、「保証やあらゆる種類の条件を含んでおらず」、
 * 「あるがまま」の状態で提供されるものとします。
 * このライセンスが適用される特定の許諾と制限については、このライセンス
 * を参照してください。
 */

package jp.sf.orangesignal.ta.data.model;

import java.io.Serializable;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.text.DateFormat;
import java.util.Date;

import jp.sf.orangesignal.ta.data.CompressType;
import jp.sf.orangesignal.ta.data.annotation.DateItem;
import jp.sf.orangesignal.ta.data.annotation.NumberCompressorType;
import jp.sf.orangesignal.ta.data.annotation.NumberItem;

/**
 * 価格データのモデルクラスを提供します。
 * 
 * @author 杉澤 浩二
 * @since 2.2
 */
public class Price implements Serializable, Comparable<Price> {

	private static final long serialVersionUID = -7071308164520373368L;

	/**
	 * 日時を保持します。
	 */
	@DateItem
	protected Date date;

	/**
	 * 始値を保持します。
	 */
	@NumberItem
	@NumberCompressorType(CompressType.FIRST)
	protected Number open;

	/**
	 * 高値を保持します。
	 */
	@NumberItem
	@NumberCompressorType(CompressType.HIGHEST)
	protected Number high;

	/**
	 * 安値を保持します。
	 */
	@NumberItem
	@NumberCompressorType(CompressType.LOWEST)
	protected Number low;

	/**
	 * 終値を保持します。
	 */
	@NumberItem
	@NumberCompressorType(CompressType.LAST)
	protected Number close;

	/**
	 * 出来高を保持します。
	 */
	@NumberItem
	@NumberCompressorType(CompressType.SUM)
	protected Number volume;

	/**
	 * デフォルトコンストラクタです。
	 */
	public Price() {}

	/**
	 * 日時を返します。
	 * 
	 * @return 日時
	 */
	public Date getDate() { return date; }

	/**
	 * 日時を設定します。
	 * 
	 * @param date 日時
	 */
	public void setDate(final Date date) { this.date = date; }

	/**
	 * 始値を返します。取組がない場合は <code>null</code> を返します。
	 * 
	 * @return 始値。または <code>null</code>
	 */
	public Number getOpen() { return open; }

	/**
	 * 始値を設定します。
	 * 
	 * @param open 始値
	 */
	public void setOpen(final Number open) { this.open = open; }

	/**
	 * 高値を返します。取組がない場合は <code>null</code> を返します。
	 * 
	 * @return 高値。または <code>null</code>
	 */
	public Number getHigh() { return high; }

	/**
	 * 高値を設定します。
	 * 
	 * @param high 高値
	 */
	public void setHigh(final Number high) { this.high = high; }

	/**
	 * 安値を返します。取組がない場合は <code>null</code> を返します。
	 * 
	 * @return 安値。または <code>null</code>
	 */
	public Number getLow() { return low; }

	/**
	 * 安値を設定します。
	 * 
	 * @param low 安値
	 */
	public void setLow(final Number low) { this.low = low; }

	/**
	 * 終値を返します。取組がない場合は <code>null</code> を返します。
	 * 
	 * @return 終値。または <code>null</code>
	 */
	public Number getClose() { return close; }

	/**
	 * 終値を設定します。
	 * 
	 * @param close 終値
	 */
	public void setClose(final Number close) { this.close = close; }

	/**
	 * 出来高を返します。取組がない場合は <code>null</code> を返します。
	 * 
	 * @return 出来高。または <code>null</code>
	 */
	public Number getVolume() { return volume; }

	/**
	 * 出来高を設定します。
	 * 
	 * @param volume 出来高
	 */
	public void setVolume(final Number volume) { this.volume = volume; }

	/**
	 * 実装は日時で比較します。
	 */
	@Override public int compareTo(final Price o) { return date.compareTo(o.date); }

	@Override
	public String toString() {
		final StringBuilder sb = new StringBuilder();
		final Field[] fields = this.getClass().getDeclaredFields();
		for (final Field field : fields) {
			if (Modifier.isStatic(field.getModifiers())) {
				continue;
			}
			if (sb.length() > 0) {
				sb.append(", ");
			}
			sb.append(field.getName()).append('=');
			try {
				final Object value = field.get(this);
				if (value == null) {
					sb.append("null");
				} else if (value instanceof Date) {
					sb.append(DateFormat.getDateTimeInstance().format((Date) value));
				} else {
					sb.append(value.toString());
				}
			} catch (IllegalAccessException  e) {
				throw new IllegalStateException(e.getMessage(), e);
			}
		}
		return new StringBuilder().append(getClass().getSimpleName()).append(' ').append(sb).toString();
	}

}
