/*
 * Copyright (c) 2009 OrangeSignal.com All rights reserved.
 * 
 * これは Apache ライセンス Version 2.0 (以下、このライセンスと記述) に
 * 従っています。このライセンスに準拠する場合以外、このファイルを使用
 * してはなりません。このライセンスのコピーは以下から入手できます。
 * 
 * http://www.apache.org/licenses/LICENSE-2.0.txt
 * 
 * 適用可能な法律がある、あるいは文書によって明記されている場合を除き、
 * このライセンスの下で配布されているソフトウェアは、明示的であるか暗黙の
 * うちであるかを問わず、「保証やあらゆる種類の条件を含んでおらず」、
 * 「あるがまま」の状態で提供されるものとします。
 * このライセンスが適用される特定の許諾と制限については、このライセンス
 * を参照してください。
 */
package jp.sf.orangesignal.ta.util;

import static org.hamcrest.core.Is.is;
import static org.junit.Assert.assertThat;

import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.text.ParseException;
import java.util.Locale;

import org.junit.Test;

/**
 * {@link NumberFormatUtils} クラスの単体テストです。
 * 
 * @author 杉澤 浩二
 */
public class NumberFormatUtilsTest {

	@Test
	public void testParseStringStringArray() throws IllegalArgumentException, ParseException {
		final Number current = 123456789.10D;
		final Number result = NumberFormatUtils.parse(
				new DecimalFormat("\u00A4\u00A4 #,##0.00").format(current),
				new String[]{ "\u00A4 #,##0.00", "\u00A4\u00A4 #,##0.00" }
			);
		assertThat(result, is(current));
	}

	@Test
	public void testParseStringStringArrayLocale() throws IllegalArgumentException, ParseException {
		final Number current = 123456789.10D;
		final Number result = NumberFormatUtils.parse(
				new DecimalFormat("\u00A4\u00A4 #,##0.00", DecimalFormatSymbols.getInstance(Locale.JAPAN)).format(current),
				new String[]{ "\u00A4 #,##0.00", "\u00A4\u00A4 #,##0.00" },
				Locale.JAPAN	// 通貨は国コード基準なので言語ではなく国で指定する。parse の場合は NaN の判定基準が影響する
			);
		assertThat(result, is(current));
	}

	@Test
	public void testParseStringStringArrayLocaleStringArray() throws IllegalArgumentException, ParseException {
		// Arrange
		final Number current = 123456789.10D;
		// Act
		final Number result = NumberFormatUtils.parse("JPY 123,456,789.10", new String[]{ "\u00A4 #,##0.00", "\u00A4\u00A4 #,##0.00" }, Locale.CANADA, new String[]{ "USD", "JPY" });
		// Assert
		assertThat(result, is(current));
	}

	@Test(expected = IllegalArgumentException.class)
	public void testParseStringStringArrayLocaleStringArrayIllegalArgumentException1() throws IllegalArgumentException, ParseException {
		// Act
		NumberFormatUtils.parse(null, new String[]{ "\u00A4 #,##0.00", "\u00A4\u00A4 #,##0.00" }, Locale.CANADA, new String[]{ "USD", "JPY" });
	}

	@Test(expected = IllegalArgumentException.class)
	public void testParseStringStringArrayLocaleStringArrayIllegalArgumentException2() throws IllegalArgumentException, ParseException {
		// Act
		NumberFormatUtils.parse("JPY 123,456,789.10", null, Locale.CANADA, new String[]{ "USD", "JPY" });
	}

	@Test(expected = IllegalArgumentException.class)
	public void testParseStringStringArrayLocaleStringArrayIllegalArgumentException3() throws IllegalArgumentException, ParseException {
		// Act
		NumberFormatUtils.parse("JPY 123,456,789.10", new String[]{ null, "\u00A4\u00A4 #,##0.00" }, Locale.CANADA, new String[]{ "USD", "JPY" });
	}

	@Test(expected = IllegalArgumentException.class)
	public void testParseStringStringArrayLocaleStringArrayIllegalArgumentException4() throws IllegalArgumentException, ParseException {
		// Act
		NumberFormatUtils.parse("JPY 123,456,789.10", new String[]{ "\u00A4 #,##0.00", "\u00A4\u00A4 #,##0.00" }, Locale.CANADA, new String[]{ null, "JPY" });
	}

	@Test
	public void testFormatNumberString() {
		// Act
		final String str = NumberFormatUtils.format(123456789.10D, "#,##0.00");
		// Assert
		assertThat(str, is("123,456,789.10"));
	}

	@Test
	public void testFormatNumberStringLocale() {
		// Act + Assert
		assertThat(NumberFormatUtils.format(123456789.10D, "#,##0.00", Locale.JAPAN), is("123,456,789.10"));
		// Act + Assert
		assertThat(NumberFormatUtils.format(123456789.10D, "#,##0.00", Locale.JAPANESE), is("123,456,789.10"));
	}

	@Test
	public void testFormatNumberStringLocaleString() {
		// Act + Assert
		assertThat(NumberFormatUtils.format(123456789.10D, "\u00A4\u00A4 #,##0.00", Locale.JAPAN, "JPY"), is("JPY 123,456,789.10"));
		// Act + Assert
		assertThat(NumberFormatUtils.format(123456789.10D, "\u00A4\u00A4 #,##0.00", Locale.JAPANESE, "JPY"), is("JPY 123,456,789.10"));
	}

}
