struct MESH_VERTEX {
	float3 PosL		: POSITION;
	float3 NormL	: NORMAL;
	float3 TangL	: TANGENT;
	float2 TCrd		: TEXCOORD;
};

struct VSOut {
	float4 PosH		: SV_POSITION;
	float3 CamW		: POSITION;
	float3 NormW	: NORMAL;
	float3 TangW	: TANGENT;

	half3 diffuse	: COLOR0;
	half3 specular	: COLOR1;

	half3 atten		: COLOR2;
	half3 insca		: COLOR3;

	float2 TCrd		: TEXCOORD;
};

//======================================
//		Shader data
//======================================

struct Sun_spec {
	float4 SunDir;
	float4 ambient;
	float4 diffuse;
	float4 specular;	//64
};

struct Light_spec {
	float4 ambient;		//						//16
	float4 diffuse;		//diffuse.w = range		//32
	float4 specular;	//specular.w = falloff	//48
	float4 PosW;		//PosW.w = theta		//64
	float4 DirW;		//DirW.w = phi			//80
	float3 att;			//attenuation			//92
	uint type;			//light source type		//96
};

cbuffer cb_VS_per_frame		: register( b0 )
{	
	uint cHazeMode;			//4		4
	float3 cFogColor;		//12	16
	float cFogDensity;		//4		20
	float3 waste0;			//12	32
	Light_spec Lights[12];	//96*12+32 = 1184
};

cbuffer cb_VS_per_object	: register( b1 )
{
	row_major matrix VP;	//64
	Sun_spec cSun;			//128
	uint ActiveLightsCount;
	float3 waste1;			//144
};

cbuffer cb_VS_per_group		: register( b2 )
{
	row_major matrix W;		//64
	float specpower;
	float3 waste2;			//80
};

void LocalVertexLight( out float3 diff, out float3 spec, in float3 NormW, in float3 PosW ) {
	float3 diffuse = 0;
	float3 specular = 0;

	[unroll(12)]
	for( uint j = 0; j < ActiveLightsCount; j++ ) {
		float dist = distance( PosW, Lights[j].PosW.xyz );
		float3 rel_posW = normalize( PosW - Lights[j].PosW.xyz );

		float att = max( 0.0f, 1.0f/(Lights[j].att.x + Lights[j].att.y*dist + Lights[j].att.z*dist*dist ) );
		float spt = 1.0f;

		[branch]
		if( Lights[j].type != 1 )
			spt = saturate( ( dot( rel_posW, Lights[j].DirW.xyz ) - Lights[j].DirW.w )*Lights[j].PosW.w );
		float d = dot( -rel_posW, NormW );
		float s = 0.0f;

		float dif = att*spt;

		[branch]
		if( d > 0.0f ) {
			diffuse += Lights[j].ambient.rgb*dif;
			diffuse += Lights[j].diffuse.rgb*(dif*max( 0, d ));

			[branch]
			if( specpower > 1.0f && d > 0.0f ) {
				s = pow( max( dot( reflect( rel_posW, NormW ), normalize( -PosW ) ), 0.0f ), specpower );
				specular += Lights[j].specular.rgb*(dif*s);
			}
		}
	}

	diff = float3( 1.5 - exp( -1.0*diffuse )*1.5h );
	spec = float3( 1.5 - exp( -1.0*specular)*1.5h );
}

void AtmosphericHaze( out half3 att, out half3 ins, in float dp ) {
	[branch]
	if( cHazeMode == 0 ) {
		att = 1;
		ins = 0;
		return;
	}
	else {
		float fogFact = 1.0f / exp( dp*cFogDensity );
		att = fogFact;
		ins = (1.0f - fogFact)*cFogColor;
		return;
	}
}

VSOut VS_Mesh( MESH_VERTEX In ) {
	VSOut Out;
	float3 PosW = mul( float4( In.PosL, 1.0f ), W ).xyz;
	Out.PosH = mul( float4( PosW, 1.0f ), VP );
	Out.NormW = mul( float4( In.NormL, 0.0f ), W ).xyz;
	Out.TangW = mul( float4( In.TangL, 0.0f ), W ).xyz;
	Out.CamW = -PosW;
	Out.TCrd = In.TCrd;

	Out.diffuse = half3( 0.0h, 0.0h, 0.0h );
	Out.specular = half3( 0.0h, 0.0h, 0.0h );

	[branch]
	if( ActiveLightsCount != 0 )
		LocalVertexLight( Out.diffuse, Out.specular, normalize( Out.NormW ), PosW );
		
	AtmosphericHaze( Out.atten, Out.insca, Out.PosH.z );

	Out.insca *= (cSun.diffuse + cSun.ambient);//??

	return Out;
}

Texture2D MainTex			: register( t0 );
Texture2D NormalTex			: register( t1 );
Texture2D SpecularTex		: register( t2 );
Texture2D EmissiveTex		: register( t3 );
Texture2D NightTex			: register( t4 );

SamplerState TextureSampler	: register( s0 );

cbuffer cb_PS_per_object	: register( b0 )
{
	Sun_spec Sun;			//64
};

cbuffer cb_PS_Mesh			: register( b1 )
{
	float3 ambient;
	float opacity;			//16

	float3 diffuse;
	uint flags;				//32

	float3 specular;
	float spec_power;		//48

	float3 emissive;
	float texEx_mix;		//64
};

//======================================
//		Pixel shader
//======================================

float4 PS_Mesh( VSOut In )		: SV_Target
{
	In.CamW = normalize( In.CamW );
	In.NormW = normalize( In.NormW );

	float4 Tex;

	//normal map
	[branch]
	if( flags & 4 ) {
		Tex = float4( NormalTex.Sample( TextureSampler, In.TCrd ) );
		Tex = 2.0f*Tex - 1.0f;

		float3 N = In.NormW;
		float3 T = normalize( In.TangW - dot( In.TangW, N)*N );
		float3 B = cross( N, T );
		float3x3 TBN = float3x3( T, B, N );
		In.NormW = normalize( mul( Tex.rgb, TBN ) );
	}

	//sample texture if textured
	[branch]
	if( flags & 1 ) {
		Tex = MainTex.Sample( TextureSampler, In.TCrd );
		[branch]
		if( flags & 2 )
			Tex.a *= opacity;
	}
	else
		Tex = float4( 1.0f, 1.0f, 1.0f, opacity );
	
	[branch]
	if( flags & 64 )
		return float4( diffuse*Tex.rgb, 1.0f );
	
	//specular map
	float4 SPEC = float4( 1.0f, 1.0f, 1.0f, 1.0f );
	[branch]
	if( flags & 8 ) {
		SPEC = SpecularTex.Sample( TextureSampler, In.TCrd );
		SPEC.a *= 256.0f;
	}
	else
		SPEC.a = spec_power;

	//compute diffuse/specular coefficients
	float d = max( dot( -Sun.SunDir.xyz, In.NormW ), 0.0f );
	float s = 0.0f;
	[branch]
	if( spec_power > 1.0 && d > 0.0f )
		s = pow( max( dot( reflect( Sun.SunDir.xyz, In.NormW ), In.CamW ), 0.0f ), SPEC.a );
	
	//emissive map
	float3 EMS = float3( 0.0f, 0.0f, 0.0f );
	[branch]
	if( flags & 16 )
		EMS = EmissiveTex.Sample( TextureSampler, In.TCrd );
	EMS += emissive;

	float3 diff = diffuse.rgb*( In.diffuse + d*Sun.diffuse.rgb ) + ambient.rgb*Sun.ambient.rgb + EMS;
	float3 spec = SPEC.rgb*specular.rgb*( In.specular + s*Sun.specular );

	float3 color = Tex.rgb*saturate( diff ) + saturate( spec );

	[branch]
	if( flags & 32 )
		color += NightTex.Sample( TextureSampler, In.TCrd ).rgb*texEx_mix;

	return float4( color*In.atten + In.insca, Tex.a );
}