
/* --------------------------------------------- */
/*  H8-3069F Main function                       */
/*                                               */
/*  CPU    : Renesus H8/3069F 25MHz              */
/*  Memory : ROM 512KB, RAM 16KB E-RAM 2MB       */
/*                (c) KAZ.Imamura                */
/* --------------------------------------------- */

#include "common.h"
#include "sc1602.h"
#include "key.h"
#include "ui.h"
#include "sl811.h"
#include "usb_ms.h"
#include "led.h"
#include "serial.h"

int counter_500us;

// -------------------------------------------
//  Proto type definitions
// -------------------------------------------
void port_initialize(void);
void register_initialize(void);

void int_CMIA0();
void int_CMIA1();
void int_IRQ0();

// Local 

// -------------------------------------------
//  Main routine
// -------------------------------------------
int main(void) {
	// uCom initialize
	port_initialize();
	register_initialize();
	
	// Function initialize
	led_initialize();
	sc1602_initialize();
	key_initialize();
	ui_initialize();
	sl811_initialize();
	usbms_initialize();
	fat_initialize();
	serial_initialize();
	
	permit_irq();
	
	while(1) {
		led_process();
		sc1602_process();
		key_process();
		ui_process();
		sl811_process();
		usbms_process();
		fat_process();
		serial_process();
	}
}

// -------------------------------------------
//  Port initialize for start up
// -------------------------------------------
void port_initialize(void) {

	// Port4
	// D7 : (OUT) LED 1
	// D6 : (OUT) LED 0
	// D5 : (OUT) LCD Driver - E
	// D4 : (OUT) LCD Driver - RS
	// D3 : (OUT) LCD Driver - DB7
	// D2 : (OUT) LCD Driver - DB6
	// D1 : (OUT) LCD Driver - DB5
	// D0 : (OUT) LCD Driver - DB4
	P4DDR = 0xFF;
	P4DR.BYTE = 0x00;
	
	// Port5
	// D3 : (IN ) Switch D (BACK)
	// D2 : (IN ) Switch C (OK)
	// D1 : (IN ) Switch B (DOWN)
	// D0 : (IN ) Switch A (UP)
	P5DDR = 0xF0;
	P5PCR.BYTE = 0xFF;
}


// -------------------------------------------
//  Register initialize for start up
// -------------------------------------------
void register_initialize(void) {
	// --------------------------------------
	// D/A converter
	// --------------------------------------
	DA.DADR0 = 0x40;
	DA.DACR.BIT.DAOE0 = 1;
	
	// --------------------------------------
	// 8 bit timer ch-0
	// --------------------------------------
	
	// Compare - A register
	//   25MHz CPU : 40ns per clock
	//   Use 8bit timer as 64interval mode. (1 interval = 2.56us)
	TMR0.TCORA = 195; // 499.2 us
	
	// Control register
	TMR0.TCR.BIT.CMIEB	= 0;			// Compare B disable
	TMR0.TCR.BIT.CMIEA	= 1;			// Compare A enable
	TMR0.TCR.BIT.OVIE 	= 0;			// Overflow disable
	TMR0.TCR.BIT.CCLR	= 0x01;			// Comp-A make counte clear
	TMR0.TCR.BIT.CKS	= 0x02;			// Use internal clock. 64 interval.
	TMR0.TCSR.BIT.CMFA  = 0;     		// Interrupt Clear 
	
	
	// --------------------------------------
	// 8 bit timer ch-1
	// --------------------------------------
	
	// Compare - A register
	//   25MHz CPU : 40ns per clock
	//   Use 8bit timer as 8192interval mode. (1 interval = 327.68us)
	TMR1.TCORA = 61; // 19.98848 ms
	
	// Control register
	TMR1.TCR.BIT.CMIEB	= 0;			// Compare B disable
	TMR1.TCR.BIT.CMIEA	= 1;			// Compare A enable
	TMR1.TCR.BIT.OVIE 	= 0;			// Overflow disable
	TMR1.TCR.BIT.CCLR	= 0x01;			// Comp-A make counte clear
	TMR1.TCR.BIT.CKS	= 0x03;			// Use internal clock. 8192 interval.
	TMR1.TCSR.BIT.CMFA  = 0;     		// Interrupt Clear 

}

// -------------------------------------------
//  Interruption handler
// -------------------------------------------
#pragma interrupt (int_CMIA0)
void int_CMIA0() {
	TMR0.TCSR.BIT.CMFA = 0;     	// Interrupt Clear 
	counter_500us++;				// interrupt per 500us.
	
	// 1ms interval 
	if( (counter_500us & 0x01) == 0x01 ) {
		sc1602_1ms_handler();
		key_1ms_handler();
		serial_1ms_handler();
	} else {
		ui_1ms_handler();
		sl811_1ms_handler();
		usbms_1ms_handler();
	}
}

#pragma interrupt (int_CMIA1)
void int_CMIA1() {
	TMR1.TCSR.BIT.CMFA = 0;     	// Interrupt Clear 
	led_20ms_handler();
}

#pragma interrupt (int_IRQ0)
void int_IRQ0() {
	sl811_chip_irq();
}


