﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Runtime.InteropServices;
using System.Runtime.CompilerServices;
using Microsoft.VisualBasic;

namespace Lugens.Components.IME
{

    [ComImport]
    [Guid("019F7152-E6DB-11D0-83C3-00C04FDDB82E")]
    [InterfaceType(ComInterfaceType.InterfaceIsIUnknown)]
    internal interface IFELanguage
    {
        [MethodImpl(MethodImplOptions.InternalCall, MethodCodeType = MethodCodeType.Runtime)]
        int Open();
        [MethodImpl(MethodImplOptions.InternalCall, MethodCodeType = MethodCodeType.Runtime)]
        int Close();
        [MethodImpl(MethodImplOptions.InternalCall, MethodCodeType = MethodCodeType.Runtime)]
        int GetJMorphResult(uint dwRequest, uint dwCMode, int cwchInput,
            [MarshalAs(UnmanagedType.LPWStr)] string pwchInput, IntPtr pfCInfo, out IntPtr ppResult);
    } // end of IFELanguage Interface

    public class ImeLanguage
    {
        private bool FInitialized = false;

        private const int S_OK = 0;
        private const int CLSCTX_LOCAL_SERVER = 4;
        private const int CLSCTX_INPROC_SERVER = 1;
        private const int CLSCTX_INPROC_HANDLER = 2;
        private const int CLSCTX_SERVER = CLSCTX_INPROC_SERVER | CLSCTX_LOCAL_SERVER;
        private const int FELANG_REQ_REV = 0x00030000;
        private const int FELANG_CMODE_PINYIN = 0x00000100;
        private const int FELANG_CMODE_NOINVISIBLECHAR = 0x40000000;

        private static Dictionary<string, string> FuriDic = null;

        [DllImport("ole32.dll")]
        public static extern int CLSIDFromString([MarshalAs(UnmanagedType.LPWStr)] string lpsz, out Guid pclsid);

        [DllImport("ole32.dll")]
        public static extern int CoCreateInstance([MarshalAs(UnmanagedType.LPStruct)] Guid rclsid,
            IntPtr pUnkOuter, uint dwClsContext, [MarshalAs(UnmanagedType.LPStruct)] Guid riid, out IntPtr rpv);

        [DllImport("ole32.dll", CallingConvention = CallingConvention.StdCall)]
        public static extern int CoInitialize(IntPtr pvReserved);

        [DllImport("ole32.dll", CallingConvention = CallingConvention.StdCall)]
        public static extern int CoUninitialize();

        [DllImport("kernel32.dll")]
        public static extern int FormatMessage(int dwFlags, IntPtr lpSource,
            int dwMessageId, int dwLanguageId,
            StringBuilder lpBuffer, int nSize, IntPtr va_list_arguments);

        public const int FORMAT_MESSAGE_IGNORE_INSERTS = 0x200;
        public const int FORMAT_MESSAGE_FROM_SYSTEM = 0x1000;
        public const int FORMAT_MESSAGE_ARGUMENT_ARRAY = 0x2000;

        public ImeLanguage()
        {
            int res = CoInitialize(IntPtr.Zero);

            if (res == S_OK)
                FInitialized = true;

            if (FuriDic == null)
            {
                FuriDic = new Dictionary<string, string>();
                FuriDic.Add("あ", "a");
                FuriDic.Add("い", "i");
                FuriDic.Add("う", "u");
                FuriDic.Add("え", "e");
                FuriDic.Add("お", "o");

                FuriDic.Add("か", "(ka|ca)");
                FuriDic.Add("き", "ki");
                FuriDic.Add("く", "(ku|cu)");
                FuriDic.Add("け", "ke");
                FuriDic.Add("こ", "(ko|co)");

                FuriDic.Add("っか", "(kka|cca)");
                FuriDic.Add("っき", "kki");
                FuriDic.Add("っく", "(kku|ccu)");
                FuriDic.Add("っけ", "kke");
                FuriDic.Add("っこ", "(kko|cco)");

                FuriDic.Add("さ", "sa");
                FuriDic.Add("し", "(shi|si|ci)");
                FuriDic.Add("す", "su");
                FuriDic.Add("せ", "(se|ce)");
                FuriDic.Add("そ", "so");

                FuriDic.Add("っさ", "ssa");
                FuriDic.Add("っし", "(ssa|cci)");
                FuriDic.Add("っす", "ssu");
                FuriDic.Add("っせ", "(sse|cce)");
                FuriDic.Add("っそ", "sso");

                FuriDic.Add("た", "ta");
                FuriDic.Add("ち", "(t|ch)i");
                FuriDic.Add("つ", "(tsu|tu)");
                FuriDic.Add("て", "te");
                FuriDic.Add("と", "to");

                FuriDic.Add("った", "tta");
                FuriDic.Add("っち", "tti");
                FuriDic.Add("っつ", "(ttu|ttsu)");
                FuriDic.Add("って", "tte");
                FuriDic.Add("っと", "tto");

                FuriDic.Add("な", "na");
                FuriDic.Add("に", "ni");
                FuriDic.Add("ぬ", "nu");
                FuriDic.Add("ね", "ne");
                FuriDic.Add("の", "no");

                FuriDic.Add("は", "ha");
                FuriDic.Add("ひ", "hi");
                FuriDic.Add("ふ", "(h|f)u");
                FuriDic.Add("へ", "he");
                FuriDic.Add("ほ", "ho");

                FuriDic.Add("っは", "hha");
                FuriDic.Add("っひ", "hhi");
                FuriDic.Add("っふ", "(hhu|ffu)");
                FuriDic.Add("っへ", "hhe");
                FuriDic.Add("っほ", "hho");

                FuriDic.Add("ま", "ma");
                FuriDic.Add("み", "mi");
                FuriDic.Add("む", "mu");
                FuriDic.Add("め", "me");
                FuriDic.Add("も", "mo");

                FuriDic.Add("っま", "mma");
                FuriDic.Add("っみ", "mmi");
                FuriDic.Add("っむ", "mmu");
                FuriDic.Add("っめ", "mme");
                FuriDic.Add("っも", "mmo");

                FuriDic.Add("や", "ya");
                FuriDic.Add("ゆ", "yu");
                FuriDic.Add("よ", "yo");

                FuriDic.Add("っや", "yya");
                FuriDic.Add("っゆ", "yyu");
                FuriDic.Add("っよ", "yyo");

                FuriDic.Add("ら", "ra");
                FuriDic.Add("り", "ri");
                FuriDic.Add("る", "ru");
                FuriDic.Add("れ", "re");
                FuriDic.Add("ろ", "ro");

                FuriDic.Add("っら", "rra");
                FuriDic.Add("っり", "rri");
                FuriDic.Add("っる", "rru");
                FuriDic.Add("っれ", "rre");
                FuriDic.Add("っろ", "rro");

                FuriDic.Add("わ", "wa");
                FuriDic.Add("を", "wo");
                FuriDic.Add("ん", "(nn|n)");

                FuriDic.Add("っわ", "wwa");
                FuriDic.Add("っを", "wwo");

                FuriDic.Add("が", "ga");
                FuriDic.Add("ぎ", "gi");
                FuriDic.Add("ぐ", "gu");
                FuriDic.Add("げ", "ge");
                FuriDic.Add("ご", "go");

                FuriDic.Add("っが", "gga");
                FuriDic.Add("っぎ", "ggi");
                FuriDic.Add("っぐ", "ggu");
                FuriDic.Add("っげ", "gge");
                FuriDic.Add("っご", "ggo");

                FuriDic.Add("ざ", "za");
                FuriDic.Add("じ", "(z|j)i");
                FuriDic.Add("ず", "zu");
                FuriDic.Add("ぜ", "ze");
                FuriDic.Add("ぞ", "zo");

                FuriDic.Add("っざ", "zza");
                FuriDic.Add("っじ", "(zz|jj)i");
                FuriDic.Add("っず", "zzu");
                FuriDic.Add("っぜ", "zze");
                FuriDic.Add("っぞ", "zzo");

                FuriDic.Add("だ", "da");
                FuriDic.Add("ぢ", "di");
                FuriDic.Add("づ", "du");
                FuriDic.Add("で", "de");
                FuriDic.Add("ど", "do");

                FuriDic.Add("っだ", "dda");
                FuriDic.Add("っぢ", "ddi");
                FuriDic.Add("っづ", "ddu");
                FuriDic.Add("っで", "dde");
                FuriDic.Add("っど", "ddo");

                FuriDic.Add("ば", "ba");
                FuriDic.Add("び", "bi");
                FuriDic.Add("ぶ", "bu");
                FuriDic.Add("べ", "be");
                FuriDic.Add("ぼ", "bo");

                FuriDic.Add("っば", "bba");
                FuriDic.Add("っび", "bbi");
                FuriDic.Add("っぶ", "bbu");
                FuriDic.Add("っべ", "bbe");
                FuriDic.Add("っぼ", "bbo");

                FuriDic.Add("ぱ", "pa");
                FuriDic.Add("ぴ", "pi");
                FuriDic.Add("ぷ", "pu");
                FuriDic.Add("ぺ", "pe");
                FuriDic.Add("ぽ", "po");

                FuriDic.Add("っぱ", "ppa");
                FuriDic.Add("っぴ", "ppi");
                FuriDic.Add("っぷ", "ppu");
                FuriDic.Add("っぺ", "ppe");
                FuriDic.Add("っぽ", "ppo");

                FuriDic.Add("ぁ", "(x|l)a");
                FuriDic.Add("ぃ", "(x|l)i");
                FuriDic.Add("ぅ", "(x|l)u");
                FuriDic.Add("ぇ", "(x|l)e");
                FuriDic.Add("ぉ", "(x|l)o");

                FuriDic.Add("っぁ", "(xx|ll)a");
                FuriDic.Add("っぃ", "(xx|ll)i");
                FuriDic.Add("っぅ", "(xx|ll)u");
                FuriDic.Add("っぇ", "(xx|ll)e");
                FuriDic.Add("っぉ", "(xx|ll)o");

                FuriDic.Add("っ", "(lts|lt|xt)u");
                FuriDic.Add("ゃ", "(x|l)ya");
                FuriDic.Add("ゅ", "(x|l)yu");
                FuriDic.Add("ょ", "(x|l)yo");

                FuriDic.Add("っっ", "(xxt|llt)u");
                FuriDic.Add("っゃ", "(xx|ll)ya");
                FuriDic.Add("っゅ", "(xx|ll)yu");
                FuriDic.Add("っょ", "(xx|ll)yo");

                FuriDic.Add("きゃ", "kya");
                FuriDic.Add("きゅ", "kyu");
                FuriDic.Add("きょ", "kyo");

                FuriDic.Add("っきゃ", "kkya");
                FuriDic.Add("っきゅ", "kkyu");
                FuriDic.Add("っきょ", "kkyo");

                FuriDic.Add("ぎゃ", "gya");
                FuriDic.Add("ぎゅ", "gyu");
                FuriDic.Add("ぎょ", "gyo");

                FuriDic.Add("っぎゃ", "ggya");
                FuriDic.Add("っぎゅ", "ggyu");
                FuriDic.Add("っぎょ", "ggyo");

                FuriDic.Add("しゃ", "ss(h|y)a");
                FuriDic.Add("しゅ", "ss(h|y)u");
                FuriDic.Add("しょ", "ss(h|y)o");

                FuriDic.Add("っしゃ", "s(h|y)a");
                FuriDic.Add("っしゅ", "s(h|y)u");
                FuriDic.Add("っしょ", "s(h|y)o");

                FuriDic.Add("じゃ", "(j|zy)a");
                FuriDic.Add("じゅ", "(j|zy)u");
                FuriDic.Add("じょ", "(j|zy)o");

                FuriDic.Add("っじゃ", "(jj|zzy)a");
                FuriDic.Add("っじゅ", "(jj|zzy)u");
                FuriDic.Add("っじょ", "(jj|zzy)o");

                FuriDic.Add("ちゃ", "(ch|ty)a");
                FuriDic.Add("ちゅ", "(ch|ty)u");
                FuriDic.Add("ちょ", "(ch|ty)o");

                FuriDic.Add("っちゃ", "(cch|tty)a");
                FuriDic.Add("っちゅ", "(cch|tty)u");
                FuriDic.Add("っちょ", "(cch|tty)o");

                FuriDic.Add("ぢゃ", "dya");
                FuriDic.Add("ぢゅ", "dyu");
                FuriDic.Add("ぢょ", "dyo");

                FuriDic.Add("っぢゃ", "ddya");
                FuriDic.Add("っぢゅ", "ddyu");
                FuriDic.Add("っぢょ", "ddyo");

                FuriDic.Add("にゃ", "nya");
                FuriDic.Add("にゅ", "nyu");
                FuriDic.Add("にょ", "nyo");

                FuriDic.Add("っにゃ", "nnya");
                FuriDic.Add("っにゅ", "nnyu");
                FuriDic.Add("っにょ", "nnyo");

                FuriDic.Add("ひゃ", "hya");
                FuriDic.Add("ひゅ", "hyu");
                FuriDic.Add("ひょ", "hyo");

                FuriDic.Add("っひゃ", "hhya");
                FuriDic.Add("っひゅ", "hhyu");
                FuriDic.Add("っひょ", "hhyo");

                FuriDic.Add("びゃ", "bya");
                FuriDic.Add("びゅ", "byu");
                FuriDic.Add("びょ", "byo");

                FuriDic.Add("っびゃ", "bbya");
                FuriDic.Add("っびゅ", "bbyu");
                FuriDic.Add("っびょ", "bbyo");

                FuriDic.Add("ぴゃ", "pya");
                FuriDic.Add("ぴゅ", "pyu");
                FuriDic.Add("ぴょ", "pyo");

                FuriDic.Add("っぴゃ", "ppya");
                FuriDic.Add("っぴゅ", "ppyu");
                FuriDic.Add("っぴょ", "ppyo");

                FuriDic.Add("みゃ", "mya");
                FuriDic.Add("みゅ", "myu");
                FuriDic.Add("みょ", "myo");

                FuriDic.Add("っみゃ", "mmya");
                FuriDic.Add("っみゅ", "mmyu");
                FuriDic.Add("っみょ", "mmyo");

                FuriDic.Add("りゃ", "rya");
                FuriDic.Add("りゅ", "ryu");
                FuriDic.Add("りょ", "ryo");

                FuriDic.Add("っりゃ", "rrya");
                FuriDic.Add("っりゅ", "rryu");
                FuriDic.Add("っりょ", "rryo");

                FuriDic.Add("ふぁ", "(fa|fuxa|fula)");
                FuriDic.Add("ふぉ", "(fo|fuxo|fulo)");

                FuriDic.Add("っふぁ", "ffa");
                FuriDic.Add("っふぉ", "ffo");

                FuriDic.Add("てぃ", "(thi|texi|teli)");

                FuriDic.Add("ってぃ", "tthi");

                FuriDic.Add("ゐ", "i");
                FuriDic.Add("ゑ", "e");

            }
        }

        public void Dispose()
        {
            if (FInitialized)
            {
                CoUninitialize();
                FInitialized = false;
            }
        }

        ~ImeLanguage()
        {
            if (FInitialized)
                CoUninitialize();
        }

        public string GetRomaji(string str)
        {
            if (String.IsNullOrEmpty(str))
                return "";

            str = GetYomi(str);
            if (String.IsNullOrEmpty(str))
                return "";

            str = Strings.StrConv(str, VbStrConv.Narrow, 0);
            int len = str.Length;
            string s;
            StringBuilder sb = new StringBuilder();
            for (int i = 0; i < len; i++)
            {
                char c = str[i];
                if (c == 'ｰ')
                    sb.Append('-');
                else if (c == '\\')
                    sb.Append(@"\\");
                else if (c == '(')
                    sb.Append(@"\(");
                else if (c == ')')
                    sb.Append(@"\)");
                else if (c == '､')
                    sb.Append(',');
                else if (c < 'ぁ' || c > 'ん')
                    sb.Append(c);
                else
                {
                    if (c == 'っ')
                    {
                        if (i + 2 < len)
                        {
                            s = str.Substring(i, 3);
                            if (FuriDic.ContainsKey(s))
                            {
                                sb.Append(FuriDic[s]);
                                i += 2;
                                continue;

                            }
                        }
                        if (i + 1 < len)
                        {
                            s = str.Substring(i, 2);
                            if (FuriDic.ContainsKey(s))
                            {
                                sb.Append(FuriDic[s]);
                                i++;
                                continue;

                            }
                        }
                        s = c.ToString();
                        if (FuriDic.ContainsKey(s))
                        {
                            sb.Append(FuriDic[s]);
                            continue;
                        }
                    }
                    if (i + 1 < len)
                    {
                        if (c == 'き' || c == 'ぎ' || c == 'し' || c == 'じ' || c == 'ち' || c == 'ぢ' || c == 'に' || c == 'ひ' || c == 'び' || c == 'ぴ' || c == 'み' || c == 'り' || c == 'ふ' || c == 'て')
                        {
                            char c2 = str[i + 1];
                            if (c2 == 'ぁ' || c2 == 'ぃ' || c2 == 'ぉ' || c2 == 'ゃ' || c2 == 'ゅ' || c2 == 'ょ')
                            {
                                s = str.Substring(i, 2);
                                if (FuriDic.ContainsKey(s))
                                {
                                    sb.Append(FuriDic[s]);
                                    i++;
                                    continue;

                                }
                            }
                        }
                    }
                    if (FuriDic.ContainsKey(c.ToString()))
                    {
                        sb.Append(FuriDic[c.ToString()]);
                    }
                }
            }


            return sb.ToString().ToLower();
        }

        public string GetYomi(string str)
        {
            string yomi = String.Empty;
            Guid pclsid;
            int res;

            // 文字列の CLSID から CLSID へのポインタを取得する
            res = CLSIDFromString("MSIME.Japan", out pclsid);

            if (res != S_OK)
            {
                this.Dispose();
                return yomi;
            }
            Guid riid = new Guid("019F7152-E6DB-11D0-83C3-00C04FDDB82E");
            IntPtr ppv;
            res = CoCreateInstance(pclsid, IntPtr.Zero, CLSCTX_SERVER, riid, out ppv);

            if (res != S_OK)
            {
                Console.WriteLine(GetMessage(res));
                this.Dispose();
                return yomi;
            }

            IFELanguage language = Marshal.GetTypedObjectForIUnknown(ppv, typeof(IFELanguage)) as IFELanguage;
            res = language.Open();

            if (res != S_OK)
            {
                this.Dispose();
                return yomi;
            }

            IntPtr result;

            res = language.GetJMorphResult(FELANG_REQ_REV, FELANG_CMODE_PINYIN | FELANG_CMODE_NOINVISIBLECHAR,
                    str.Length, str, IntPtr.Zero, out result);

            if (res != S_OK || result == IntPtr.Zero)
            {
                this.Dispose();
                return yomi;
            }

            yomi = Marshal.PtrToStringUni(Marshal.ReadIntPtr(result, 4), Marshal.ReadInt16(result, 8));

            language.Close();

            return yomi;
        }

        private string[] Split(string str)
        {
            char c;
            List<string> list = new List<string>();
            StringBuilder sb = new StringBuilder();

            for (int i = 0; i < str.Length; i++)
            {
                c = str[i];
                if (c == '\\')
                {
                    if (i + 1 < str.Length)
                    {
                        sb.Append(str[i + 1]);
                        i++;
                        continue;
                    }
                }
                if (c == '(' || c == ')')
                {
                    if (sb.Length > 0)
                    {
                        list.Add(sb.ToString());
                        sb.Length = 0;
                    }
                    continue;
                }
                sb.Append(c);
            }
            if (sb.Length > 0)
                list.Add(sb.ToString());

            return list.ToArray();
        }

        public bool StartsWith(string regex, string str)
        {
            if (String.IsNullOrEmpty(regex))
                return false;

            bool flag;
            string[] regexs = Split(regex);

            foreach (string reg in regexs)
            {
                if (String.IsNullOrEmpty(reg))
                    continue;

                string[] lines = reg.Split('|');
                flag = false;
                foreach (string line in lines)
                {
                    if (str.Length > line.Length)
                    {
                        if (!str.StartsWith(line))
                        {
                            continue;
                        }
                        str = str.Substring(line.Length);
                        flag = true;
                        break;
                    }
                    else
                    {
                        if (line.StartsWith(str))
                        {
                            return true;
                        }

                    }
                }
                if (!flag)
                    return false;
            }
            return (str.Length == 0);
        }

        public static string GetMessage(int errorCode)
        {
            StringBuilder lpBuffer = new StringBuilder(0x200);
            if (FormatMessage(FORMAT_MESSAGE_IGNORE_INSERTS | FORMAT_MESSAGE_FROM_SYSTEM |
                FORMAT_MESSAGE_ARGUMENT_ARRAY,
                IntPtr.Zero, errorCode, 0, lpBuffer, lpBuffer.Capacity, IntPtr.Zero) != 0)
            {
                return lpBuffer.ToString();
            }
            return "Unknown";
        }

/*
        [STAThread]
        static void Main(string[] args)
        {

            ImeLanguage lang = new ImeLanguage();
            //Console.WriteLine(lang.GetRomaji("<＜！朝鮮"));
            //Console.WriteLine(lang.GetYomi("虚心坦懐"));
            string s = lang.GetRomaji("ばってん");
            Console.WriteLine(s);
            Console.Read();
        }*/

    }
}
