
// ------------------------------------------------------------------------- //
// ------------ local variables -------------------------------------------- //
// ------------------------------------------------------------------------- //
var
	/**
	 * 全てのクラスのスーパークラスのようなもの。(ライブラリ内にカプセル化されているため、ユーザが触ることはありません)<br>
	 * X.Class.create() で定義されたクラスのインスタンスが共通で備えるメソッド を確認してください。
	 * @class __ClassBase__
	 * @private
	 * @abstract
	 */
	__ClassBase__ = {
			/**
			 * クラス名
			 * @type {string}
			 */
			name         : ''
		},

	X_Class_CLASS_LIST         = [],
	X_Class_DEF_LIST           = [],
	X_Class_CALLING_SUPER      = [],
	X_Class_CALL_SUPER_STACK   = [],
	X_Class_traits             = null,
	X_Class_useObjectCreate    = false, // !!Object.create, http://jsperf.com/prototype-vs-object-create-perf
	X_Class_use_proto_         = !X_UA[ 'OperaMobile' ] && !X_UA[ 'OperaTablet' ] && !!X_emptyFunction.prototype.__proto__,
		// Opera Mobile 12.10 Android11 IS01 でクラスのメンバが欠落する問題に遭遇。__proto__ を辞めると動作,,,

X_Class_CommonMethods =
/** @lends __ClassBase__.prototype */
{
	/**
	 * 全ての動的メンバを削除して、インスタンスを破棄する。<br>
	 * インスタンスが X.EventDispatcher とそのサブクラスの場合、次の動作をする。
	 * <ol>
	 * <li>X.Event.BEFORE_KILL_INSTANCE を発火する。戻り値のビットフラグに X_Callback.PREVENT_DEFAULT が立つ場合、破棄をキャンセルし X.Event.KILL_INSTANCE_CANCELED を発火する。
	 * <li>破棄に進む場合は、X.Event.KILL_INSTANCE を発火する。
	 * <li>dispatch 中は、インスタンスの全ての dispatch が終了するまで実際の破棄を待つ。
	 * <li>実際の破棄では、インスタンスのメンバの削除に加えて全てのイベントリスナを解除する。
	 */
	// TODO kill したインスタンスのイベントが残っていないか？これは開発用のみ
	'kill' : function(){
		var def, listeners, p;
		
		// TODO kill 中の kill の呼び出しを防ぐ, 破棄済のインスタンスへの kill
		
		if( this[ 'instanceOf' ]( X_EventDispatcher ) ){

			if( this[ 'dispatch' ]( X_EVENT_BEFORE_KILL_INSTANCE ) & X_Callback_PREVENT_DEFAULT ){
				this[ 'dispatch' ]( X_EVENT_KILL_INSTANCE_CANCELED );
				return;
			};
			
			listeners = this[ '_listeners' ];
			
			if( listeners && listeners[ X_LISTENERS_DISPATCHING ] ){
				listeners[ X_LISTENERS_KILL_RESERVED ] = true;
				return;
			};

			// asyncDispatch の削除
			for( p in X_EventDispatcher_LAZY_TIMERS ){
				if( X_EventDispatcher_LAZY_TIMERS[ p ] === this ){
					 // delete X_EventDispatcher_LAZY_TIMERS[ p ];　コレ不要
					X_Timer_remove( p );
				};
			};
			
			this[ 'dispatch' ]( X_EVENT_KILL_INSTANCE );
			listeners && X_EventDispatcher_unlistenAll( this );
		};
		
		X_Object_clear( this );
		
		def = X_Class_getClassDef( this );
		
		if( def.pool ){
			def.live.splice( def.live.indexOf( this ), 1 );
			def.pool[ def.pool.length ] = this;
		};
	},
	
	/**
	 * 関数は Constructor 内で使用します。クラス定義を辿ってスーパークラスのコンストラクタを探します。<br>
	 * 内部的には、呼び出したコンストラクタは配列に控え(X_Class_CALLING_SUPER)、呼び出したコンストラクタ内でさらに Super が呼ばれた場合、配列を元にさらにスーパーなコンストラクタを辿ります。
	 * @example Constructor : function( arg1, arg2 ){
	 * 	this.Super( aeg1, arg2 );
	 * }
	 * @param var_args {...?} 親コンストラクタを呼ぶ際に渡す任意の数の引数
	 * @return {*}
	 */
	// TODO 現在 new しているインスタンスを保持してチェックする
	'Super' : function( var_args ){
		var sClass = this,
			i      = X_Class_CALLING_SUPER.indexOf( sClass ),
			n      = -1,
			l, sList, def, sConst, ret;
	
		if( i === -1 ){
			X_Class_CALLING_SUPER[ l = X_Class_CALLING_SUPER.length ] = sClass;
			X_Class_CALL_SUPER_STACK[ l ] = sList = [];
		} else {
			sList = X_Class_CALL_SUPER_STACK[ i ];
		};
		
		while( sClass ){
			def    = X_Class_getClassDef( sClass );
			sClass = def.SuperClass;
			sConst = def.SuperConstructor;
			if( sConst && sList[ ++n ] !== sConst ){
				sList[ n ] = sConst;
				ret = sConst.apply( this, arguments );
				--sList.length;
				if( !sList.length ){
					X_Class_CALLING_SUPER.splice( i, 1 );
					X_Class_CALL_SUPER_STACK.splice( i, 1 );
				};
				return ret;
			};
		};
		console.log( 'スーパークラスのコンストラクタが見つかりません' );
	},

	/**
	 * func について、親クラスで設定されている同名の関数メンバーを呼び出す。<br>
	 * 第一引数に関数を指定し、2つ以上の異なる名前で同じ関数がメンバーがいた場合、動作が不確実になります。<br>
	 * 参考:<a href="http://qiita.com/no22@github/items/d3bead2acbb7ff1fb86b" target="_blank">ES5なJavascriptでモダンなクラス的継承＆スーパー呼び出し </a>
	 * @param funcNameOrFunc {Function|string} スーパークラスの関数名 または、オーバーライド済の自身の関数。
	 * @param var_args {...*} オーバーライド元関数に渡す任意の数の引数
	 * @example return this.superCall( arguments.callee, param0, param1, ... );
	 * @return {*} オーバーライド元の関数を呼び出した戻り値。
	 */
	'superCall' : function( funcNameOrFunc, var_args ){
		var sClass = this,
			args   = arguments,
			name, p, sFunc, hit = false;
		if( X_Type_isFunction( funcNameOrFunc ) ){
			for( p in this.constructor.prototype ){
				if( this.constructor.prototype[ p ] === funcNameOrFunc ){
					name = p;
					break;
				};
			};
			if( !name ) return;
		} else {
			return;
		};
		
		if( X_EMPTY_OBJECT[ name ] ) return;
		
		while( sClass ){
			def    = X_Class_getClassDef( sClass );
			sClass = def.SuperClass;
			sFunc  = sClass.prototype[ name ];
			if( sFunc === funcNameOrFunc ){
				hit = true; // 現在の関数にヒット
			} else
			if( hit && X_Object_inObject( name, this ) ){
				if( X_Type_isFunction( sFunc ) ){
					switch( args.length ){
						case 1 :
							return sFunc.call( this );
						case 2 :
							return sFunc.call( this, args[ 1 ] );
						case 3 :
							return sFunc.call( this, args[ 1 ], args[ 2 ] );
						case 4 :
							return sFunc.call( this, args[ 1 ], args[ 2 ], args[ 3 ] );
						default :
							args = X_Object_cloneArray( args );
							args.shift();
							return sFunc.apply( this, args );
					};
				};
				break;
			};
		};
	},
	
	/**
	 * インスタンスのクラスか？またはスーパークラスか？調べる。<br>
	 * instanceof 構文をサポートしない環境(IE4,Mac IE5)を想定する場合、必ずこのメソッドを使用すること。<br>
	 * クラスのインスタンスか？だけ調べたい場合は this.constructor === klass が高速。
	 * @param klass {__ClassBase__} クラス定義
	 * @return {boolean}
	 */
	// TODO instanceof に対応したブラウザはそちらを使用
	'instanceOf' : function( klass ){
		var Super = this;
		if( this.constructor === klass ) return true;
		while( Super = X_Class_getClassDef( Super ).SuperClass ){
			if( Super === klass ) return true;
		};
		return false;
	}
};

// ------------------------------------------------------------------------- //
// --- interface ----------------------------------------------------------- //
// ------------------------------------------------------------------------- //

/*
 * @enum {number}
 * @const
 */
var X_Class = {
	NONE         :  0,
	POOL_OBJECT  :  1,
	ABSTRACT     :  2,
	FINAL        :  4,
	SINGLETON    :  8
};

/**
 * <p>Class を定義し システムの管理下に置く。
 * <p>prototype 継承のブラウザ毎の差異も吸収し、 以下から最適な方法をしてくれる。
 * 
 * <ol>
 * <li>Object.create はパフォーマンスが悪そうなので現在は使っていない。
 * <li>SubClass.prototype.__proto__ = SuperClass.prototype;
 * <li>SubClass.prototype = new SuperClass;
 * </ol>
 * 
 * <ol>
 * <li>X.Class.create( opt_settings, opt_name, opt_props ) でクラスを登録．
 * <li>コンストラクタ となるメソッドは、opt_props 内の Constructor : function( arg ){ ... }, に書く．
 * <li>通常通り new で インスタンス生成
 * <li>kill() でオブジェクトをクリーンして削除、pool が有効の場合は pool される．
 * <li>pool が有効の場合、new で pool されたインスタンスが返される．
 * </ol>
 * @namespace X.Class
 * @alias X.Class
 */ 
X[ 'Class' ] = /** @lends X.Class */ {

    /**
     * 設定なし。
     * @const
     */	
	'NONE'         : X_Class.NONE,
	
    /**
     * インスタンスは破棄時(this.kill())に回収され、次回の new MyClass() 時に再利用されます。
     * @const
     */
	'POOL_OBJECT'  :  X_Class.POOL_OBJECT,
	
	/**
	 * 定義するクラスは抽象クラスになります。new AbstractClass() とするとエラーになります。
	 * @const
	 */
	'ABSTRACT'     :  X_Class.ABSTRACT,

	/**
	 * クラスの継承を禁止する。
	 * @const
	 */
	'FINAL'        :  X_Class.FINAL,

	/**
	 * 未実装。でも目印になるので付けておきましょう。
	 * @const
	 */
	'SINGLETON'    : X_Class.SINGLETON,

	'create'       : X_Class_create
	
	// TODO collect
};



// ------------------------------------------------------------------------- //
// --- implements ---------------------------------------------------------- //
// ------------------------------------------------------------------------- //
	/**
	 * クラスを定義する。<br>
	 * X.Class.create() によるクラス定義は必ずしもコンストラクタ('Constructor')を必要としません。クラス定義時にコンストラクタが未設定の場合、スーパークラスがあればそのコンストラクタを使用します。
	 * @alias X.Class.create
	 * @param {string} [displayName] クラスの名前
	 * @param {number} [classSetting=0] X_Class.POOL_OBJECT | X_Class.FINAL など
	 * @param {object} [props={}] このクラスのメンバと関数。コンストラクタは Constructor と書くこと
	 * @return {__ClassBase__}
	 * @example var myClass = X.Class.create(
	 * 	'myClass',
	 *  X.Class.FINAL,
	 *  {
	 * 	 name : '',
	 * 	 Constructor : function( obj ){
	 * 	  this.name = obj.name;
	 * 	 },
	 * 	 getName : function(){
	 * 	  return this.name;
	 * 	 },
	 * 	 setName : function(v){
	 * 	  this.name = v;
	 * 	 }
	 *  }
	 * );
	 */
	function X_Class_create( /* displayName, classSetting, privateClass, props */ ){
		var args        = X_Object_cloneArray( arguments ),
			displayName = args[ 0 ],
			classSetting,
			opt_pool, opt_abstract, opt_final,
			privateDef,
			props,
			klass,
			classDef = {},
			cbHash = { proxy : X_Class_actualConstructor, classDef : classDef };

		if( X_Type_isString( displayName ) === true ){
			classDef.displayName = displayName;
			args.shift();
		};
		
		// クラス設定
		classDef.setting = classSetting = args[ 0 ];
		if( X_Type_isNumber( classSetting ) ){
			opt_pool     = !!( classSetting & X_Class.POOL_OBJECT  );
			opt_abstract = !!( classSetting & X_Class.ABSTRACT     );
			opt_final    = !!( classSetting & X_Class.FINAL        );
			if( opt_final && opt_abstract ){
				X.Logger.critical( 'final & Abstract!' );
				return;
			};	
			args.shift();
		} else {
			classDef.setting = 0;
		};
		
		// インスタンスのメンバー
		props = args[ 0 ];
		if( !X_Type_isObject( props ) ){
			// クラスメンバ用オブジェクトが無しでもクラスは作成可能
			props = {};
		} else
		if( props[ 'Constructor' ] ){
			//{+dev
			if( !X_Type_isFunction( props[ 'Constructor' ] ) ){
				alert( '"Constructor" is not function.' );
				return;
			};
			//}+dev
			classDef.Constructor = props[ 'Constructor' ];
		};

		klass  = X_Callback_actualClosure( cbHash ); // TODO callbackHash を class定義の置き場所にしてしまう！なるほど…
		cbHash.klass = klass;
		klass[ 'superClassOf' ] = X_Class_superClassOf;
		klass[ 'subClassOf' ]   = X_Class_subClassOf;
		
		if( X_Class_useObjectCreate ){
			klass.prototype = X_Class_override( X_Class_override( X_Class_traits || klass.prototype, props, true ), X_Class_CommonMethods, false );
			klass.prototype.constructor = klass;
		} else
		if( X_Class_use_proto_ ){
			X_Class_override( klass.prototype, props, true );
			if( X_Class_traits ){
				klass.prototype.__proto__ = X_Class_traits;
			} else {
				X_Class_override( klass.prototype, X_Class_CommonMethods, false );
			};
		} else {
			klass.prototype = X_Class_override( X_Class_override( X_Class_traits || klass.prototype, props, true ), X_Class_CommonMethods, false );
			klass.prototype.constructor = klass;
		};
		
		klass[ 'name' ] = displayName;
		
		if( opt_abstract ){
			classDef.Abstract = true;
		} else
		if( opt_pool ){
			classDef.pool = [];
			classDef.live = [];
		};			
		if( opt_final ){
			classDef.Final = true;
		} else {
			klass[ 'inherits' ] = X_Class_inherits;
		};			
		
		X_Class_CLASS_LIST.push( klass );
		X_Class_DEF_LIST.push( classDef );				

		return klass;
	};



function X_Class_getClass( instance ){
	var cList    = X_Class_CLASS_LIST,
		i        = cList.length,
		klass;
	for( ; i; ){
		klass = cList[ --i ];
		if( instance.constructor === klass ) return klass;
	};
	if( cList.indexOf( instance ) !== -1 ) return instance;
};

function X_Class_getClassDef( KlassOrInstance ){
	var i = X_Class_CLASS_LIST.indexOf( KlassOrInstance );
	if( i === -1 ) i = X_Class_CLASS_LIST.indexOf( X_Class_getClass( KlassOrInstance ) );
	if( i !== -1 ) return X_Class_DEF_LIST[ i ];
	
	if( X_Class_DEF_LIST.indexOf( KlassOrInstance ) !== -1 ) return KlassOrInstance;
};

/* over のプロパティを target にコピーする．ただし target の プロパティが優先, force で解除 */
function X_Class_override( target, src, force ){
	var p;
	for( p in src ){
		if( p === 'Constructor' ) continue;
		if( p === '__proto__' || p === 'prototype' || p === 'constructor' ){
			X.Logger.critical( p + ' is reserved!' );
			return;
		};
		if( force || target[ p ] === undefined ){
			target[ p ] = src[ p ];
		};
	};
	return target;
};

/**
 * スーパークラスか？調べます。
 * @alias __ClassBase__.superClassOf
 * @param klass {__ClassBase__}
 * @return {boolean}
 */
function X_Class_superClassOf( klass ){
	var myDef      = X_Class_getClassDef( this ),
		targetDef  = X_Class_getClassDef( klass ),
		SuperClass = klass;

	if( !myDef || !targetDef || this === klass ) return false;
	
	while( SuperClass = X_Class_getClassDef( SuperClass ).SuperClass ){
		if( SuperClass === this ) return true;
	};
	
	return false;
};

/**
 * サブクラスか？調べます。
 * @alias __ClassBase__.subClassOf
 * @type {Function}
 * @param klass {__ClassBase__}
 * @return {boolean}
 */
function X_Class_subClassOf( klass ){
	return klass && X_Class_superClassOf.call( klass, this );
};
			
/**
 * サブクラスを作ります。与える引数は X_Class.create と同じです。http://d.hatena.ne.jp/m-hiyama/20051018/1129605002
 * @alias __ClassBase__.inherits
 * @example var SubClass = SuperClass.inherits( 'Sub', X_Class.FINAL, { ... } );
 * @param {string} [displayName] クラスの名前
 * @param {number} [classSetting=0] X_Class.POOL_OBJECT | X_Class.FINAL など
 * @param {__ClassBase__=} [privateClass] このクラスとペアで動作するシャドウクラス
 * @param {object} [props={}] このクラスのメンバと関数。コンストラクタは Constructor と書くこと
 * @return {__ClassBase__}
 */
function X_Class_inherits( /* displayName, classSetting, opt_PrivateClass, props */ ){
	var args        = X_Object_cloneArray( arguments ),
		params      = [],
		Super       = this,
		superDef    = X_Class_getClassDef( Super ),
		displayName = args[ 0 ],
		classSetting,
		//opt_super,
		klass, def;
	if( superDef.Final ) X.Logger.critical( 'X.Class inherits, Class is final!' );
	
	// サブクラス名
	if( X_Type_isString( displayName ) ){
		args.shift();
	} else {
		displayName = 'SubClass of ' + superDef.displayName;
	};
	params.push( displayName );
	
	// サブクラス設定
	classSetting = args[ 0 ];
	if( X_Type_isNumber( classSetting ) ){
		args.shift();
	} else {
		// クラス設定がない場合、親からコピーして、Abstract flag は落とす??
		classSetting = superDef.setting;// &= ~X_Class.ABSTRACT;
	};

	params.push( classSetting );

	// サブクラスのシャドウ
	if( args[ 0 ] && X_Class_getClass( args[ 0 ] ) ){
		params.push( args.shift() );
	};
	
	/* props 未定義でも可 */
	params.push( args[ 0 ] );
	
	// 継承クラスの作成
	if( X_Class_useObjectCreate ){
		X_Class_traits = Object.create( Super.prototype );
	} else
	if( X_Class_use_proto_ ){
		X_Class_traits = Super.prototype;
	} else {
		X_Class_traits = new Super( X_Closure_COMMAND_DROP );
	};
	klass  = X_Class_create.apply( X.Class, params );
	X_Class_traits = null;
	
	def    = X_Class_getClassDef( klass );
	// 継承用プロパティを控える
	def.SuperClass       = Super;
	def.SuperProto       = Super.prototype;
	def.SuperConstructor = superDef.Constructor || superDef.SuperConstructor;
	
	return klass;
};
	
/*
 * new の実体．コンストラクタの機能は instance.Constructor に書く．
 * これにより pool された オブジェクト（破棄されたインスタンス） を再利用できる
 */
function X_Class_actualConstructor( f, args ){
	var klass    = f.klass,
		def      = f.classDef,
		instance, obj;

	if( def.Abstract ){
		X.Logger.critical( 'AbstractClass!' );
		return;
	};
	
	instance = def.pool && def.pool.length ?
					def.pool.pop() :
				X_Class_useObjectCreate ?
					Object.create( klass.prototype ) :
					new klass( X_Closure_COMMAND_DROP );
	
	def.live && def.live.push( instance );

	obj = def.Constructor ?
			def.Constructor.apply( instance, args ) :
		def.SuperConstructor &&
			def.SuperConstructor.apply( instance, args );

	if( obj !== instance && ( X_Type_isObject( obj ) || X_Type_isFunction( obj ) ) ){ // Class
		instance[ 'kill' ]();
		return obj;
	};
	return instance;
};

console.log( 'X.Core.Class' );
