------------------------------------------------------------------------------
-- Copyright 2018 Levashev Ivan Aleksandrovich                              --
--                                                                          --
-- Licensed under the Apache License, Version 2.0 (the "License");          --
-- you may not use this file except in compliance with the License.         --
-- You may obtain a copy of the License at                                  --
--                                                                          --
--     http://www.apache.org/licenses/LICENSE-2.0                           --
--                                                                          --
-- Unless required by applicable law or agreed to in writing, software      --
-- distributed under the License is distributed on an "AS IS" BASIS,        --
-- WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. --
-- See the License for the specific language governing permissions and      --
-- limitations under the License.                                           --
------------------------------------------------------------------------------

with Ada.Finalization;

with System.Address_To_Access_Conversions;
with System.Atomic_Counters;

package body Fast_Strings.Unbounded is

   type String_Constraints_Access is access constant String_Constraints;

   type String_Access_Structure is record
      Data : System.Address;
      Constraints : String_Constraints_Access;
   end record;

   pragma Assert (String_Access'Size = String_Access_Structure'Size);
   pragma Assert (String_Access'Alignment = String_Access_Structure'Alignment);

   type Unbounded_String_Constant_Access is
     access constant Unbounded_String;

   type Mutable_Constant_Unbounded_String_Container is limited record
      Data : Unbounded_String_Constant_Access;
      Constraints : aliased String_Constraints;
      Item : aliased Unbounded_String;
   end record;

   pragma Assert (Constant_Unbounded_String_Container'Size = Mutable_Constant_Unbounded_String_Container'Size);
   pragma Assert (Constant_Unbounded_String_Container'Alignment = Mutable_Constant_Unbounded_String_Container'Alignment);

   type Mutable_String_View is limited record
      Data : String_Access_Structure;
      To_Unbounded_String : Mutable_Constant_Unbounded_String_Container;
   end record;

   pragma Assert (String_View'Size = Mutable_String_View'Size);
   pragma Assert (String_View'Alignment = Mutable_String_View'Alignment);

   package String_View_Conversions is new
     System.Address_To_Access_Conversions
     (String_View);

   type Shared_String_Structure (Max_Length : Natural) is limited record
      Counter : System.Atomic_Counters.Atomic_Counter;
      --  Reference counter

      Last : Natural := 0;
      Data : String (1 .. Max_Length);
      --  Last is the index of last significant element of the Data. All
      --  elements with larger indexes are currently insignificant.
   end record;

   type Shared_String_Access_Structure is access all Shared_String_Structure;

   type Unbounded_String_Structure is new Ada.Finalization.Controlled with record
      Reference : Shared_String_Access_Structure;
   end record;

   pragma Assert (Unbounded_String'Size = Unbounded_String_Structure'Size);
   pragma Assert (Unbounded_String'Alignment = Unbounded_String_Structure'Alignment);

   Null_Unbounded_String_Structure : aliased Unbounded_String_Structure
     with Import, Address => Null_Unbounded_String'Address;
   Null_Unbounded_String_Aliased : aliased Unbounded_String
     with Import, Address => Null_Unbounded_String'Address;
   Empty_Shared_String_Access : Shared_String_Access_Structure renames
     Null_Unbounded_String_Structure.Reference;

   --------------------
   -- To_String_View --
   --------------------

   function To_String_View
     (Source : in Unbounded_String)
     return String_View
   is
      Source_Structure : Unbounded_String_Structure with Import, Address => Source'Address;
      Last : constant Natural :=
        (if Source_Structure.Reference = Empty_Shared_String_Access then
            0
         else
            Source_Structure.Reference.all.Last);
   begin
      return Result : aliased String_View :=
        (Data                => Internal.Aliased_Empty_String'Access,
         To_Unbounded_String =>
           (Data             => Null_Unbounded_String_Aliased'Unchecked_Access,
            Constraints      => (1, Last),
            Item             => Source))
      do
         if Last = 0 then
            return;
         end if;

         declare
            Mutable_Result : Mutable_String_View
              with Import,
                   Address => String_View_Conversions.To_Address
                                (Result'Unchecked_Access);
            Item_Address : constant System.Address :=
              Mutable_Result.To_Unbounded_String.Item'Address;
            Item_Structure : Unbounded_String_Structure
              with Import, Address => Item_Address;
            Shared_Data : Shared_String_Structure renames
              Item_Structure.Reference.all;
         begin
            Mutable_Result.Data.Data := Shared_Data.Data'Address;
            Mutable_Result.Data.Constraints := Result.To_Unbounded_String.Constraints'Unchecked_Access;
            Mutable_Result.To_Unbounded_String.Data :=
              Mutable_Result.To_Unbounded_String.Item'Unchecked_Access;
         end;
      end return;
   end To_String_View;

   ----------------------
   -- Force_Uniqueness --
   ----------------------

   procedure Force_Uniqueness (Target : in out Unbounded_String; Last : out Natural) is
      Target_Structure : Unbounded_String_Structure with Import, Address => Target'Address;
      Last_Constant : constant Natural :=
        (if Target_Structure.Reference = Empty_Shared_String_Access then
            0
         else
            Target_Structure.Reference.all.Last);
   begin
      Last := Last_Constant;
      if Last_Constant = 0 then
         return;
      end if;

      declare
         Saved_Reference : constant Shared_String_Access_Structure :=
           Target_Structure.Reference;
      begin
         if not System.Atomic_Counters.Is_One (Saved_Reference.all.Counter) then
            Set_Unbounded_String (Target, Saved_Reference.Data (1 .. Last));
         end if;
      end;
   end Force_Uniqueness;

   type Unbounded_String_Access is access all Unbounded_String;

   type Mutable_String_Editor_Reference is
     new Ada.Finalization.Limited_Controlled with
   record
      Data : String_Access_Structure;
      Target : Unbounded_String_Access;
      Constraints : aliased String_Constraints;
      Target_Shadow : aliased Unbounded_String;
   end record;

   pragma Assert (String_Editor_Reference'Size = Mutable_String_Editor_Reference'Size);
   pragma Assert (String_Editor_Reference'Alignment = Mutable_String_Editor_Reference'Alignment);

   package String_Editor_Reference_Conversions is new
     System.Address_To_Access_Conversions
     (String_Editor_Reference);

   ----------------------------------------
   -- String_Editor_Reference.Initialize --
   ----------------------------------------

   overriding
   procedure Initialize (Object : in out String_Editor_Reference) is
   begin
      -- Target_Shadow is already assigned to Target
      Set_Unbounded_String (Object.Target.all, "");
      declare
         Last : Natural;
      begin
         Force_Uniqueness (Object.Target_Shadow, Last);

         if Last = 0 then
            return;
         end if;

         Object.Constraints.Last := Last;

         declare
            Mutable_Object : Mutable_String_Editor_Reference
              with Import,
                   Address => String_Editor_Reference_Conversions.To_Address
                                (Object'Unchecked_Access);
            Target_Shadow_Address : constant System.Address :=
              Mutable_Object.Target_Shadow'Address;
            Target_Shadow_Structure : Unbounded_String_Structure
              with Import, Address => Target_Shadow_Address;
            Shared_Data : Shared_String_Structure renames Target_Shadow_Structure.Reference.all;
         begin
            Mutable_Object.Data.Data := Shared_Data.Data'Address;
            Mutable_Object.Data.Constraints := Mutable_Object.Constraints'Unchecked_Access;
         end;
      end;
   end Initialize;

   --------------------------------------
   -- String_Editor_Reference.Finalize --
   --------------------------------------

   overriding
   procedure Finalize (Object : in out String_Editor_Reference) is
      Mutable_Object : Mutable_String_Editor_Reference
        with Import,
             Address => String_Editor_Reference_Conversions.To_Address
                          (Object'Unchecked_Access);
      Target_Shadow_Address : constant System.Address :=
        Mutable_Object.Target_Shadow'Address;
      Target_Shadow_Structure : Unbounded_String_Structure
        with Import, Address => Target_Shadow_Address;
      Target_Address : constant System.Address :=
        Object.Target.all'Address;
      Target_Structure : Unbounded_String_Structure
        with Import, Address => Target_Address;
      Saved_Target_Reference : constant Shared_String_Access_Structure :=
        Target_Structure.Reference;
   begin
      -- swap
      Target_Structure.Reference := Target_Shadow_Structure.Reference;
      Target_Shadow_Structure.Reference := Saved_Target_Reference;
      -- Finalize on Target_Shadow will do the rest if required
   end Finalize;

end Fast_Strings.Unbounded;
