package net.morilib.db.jdbc;

import java.io.IOException;
import java.sql.BatchUpdateException;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.sql.SQLWarning;
import java.sql.Statement;
import java.util.List;

import net.morilib.db.misc.ErrorBundle;
import net.morilib.db.relations.Relation;

public class RelationsStatement implements Statement {

	RelationsConnection conn;
	int maxFieldSize = 0, maxRows = 0;
	Relation select0 = null;
	int count0 = -1;
	List<String> batch;

	RelationsStatement(RelationsConnection c) {
		conn = c;
	}

	@Override
	public <T> T unwrap(Class<T> iface) throws SQLException {
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public boolean isWrapperFor(Class<?> iface) throws SQLException {
		return false;
	}

	@Override
	public ResultSet executeQuery(String sql) throws SQLException {
		Object o;

		try {
			o = conn.engine.execute(conn.schema, sql);
			if(o instanceof Relation) {
				return new RelationsResultSet(this, (Relation)o);
			} else {
				throw ErrorBundle.getDefault(10038);
			}
		} catch (IOException e) {
			throw ErrorBundle.getDefault(10037);
		}
	}

	@Override
	public int executeUpdate(String sql) throws SQLException {
		Object o;

		try {
			o = conn.engine.execute(conn.schema, sql);
			if(o instanceof Number) {
				return ((Number)o).intValue();
			} else {
				return 0;
			}
		} catch (IOException e) {
			throw ErrorBundle.getDefault(10037);
		}
	}

	void checkclose() throws SQLException {
		if(conn == null) {
			throw ErrorBundle.getDefault(20002);
		}
	}

	@Override
	public void close() throws SQLException {
		checkclose();
		conn = null;
	}

	@Override
	public int getMaxFieldSize() throws SQLException {
		return maxFieldSize;
	}

	@Override
	public void setMaxFieldSize(int max) throws SQLException {
		if((maxFieldSize = max) < 0) {
			throw ErrorBundle.getDefault(20001);
		}
	}

	@Override
	public int getMaxRows() throws SQLException {
		return maxRows;
	}

	@Override
	public void setMaxRows(int max) throws SQLException {
		if((maxRows = max) < 0) {
			throw ErrorBundle.getDefault(20001);
		}
	}

	@Override
	public void setEscapeProcessing(
			boolean enable) throws SQLException {
		// ignore it
	}

	@Override
	public int getQueryTimeout() throws SQLException {
		return 0;
//		return queryTimeout;
	}

	@Override
	public void setQueryTimeout(int seconds) throws SQLException {
		// ignore it
//		if((queryTimeout = seconds) < 0) {
//			throw ErrorBundle.getDefault(20001);
//		}
	}

	@Override
	public void cancel() throws SQLException {
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public SQLWarning getWarnings() throws SQLException {
		checkclose();
		return null;
	}

	@Override
	public void clearWarnings() throws SQLException {
		checkclose();
		// ignore it
	}

	@Override
	public void setCursorName(String name) throws SQLException {
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public boolean execute(String sql) throws SQLException {
		Object o;

		checkclose();
		try {
			o = conn.engine.execute(conn.schema, sql);
			if(o instanceof Relation) {
				select0 = (Relation)o;
				return true;
			} else if(o instanceof Number) {
				count0 = ((Number)o).intValue();
				return false;
			} else {
				count0 = 0;
				return false;
			}
		} catch (IOException e) {
			throw ErrorBundle.getDefault(10037);
		}
	}

	@Override
	public ResultSet getResultSet() throws SQLException {
		Relation s;

		checkclose();
		if(select0 != null) {
			s = select0;
			select0 = null;
			return new RelationsResultSet(this, s);
		} else {
			return null;
		}
	}

	@Override
	public int getUpdateCount() throws SQLException {
		int s;

		checkclose();
		if(select0 != null) {
			s = count0;
			count0 = -1;
			return s;
		} else {
			return -1;
		}
	}

	@Override
	public boolean getMoreResults() throws SQLException {
		checkclose();
		return select0 != null;
	}

	@Override
	public void setFetchDirection(int direction) throws SQLException {
		// ignore it
	}

	@Override
	public int getFetchDirection() throws SQLException {
		// ignore it
		return ResultSet.FETCH_FORWARD;
	}

	@Override
	public void setFetchSize(int rows) throws SQLException {
		// ignore it
	}

	@Override
	public int getFetchSize() throws SQLException {
		// ignore it
		return 0;
	}

	@Override
	public int getResultSetConcurrency() throws SQLException {
		return ResultSet.CONCUR_READ_ONLY;
	}

	@Override
	public int getResultSetType() throws SQLException {
		return ResultSet.TYPE_FORWARD_ONLY;
	}

	@Override
	public void addBatch(String sql) throws SQLException {
		checkclose();
		synchronized(this) {
			batch.add(sql);
		}
	}

	@Override
	public void clearBatch() throws SQLException {
		checkclose();
		synchronized(this) {
			batch.clear();
		}
	}

	@Override
	public int[] executeBatch() throws SQLException {
		int[] r = new int[batch.size()];
		Object o;

		checkclose();
		for(int i = 0; i < batch.size(); i++) {
			try {
				o = conn.engine.execute(conn.schema, batch.get(i));
				if(o instanceof Relation) {
					throw new BatchUpdateException();
				} else if(o instanceof Number) {
					r[i] = ((Number)o).intValue();
				} else {
					r[i] = Statement.SUCCESS_NO_INFO;
				}
			} catch(SQLException e) {
				throw new BatchUpdateException(e);
			} catch (IOException e) {
				throw ErrorBundle.getDefault(10037);
			}
		}
		return r;
	}

	@Override
	public Connection getConnection() throws SQLException {
		checkclose();
		return conn;
	}

	@Override
	public boolean getMoreResults(int current) throws SQLException {
		checkclose();
		return getMoreResults();
	}

	@Override
	public ResultSet getGeneratedKeys() throws SQLException {
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public int executeUpdate(String sql,
			int autoGeneratedKeys) throws SQLException {
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public int executeUpdate(String sql,
			int[] columnIndexes) throws SQLException {
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public int executeUpdate(String sql,
			String[] columnNames) throws SQLException {
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public boolean execute(String sql,
			int autoGeneratedKeys) throws SQLException {
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public boolean execute(String sql,
			int[] columnIndexes) throws SQLException {
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public boolean execute(String sql,
			String[] columnNames) throws SQLException {
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public int getResultSetHoldability() throws SQLException {
		return ResultSet.HOLD_CURSORS_OVER_COMMIT;
	}

	@Override
	public boolean isClosed() throws SQLException {
		return conn == null;
	}

	@Override
	public void setPoolable(boolean poolable) throws SQLException {
		// ignore it
	}

	@Override
	public boolean isPoolable() throws SQLException {
		return false;
	}

}
