package net.morilib.db.jdbc;

import java.io.IOException;
import java.sql.BatchUpdateException;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.sql.SQLWarning;
import java.sql.Statement;
import java.util.Arrays;
import java.util.List;
import java.util.logging.Logger;

import net.morilib.db.misc.ErrorBundle;
import net.morilib.db.relations.Relation;

public class RelationsStatement implements Statement {

	private static Logger logger = RelationsJDBCUtils.logger;

	RelationsConnection conn;
	int maxFieldSize = 0, maxRows = 0;
	Relation select0 = null;
	int count0 = -1;
	List<String> batch;

	RelationsStatement(RelationsConnection c) {
		conn = c;
	}

	@Override
	public <T> T unwrap(Class<T> iface) throws SQLException {
		logger.finer("unwrap");
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public boolean isWrapperFor(Class<?> iface) throws SQLException {
		logger.finer("isWrapperFor");
		return false;
	}

	@Override
	public ResultSet executeQuery(String sql) throws SQLException {
		Object o;

		logger.finer("executeQuery");
		try {
			o = conn.engine.execute(conn.schema, sql);
			if(o instanceof Relation) {
				return new RelationsResultSet(this, (Relation)o);
			} else {
				throw ErrorBundle.getDefault(10038);
			}
		} catch (IOException e) {
			throw ErrorBundle.getDefault(10037);
		}
	}

	@Override
	public int executeUpdate(String sql) throws SQLException {
		Object o;

		logger.finer("executeUpdate");
		try {
			o = conn.engine.execute(conn.schema, sql);
			if(o instanceof Number) {
				return ((Number)o).intValue();
			} else {
				return 0;
			}
		} catch (IOException e) {
			throw ErrorBundle.getDefault(10037);
		}
	}

	void checkclose() throws SQLException {
		logger.finer("checkclose");
		if(conn == null) {
			throw ErrorBundle.getDefault(20002);
		}
	}

	@Override
	public void close() throws SQLException {
		logger.finer("close");
		checkclose();
		conn = null;
	}

	@Override
	public int getMaxFieldSize() throws SQLException {
		logger.finer("getMaxFieldSize");
		return maxFieldSize;
	}

	@Override
	public void setMaxFieldSize(int max) throws SQLException {
		logger.finer("setMaxFieldSize");
		if((maxFieldSize = max) < 0) {
			throw ErrorBundle.getDefault(20001);
		}
	}

	@Override
	public int getMaxRows() throws SQLException {
		logger.finer("getMaxRows");
		return maxRows;
	}

	@Override
	public void setMaxRows(int max) throws SQLException {
		logger.finer("setMaxRows");
		if((maxRows = max) < 0) {
			throw ErrorBundle.getDefault(20001);
		}
	}

	@Override
	public void setEscapeProcessing(
			boolean enable) throws SQLException {
		logger.finer("setEscapeProcessing");
		// ignore it
	}

	@Override
	public int getQueryTimeout() throws SQLException {
		logger.finer("getQueryTimeout");
		return 0;
//		return queryTimeout;
	}

	@Override
	public void setQueryTimeout(int seconds) throws SQLException {
		logger.finer("setQueryTimeout");
		// ignore it
//		if((queryTimeout = seconds) < 0) {
//			throw ErrorBundle.getDefault(20001);
//		}
	}

	@Override
	public void cancel() throws SQLException {
		logger.finer("cancel");
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public SQLWarning getWarnings() throws SQLException {
		logger.finer("getWarnings");
		checkclose();
		return null;
	}

	@Override
	public void clearWarnings() throws SQLException {
		logger.finer("clearWarnings");
		checkclose();
		// ignore it
	}

	@Override
	public void setCursorName(String name) throws SQLException {
		logger.finer("setCursorName");
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public boolean execute(String sql) throws SQLException {
		Object o;

		logger.finer("execute");
		checkclose();
		try {
			o = conn.engine.execute(conn.schema, sql);
			if(o instanceof Relation) {
				select0 = (Relation)o;
				return true;
			} else if(o instanceof Number) {
				count0 = ((Number)o).intValue();
				return false;
			} else {
				count0 = 0;
				return false;
			}
		} catch (IOException e) {
			throw ErrorBundle.getDefault(10037);
		}
	}

	@Override
	public ResultSet getResultSet() throws SQLException {
		Relation s;

		logger.finer("getResultSet");
		checkclose();
		if(select0 != null) {
			s = select0;
			select0 = null;
			return new RelationsResultSet(this, s);
		} else {
			return null;
		}
	}

	@Override
	public int getUpdateCount() throws SQLException {
		int s;

		logger.finer("getUpdateCount");
		checkclose();
		if(select0 != null) {
			s = count0;
			count0 = -1;
			return s;
		} else {
			return -1;
		}
	}

	@Override
	public boolean getMoreResults() throws SQLException {
		logger.finer("getMoreResults");
		checkclose();
		return select0 != null;
	}

	@Override
	public void setFetchDirection(int direction) throws SQLException {
		logger.finer("setFetchDirection");
		// ignore it
	}

	@Override
	public int getFetchDirection() throws SQLException {
		logger.finer("getFetchDirection");
		return ResultSet.FETCH_FORWARD;
	}

	@Override
	public void setFetchSize(int rows) throws SQLException {
		logger.finer("setFetchSize");
		// ignore it
	}

	@Override
	public int getFetchSize() throws SQLException {
		logger.finer("getFetchSize");
		return 0;
	}

	@Override
	public int getResultSetConcurrency() throws SQLException {
		logger.finer("getResultSetConcurrency");
		return ResultSet.CONCUR_READ_ONLY;
	}

	@Override
	public int getResultSetType() throws SQLException {
		logger.finer("getResultSetType");
		return ResultSet.TYPE_FORWARD_ONLY;
	}

	@Override
	public void addBatch(String sql) throws SQLException {
		logger.finer("addBatch");
		checkclose();
		synchronized(this) {
			batch.add(sql);
		}
	}

	@Override
	public void clearBatch() throws SQLException {
		logger.finer("clearBatch");
		checkclose();
		synchronized(this) {
			batch.clear();
		}
	}

	@Override
	public int[] executeBatch() throws SQLException {
		int[] r = new int[batch.size()];
		Object o;

		logger.finer("executeBatch");
		checkclose();
		for(int i = 0; i < batch.size(); i++) {
			try {
				o = conn.engine.execute(conn.schema, batch.get(i));
				if(o instanceof Relation) {
					throw new BatchUpdateException();
				} else if(o instanceof Number) {
					r[i] = ((Number)o).intValue();
				} else {
					r[i] = Statement.SUCCESS_NO_INFO;
				}
			} catch(SQLException e) {
				throw new BatchUpdateException(e);
			} catch (IOException e) {
				throw ErrorBundle.getDefault(10037);
			}
		}
		return r;
	}

	@Override
	public Connection getConnection() throws SQLException {
		logger.finer("getConnection");
		checkclose();
		return conn;
	}

	@Override
	public boolean getMoreResults(int current) throws SQLException {
		logger.finer("getMoreResults");
		checkclose();
		return getMoreResults();
	}

	@Override
	public ResultSet getGeneratedKeys() throws SQLException {
		logger.finer("getGeneratedKeys");
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public int executeUpdate(String sql,
			int autoGeneratedKeys) throws SQLException {
		logger.finer("executeUpdate");
		logger.finer("sql:" + sql);
		logger.finer("autoGeneratedKeys:" + autoGeneratedKeys);
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public int executeUpdate(String sql,
			int[] columnIndexes) throws SQLException {
		logger.finer("executeUpdate");
		logger.finer("sql:" + sql);
		logger.finer("columnIndexes:" + Arrays.toString(columnIndexes));
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public int executeUpdate(String sql,
			String[] columnNames) throws SQLException {
		logger.finer("executeUpdate");
		logger.finer("sql:" + sql);
		logger.finer("columnNames:" + Arrays.toString(columnNames));
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public boolean execute(String sql,
			int autoGeneratedKeys) throws SQLException {
		logger.finer("execute");
		logger.finer("sql:" + sql);
		logger.finer("autoGeneratedKeys:" + autoGeneratedKeys);
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public boolean execute(String sql,
			int[] columnIndexes) throws SQLException {
		logger.finer("execute");
		logger.finer("sql:" + sql);
		logger.finer("columnIndexes:" + Arrays.toString(columnIndexes));
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public boolean execute(String sql,
			String[] columnNames) throws SQLException {
		logger.finer("execute");
		logger.finer("sql:" + sql);
		logger.finer("columnNames:" + Arrays.toString(columnNames));
		throw new SQLFeatureNotSupportedException();
	}

	@Override
	public int getResultSetHoldability() throws SQLException {
		logger.finer("getResultSetHoldability");
		return ResultSet.HOLD_CURSORS_OVER_COMMIT;
	}

	@Override
	public boolean isClosed() throws SQLException {
		logger.finer("isClosed");
		return conn == null;
	}

	@Override
	public void setPoolable(boolean poolable) throws SQLException {
		logger.finer("setPoolable");
		// ignore it
	}

	@Override
	public boolean isPoolable() throws SQLException {
		logger.finer("isPoolable");
		return false;
	}

}
