/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.db.sql;

import java.io.IOException;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import net.morilib.db.misc.ErrorBundle;
import net.morilib.db.misc.Rational;
import net.morilib.db.sqlcs.dml.SqlBinaryOperation;
import net.morilib.db.sqlcs.dml.SqlBinaryOperator;
import net.morilib.db.sqlcs.dml.SqlExists;
import net.morilib.db.sqlcs.dml.SqlExpression;
import net.morilib.db.sqlcs.dml.SqlExpressions;
import net.morilib.db.sqlcs.dml.SqlFunction;
import net.morilib.db.sqlcs.dml.SqlIn;
import net.morilib.db.sqlcs.dml.SqlInSubquery;
import net.morilib.db.sqlcs.dml.SqlNumeric;
import net.morilib.db.sqlcs.dml.SqlPlaceHolder;
import net.morilib.db.sqlcs.dml.SqlSelect;
import net.morilib.db.sqlcs.dml.SqlSetExpression;
import net.morilib.db.sqlcs.dml.SqlString;
import net.morilib.db.sqlcs.dml.SqlSubqueryLiteral;
import net.morilib.db.sqlcs.dml.SqlSymbol;
import net.morilib.db.sqlcs.dml.SqlTernaryOperation;
import net.morilib.db.sqlcs.dml.SqlTernaryOperator;
import net.morilib.db.sqlcs.dml.SqlUnaryOperation;
import net.morilib.db.sqlcs.dml.SqlUnaryOperator;

public class DbSqlExprParser {

	//
	private List<Object> stack = new ArrayList<Object>();
	private DbSqlNonterminal gt = null;
	private int stat;

	//
	private Object pop() {
		stack.remove(stack.size() - 1);
		return stack.remove(stack.size() - 1);
	}

	//
	private void shift(int n) {
		stat = n;
		gt   = null;
		stack.add(null);  stack.add(n);
	}

	//
	private void shift(int n, Object o) {
		stat = n;
		gt   = null;
		stack.add(o);  stack.add(n);
	}

	//
	private void setGoto(DbSqlNonterminal nt, Object o) {
		stat = ((Integer)stack.get(stack.size() - 1)).intValue();
		gt   = nt;
		stack.add(o);
	}

	//
	private void goTo(int n) {
		stat = n;
		gt   = null;
		stack.add(n);
	}

	/**
	 * 
	 * @param lexer
	 */
	@SuppressWarnings("unchecked")
	public SqlExpression parse(DbSqlLexer lex,
			Map<String, SqlSetExpression> m
			) throws IOException, SQLException {
		/* S -> E
		 * E -> ( E )
		 *    | + E
		 *    | - E
		 *    | not E
		 *    | E + E
		 *    | E - E
		 *    | E * E
		 *    | E / E
		 *    | E || E
		 *    | E relop E
		 *    | E and E
		 *    | E or E
		 *    | E in ( L )
		 *    | E in ( <select-clause> )
		 *    | id ( L )
		 *    | id ( [*] )
		 *    | id ( )
		 *    | E between E and E
		 *    | exists ( <select-clause> )
		 *    | E like E
		 *    | E is-null
		 *    | V
		 * L -> L , E
		 *    | E
		 * V -> string
		 *    | number
		 *    | id
		 *    | *
		 *    | ( <select-clause> )
		 */
		List<SqlExpression> l;
		SqlExpression x, y, z;
		SqlBinaryOperator r;
		SqlSelect t;
		Rational n;
		String s;
		Object o;

		stack.add(stat = 100000);
		while(true) {
			switch(stat) {
			case 100000:
				/* S -> *E
				 * E -> *( E )
				 *    | *+ E
				 *    | *- E
				 *    | *not E
				 *    | *E + E
				 *    | *E - E
				 *    | *E [*] E
				 *    | *E [/] E
				 *    | *E || E
				 *    | *E relop E
				 *    | *E and E
				 *    | *E or E
				 *    | *E in ( L )
				 *    | *E like E
				 *    | *id ( L )
				 *    | *id ( [*] )
				 *    | *id ( )
				 *    | *E between E and E
				 *    | *exist ( <select-clause> )
				 *    | *E is-null
				 *    | *V
				 * V -> *string
				 *    | *number
				 *    | *id
				 *    | *[*]
				 *    | *( <select-clause> )
				 */
				if(gt == DbSqlNonterminal.EXPR) {
					goTo(100100);
				} else if(lex.eqchar('(')) {
					shift(101100);
				} else if(lex.eqchar('+')) {
					shift(102100);
				} else if(lex.eqchar('-')) {
					shift(103100);
				} else if(lex.eqsym("NOT")) {
					shift(104100);
				} else if(lex.eqsym("EXISTS")) {
					shift(119100);
				} else if((s = lex.getsym()) != null) {
					shift(115100, s);
				} else if((s = lex.getstr()) != null) {
					shift(300100, s);
				} else if((n = lex.getnum()) != null) {
					shift(300200, n);
				} else if(lex.eqchar('*')) {
					shift(300300);
				} else if(lex.eqchar('?')) {
					shift(300400);
				} else {
					throw ErrorBundle.getDefault(10019,
							lex.get().toString());
				}
				break;
			case 100100:
				/* S -> E*
				 * E -> E *+ E
				 *    | E *- E
				 *    | E *[*] E
				 *    | E *[/] E
				 *    | E *|| E
				 *    | E *relop E
				 *    | E *and E
				 *    | E *or E
				 *    | E *in ( L )
				 *    | E *between E and E
				 *    | E *like E
				 */
				if(lex.eqchar('+')) {
					shift(105200);
				} else if(lex.eqchar('-')) {
					shift(106200);
				} else if(lex.eqchar('*')) {
					shift(107200);
				} else if(lex.eqchar('/')) {
					shift(108200);
				} else if(lex.eq(DbSqlReserved.CONCAT)) {
					shift(109200);
				} else if((r = lex.getrelop()) != null) {
					shift(110200, r);
				} else if(lex.eqsym("AND")) {
					shift(111200);
				} else if(lex.eqsym("OR")) {
					shift(112200);
				} else if(lex.eqsym("IN")) {
					shift(114200);
				} else if(lex.eqsym("BETWEEN")) {
					shift(116100);
				} else if(lex.eqsym("IS")) {
					shift(118100);
				} else {
					return (SqlExpression)pop();
				}
				break;
			case 101100:
				/* E -> ( *E )
				 * V -> ( *<select-clause> )
				 * E -> *( E )
				 *    | *+ E
				 *    | *- E
				 *    | *not E
				 *    | *E + E
				 *    | *E - E
				 *    | *E [*] E
				 *    | *E [/] E
				 *    | *E || E
				 *    | *E relop E
				 *    | *E and E
				 *    | *E or E
				 *    | *E in ( L )
				 *    | *E like E
				 *    | *id ( L )
				 *    | *id ( )
				 *    | *E between E and E
				 *    | [*exist ( <select-clause> )]
				 *    | *is-null
				 *    | *V
				 * V -> *string
				 *    | *number
				 *    | *id
				 */
				if(gt == DbSqlNonterminal.EXPR) {
					goTo(101200);
				} else if(lex.eqchar('(')) {
					shift(101100);
				} else if(lex.eqchar('+')) {
					shift(102100);
				} else if(lex.eqchar('-')) {
					shift(103100);
				} else if(lex.eqsym("NOT")) {
					shift(104100);
				} else if(lex.eqsym("EXISTS")) {
					shift(119100);
				} else if(lex.eqsym("SELECT")) {
					o = new DbSqlParser()._select(lex, m);
					shift(101200,
							new SqlSubqueryLiteral((SqlSelect)o));
				} else if((s = lex.getsym()) != null) {
					shift(115100, s);
				} else if((s = lex.getstr()) != null) {
					shift(300100, s);
				} else if((n = lex.getnum()) != null) {
					shift(300200, n);
				} else if(lex.eqchar('*')) {
					shift(300300);
				} else if(lex.eqchar('?')) {
					shift(300400);
				} else {
					throw ErrorBundle.getDefault(10019,
							lex.get().toString());
				}
				break;
			case 101200:
				/* E -> ( E *)
				 * E -> E *+ E
				 *    | E *- E
				 *    | E *[*] E
				 *    | E *[/] E
				 *    | E *|| E
				 *    | E *relop E
				 *    | E *and E
				 *    | E *or E
				 *    | E *in ( L )
				 *    | E *between E and E
				 *    | E *like E
				 */
				if(lex.eqchar(')')) {
					shift(101300);
				} else if(lex.eqchar('+')) {
					shift(105200);
				} else if(lex.eqchar('-')) {
					shift(106200);
				} else if(lex.eqchar('*')) {
					shift(107200);
				} else if(lex.eqchar('/')) {
					shift(108200);
				} else if(lex.eq(DbSqlReserved.CONCAT)) {
					shift(109200);
				} else if((r = lex.getrelop()) != null) {
					shift(110200, r);
				} else if(lex.eqsym("AND")) {
					shift(111200);
				} else if(lex.eqsym("OR")) {
					shift(112200);
				} else if(lex.eqsym("IN")) {
					shift(114200);
				} else if(lex.eqsym("BETWEEN")) {
					shift(116100);
				} else if(lex.eqsym("IS")) {
					shift(118100);
				} else {
					throw ErrorBundle.getDefault(10019,
							lex.get().toString());
				}
				break;
			case 101300:
				/* E -> ( E )*
				 */
				pop();  o = pop();  pop();
				setGoto(DbSqlNonterminal.EXPR, o);
				break;
			case 102100:
				/* E -> + *E
				 * E -> *( E )
				 *    | *+ E
				 *    | *- E
				 *    | *not E
				 *    | *E + E
				 *    | *E - E
				 *    | *E [*] E
				 *    | *E [/] E
				 *    | *E || E
				 *    | *E relop E
				 *    | *E and E
				 *    | *E or E
				 *    | *E in ( L )
				 *    | *id ( L )
				 *    | *id ( )
				 *    | *E between E and E
				 *    | [*exist ( <select-clause> )]
				 *    | *E like E
				 *    | *is-null
				 *    | *V
				 * V -> *string
				 *    | *number
				 *    | *id
				 */
				if(gt == DbSqlNonterminal.EXPR) {
					goTo(102200);
				} else if(lex.eqchar('(')) {
					shift(101100);
	//			} else if(lex.eqchar('+')) {
	//				shift(102100);
	//			} else if(lex.eqchar('-')) {
	//				shift(103100);
	//			} else if(lex.eqsym("NOT")) {
	//				shift(104100);
				} else if((s = lex.getsym()) != null) {
					shift(115100, s);
				} else if((s = lex.getstr()) != null) {
					shift(300100, s);
				} else if((n = lex.getnum()) != null) {
					shift(300200, n);
				} else if(lex.eqchar('*')) {
					shift(300300);
				} else if(lex.eqchar('?')) {
					shift(300400);
				} else {
					throw ErrorBundle.getDefault(10019,
							lex.get().toString());
				}
				break;
			case 102200:
				/* E -> + E*
				 * E -> E *+ E
				 *    | E *- E
				 *    | E *[*] E
				 *    | E *[/] E
				 *    | E *|| E
				 *    | E *relop E
				 *    | E *and E
				 *    | E *or E
				 *    | E *in ( L )
				 *    | E *between E and E
				 *    | E *like E
				 */
				x = (SqlExpression)pop();  pop();
				x = new SqlUnaryOperation(SqlUnaryOperator.PLUS,
						x);
				setGoto(DbSqlNonterminal.EXPR, x);
				break;
			case 103100:
				/* E -> - *E
				 * E -> *( E )
				 *    | *+ E
				 *    | *- E
				 *    | *not E
				 *    | *E + E
				 *    | *E - E
				 *    | *E [*] E
				 *    | *E [/] E
				 *    | *E || E
				 *    | *E relop E
				 *    | *E and E
				 *    | *E or E
				 *    | *E in ( L )
				 *    | *id ( L )
				 *    | *id ( )
				 *    | *E between E and E
				 *    | [*exist ( <select-clause> )]
				 *    | *E like E
				 *    | *is-null
				 *    | *V
				 * V -> *string
				 *    | *number
				 *    | *id
				 */
				if(gt == DbSqlNonterminal.EXPR) {
					goTo(103200);
				} else if(lex.eqchar('(')) {
					shift(101100);
	//			} else if(lex.eqchar('+')) {
	//				shift(102100);
	//			} else if(lex.eqchar('-')) {
	//				shift(103100);
	//			} else if(lex.eqsym("NOT")) {
	//				shift(104100);
				} else if((s = lex.getsym()) != null) {
					shift(115100, s);
				} else if((s = lex.getstr()) != null) {
					shift(300100, s);
				} else if((n = lex.getnum()) != null) {
					shift(300200, n);
				} else if(lex.eqchar('*')) {
					shift(300300);
				} else if(lex.eqchar('?')) {
					shift(300400);
				} else {
					throw ErrorBundle.getDefault(10019,
							lex.get().toString());
				}
				break;
			case 103200:
				/* E -> - E*
				 * E -> E *+ E
				 *    | E *- E
				 *    | E *[*] E
				 *    | E *[/] E
				 *    | E *|| E
				 *    | E *relop E
				 *    | E *and E
				 *    | E *or E
				 *    | E *in ( L )
				 *    | E *between E and E
				 *    | E *like E
				 */
				x = (SqlExpression)pop();  pop();
				x = new SqlUnaryOperation(SqlUnaryOperator.MINUS,
						x);
				setGoto(DbSqlNonterminal.EXPR, x);
				break;
			case 104100:
				/* E -> not *E
				 * E -> *( E )
				 *    | *+ E
				 *    | *- E
				 *    | *not E
				 *    | *E + E
				 *    | *E - E
				 *    | *E [*] E
				 *    | *E [/] E
				 *    | *E || E
				 *    | *E relop E
				 *    | *E and E
				 *    | *E or E
				 *    | *E in ( L )
				 *    | *id ( L )
				 *    | *id ( )
				 *    | *E between E and E
				 *    | [*exist ( <select-clause> )]
				 *    | *E like E
				 *    | *is-null
				 *    | *V
				 * V -> *string
				 *    | *number
				 *    | *id
				 */
				if(gt == DbSqlNonterminal.EXPR) {
					goTo(104200);
				} else if(lex.eqchar('(')) {
					shift(101100);
				} else if(lex.eqchar('+')) {
					shift(102100);
				} else if(lex.eqchar('-')) {
					shift(103100);
	//			} else if(lex.eqsym("NOT")) {
	//				shift(104100);
				} else if(lex.eqsym("EXISTS")) {
					shift(119100);
				} else if((s = lex.getsym()) != null) {
					shift(115100, s);
				} else if((s = lex.getstr()) != null) {
					shift(300100, s);
				} else if((n = lex.getnum()) != null) {
					shift(300200, n);
				} else if(lex.eqchar('*')) {
					shift(300300);
				} else if(lex.eqchar('?')) {
					shift(300400);
				} else {
					throw ErrorBundle.getDefault(10019,
							lex.get().toString());
				}
				break;
			case 104200:
				/* E -> not E*
				 * E -> E *+ E
				 *    | E *- E
				 *    | E *[*] E
				 *    | E *[/] E
				 *    | E *|| E
				 *    | E *relop E
				 *    | E *and E
				 *    | E *or E
				 *    | E *in ( L )
				 *    | E *between E and E
				 *    | E *like E
				 */
				if(lex.eqchar('+')) {
					shift(105200);
				} else if(lex.eqchar('-')) {
					shift(106200);
				} else if(lex.eqchar('*')) {
					shift(107200);
				} else if(lex.eqchar('/')) {
					shift(108200);
				} else if(lex.eq(DbSqlReserved.CONCAT)) {
					shift(109200);
				} else if((r = lex.getrelop()) != null) {
					shift(110200, r);
				} else if(lex.eqsym("IN")) {
					shift(114200);
				} else if(lex.eqsym("BETWEEN")) {
					shift(116100);
				} else if(lex.eqsym("IS")) {
					shift(118100);
				} else {
					x = (SqlExpression)pop();  pop();
					x = new SqlUnaryOperation(SqlUnaryOperator.NOT,
							x);
					setGoto(DbSqlNonterminal.EXPR, x);
				}
				break;
			case 105200:
				/* E -> E + *E
				 * E -> *( E )
				 *    | *+ E
				 *    | *- E
				 *    | *not E
				 *    | *E + E
				 *    | *E - E
				 *    | *E [*] E
				 *    | *E [/] E
				 *    | *E || E
				 *    | *E relop E
				 *    | *E and E
				 *    | *E or E
				 *    | *E in ( L )
				 *    | *id ( L )
				 *    | *id ( )
				 *    | *E between E and E
				 *    | [*exist ( <select-clause> )]
				 *    | *E like E
				 *    | *is-null
				 *    | *V
				 * V -> *string
				 *    | *number
				 *    | *id
				 */
				if(gt == DbSqlNonterminal.EXPR) {
					goTo(105300);
				} else if(lex.eqchar('(')) {
					shift(101100);
				} else if(lex.eqchar('+')) {
					shift(102100);
				} else if(lex.eqchar('-')) {
					shift(103100);
				} else if(lex.eqsym("NOT")) {
					shift(104100);
				} else if(lex.eqsym("EXISTS")) {
					shift(119100);
				} else if((s = lex.getsym()) != null) {
					shift(115100, s);
				} else if((s = lex.getstr()) != null) {
					shift(300100, s);
				} else if((n = lex.getnum()) != null) {
					shift(300200, n);
				} else if(lex.eqchar('*')) {
					shift(300300);
				} else if(lex.eqchar('?')) {
					shift(300400);
				} else {
					throw ErrorBundle.getDefault(10019,
							lex.get().toString());
				}
				break;
			case 105300:
				/* E -> E + E*
				 * E -> E *+ E
				 *    | E *- E
				 *    | E *[*] E
				 *    | E *[/] E
				 *    | E *|| E
				 *    | E *relop E
				 *    | E *and E
				 *    | E *or E
				 *    | E *in ( L )
				 *    | E *between E and E
				 *    | E *like E
				 */
				if(lex.eqchar('*')) {
					shift(107200);
				} else if(lex.eqchar('/')) {
					shift(108200);
				} else {
					y = (SqlExpression)pop();  pop();
					x = (SqlExpression)pop();
					x = new SqlBinaryOperation(SqlBinaryOperator.ADD,
							x, y);
					setGoto(DbSqlNonterminal.EXPR, x);
				}
				break;
			case 106200:
				/* E -> E - *E
				 * E -> *( E )
				 *    | *+ E
				 *    | *- E
				 *    | *not E
				 *    | *E + E
				 *    | *E - E
				 *    | *E [*] E
				 *    | *E [/] E
				 *    | *E || E
				 *    | *E relop E
				 *    | *E and E
				 *    | *E or E
				 *    | *E in ( L )
				 *    | *id ( L )
				 *    | *id ( )
				 *    | *E between E and E
				 *    | [*exist ( <select-clause> )]
				 *    | *E like E
				 *    | *is-null
				 *    | *V
				 * V -> *string
				 *    | *number
				 *    | *id
				 */
				if(gt == DbSqlNonterminal.EXPR) {
					goTo(106300);
				} else if(lex.eqchar('(')) {
					shift(101100);
				} else if(lex.eqchar('+')) {
					shift(102100);
				} else if(lex.eqchar('-')) {
					shift(103100);
				} else if(lex.eqsym("NOT")) {
					shift(104100);
				} else if(lex.eqsym("EXISTS")) {
					shift(119100);
				} else if((s = lex.getsym()) != null) {
					shift(115100, s);
				} else if((s = lex.getstr()) != null) {
					shift(300100, s);
				} else if((n = lex.getnum()) != null) {
					shift(300200, n);
				} else if(lex.eqchar('*')) {
					shift(300300);
				} else if(lex.eqchar('?')) {
					shift(300400);
				} else {
					throw ErrorBundle.getDefault(10019,
							lex.get().toString());
				}
				break;
			case 106300:
				/* E -> E - E*
				 * E -> E *+ E
				 *    | E *- E
				 *    | E *[*] E
				 *    | E *[/] E
				 *    | E *|| E
				 *    | E *relop E
				 *    | E *and E
				 *    | E *or E
				 *    | E *in ( L )
				 *    | E *between E and E
				 *    | E *like E
				 */
				if(lex.eqchar('*')) {
					shift(107200);
				} else if(lex.eqchar('/')) {
					shift(108200);
				} else {
					y = (SqlExpression)pop();  pop();
					x = (SqlExpression)pop();
					x = new SqlBinaryOperation(SqlBinaryOperator.SUB,
							x, y);
					setGoto(DbSqlNonterminal.EXPR, x);
				}
				break;
			case 107200:
				/* E -> E [*] *E
				 * E -> *( E )
				 *    | *+ E
				 *    | *- E
				 *    | *not E
				 *    | *E + E
				 *    | *E - E
				 *    | *E [*] E
				 *    | *E [/] E
				 *    | *E || E
				 *    | *E relop E
				 *    | *E and E
				 *    | *E or E
				 *    | *E in ( L )
				 *    | *id ( L )
				 *    | *id ( )
				 *    | *E between E and E
				 *    | [*exist ( <select-clause> )]
				 *    | *E like E
				 *    | *is-null
				 *    | *V
				 * V -> *string
				 *    | *number
				 *    | *id
				 */
				if(gt == DbSqlNonterminal.EXPR) {
					goTo(107300);
				} else if(lex.eqchar('(')) {
					shift(101100);
				} else if(lex.eqchar('+')) {
					shift(102100);
				} else if(lex.eqchar('-')) {
					shift(103100);
				} else if(lex.eqsym("NOT")) {
					shift(104100);
				} else if(lex.eqsym("EXISTS")) {
					shift(119100);
				} else if((s = lex.getsym()) != null) {
					shift(115100, s);
				} else if((s = lex.getstr()) != null) {
					shift(300100, s);
				} else if((n = lex.getnum()) != null) {
					shift(300200, n);
				} else if(lex.eqchar('*')) {
					shift(300300);
				} else if(lex.eqchar('?')) {
					shift(300400);
				} else {
					throw ErrorBundle.getDefault(10019,
							lex.get().toString());
				}
				break;
			case 107300:
				/* E -> E [*] E*
				 * E -> E *+ E
				 *    | E *- E
				 *    | E *[*] E
				 *    | E *[/] E
				 *    | E *|| E
				 *    | E *relop E
				 *    | E *and E
				 *    | E *or E
				 *    | E *in ( L )
				 *    | E *between E and E
				 *    | E *like E
				 */
				y = (SqlExpression)pop();  pop();
				x = (SqlExpression)pop();
				x = new SqlBinaryOperation(SqlBinaryOperator.MUL,
						x, y);
				setGoto(DbSqlNonterminal.EXPR, x);
				break;
			case 108200:
				/* E -> E [/] *E
				 * E -> *( E )
				 *    | *+ E
				 *    | *- E
				 *    | *not E
				 *    | *E + E
				 *    | *E - E
				 *    | *E [*] E
				 *    | *E [/] E
				 *    | *E || E
				 *    | *E relop E
				 *    | *E and E
				 *    | *E or E
				 *    | *E in ( L )
				 *    | *id ( L )
				 *    | *id ( )
				 *    | *E between E and E
				 *    | [*exist ( <select-clause> )]
				 *    | *E like E
				 *    | *is-null
				 *    | *V
				 * V -> *string
				 *    | *number
				 *    | *id
				 */
				if(gt == DbSqlNonterminal.EXPR) {
					goTo(108300);
				} else if(lex.eqchar('(')) {
					shift(101100);
				} else if(lex.eqchar('+')) {
					shift(102100);
				} else if(lex.eqchar('-')) {
					shift(103100);
				} else if(lex.eqsym("NOT")) {
					shift(104100);
				} else if(lex.eqsym("EXISTS")) {
					shift(119100);
				} else if((s = lex.getsym()) != null) {
					shift(115100, s);
				} else if((s = lex.getstr()) != null) {
					shift(300100, s);
				} else if((n = lex.getnum()) != null) {
					shift(300200, n);
				} else if(lex.eqchar('*')) {
					shift(300300);
				} else if(lex.eqchar('?')) {
					shift(300400);
				} else {
					throw ErrorBundle.getDefault(10019,
							lex.get().toString());
				}
				break;
			case 108300:
				/* E -> E [/] E*
				 * E -> E *+ E
				 *    | E *- E
				 *    | E *[*] E
				 *    | E *[/] E
				 *    | E *|| E
				 *    | E *relop E
				 *    | E *and E
				 *    | E *or E
				 *    | E *in ( L )
				 *    | E *between E and E
				 *    | E *like E
				 */
				y = (SqlExpression)pop();  pop();
				x = (SqlExpression)pop();
				x = new SqlBinaryOperation(SqlBinaryOperator.DIV,
						x, y);
				setGoto(DbSqlNonterminal.EXPR, x);
				break;
			case 109200:
				/* E -> E || *E
				 * E -> *( E )
				 *    | *+ E
				 *    | *- E
				 *    | *not E
				 *    | *E + E
				 *    | *E - E
				 *    | *E [*] E
				 *    | *E [/] E
				 *    | *E || E
				 *    | *E relop E
				 *    | *E and E
				 *    | *E or E
				 *    | *E in ( L )
				 *    | *id ( L )
				 *    | *id ( )
				 *    | *E between E and E
				 *    | [*exist ( <select-clause> )]
				 *    | *E like E
				 *    | *is-null
				 *    | *V
				 * V -> *string
				 *    | *number
				 *    | *id
				 */
				if(gt == DbSqlNonterminal.EXPR) {
					goTo(109300);
				} else if(lex.eqchar('(')) {
					shift(101100);
				} else if(lex.eqchar('+')) {
					shift(102100);
				} else if(lex.eqchar('-')) {
					shift(103100);
				} else if(lex.eqsym("NOT")) {
					shift(104100);
				} else if(lex.eqsym("EXISTS")) {
					shift(119100);
				} else if((s = lex.getsym()) != null) {
					shift(115100, s);
				} else if((s = lex.getstr()) != null) {
					shift(300100, s);
				} else if((n = lex.getnum()) != null) {
					shift(300200, n);
				} else if(lex.eqchar('*')) {
					shift(300300);
				} else if(lex.eqchar('?')) {
					shift(300400);
				} else {
					throw ErrorBundle.getDefault(10019,
							lex.get().toString());
				}
				break;
			case 109300:
				/* E -> E || E*
				 * E -> E *+ E
				 *    | E *- E
				 *    | E *[*] E
				 *    | E *[/] E
				 *    | E *|| E
				 *    | E *relop E
				 *    | E *and E
				 *    | E *or E
				 *    | E *in ( L )
				 *    | E *between E and E
				 *    | E *like E
				 */
				if(lex.eqchar('+')) {
					shift(105200);
				} else if(lex.eqchar('-')) {
					shift(106200);
				} else if(lex.eqchar('*')) {
					shift(107200);
				} else if(lex.eqchar('/')) {
					shift(108200);
				} else {
					y = (SqlExpression)pop();  pop();
					x = (SqlExpression)pop();
					x = new SqlBinaryOperation(
							SqlBinaryOperator.CONCAT, x, y);
					setGoto(DbSqlNonterminal.EXPR, x);
				}
				break;
			case 110200:
				/* E -> E relop *E
				 * E -> *( E )
				 *    | *+ E
				 *    | *- E
				 *    | *not E
				 *    | *E + E
				 *    | *E - E
				 *    | *E [*] E
				 *    | *E [/] E
				 *    | *E || E
				 *    | *E relop E
				 *    | *E and E
				 *    | *E or E
				 *    | *E in ( L )
				 *    | *id ( L )
				 *    | *id ( )
				 *    | *E between E and E
				 *    | [*exist ( <select-clause> )]
				 *    | *E like E
				 *    | *is-null
				 *    | *V
				 * V -> *string
				 *    | *number
				 *    | *id
				 */
				if(gt == DbSqlNonterminal.EXPR) {
					goTo(110300);
				} else if(lex.eqchar('(')) {
					shift(101100);
				} else if(lex.eqchar('+')) {
					shift(102100);
				} else if(lex.eqchar('-')) {
					shift(103100);
				} else if(lex.eqsym("NOT")) {
					shift(104100);
				} else if(lex.eqsym("EXISTS")) {
					shift(119100);
				} else if((s = lex.getsym()) != null) {
					shift(115100, s);
				} else if((s = lex.getstr()) != null) {
					shift(300100, s);
				} else if((n = lex.getnum()) != null) {
					shift(300200, n);
				} else if(lex.eqchar('*')) {
					shift(300300);
				} else if(lex.eqchar('?')) {
					shift(300400);
				} else {
					throw ErrorBundle.getDefault(10019,
							lex.get().toString());
				}
				break;
			case 110300:
				/* E -> E relop E*
				 * E -> E *+ E
				 *    | E *- E
				 *    | E *[*] E
				 *    | E *[/] E
				 *    | E *|| E
				 *    | E *relop E
				 *    | E *and E
				 *    | E *or E
				 *    | E *in ( L )
				 *    | E *between E and E
				 *    | E *like E
				 */
				if(lex.eqchar('+')) {
					shift(105200);
				} else if(lex.eqchar('-')) {
					shift(106200);
				} else if(lex.eqchar('*')) {
					shift(107200);
				} else if(lex.eqchar('/')) {
					shift(108200);
				} else if(lex.eq(DbSqlReserved.CONCAT)) {
					shift(109200);
				} else {
					y = (SqlExpression)pop();
					r = (SqlBinaryOperator)pop();
					x = (SqlExpression)pop();
					x = new SqlBinaryOperation(r, x, y);
					setGoto(DbSqlNonterminal.EXPR, x);
				}
				break;
			case 111200:
				/* E -> E and *E
				 * E -> *( E )
				 *    | *+ E
				 *    | *- E
				 *    | *not E
				 *    | *E + E
				 *    | *E - E
				 *    | *E [*] E
				 *    | *E [/] E
				 *    | *E || E
				 *    | *E relop E
				 *    | *E and E
				 *    | *E or E
				 *    | *E in ( L )
				 *    | *id ( L )
				 *    | *id ( )
				 *    | *E between E and E
				 *    | [*exist ( <select-clause> )]
				 *    | *E like E
				 *    | *is-null
				 *    | *V
				 * V -> *string
				 *    | *number
				 *    | *id
				 */
				if(gt == DbSqlNonterminal.EXPR) {
					goTo(111300);
				} else if(lex.eqchar('(')) {
					shift(101100);
				} else if(lex.eqchar('+')) {
					shift(102100);
				} else if(lex.eqchar('-')) {
					shift(103100);
				} else if(lex.eqsym("NOT")) {
					shift(104100);
				} else if(lex.eqsym("EXISTS")) {
					shift(119100);
				} else if((s = lex.getsym()) != null) {
					shift(115100, s);
				} else if((s = lex.getstr()) != null) {
					shift(300100, s);
				} else if((n = lex.getnum()) != null) {
					shift(300200, n);
				} else if(lex.eqchar('*')) {
					shift(300300);
				} else if(lex.eqchar('?')) {
					shift(300400);
				} else {
					throw ErrorBundle.getDefault(10019,
							lex.get().toString());
				}
				break;
			case 111300:
				/* E -> E and E*
				 * E -> E *+ E
				 *    | E *- E
				 *    | E *[*] E
				 *    | E *[/] E
				 *    | E *|| E
				 *    | E *relop E
				 *    | E *and E
				 *    | E *or E
				 *    | E *in ( L )
				 *    | E *between E and E
				 *    | E *like E
				 */
				if(lex.eqchar('+')) {
					shift(105200);
				} else if(lex.eqchar('-')) {
					shift(106200);
				} else if(lex.eqchar('*')) {
					shift(107200);
				} else if(lex.eqchar('/')) {
					shift(108200);
				} else if(lex.eq(DbSqlReserved.CONCAT)) {
					shift(109200);
				} else if((r = lex.getrelop()) != null) {
					shift(110200, r);
				} else if(lex.eqsym("IN")) {
					shift(114200);
				} else if(lex.eqsym("BETWEEN")) {
					shift(116100);
				} else if(lex.eqsym("IS")) {
					shift(118100);
				} else {
					y = (SqlExpression)pop();  pop();
					x = (SqlExpression)pop();
					x = new SqlBinaryOperation(SqlBinaryOperator.AND,
							x, y);
					setGoto(DbSqlNonterminal.EXPR, x);
				}
				break;
			case 112200:
				/* E -> E or *E
				 * E -> *( E )
				 *    | *+ E
				 *    | *- E
				 *    | *not E
				 *    | *E + E
				 *    | *E - E
				 *    | *E [*] E
				 *    | *E [/] E
				 *    | *E || E
				 *    | *E relop E
				 *    | *E and E
				 *    | *E or E
				 *    | *E in ( L )
				 *    | *id ( L )
				 *    | *id ( )
				 *    | *E between E and E
				 *    | [*exist ( <select-clause> )]
				 *    | *E like E
				 *    | *is-null
				 *    | *V
				 * V -> *string
				 *    | *number
				 *    | *id
				 */
				if(gt == DbSqlNonterminal.EXPR) {
					goTo(112300);
				} else if(lex.eqchar('(')) {
					shift(101100);
				} else if(lex.eqchar('+')) {
					shift(102100);
				} else if(lex.eqchar('-')) {
					shift(103100);
				} else if(lex.eqsym("NOT")) {
					shift(104100);
				} else if(lex.eqsym("EXISTS")) {
					shift(119100);
				} else if((s = lex.getsym()) != null) {
					shift(115100, s);
				} else if((s = lex.getstr()) != null) {
					shift(300100, s);
				} else if((n = lex.getnum()) != null) {
					shift(300200, n);
				} else if(lex.eqchar('*')) {
					shift(300300);
				} else if(lex.eqchar('?')) {
					shift(300400);
				} else {
					throw ErrorBundle.getDefault(10019,
							lex.get().toString());
				}
				break;
			case 112300:
				/* E -> E or E*
				 * E -> E *+ E
				 *    | E *- E
				 *    | E *[*] E
				 *    | E *[/] E
				 *    | E *|| E
				 *    | E *relop E
				 *    | E *and E
				 *    | E *or E
				 *    | E *in ( L )
				 *    | E *between E and E
				 *    | E *like E
				 */
				if(lex.eqchar('+')) {
					shift(105200);
				} else if(lex.eqchar('-')) {
					shift(106200);
				} else if(lex.eqchar('*')) {
					shift(107200);
				} else if(lex.eqchar('/')) {
					shift(108200);
				} else if(lex.eq(DbSqlReserved.CONCAT)) {
					shift(109200);
				} else if((r = lex.getrelop()) != null) {
					shift(110200, r);
				} else if(lex.eqsym("AND")) {
					shift(111200);
				} else if(lex.eqsym("IN")) {
					shift(114200);
				} else if(lex.eqsym("BETWEEN")) {
					shift(116100);
				} else if(lex.eqsym("IS")) {
					shift(118100);
				} else {
					y = (SqlExpression)pop();  pop();
					x = (SqlExpression)pop();
					x = new SqlBinaryOperation(SqlBinaryOperator.OR,
							x, y);
					setGoto(DbSqlNonterminal.EXPR, x);
				}
				break;
			case 114200:
				/* E -> E in *( L )
				 */
				if(lex.eqchar('(')) {
					shift(114300);
				} else {
					throw ErrorBundle.getDefault(10019,
							lex.get().toString());
				}
				break;
			case 114300:
				/* E -> E in ( *L )
				 * L -> *L , E
				 *    | *E
				 * E -> *( E )
				 *    | *+ E
				 *    | *- E
				 *    | *not E
				 *    | *E + E
				 *    | *E - E
				 *    | *E [*] E
				 *    | *E [/] E
				 *    | *E || E
				 *    | *E relop E
				 *    | *E and E
				 *    | *E or E
				 *    | *E in ( L )
				 *    | *id ( L )
				 *    | *id ( )
				 *    | *E between E and E
				 *    | [*exist ( <select-clause> )]
				 *    | *E like E
				 *    | *is-null
				 *    | *V
				 * V -> *string
				 *    | *number
				 *    | *id
				 */
				if(gt == DbSqlNonterminal.EXPR_LIST) {
					goTo(114400);
				} else if(gt == DbSqlNonterminal.EXPR) {
					goTo(201100);
				} else if(lex.eqchar('(')) {
					shift(101100);
				} else if(lex.eqchar('+')) {
					shift(102100);
				} else if(lex.eqchar('-')) {
					shift(103100);
				} else if(lex.eqsym("NOT")) {
					shift(104100);
				} else if(lex.eqsym("EXISTS")) {
					shift(119100);
				} else if(lex.eqsym("SELECT")) {
					o = new DbSqlParser()._select(lex, m);
					shift(114410, (SqlSelect)o);
				} else if((s = lex.getsym()) != null) {
					shift(115100, s);
				} else if((s = lex.getstr()) != null) {
					shift(300100, s);
				} else if((n = lex.getnum()) != null) {
					shift(300200, n);
				} else if(lex.eqchar('*')) {
					shift(300300);
				} else if(lex.eqchar('?')) {
					shift(300400);
				} else {
					throw ErrorBundle.getDefault(10019,
							lex.get().toString());
				}
				break;
			case 114400:
				/* E -> E in ( L *)
				 * L -> L *, E
				 */
				if(lex.eqchar(')')) {
					shift(114500);
				} else if(lex.eqchar(',')) {
					shift(200200);
				} else {
					throw ErrorBundle.getDefault(10019,
							lex.get().toString());
				}
				break;
			case 114500:
				/* E -> E in ( L )*
				 */
				pop();  l = (List<SqlExpression>)pop();
				pop();  pop();  x = (SqlExpression)pop();
				x = new SqlIn(x, l);
				setGoto(DbSqlNonterminal.EXPR, x);
				break;
			case 114410:
				/* E -> E in ( <select-clause> *)
				 */
				if(lex.eqchar(')')) {
					shift(114510);
				} else {
					throw ErrorBundle.getDefault(10019,
							lex.get().toString());
				}
				break;
			case 114510:
				/* E -> E in ( <select-clause> )*
				 */
				pop();  t = (SqlSelect)pop();
				pop();  pop();  x = (SqlExpression)pop();
				x = new SqlInSubquery(x, t);
				setGoto(DbSqlNonterminal.EXPR, x);
				break;
			case 115100:
				/* E -> id *( L )
				 *    | id *( [*] )
				 *    | id *( )
				 * V -> id*
				 */
				if(lex.eqchar('(')) {
					shift(115200);
				} else {
					setGoto(DbSqlNonterminal.EXPR,
							new SqlSymbol(pop().toString()));
				}
				break;
			case 115200:
				/* E -> id ( *L )
				 *    | id ( *[*] )
				 *    | id ( *)
				 * L -> *L , E
				 *    | *E
				 * E -> *( E )
				 *    | *+ E
				 *    | *- E
				 *    | *not E
				 *    | *E + E
				 *    | *E - E
				 *    | *E [*] E
				 *    | *E [/] E
				 *    | *E || E
				 *    | *E relop E
				 *    | *E and E
				 *    | *E or E
				 *    | *E in ( L )
				 *    | *id ( L )
				 *    | *id ( )
				 *    | *E between E and E
				 *    | [*exist ( <select-clause> )]
				 *    | *E like E
				 *    | *is-null
				 *    | *V
				 * V -> *string
				 *    | *number
				 *    | *id
				 */
				if(gt == DbSqlNonterminal.EXPR_LIST) {
					goTo(115300);
				} else if(gt == DbSqlNonterminal.EXPR) {
					goTo(201100);
				} else if(lex.eqchar(')')) {
					shift(115310);
				} else if(lex.eqchar('(')) {
					shift(101100);
				} else if(lex.eqchar('+')) {
					shift(102100);
				} else if(lex.eqchar('-')) {
					shift(103100);
				} else if(lex.eqsym("NOT")) {
					shift(104100);
				} else if(lex.eqsym("EXISTS")) {
					shift(119100);
				} else if(lex.eqchar('*')) {
					shift(115320);
				} else if((s = lex.getsym()) != null) {
					shift(115100, s);
				} else if((s = lex.getstr()) != null) {
					shift(300100, s);
				} else if((n = lex.getnum()) != null) {
					shift(300200, n);
				} else if(lex.eqchar('*')) {
					shift(300300);
				} else if(lex.eqchar('?')) {
					shift(300400);
				} else {
					throw ErrorBundle.getDefault(10019,
							lex.get().toString());
				}
				break;
			case 115300:
				/* E -> id ( L *)
				 * L -> L *, E
				 */
				if(lex.eqchar(')')) {
					shift(115400);
				} else if(lex.eqchar(',')) {
					shift(200200);
				} else {
					throw ErrorBundle.getDefault(10019,
							lex.get().toString());
				}
				break;
			case 115310:
				/* E -> id ( )*
				 */
				pop();  pop();  s = pop().toString();
				setGoto(DbSqlNonterminal.EXPR, new SqlFunction(s));
				break;
			case 115320:
				/* E -> id ( [*] *)
				 */
				if(lex.eqchar(')')) {
					shift(115420);
				} else {
					throw ErrorBundle.getDefault(10019,
							lex.get().toString());
				}
				break;
			case 115400:
				/* E -> id ( L )*
				 */
				pop();  l = (List<SqlExpression>)pop();
				pop();  s = pop().toString();
				setGoto(DbSqlNonterminal.EXPR, new SqlFunction(s, l));
				break;
			case 115420:
				/* E -> id ( [*] )*
				 */
				pop();  pop();  pop();  s = pop().toString();
				l = Collections.singletonList(SqlExpressions.ANY);
				setGoto(DbSqlNonterminal.EXPR, new SqlFunction(s, l));
				break;
			case 116100:
				/* E -> E between *E and E
				 * E -> *( E )
				 *    | *+ E
				 *    | *- E
				 *    | *not E
				 *    | *E + E
				 *    | *E - E
				 *    | *E [*] E
				 *    | *E [/] E
				 *    | *E || E
				 *    | *E relop E
				 *    | *E and E
				 *    | *E or E
				 *    | *E in ( L )
				 *    | *id ( L )
				 *    | *id ( )
				 *    | *E between E and E
				 *    | [*exist ( <select-clause> )]
				 *    | *E like E
				 *    | *is-null
				 *    | *V
				 * V -> *string
				 *    | *number
				 *    | *id
				 */
				if(gt == DbSqlNonterminal.EXPR) {
					goTo(116200);
				} else if(lex.eqchar('(')) {
					shift(101100);
				} else if(lex.eqchar('+')) {
					shift(102100);
				} else if(lex.eqchar('-')) {
					shift(103100);
				} else if(lex.eqsym("NOT")) {
					shift(104100);
				} else if(lex.eqsym("EXISTS")) {
					shift(119100);
				} else if((s = lex.getsym()) != null) {
					shift(115100, s);
				} else if((s = lex.getstr()) != null) {
					shift(300100, s);
				} else if((n = lex.getnum()) != null) {
					shift(300200, n);
				} else if(lex.eqchar('*')) {
					shift(300300);
				} else if(lex.eqchar('?')) {
					shift(300400);
				} else {
					throw ErrorBundle.getDefault(10019,
							lex.get().toString());
				}
				break;
			case 116200:
				/* E -> E between E *and E
				 * E -> E *+ E
				 *    | E *- E
				 *    | E *[*] E
				 *    | E *[/] E
				 *    | E *|| E
				 *    | E *relop E
				 *    | E *or E
				 *    | E *in ( L )
				 *    | E *between E and E
				 *    | E *like E
				 */
				if(lex.eqchar('+')) {
					shift(105200);
				} else if(lex.eqchar('-')) {
					shift(106200);
				} else if(lex.eqchar('*')) {
					shift(107200);
				} else if(lex.eqchar('/')) {
					shift(108200);
				} else if(lex.eq(DbSqlReserved.CONCAT)) {
					shift(109200);
				} else if((r = lex.getrelop()) != null) {
					shift(110200, r);
				} else if(lex.eqsym("AND")) {
					shift(116300);
				} else if(lex.eqsym("IS")) {
					shift(118100);
				} else {
					throw ErrorBundle.getDefault(10019,
							lex.get().toString());
				}
				break;
			case 116300:
				/* E -> E between E and *E
				 * E -> *( E )
				 *    | *+ E
				 *    | *- E
				 *    | *not E
				 *    | *E + E
				 *    | *E - E
				 *    | *E [*] E
				 *    | *E [/] E
				 *    | *E || E
				 *    | *E relop E
				 *    | *E and E
				 *    | *E or E
				 *    | *E in ( L )
				 *    | *id ( L )
				 *    | *id ( )
				 *    | *E between E and E
				 *    | [*exist ( <select-clause> )]
				 *    | *E like E
				 *    | *is-null
				 *    | *V
				 * V -> *string
				 *    | *number
				 *    | *id
				 */
				if(gt == DbSqlNonterminal.EXPR) {
					goTo(116400);
				} else if(lex.eqchar('(')) {
					shift(101100);
				} else if(lex.eqchar('+')) {
					shift(102100);
				} else if(lex.eqchar('-')) {
					shift(103100);
				} else if(lex.eqsym("NOT")) {
					shift(104100);
				} else if(lex.eqsym("EXISTS")) {
					shift(119100);
				} else if((s = lex.getsym()) != null) {
					shift(115100, s);
				} else if((s = lex.getstr()) != null) {
					shift(300100, s);
				} else if((n = lex.getnum()) != null) {
					shift(300200, n);
				} else if(lex.eqchar('*')) {
					shift(300300);
				} else if(lex.eqchar('?')) {
					shift(300400);
				} else {
					throw ErrorBundle.getDefault(10019,
							lex.get().toString());
				}
				break;
			case 116400:
				/* E -> E between E and E*
				 *    | E *+ E
				 *    | E *- E
				 *    | E *[*] E
				 *    | E *[/] E
				 *    | E *|| E
				 *    | E *relop E
				 *    | E *and E
				 *    | E *or E
				 *    | E *in ( L )
				 *    | E *between E and E
				 *    | E *like E
				 */
				if(lex.eqchar('+')) {
					shift(105200);
				} else if(lex.eqchar('-')) {
					shift(106200);
				} else if(lex.eqchar('*')) {
					shift(107200);
				} else if(lex.eqchar('/')) {
					shift(108200);
				} else if(lex.eq(DbSqlReserved.CONCAT)) {
					shift(109200);
				} else if((r = lex.getrelop()) != null) {
					shift(110200, r);
				} else if(lex.eqsym("IN")) {
					shift(114200);
				} else if(lex.eqsym("BETWEEN")) {
					shift(116100);
				} else if(lex.eqsym("IS")) {
					shift(118100);
				} else {
					z = (SqlExpression)pop();  pop();
					y = (SqlExpression)pop();  pop();
					x = (SqlExpression)pop();
					x = new SqlTernaryOperation(
							SqlTernaryOperator.BETWEEN, x, y, z);
					setGoto(DbSqlNonterminal.EXPR, x);
				}
				break;
			case 118100:
				/* E -> E is *null
				 *    | E is *not null
				 */
				if(lex.eqsym("NULL")) {
					shift(118200);
				} else if(lex.eqsym("NOT")) {
					shift(118210);
				} else {
					throw ErrorBundle.getDefault(10019,
							lex.get().toString());
				}
				break;
			case 118200:
				/* E -> E is null*
				 */
				pop();  pop();  x = (SqlExpression)pop();
				x = new SqlUnaryOperation(SqlUnaryOperator.IS_NULL, x);
				setGoto(DbSqlNonterminal.EXPR, x);
				break;
			case 118210:
				/* E -> E is not *null
				 */
				if(lex.eqsym("NULL")) {
					shift(118310);
				} else {
					throw ErrorBundle.getDefault(10019,
							lex.get().toString());
				}
				break;
			case 118310:
				/* E -> E is not null*
				 */
				pop();  pop();  pop();  x = (SqlExpression)pop();
				x = new SqlUnaryOperation(SqlUnaryOperator.IS_NOT_NULL,
						x);
				setGoto(DbSqlNonterminal.EXPR, x);
				break;
			case 119100:
				/* E -> exists *( select-clause )
				 */
				if(lex.eqchar('(')) {
					shift(119200);
				} else {
					throw ErrorBundle.getDefault(10019,
							lex.get().toString());
				}
				break;
			case 119200:
				/* E -> exists ( *select-clause )
				 */
				lex.eatsym("SELECT");
				shift(119300, new DbSqlParser()._select(lex, m));
				break;
			case 119300:
				/* E -> exists ( select-clause *)
				 */
				if(lex.eqchar(')')) {
					shift(119400);
				} else {
					throw ErrorBundle.getDefault(10019,
							lex.get().toString());
				}
				break;
			case 119400:
				/* E -> exists ( select-clause )*
				 */
				pop();  o = pop();  pop();  pop();
				setGoto(DbSqlNonterminal.EXPR,
						new SqlExists((SqlSelect)o));
				break;
			case 200200:
				/* L -> L , *E
				 * E -> *( E )
				 *    | *+ E
				 *    | *- E
				 *    | *not E
				 *    | *E + E
				 *    | *E - E
				 *    | *E [*] E
				 *    | *E [/] E
				 *    | *E || E
				 *    | *E relop E
				 *    | *E and E
				 *    | *E or E
				 *    | *E in ( L )
				 *    | *id ( L )
				 *    | *id ( )
				 *    | *E between E and E
				 *    | [*exist ( <select-clause> )]
				 *    | *E like E
				 *    | *is-null
				 *    | *V
				 * V -> *string
				 *    | *number
				 *    | *id
				 */
				if(gt == DbSqlNonterminal.EXPR) {
					goTo(200300);
				} else if(lex.eqchar('(')) {
					shift(101100);
				} else if(lex.eqchar('+')) {
					shift(102100);
				} else if(lex.eqchar('-')) {
					shift(103100);
				} else if(lex.eqsym("NOT")) {
					shift(104100);
				} else if(lex.eqsym("EXISTS")) {
					shift(119100);
				} else if((s = lex.getsym()) != null) {
					shift(115100, s);
				} else if((s = lex.getstr()) != null) {
					shift(300100, s);
				} else if((n = lex.getnum()) != null) {
					shift(300200, n);
				} else if(lex.eqchar('*')) {
					shift(300300);
				} else if(lex.eqchar('?')) {
					shift(300400);
				} else {
					throw ErrorBundle.getDefault(10019,
							lex.get().toString());
				}
				break;
			case 200300:
				/* L -> L , E*
				 * E -> E *+ E
				 *    | E *- E
				 *    | E *[*] E
				 *    | E *[/] E
				 *    | E *|| E
				 *    | E *relop E
				 *    | E *and E
				 *    | E *or E
				 *    | E *in ( L )
				 *    | E *between E and E
				 *    | E *like E
				 */
				if(lex.eqchar('+')) {
					shift(105200);
				} else if(lex.eqchar('-')) {
					shift(106200);
				} else if(lex.eqchar('*')) {
					shift(107200);
				} else if(lex.eqchar('/')) {
					shift(108200);
				} else if(lex.eq(DbSqlReserved.CONCAT)) {
					shift(109200);
				} else if((r = lex.getrelop()) != null) {
					shift(110200, r);
				} else if(lex.eqsym("AND")) {
					shift(111200);
				} else if(lex.eqsym("OR")) {
					shift(112200);
				} else if(lex.eqsym("IN")) {
					shift(114200);
				} else if(lex.eqsym("BETWEEN")) {
					shift(116100);
				} else if(lex.eqsym("IS")) {
					shift(118100);
				} else {
					x = (SqlExpression)pop();  pop();
					l = (List<SqlExpression>)pop();
					l.add(x);
					setGoto(DbSqlNonterminal.EXPR_LIST, l);
				}
				break;
			case 201100:
				/* L -> E*
				 * E -> E *+ E
				 *    | E *- E
				 *    | E *[*] E
				 *    | E *[/] E
				 *    | E *|| E
				 *    | E *relop E
				 *    | E *and E
				 *    | E *or E
				 *    | E *in ( L )
				 *    | E *between E and E
				 *    | E *like E
				 */
				if(lex.eqchar('+')) {
					shift(105200);
				} else if(lex.eqchar('-')) {
					shift(106200);
				} else if(lex.eqchar('*')) {
					shift(107200);
				} else if(lex.eqchar('/')) {
					shift(108200);
				} else if(lex.eq(DbSqlReserved.CONCAT)) {
					shift(109200);
				} else if((r = lex.getrelop()) != null) {
					shift(110200, r);
				} else if(lex.eqsym("AND")) {
					shift(111200);
				} else if(lex.eqsym("OR")) {
					shift(112200);
				} else if(lex.eqsym("IN")) {
					shift(114200);
				} else if(lex.eqsym("BETWEEN")) {
					shift(116100);
				} else if(lex.eqsym("IS")) {
					shift(118100);
				} else {
					x = (SqlExpression)pop();
					l = new ArrayList<SqlExpression>();
					l.add(x);
					setGoto(DbSqlNonterminal.EXPR_LIST, l);
				}
				break;
			case 300100:
				/* V -> string*
				 */
				setGoto(DbSqlNonterminal.EXPR,
						new SqlString((String)pop()));
				break;
			case 300200:
				/* V -> number*
				 */
				setGoto(DbSqlNonterminal.EXPR,
						new SqlNumeric((Rational)pop()));
				break;
			case 300300:
				/* V -> [*]*
				 */
				pop();
				setGoto(DbSqlNonterminal.EXPR, new SqlSymbol("*"));
				break;
			case 300400:
				/* V -> [*]*
				 */
				pop();
				setGoto(DbSqlNonterminal.EXPR,
						new SqlPlaceHolder(lex.nextPlaceNumber()));
				break;
			}
		}
	}
}
