/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.db.engine;

import java.io.IOException;
import java.math.BigDecimal;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.SortedSet;
import java.util.TreeSet;

import net.morilib.db.misc.ErrorBundle;
import net.morilib.db.misc.SqlResponse;
import net.morilib.db.relations.DefaultRelationTuple;
import net.morilib.db.relations.IsolatedRelation;
import net.morilib.db.relations.NamedRelation;
import net.morilib.db.relations.OperatedRelation;
import net.morilib.db.relations.Relation;
import net.morilib.db.relations.RelationAggregate;
import net.morilib.db.relations.RelationCursor;
import net.morilib.db.relations.RelationInSubquery;
import net.morilib.db.relations.RelationTuple;
import net.morilib.db.relations.SingleTableRelation;
import net.morilib.db.relations.TableRelation;
import net.morilib.db.relations.TableRenameRelation;
import net.morilib.db.relations.RelationBinaryOperate;
import net.morilib.db.relations.RelationBinaryOperator;
import net.morilib.db.relations.RelationConst;
import net.morilib.db.relations.RelationExists;
import net.morilib.db.relations.RelationFunction;
import net.morilib.db.relations.RelationFunctionApply;
import net.morilib.db.relations.RelationIn;
import net.morilib.db.relations.RelationExpression;
import net.morilib.db.relations.RelationRefer;
import net.morilib.db.relations.RelationReferSubquery;
import net.morilib.db.relations.RelationTernaryOperate;
import net.morilib.db.relations.RelationTernaryOperator;
import net.morilib.db.relations.RelationUnaryOperate;
import net.morilib.db.relations.RelationUnaryOperator;
import net.morilib.db.relations.Relations;
import net.morilib.db.relations.UnionAllRelation;
import net.morilib.db.relations.VirtualCrossJoinRelation;
import net.morilib.db.schema.SqlSchema;
import net.morilib.db.sqlcs.ddl.SqlAlterTableAdd;
import net.morilib.db.sqlcs.ddl.SqlAlterTableDrop;
import net.morilib.db.sqlcs.ddl.SqlAlterTableModify;
import net.morilib.db.sqlcs.ddl.SqlAlterTableRenameColumn;
import net.morilib.db.sqlcs.ddl.SqlColumnDefinition;
import net.morilib.db.sqlcs.ddl.SqlCreateTable;
import net.morilib.db.sqlcs.ddl.SqlDropTable;
import net.morilib.db.sqlcs.ddl.SqlTruncateTable;
import net.morilib.db.sqlcs.dml.SqlBinaryOperation;
import net.morilib.db.sqlcs.dml.SqlDelete;
import net.morilib.db.sqlcs.dml.SqlExists;
import net.morilib.db.sqlcs.dml.SqlExpression;
import net.morilib.db.sqlcs.dml.SqlExpressions;
import net.morilib.db.sqlcs.dml.SqlFunction;
import net.morilib.db.sqlcs.dml.SqlIn;
import net.morilib.db.sqlcs.dml.SqlInSubquery;
import net.morilib.db.sqlcs.dml.SqlInsertValues;
import net.morilib.db.sqlcs.dml.SqlJoin;
import net.morilib.db.sqlcs.dml.SqlNumeric;
import net.morilib.db.sqlcs.dml.SqlPlaceHolder;
import net.morilib.db.sqlcs.dml.SqlRelation;
import net.morilib.db.sqlcs.dml.SqlSelect;
import net.morilib.db.sqlcs.dml.SqlSetBinaryOperation;
import net.morilib.db.sqlcs.dml.SqlSetExpression;
import net.morilib.db.sqlcs.dml.SqlString;
import net.morilib.db.sqlcs.dml.SqlSubqueryLiteral;
import net.morilib.db.sqlcs.dml.SqlSubqueryRelation;
import net.morilib.db.sqlcs.dml.SqlSymbol;
import net.morilib.db.sqlcs.dml.SqlTable;
import net.morilib.db.sqlcs.dml.SqlTernaryOperation;
import net.morilib.db.sqlcs.dml.SqlUnaryOperation;
import net.morilib.db.sqlcs.dml.SqlUpdate;

public class DefaultSqlEngine extends SqlEngine {

	public Relation visit(SqlSchema f, SqlRelation t,
			List<Object> h) throws IOException, SQLException {
		SqlSubqueryRelation ts;
		Relation a, b;
		SqlTable tt;
		SqlJoin tj;

		if(t instanceof SqlTable) {
			tt = (SqlTable)t;
			return f.readRelation(tt.getName(), tt.getAs());
		} else if(t instanceof SqlJoin) {
			tj = (SqlJoin)t;
			a  = visit(f, tj.getJoinee(), h);
			b  = visit(f, tj.getJoiner(), h);
			return Relations.join(
					this, f, visit(tj.getOn(), h), a, b, null, null,
					tj.getType(), h);
		} else if(t instanceof SqlSubqueryRelation) {
			ts = (SqlSubqueryRelation)t;
			return new TableRenameRelation(
					visit(f, ts.getSubquery(), Relations.NULLTUPLE, h),
					ts.getAs());
		} else {
			throw new RuntimeException();
		}
	}

	public Relation visit(SqlSchema f, SqlSelect t,
			RelationTuple e,
			List<Object> hl) throws IOException, SQLException {
		List<RelationAggregate> m;
		boolean[] ag = new boolean[1];
		SortedSet<RelationTuple> s;
		Collection<RelationTuple> l;
		Map<String, Object> n;
		Relation r = null, a;
		RelationAggregate p;
		OperatedRelation o;
		RelationCursor c;
		RelationTuple w;
		List<String> h;
		String v;

		if(t.getTables() != null && t.getTables().size() > 0) {
			for(SqlRelation x : t.getTables()) {
				a = visit(f, x, hl);
				r = (r != null) ?
						new VirtualCrossJoinRelation(r, a) : a;
			}

			if(r instanceof SingleTableRelation) {
				r = new IsolatedRelation(
						(TableRelation)r,
						((SingleTableRelation)r).getName());
			} else if(r instanceof TableRenameRelation) {
				r = new IsolatedRelation(
						((TableRenameRelation)r).getRelation(),
						((TableRenameRelation)r).getName());
			}
		} else {
			r = Relations.DUAL;
		}

		// where
		r = r.select(this, f, visit(t.getWhere(), hl), null,
				t.getDistinct(), e, null, hl);
		o = Relations.operate(this, f, r, t.getData(),
				t.getGroupby(), hl, ag);

		// group by
		if((t.getGroupby() != null && t.getGroupby().size() > 0) ||
				ag[0]) {
			for(c = o.iterator(); c.hasNext(); c.next());
			n = new LinkedHashMap<String, Object>();
			l = t.getDistinct().create();
			h = new ArrayList<String>();
			for(int i = 0; i < t.getData().size(); i++) {
				v = t.getData().get(i).getAs();
				v = v != null ? v : i + "";
				h.add(v);
			}

			if(!(m = o.getMap()).get(0).isEmpty()) {
				for(Object z : m.get(0).keySet()) {
					for(int i = 0; i < t.getData().size(); i++) {
						p = m.get(i);
						v = t.getData().get(i).getAs();
						v = v != null ? v : i + "";
						if(p.containsKey(z)) {
							n.put(v, p.get(z).force());
						} else {
							n.put(v, "");
						}
					}
					l.add(new DefaultRelationTuple(n));
				}
			} else if(t.getGroupby() == null ||
					t.getGroupby().size() == 0) {
				for(int i = 0; i < t.getData().size(); i++) {
					v = t.getData().get(i).getAs();
					v = v != null ? v : i + "";
					n.put(v, m.get(i).getInit().force());
				}
				l.add(new DefaultRelationTuple(n));
			}
			r = new TableRelation(h, l);

			// having
			r = r.select(this, f, visit(t.getHaving(), hl), null,
					t.getDistinct(), e, null, hl);
		} else {
			r = o;
		}

		// order by
		if(t.getOrderby() != null && t.getOrderby().size() > 0) {
			s = new TreeSet<RelationTuple>(
					Relations.getComparator(r, t.getOrderby()));
			c = r.iterator();
			while(c.hasNext())  s.add(c.next());
			r = new TableRelation(r.getColumnNames(), s);
		}

		// projection
		l = t.getDistinct().create();
		n = new LinkedHashMap<String, Object>();
		c = r.iterator();
		while(c.hasNext()) {
			w = c.next();
			for(String q : r.getColumnNames()) {
				n.put(q, w.get(q));
			}
			l.add(new DefaultRelationTuple(n));
		}
		r = new TableRelation(r.getColumnNames(), l);
		return r;
	}

	public RelationExpression visit(SqlExpression e,
			List<Object> h) throws SQLException {
		List<RelationExpression> l = new ArrayList<RelationExpression>();
		SqlTernaryOperation t;
		SqlBinaryOperation b;
		SqlUnaryOperation u;
		SqlInSubquery q;
		SqlIn n;
		int j;

		if(e == null) {
			return new RelationConst(BigDecimal.ONE);
		} else if(e instanceof SqlNumeric) {
			return new RelationConst(((SqlNumeric)e).getValue());
		} else if(e instanceof SqlString) {
			return new RelationConst(((SqlString)e).getValue());
		} else if(e instanceof SqlSymbol) {
			return new RelationRefer(((SqlSymbol)e).getValue());
		} else if(e instanceof SqlSubqueryLiteral) {
			return new RelationReferSubquery(
					((SqlSubqueryLiteral)e).getQuery());
		} else if(e instanceof SqlPlaceHolder) {
			j = ((SqlPlaceHolder)e).getNumber() - 1;
			if(j < 0 || j > h.size()) {
				throw ErrorBundle.getDefault(10036);
			}
			return new RelationConst(h.get(j));
		} else if(e instanceof SqlBinaryOperation) {
			b = (SqlBinaryOperation)e;
			return new RelationBinaryOperate(
					RelationBinaryOperator.get(b.getOperator()),
					visit(b.getOperand1(), h),
					visit(b.getOperand2(), h));
		} else if(e instanceof SqlUnaryOperation) {
			u = (SqlUnaryOperation)e;
			return new RelationUnaryOperate(
					RelationUnaryOperator.get(u.getOperator()),
					visit(u.getOperand1(), h));
		} else if(e instanceof SqlTernaryOperation) {
			t = (SqlTernaryOperation)e;
			return new RelationTernaryOperate(
					RelationTernaryOperator.get(t.getOperator()),
					visit(t.getOperand1(), h),
					visit(t.getOperand2(), h),
					visit(t.getOperand3(), h));
		} else if(e instanceof SqlIn) {
			n = (SqlIn)e;
			for(SqlExpression p : n.getValues())  l.add(visit(p, h));
			return new RelationIn(visit(n.getExpression(), h), l);
		} else if(e instanceof SqlInSubquery) {
			q = (SqlInSubquery)e;
			return new RelationInSubquery(
					visit(q.getExpression(), h),
					q.getSelect());
		} else if(e instanceof SqlExists) {
			return new RelationExists(((SqlExists)e).getQuery());
		} else if(e instanceof SqlFunction) {
			for(SqlExpression p : ((SqlFunction)e).getArguments()) {
				l.add(visit(p, h));
			}
			return new RelationFunctionApply(
					RelationFunction.get(((SqlFunction)e).getName()),
					l);
		} else if(e == SqlExpressions.ANY) {
			return Relations.ANY;
		} else {
			throw new RuntimeException();
		}
	}

	private boolean iskey(RelationTuple t, Map<String, Object> is,
			List<String> ky) throws SQLException {
		for(String s : ky) {
			if(!t.get(s).equals(is.get(s))) {
				return false;
			}
		}
		return true;
	}

	@Override
	public synchronized int visit(SqlSchema f, SqlInsertValues t,
			List<Object> h) throws IOException, SQLException {
		Map<String, Object> m = new LinkedHashMap<String, Object>();
		List<RelationTuple> z = new ArrayList<RelationTuple>();
		NamedRelation r = f.readRelation(t.getName(), null);
		SqlCreateTable c = f.getCreateTable(t.getName());
		RelationCursor j = r.iterator();
		RelationExpression x;
		RelationTuple p;
		List<String> l;
		String s;

		if((l = t.getColumns()) == null)  l = r.getColumnNames();
		for(int i = 0; i < t.getExprs().size(); i++) {
			x = visit((SqlExpression)t.getExprs().get(i), h);
			m.put(l.get(i),
					x.eval(this, f, Relations.NULLTUPLE, null, null, h));
		}

		for(int i = 0; i < c.getColumnDefinitions().size(); i++) {
			s = c.getColumnDefinitions().get(i).getName();
			if(!m.containsKey(s))  m.put(s, "");
		}

		while(j.hasNext()) {
			p = j.next();
			if(iskey(p, m, c.getKeys())) {
				throw ErrorBundle.getDefault(10001);
			}
			z.add(p);
		}
		z.add(new DefaultRelationTuple(m));
		f.writeRelation(t.getName(), z);
		return 1;
	}

	/* (non-Javadoc)
	 * @see net.morilib.db.engine.SqlEngine#visit(net.morilib.db.engine.SqlFileSystem, net.morilib.db.sqlcs.dml.SqlUpdate)
	 */
	@Override
	public int visit(SqlSchema f, SqlUpdate t,
			List<Object> h) throws IOException, SQLException {
		List<RelationTuple> z = new ArrayList<RelationTuple>();
		NamedRelation r = f.readRelation(t.getTable(), null);
		RelationCursor j = r.iterator();
		RelationExpression x = null;
		List<RelationExpression> y;
		Map<String, Object> m;
		RelationTuple p;
		int n = 0;

		if(t.getWhere() != null)  x = visit(t.getWhere(), h);
		y = new ArrayList<RelationExpression>();
		for(SqlExpression b : t.getExpressions())  y.add(visit(b, h));
		while(j.hasNext()) {
			p = j.next();
			if(x != null &&
					!x.test(this, f, p, null, null, h).isTrue()) {
				z.add(p);
			} else {
				m = new LinkedHashMap<String, Object>(p.toMap());
				for(int i = 0; i < y.size(); i++) {
					m.put(t.getNames().get(i),
							y.get(i).eval(this, f, p, null, null, h));
				}
				z.add(new DefaultRelationTuple(m));
				n++;
			}
		}
		f.writeRelation(t.getTable(), z);
		return n;
	}

	@Override
	public int visit(SqlSchema f, SqlDelete t,
			List<Object> h) throws IOException, SQLException {
		List<RelationTuple> z = new ArrayList<RelationTuple>();
		NamedRelation r = f.readRelation(t.getName(), null);
		RelationCursor j = r.iterator();
		RelationExpression x = null;
		RelationTuple p;
		int n = 0;

		if(t.getExpression() != null)  x = visit(t.getExpression(), h);
		while(j.hasNext()) {
			p = j.next();
			if(x != null &&
					!x.test(this, f, p, null, null, h).isTrue()) {
				z.add(p);
			} else {
				n++;
			}
		}
		f.writeRelation(t.getName(), z);
		return n;
	}

	Relation _union(SqlSchema f,
			Relation a, Relation b) throws IOException, SQLException {
		Collection<RelationTuple> l;
		Map<String, Object> m, n;
		RelationCursor c;
		List<String> p;
		int i;

		p = a.getColumnNames();
		if(p.size() != b.getColumnNames().size()) {
			throw ErrorBundle.getDefault(10002);
		}

		l = new LinkedHashSet<RelationTuple>();
		c = a.iterator();
		while(c.hasNext()) {
			l.add(new DefaultRelationTuple(c.next().toMap()));
		}

		c = b.iterator();
		n = new LinkedHashMap<String, Object>();
		while(c.hasNext()) {
			i = 0;
			m = c.next().toMap();
			for(String s : m.keySet())  n.put(p.get(i++), m.get(s));
			l.add(new DefaultRelationTuple(n));
		}
		return new TableRelation(p, l);
	}

	Relation _intersect(SqlSchema f,
			Relation a, Relation b) throws IOException, SQLException {
		Collection<RelationTuple> l, q;
		Map<String, Object> m, n;
		RelationCursor c;
		List<String> p;
		int i;

		p = a.getColumnNames();
		if(p.size() != b.getColumnNames().size()) {
			throw ErrorBundle.getDefault(10002);
		}

		l = new LinkedHashSet<RelationTuple>();
		c = a.iterator();
		while(c.hasNext()) {
			l.add(new DefaultRelationTuple(c.next().toMap()));
		}

		q = new LinkedHashSet<RelationTuple>();
		c = b.iterator();
		n = new LinkedHashMap<String, Object>();
		while(c.hasNext()) {
			i = 0;
			m = c.next().toMap();
			for(String s : m.keySet())  n.put(p.get(i++), m.get(s));
			q.add(new DefaultRelationTuple(n));
		}

		l.retainAll(q);
		return new TableRelation(p, l);
	}

	Relation _except(SqlSchema f,
			Relation a, Relation b) throws IOException, SQLException {
		Collection<RelationTuple> l;
		Map<String, Object> m, n;
		RelationCursor c;
		RelationTuple t;
		List<String> p;
		int i;

		p = a.getColumnNames();
		if(p.size() != b.getColumnNames().size()) {
			throw ErrorBundle.getDefault(10002);
		}

		l = new LinkedHashSet<RelationTuple>();
		c = a.iterator();
		while(c.hasNext()) {
			l.add(new DefaultRelationTuple(c.next().toMap()));
		}

		c = b.iterator();
		n = new LinkedHashMap<String, Object>();
		while(c.hasNext()) {
			i = 0;
			m = c.next().toMap();
			for(String s : m.keySet())  n.put(p.get(i++), m.get(s));
			t = new DefaultRelationTuple(n);
			if(l.contains(t))  l.remove(t);
		}
		return new TableRelation(p, l);
	}

	private boolean _in(RelationTuple t, RelationTuple v,
			Relation b) throws SQLException {
		for(String s : b.getColumnNames()) {
			if(!t.get(s).equals(v.get(s)))  return false;
		}
		return true;
	}

	Relation _divide(SqlSchema f,
			Relation a, Relation b) throws IOException, SQLException {
		Map<String, Object> n = new LinkedHashMap<String, Object>();
		Collection<RelationTuple> l = null, m;
		RelationCursor c, j;
		RelationTuple t, v;
		List<String> p;

		p = new ArrayList<String>(a.getColumnNames());
		p.removeAll(b.getColumnNames());
		c = b.iterator();
		for(; c.hasNext(); ) {
			t = c.next();
			j = a.iterator();
			m = new LinkedHashSet<RelationTuple>();
			while(j.hasNext()) {
				v = j.next();
				if(_in(v, t, b)) {
					for(String s : p)  n.put(s, v.get(s));
					m.add(new DefaultRelationTuple(n));
				}
			}

			if(l == null) {
				l = m;
			} else {
				l.retainAll(m);
			}
		}

		if(l == null) {
			throw ErrorBundle.getDefault(10003);
		}
		return new TableRelation(p, l);
	}

	@Override
	public Relation visit(SqlSchema f, SqlSetExpression s,
			List<Object> h) throws IOException, SQLException {
		SqlSetBinaryOperation b;
		Relation x, y;

		if(s instanceof SqlSetBinaryOperation) {
			b = (SqlSetBinaryOperation)s;
			x = visit(f, b.getExpression1(), h);
			y = visit(f, b.getExpression2(), h);
			switch(b.getOpeator()) {
			case UNION:      return _union(f, x, y);
			case UNION_ALL:  return new UnionAllRelation(x, y);
			case INTERSECT:  return _intersect(f, x, y);
			case MINUS:      return _except(f, x, y);
			case DIVIDE:     return _divide(f, x, y);
			}
		} else if(s instanceof SqlSelect) {
			return visit(f, (SqlSelect)s, Relations.NULLTUPLE, h);
		} else {
			throw new NullPointerException();
		}
		return null;
	}

	@Override
	public Object visit(SqlSchema f,
			SqlCreateTable c) throws IOException, SQLException {
		f.putCreateTable(c.getName(), c);
		return new SqlResponse(true, "Table created");
	}

	@Override
	public Object visit(SqlSchema f,
			SqlDropTable c) throws IOException, SQLException {
		f.removeCreateTable(c.getName());
		return new SqlResponse(true, "Table dropped");
	}

	@Override
	public Object visit(SqlSchema f,
			SqlTruncateTable c) throws IOException, SQLException {
		f.truncateTable(c.getName());
		return new SqlResponse(true, "Table truncated");
	}

	@Override
	public Object visit(SqlSchema f,
			SqlAlterTableAdd c) throws IOException, SQLException {
		SqlCreateTable t = f.getCreateTable(c.getName());
		List<SqlColumnDefinition> l;
		Map<String, Object> m;
		List<RelationTuple> z;
		RelationCursor s;
		RelationTuple v;
		Relation r;

		l = t.getColumnDefinitions();
		l = new ArrayList<SqlColumnDefinition>(l);
		for(SqlColumnDefinition x : c.getColumnDefinitions()) {
			if(t.findColumn(x.getName()) != null) {
				throw ErrorBundle.getDefault(10005);
			}
			l.add(x);
		}

		z = new ArrayList<RelationTuple>();
		r = f.readRelation(c.getName(), null);
		s = r.iterator();
		m = new LinkedHashMap<String, Object>();
		while(s.hasNext()) {
			v = s.next();
			for(SqlColumnDefinition x : t.getColumnDefinitions()) {
				m.put(x.getName(), v.get(x.getName()));
			}

			for(SqlColumnDefinition x : c.getColumnDefinitions()) {
				m.put(x.getName(), "");
			}
			z.add(new DefaultRelationTuple(m));
		}

		t = new SqlCreateTable(c.getName(), l);
		f.alterCreateTable(c.getName(), t);
		f.writeRelation(c.getName(), z);
		return new SqlResponse(true, "Table columns added");
	}

	@Override
	public Object visit(SqlSchema f,
			SqlAlterTableModify c) throws IOException, SQLException {
		SqlCreateTable t = f.getCreateTable(c.getName());
		List<SqlColumnDefinition> l, p;
		SqlColumnDefinition y;
		List<RelationTuple> z;
		Map<String, Object> m;
		RelationCursor s;
		RelationTuple v;
		Relation r;

		l = p = t.getColumnDefinitions();
		l = new ArrayList<SqlColumnDefinition>(l);
		for(SqlColumnDefinition x : c.getColumnDefinitions()) {
			if((y = t.findColumn(x.getName())) == null) {
				throw ErrorBundle.getDefault(10006);
			}
			l.set(l.indexOf(y), x);
		}

		z = new ArrayList<RelationTuple>();
		r = f.readRelation(c.getName(), null);
		s = r.iterator();
		m = new LinkedHashMap<String, Object>();
		while(s.hasNext()) {
			v = s.next();
			for(int i = 0; i < l.size(); i++) {
				m.put(l.get(i).getName(), v.get(p.get(i).getName()));
			}
			z.add(new DefaultRelationTuple(m));
		}

		t = new SqlCreateTable(c.getName(), l);
		f.alterCreateTable(c.getName(), t);
		f.writeRelation(c.getName(), z);
		return new SqlResponse(true, "Table columns modified");
	}

	@Override
	public Object visit(SqlSchema f,
			SqlAlterTableDrop c) throws IOException, SQLException {
		SqlCreateTable t = f.getCreateTable(c.getName());
		List<SqlColumnDefinition> l;
		SqlColumnDefinition y;
		List<RelationTuple> z;
		Map<String, Object> m;
		RelationCursor s;
		RelationTuple v;
		Relation r;

		l = t.getColumnDefinitions();
		l = new ArrayList<SqlColumnDefinition>(l);
		for(String x : c.getColumnNames()) {
			if((y = t.findColumn(x)) == null) {
				throw ErrorBundle.getDefault(10006);
			}
			l.remove(y);
		}

		z = new ArrayList<RelationTuple>();
		r = f.readRelation(c.getName(), null);
		s = r.iterator();
		m = new LinkedHashMap<String, Object>();
		while(s.hasNext()) {
			v = s.next();
			for(int i = 0; i < l.size(); i++) {
				m.put(l.get(i).getName(), v.get(l.get(i).getName()));
			}
			z.add(new DefaultRelationTuple(m));
		}

		t = new SqlCreateTable(c.getName(), l);
		f.alterCreateTable(c.getName(), t);
		f.writeRelation(c.getName(), z);
		return new SqlResponse(true, "Table columns dropped");
	}

	@Override
	public Object visit(SqlSchema f,
			SqlAlterTableRenameColumn c
			) throws IOException, SQLException {
		SqlCreateTable t = f.getCreateTable(c.getTableName());
		List<SqlColumnDefinition> l, p;
		SqlColumnDefinition y, d;
		List<RelationTuple> z;
		Map<String, Object> m;
		RelationCursor s;
		RelationTuple v;
		Relation r;

		l = p = t.getColumnDefinitions();
		l = new ArrayList<SqlColumnDefinition>(l);
		if((y = t.findColumn(c.getColumnNameFrom())) == null) {
			throw ErrorBundle.getDefault(10006);
		}
		d = new SqlColumnDefinition(c.getColumnNameTo(), y.getType(),
				y.getAttributes());
		l.set(l.indexOf(y), d);

		z = new ArrayList<RelationTuple>();
		r = f.readRelation(c.getTableName(), null);
		s = r.iterator();
		m = new LinkedHashMap<String, Object>();
		while(s.hasNext()) {
			v = s.next();
			for(int i = 0; i < l.size(); i++) {
				m.put(l.get(i).getName(), v.get(p.get(i).getName()));
			}
			z.add(new DefaultRelationTuple(m));
		}

		t = new SqlCreateTable(c.getTableName(), l);
		f.alterCreateTable(c.getTableName(), t);
		f.writeRelation(c.getTableName(), z);
		return new SqlResponse(true, "Table column renamed");
	}

}
